/*****************************************************************************/
/*                                                                           */
/*                                   USER.CC                                 */
/*                                                                           */
/* (C) 1993,94  Michael Peschel                                              */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSPROCESS


#include <os2.h>
#include <stdio.h>

#include "check.h"
#include "streamid.h"
#include "user.h"


// additional commands, not implemented in bsedev.h
#define ASYNC_SETEXTBAUDRATE    0x0043
#define ASYNC_SETENHPARAMETER   0x0054
#define ASYNC_GETEXTBAUDRATE    0x0063
#define ASYNC_GETENHPARAMETER   0x0074



// extended mode structures

// for setting baudrates > 57500 baud
typedef struct _EXTBAUDRATE {
        ULONG   BAUDRATE;
        BYTE    FRACTION;
} EXTBAUDRATE;
typedef EXTBAUDRATE *PEXTBAUDRATE;



// checking the actual baudrate and the limits
typedef struct _QEXTBAUDRATE {
        ULONG   BAUDRATE;
        BYTE    FRACTBAUD;
        ULONG   MINBAUDRATE;
        BYTE    MINFRACTBAUD;
        ULONG   MAXBAUDRATE;
        BYTE    MAXFRACTBAUD;
} QEXTBAUDRATE;
typedef QEXTBAUDRATE *PQEXTBAUDRATE;



// checking the enhanced parameters
typedef struct _ENHPARAMETER {
        BYTE    EnhFlags;
        ULONG   Reserverd;
} ENHPARAMETER;
typedef ENHPARAMETER *PENHPARAMETER;



// enhanced parameter masks, ASYNC_GETENHPARAMETER
#define ENHANCEDMODE_SUPPORTED  0x01
#define ENHANCEDMODE_ENABLE     0x02
#define DMA_RX_ENABLE           0x04
#define DMA_RX_DEDICATE         0x08
#define DMA_TX_ENABLE           0x10
#define DMA_TX_DEDICATE         0x20
#define DMA_RX_MODE             0x40
#define DMA_TX_MODE             0x80



// internal used data
struct _ComData {

    // Port-Handle
    HFILE   hf;

    // Info
    u16     RXBufSize;      // set when port opened
    u16     TXBufSize;      //  -- " --
    u16     RXCount;        // characters in receive queue
    u16     TXCount;        // characters in transmit queue
    DCBINFO dcb;            // a copy of the actual dcb

    // returncode of the last IOCtl-command
    u16     APICode;
};





/******************************************************************************/
/*                           Funktionen                                       */
/******************************************************************************/


ComPort::ComPort () :
    PortName (),
    ComData (NULL)
{
};


ComPort::ComPort (const  String &aPortName,
                  u16    aRXBufSize,
                  u16    aTXBufSize,
                  u32    aBaudrate,
                  char   aDataBits,           /* 5..8 */
                  char   aParity,             /* <N>one, <O>dd, <E>ven, <S>pace, <M>ark */
                  char   aStopBits,           /* 1, 2 */
                  char   aConnection,         /* <D>irect, <M>odem */
                  char   aXonXoff             /* <E>nabled, <D>isabled */
                ):
    PortName   (aPortName),
    Baudrate   (aBaudrate),
    DataBits   (aDataBits),
    Parity     (aParity),
    StopBits   (aStopBits),
    Connection (aConnection),
    XonXoff    (aXonXoff)
{
    // reset error counters
    memset (ErrorCounter, 0, sizeof (ErrorCounter));

    // reserve memory for internal data structure
    ComData = new _ComData;

    // setting up ComData
    ComData->hf        = -1;                // Port not open
    ComData->RXBufSize = aRXBufSize;
    ComData->TXBufSize = aTXBufSize;
    ComData->RXCount   = 0;
    ComData->TXCount   = 0;
};



ComPort::ComPort (StreamableInit) :
    PortName (Empty)
{
    // reserve memory for internal data structure
    ComData = new _ComData;
};



ComPort::~ComPort ()
{
    if (ComIsOpen()) {
        ComClose();
    }
    delete ComData;
};



void ComPort::ComError (u32 ErrorCode)
/* called for API Errors */
{
    FAIL (FormatStr ("ComPort Error# %d, %s", ErrorCode, PortName.GetStr()).GetStr ());
}



u16 ComPort::ComOpen ()
/* opens the port, returns 0 if success */
{
    APIRET      rc;
    HFILE       hf;
    ULONG       ulAction;
    DCBINFO     DCBInfo;
    EXTBAUDRATE Baud;
    LINECONTROL LineCtrl;
    RXQUEUE     Queue;


    // Open the port
    rc = DosOpen ((PSZ) PortName.GetStr (), &hf, &ulAction,
                  0, FILE_NORMAL, FILE_OPEN,
                  OPEN_ACCESS_READWRITE | OPEN_SHARE_DENYNONE, (PEAOP2) NULL);

    if (rc) {
        return (rc);
    }

    // Port-Handle setzen
    ComData->hf = hf;

    // actual settings
    ASYNC_IOCtl (ASYNC_GETDCBINFO, NULL, 0, &DCBInfo, sizeof (DCBInfo));

    /* setting timeouts to approx. 30 character times (assuming 1 char = 10 bit) */
    DCBInfo.usWriteTimeout = (30 * 100) / (Baudrate / 10);
    DCBInfo.usReadTimeout  = DCBInfo.usWriteTimeout;

    /* Flow Control */
    if (XonXoff =='E') {                      // enable XON/XOFF
        DCBInfo.fbFlowReplace = 0xA3;         // 1010 0011
    } else {                                  // disable XON/XOFF
        DCBInfo.fbFlowReplace = 0xA0;         // 1010 0000
    }
    if (Connection =='D') {                   // direct connection
        DCBInfo.fbFlowReplace &= 0x3F;        // disable RTS handshake
    }

    /* Handshake */
    if (Connection =='M') {                   // using RTS/CTS, DTR/DSR handshake
        DCBInfo.fbCtlHndShake = 0x58;         // 01001000
    } else {
        DCBInfo.fbCtlHndShake = 0x00;         // 00000000
    }

    /* timeouts and buffers */                // normal read/write timeout
    DCBInfo.fbTimeout = 0xD2;                 // 11010010

    /* setting new dcb */
    ASYNC_IOCtl (ASYNC_SETDCBINFO, &DCBInfo, sizeof (DCBInfo), NULL, 0);

    /* holding a copy */
    ComData->dcb = DCBInfo;

    /* set up the baudrate */
    Baud.BAUDRATE = Baudrate;
    Baud.FRACTION = 0;

    ASYNC_IOCtl (ASYNC_SETEXTBAUDRATE, &Baud, sizeof (Baud), NULL, 0);

    /* line control */
    LineCtrl.bDataBits  = DataBits;
    LineCtrl.bStopBits  = StopBits -1;

    switch (Parity) {
        case 'N' : LineCtrl.bParity = 0; break;
        case 'O' : LineCtrl.bParity = 1; break;
        case 'E' : LineCtrl.bParity = 2; break;
        case 'M' : LineCtrl.bParity = 3; break;
        case 'S' : LineCtrl.bParity = 4; break;
        default  : LineCtrl.bParity = 0;
    }
    ASYNC_IOCtl (ASYNC_SETLINECTRL, &LineCtrl, sizeof (LineCtrl), NULL, 0);

    /* get buffer sizes */
    ASYNC_IOCtl (ASYNC_GETINQUECOUNT, NULL, 0, &Queue, sizeof (Queue));
    ComData->RXBufSize = Queue.cb;

    ASYNC_IOCtl (ASYNC_GETOUTQUECOUNT, NULL, 0, &Queue, sizeof (Queue));
    ComData->TXBufSize = Queue.cb;

    return (0);
}



void ComPort::ComClose ()
/* closes the port */
{
    /* allow changing RTS. RTS is set to OFF when closing the port */
    ComData->dcb.fbFlowReplace &= 0x3F;
    ComData->dcb.fbFlowReplace |= 0x40;
    ASYNC_IOCtl (ASYNC_SETDCBINFO, &ComData->dcb, sizeof (ComData->dcb), NULL, 0);

    /* closing */
    DosClose (ComData->hf);

    /* reset handle */
    ComData->hf = -1;
}



u16 ComPort::ComIsOpen ()
/* returns 1 if the port is open, else 0. */
{
    return (ComData->hf != -1) ? 1 : 0;
}


/* setting timeout for read and write operations */
void ComPort::SetWriteTimeout (double timeout)
{
    if (timeout < 0.01) {
        timeout = 0.01;
    }
    ComData->dcb.usWriteTimeout = (timeout * 100) - 1;
    ASYNC_IOCtl (ASYNC_SETDCBINFO, &ComData->dcb, sizeof (ComData->dcb), NULL, 0);
}



void ComPort::SetReadTimeout (double timeout)
{
    if (timeout < 0.01) {
        timeout = 0.01;
    }
    ComData->dcb.usReadTimeout = (timeout * 100) - 1;
    ASYNC_IOCtl (ASYNC_SETDCBINFO, &ComData->dcb, sizeof (ComData->dcb), NULL, 0);
}



/* actual timeout settings in seconds */
double ComPort::GetWriteTimeout ()
{
    return (ComData->dcb.usWriteTimeout + 1) / 100;
}



double ComPort::GetReadTimeout ()
{
    return (ComData->dcb.usReadTimeout + 1) / 100;
}



void ComPort::ComDTROn ()
/* activates DTR */
{
    MODEMSTATUS ModemState;
    USHORT      ErrorWord;

    /* setting bitmasks */
    ModemState.fbModemOn  = DTR_ON;
    ModemState.fbModemOff = 0xFF;

    /* calling the driver */
    ASYNC_IOCtl (ASYNC_SETMODEMCTRL,
                 &ModemState, sizeof (ModemState),
                 &ErrorWord, sizeof (ErrorWord));


    /* get and reset errors */
    if (ErrorWord) {
        GetComError ();
    }
}




void ComPort::ComDTROff ()
/* deactivates DTR */
{
    MODEMSTATUS ModemState;
    USHORT      ErrorWord;

    /* serrtting masks */
    ModemState.fbModemOn  = 0;
    ModemState.fbModemOff = DTR_OFF;

    /* calling the driver */
    ASYNC_IOCtl (ASYNC_SETMODEMCTRL,
                 &ModemState, sizeof (ModemState),
                 &ErrorWord, sizeof (ErrorWord));

    /* get and reset errors */
    if (ErrorWord) {
        GetComError ();
    }
}



u16 ComPort::ComRXSize ()
/* returns the size of the receive queue */
{
    return (ComData->RXBufSize);
}



u16 ComPort::ComRXCount ()
/* returns the number of characters in the receive queue */
{
    RXQUEUE RXQueue;

    ASYNC_IOCtl (ASYNC_GETINQUECOUNT, NULL, 0, &RXQueue, sizeof (RXQueue));
    return (RXQueue.cch);
}


void ComPort::ComRXFlush ()
/* flushes the receive queue */
{
    BYTE  ParmList     = 0;
    BYTE  DataList     = 0;

    GENERAL_IOCtl (DEV_FLUSHINPUT,
                   &ParmList, sizeof (ParmList), &DataList, sizeof (DataList));
}



u16 ComPort::ComTXSize ()
/* returns the size of the transmit queue */
{
    return (ComData->TXBufSize);
}


u16 ComPort::ComTXCount ()
/* returns the number of charcters in the transmit queue */
{
    RXQUEUE TXQueue;    // same data format as RXQueue

    ASYNC_IOCtl (ASYNC_GETOUTQUECOUNT, NULL, 0, &TXQueue, sizeof (TXQueue));
    return (TXQueue.cch);
}


u16 ComPort::ComTXFree ()
/* returns the free space in the transmit queue */
{
    return (ComTXSize() - ComTXCount());
}



void ComPort::ComTXFlush ()
/* flushes the transmit queue */
{

    BYTE  ParmList     = 0;
    BYTE  DataList     = 0;

    GENERAL_IOCtl (DEV_FLUSHOUTPUT,
                   &ParmList, sizeof (ParmList), &DataList, sizeof (DataList));
}


i16 ComPort::ComReceive ()
/* reads and returns a character from the receive queue.
 * -1 is returned if the receive queue is empty.
 */
{
    char    B;
    ULONG   chRead;

    // queue is empty, return -1
    if (ComRXCount () == 0) {
        return (-1);
    }

    // read one character
    DosRead (ComData->hf, &B, 1, &chRead);

    // get errors
    GetComError ();

    // return character
    return (B);
}



i16 ComPort::TimedReceive ()
/* waits with timeout for a character. If a character is received within the
 * timeout-interval this character is returned, if not -1 is returned.
 */
{
    char    B;
    ULONG   chRead;

    DosRead (ComData->hf, &B, 1, &chRead);

    // get errors
    GetComError ();

    // retrun character or -1
    return (chRead == 0) ? -1 : B;
}



void ComPort::TimedReceiveBlock (char Buffer[], u32 Count, u32 &ChRead)
/* waits with timeout for Count characters. If the characters are received
 * within the timeout interval ChRead is equal Count. In case of timeout
 * ChRead holds the number of characters received.
 */
{
    DosRead (ComData->hf, Buffer, Count, PULONG (&ChRead));

    // get errors
    GetComError ();
}



void ComPort::ComSend (unsigned char B)
/* sends a character. If the transmit queue ist full the errorcounter
 * TXOverflow is incremented.
 */
{
    ULONG chWrite;

    if (ComTXFree () == 0) {
        ErrorCounter [ceTXOverflow]++;
    } else {
         DosWrite (ComData->hf, &B, 1, &chWrite);
    }
    GetComError ();
}



void ComPort::TimedSend (unsigned char B)
/* waits with timeout to send a character. If timeout occurs the errorcounter
 * TXOverflow is incremented. The character is ignored.
 */
{
    ULONG chWrite;

    DosWrite (ComData->hf, &B, 1, &chWrite);
    if (chWrite != 1) {
        ErrorCounter [ceTXOverflow]++;
    }
    GetComError ();
}



void ComPort::TimedSendBlock  (const char Buffer[], u32 Count, u32 &ChWritten)
/* waits with timeout to send the given block. If timout occurs the errorcounter
 * TXOverflow is incremented. ChWritten holds the number of characters sended.
 */
{
    DosWrite (ComData->hf, (char*) Buffer, Count, PULONG (&ChWritten));
    if (ChWritten != Count) {
        ErrorCounter [ceTXOverflow]++;
    }
    GetComError ();
}




void ComPort::ComBreak (double Duration)
/* sends break with the given duration */
{
    USHORT  ErrorWord;

    // BREAK ON
    ASYNC_IOCtl (ASYNC_SETBREAKON, NULL, 0, &ErrorWord, sizeof (ErrorWord));

    // Waiting
    DosSleep (Duration * 1000);

    // BREAK OFF
    ASYNC_IOCtl (ASYNC_SETBREAKOFF, NULL, 0, &ErrorWord, sizeof (ErrorWord));
}



ComErrorCounter* ComPort::ComErrors ()
/* returns a pointer to the errorcounter array. */
{
    return (&ErrorCounter);
}



u16 ComPort::ComModemStatus ()
/* returns the state of the modem input signals. (see constants) */
{
    BYTE    Inputs;
    USHORT  Events;

    /* reading inputs */
    ASYNC_IOCtl (ASYNC_GETMODEMINPUT, NULL, 0, &Inputs, sizeof (Inputs));

    /* reading events */
    ASYNC_IOCtl (ASYNC_GETCOMMEVENT, NULL, 0, &Events, sizeof (Events));

    /* copy RI bit to bit 6 */
    if (Events & 0x0100) {
        Events |= 0x0040;
    } else {
        Events &= 0xFFBF;
    }

    /* shift to bits 0 - 3 */
    Events >>= 3;

    /* returning the bitmask */
    return ((Events & 0x0F) | (Inputs & 0xF0));
}




// Derived from class Streamable
void ComPort::Load (Stream &S)
{
    S >> Baudrate >> DataBits >> Parity >> StopBits >> Connection >> XonXoff;
    // ##
};



void ComPort::Store (Stream &S) const
{
    S << Baudrate << DataBits << Parity << StopBits << Connection << XonXoff;
};



u16 ComPort::StreamableID () const
{
    return ID_ComPort;
}



Streamable* ComPort::Build ()
{
    return new ComPort (Empty);
};



// private methods

// simple interface for DosDevIOCTL call

// async device control
void ComPort::ASYNC_IOCtl (u32 Function, void *ParmList, u32 ParmLen,
                                         void *DataList, u32 DataLen)
{
    ULONG ParmLenInOut = ParmLen;
    ULONG DataLenInOut = DataLen;

    if ((ComData->APICode =
            DosDevIOCtl (ComData->hf, IOCTL_ASYNC, Function,
                         ParmList, ParmLen, &ParmLenInOut,
                         DataList, DataLen, &DataLenInOut)
        ) != NO_ERROR) {

        ComError (ComData->APICode);
    }
}



// general device control
void ComPort::GENERAL_IOCtl (u32 Function, void *ParmList, u32 ParmLen,
                                           void *DataList, u32 DataLen)
{
    ULONG ParmLenInOut = ParmLen;
    ULONG DataLenInOut = DataLen;

    if ((ComData->APICode =
            DosDevIOCtl (ComData->hf, IOCTL_GENERAL, Function,
                         ParmList, ParmLen, &ParmLenInOut,
                         DataList, DataLen, &DataLenInOut)
        ) != NO_ERROR) {

        ComError (ComData->APICode);
    }
}


// error counter update
void ComPort::UpdateErrorCounter (u16 ErrorWord)
{
    if (ErrorWord & RX_QUE_OVERRUN)       ErrorCounter [ceRXOverflow]++;
    if (ErrorWord & RX_HARDWARE_OVERRUN)  ErrorCounter [ceOverrun]++;
    if (ErrorWord & PARITY_ERROR)         ErrorCounter [ceParity]++;
    if (ErrorWord & FRAMING_ERROR)        ErrorCounter [ceFrame]++;
}



// gets the error state of the port and updates the error counters
void ComPort::GetComError ()
{
    u16     Error;

    /* reading error state */
    ASYNC_IOCtl (ASYNC_GETCOMMERROR, NULL, 0, &Error, sizeof (Error));
    UpdateErrorCounter (Error);
};

