/*****************************************************************************/
/*									     */
/*				   ICDEVS.CC				     */
/*									     */
/* (C) 1995	Ullrich von Bassewitz					     */
/*		Zwehrenbuehlstrasse 33					     */
/*		D-72070 Tuebingen					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include "listbox.h"
#include "progutil.h"
#include "menue.h"
#include "menuitem.h"
#include "statline.h"
#include "strcvt.h"
#include "stdmsg.h"
#include "stdmenue.h"
#include "settings.h"

#include "icmsg.h"
#include "icconfig.h"



/*****************************************************************************/
/*			       Message constants			     */
/*****************************************************************************/



const u16 msKeine			= MSGBASE_ICDEVS +  0;
const u16 msInland			= MSGBASE_ICDEVS +  1;
const u16 msOrt				= MSGBASE_ICDEVS +  2;
const u16 msHalbamt			= MSGBASE_ICDEVS +  3;
const u16 msNichtamt			= MSGBASE_ICDEVS +  4;
const u16 msFernsprechen		= MSGBASE_ICDEVS +  5;
const u16 msFaxG3			= MSGBASE_ICDEVS +  6;
const u16 msDatenModem			= MSGBASE_ICDEVS +  7;
const u16 msDatexJModem			= MSGBASE_ICDEVS +  8;
const u16 msAnrufbeantworter		= MSGBASE_ICDEVS +  9;
const u16 msKombiDienste		= MSGBASE_ICDEVS + 10;
const u16 msOn				= MSGBASE_ICDEVS + 11;
const u16 msOff				= MSGBASE_ICDEVS + 12;
const u16 msNoReroute			= MSGBASE_ICDEVS + 13;
const u16 msExtReroute			= MSGBASE_ICDEVS + 14;
const u16 msIntReroute			= MSGBASE_ICDEVS + 15;
const u16 msDeviceHeader		= MSGBASE_ICDEVS + 16;



/*****************************************************************************/
/*			Explicit template instantiation			     */
/*****************************************************************************/



#ifdef EXPLICIT_TEMPLATES
template class Collection<class DevInfo>;
template class SortedCollection<class DevInfo, unsigned char>;
template class ListBox<class DevInfo>;
#endif



/*****************************************************************************/
/*				 class DevInfo				     */
/*****************************************************************************/



class DevInfo: public Streamable {

public:
    DevConfig	DC;		// Device configuration
    unsigned	Charges;	// Device charges
    unsigned	Index;		// Index into the device array


    DevInfo (const DevConfig& aDC, unsigned aCharges, unsigned aIndex);

};



DevInfo::DevInfo (const DevConfig& aDC, unsigned aCharges, unsigned aIndex):
    DC (aDC),
    Charges (aCharges),
    Index (aIndex)
{
}



/*****************************************************************************/
/*			       class DevInfoColl			     */
/*****************************************************************************/



class DevInfoColl: public SortedCollection<DevInfo, unsigned char> {

protected:
    virtual int Compare (const unsigned char* Key1, const unsigned char* Key2);
    virtual const unsigned char* KeyOf (const DevInfo* Item);

public:
    DevInfoColl ();
    DevInfoColl (StreamableInit);

    int FindDev (unsigned char Num);
    // Return the index of the device with number Num

    virtual DevInfo* At (int Index);
    // Return a pointer to the item at position Index.
    // OVERRIDE FOR DEBUGGING

};



int DevInfoColl::Compare (const unsigned char* Key1, const unsigned char* Key2)
{
    if (*Key1 < *Key2) {
	return -1;
    } else if (*Key1 > *Key2) {
	return 1;
    } else {
	return 0;
    }
}



const unsigned char* DevInfoColl::KeyOf (const DevInfo* Item)
{
    return &Item->DC.DevNum;
}



DevInfoColl::DevInfoColl ():
    SortedCollection <DevInfo, unsigned char> (20, 10, 1)
{
}



DevInfoColl::DevInfoColl (StreamableInit):
    SortedCollection <DevInfo, unsigned char> (Empty)
{
}



int DevInfoColl::FindDev (unsigned char Num)
// Return the index of the device with number Num
{
    int Index;
    return Search (&Num, Index) ? Index : -1;
}



DevInfo* DevInfoColl::At (int Index)
// Return a pointer to the item at position Index.
// OVERRIDE FOR DEBUGGING
{
    // Check range
    if (Index < 0 || Index >= Count) {
	FAIL ("DevInfoColl::At: Index out of bounds");
	return NULL;
    }

    return SortedCollection<DevInfo, unsigned char>::At (Index);
}



/*****************************************************************************/
/*			       class DevListBox				     */
/*****************************************************************************/



class DevListBox: public ListBox<DevInfo> {


private:
    static const String& DialCapsName (unsigned DialCaps);
    // Map the number of a dial capability to a name of fixed length

    static const String& ServiceName (unsigned Service);
    // Map the number of a service to a name of fixed length

    static String RerouteName (unsigned Val, unsigned char* Num);
    // Map the reroute capability to a string with fixed length

    static const String& BoolName (unsigned B);
    // Map a boolean flag to a string with fixed length


protected:
    virtual void Print (int Index, int X, int Y, u16 Attr);
    // Display one of the listbox entries

public:
    DevListBox (i16 aID, const Point& aSize, WindowItem* NextItem = NULL);

};



DevListBox::DevListBox (i16 aID, const Point& aSize, WindowItem* NextItem):
    ListBox <DevInfo> ("", aID, aSize, atEditNormal, atEditBar, atEditHigh, NextItem)
{
}



const String& DevListBox::DialCapsName (unsigned DialCaps)
// Map the number of a dial capability to a name of fixed length
{
    unsigned MsgNum = msKeine;
    switch (DialCaps) {

	case dcKeine:
	    MsgNum = msKeine;
	    break;

	case dcInland:
	    MsgNum = msInland;
	    break;

	case dcOrt:
	    MsgNum = msOrt;
	    break;

	case dcHalbamt:
	    MsgNum = msHalbamt;
	    break;

	case dcNichtamt:
	    MsgNum = msNichtamt;
	    break;
    }
    return LoadAppMsg (MsgNum);
}



const String& DevListBox::ServiceName (unsigned Service)
// Map the number of a service to a name of fixed length
{
    unsigned MsgNum = msFernsprechen;
    switch (Service) {

	case svFernsprechen:
	    MsgNum = msFernsprechen;
	    break;

	case svFaxG3:
	    MsgNum = msFaxG3;
	    break;

	case svDatenModem:
	    MsgNum = msDatenModem;
	    break;

	case svDatexJModem:
	    MsgNum = msDatexJModem;
	    break;

	case svAnrufbeantworter:
	    MsgNum = msAnrufbeantworter;
	    break;

	case svKombi:
	    MsgNum = msKombiDienste;
	    break;

    }
    return LoadAppMsg (MsgNum);
}



String DevListBox::RerouteName (unsigned Val, unsigned char* Num)
// Map the reroute capability to a string with fixed length
{
    const StringLength = 11;
    const PadLength = 12;
    String Res (PadLength);

    if (Val == 0x00) {
	// No reroute
	Res = LoadAppMsg (msNoReroute);
    } else if (Val == 0x01) {
	// External reroute. Bug: Watcom does not accept the sizeof op here,
	// so use absolute values
	char Buf [21];
	Res = FromBCD (Buf, Num, 10);
	if (Res.Len () > StringLength) {
	    // Too long for display
	    Res = LoadAppMsg (msExtReroute);
	}
    } else {
	Res = FormatStr (LoadAppMsg (msIntReroute).GetStr (), Val);
    }

    // Pad to length
    Res.Pad (String::Right, PadLength);

    // Return the result
    return Res;
}



const String& DevListBox::BoolName (unsigned B)
// Map a boolean flag to a string with fixed length
{
    return LoadAppMsg (B ? msOn : msOff);
}



void DevListBox::Print (int Index, int X, int Y, u16 Attr)
{
    // Get the entry
    DevInfo* Info = Coll->At (Index);

    // Create a string with the correct length
    String Line (Size.X);

    // Build the line
    char Buf [100];
    String PIN = FromBCD (Buf, Info->DC.PIN, sizeof (Info->DC.PIN));
    PIN.Pad (String::Left, 4);
    Line = FormatStr (" %2u   ", Info->DC.DevNum + 21);
    Line += DialCapsName (Info->DC.DialCaps);
    Line += ServiceName (Info->DC.Service);
    Line += RerouteName (Info->DC.Reroute, Info->DC.ExtNum);
    Line += BoolName (Info->DC.ChargePulse);
    Line += PIN;
    Line += FormatStr ("   %6u", Info->Charges);

    // Pad the line
    Line.Pad (String::Right, Size.X - 1);
    Line += ' ';

    // Print the name
    Owner->Write (X, Y, Line, Attr);
}



/*****************************************************************************/
/*			       class DevListBox				     */
/*****************************************************************************/



class ChargesBox: public ListBox<DevInfo> {

protected:
    virtual void Print (int Index, int X, int Y, u16 Attr);
    // Display one of the listbox entries

public:
    ChargesBox (i16 aID, const Point& aSize, WindowItem* NextItem = NULL);

};



ChargesBox::ChargesBox (i16 aID, const Point& aSize, WindowItem* NextItem):
    ListBox <DevInfo> ("", aID, aSize, atEditNormal, atEditBar, atEditHigh, NextItem)
{
}



void ChargesBox::Print (int Index, int X, int Y, u16 Attr)
{
    // Get the entry
    DevInfo* Info = Coll->At (Index);

    // Create a string with the correct length
    String Line (Size.X);

    // Build the line
    Line = FormatStr (" %2u   %6u", Info->DC.DevNum + 21, Info->Charges);

    // Pad the line
    Line.Pad (String::Right, Size.X - 1);
    Line += ' ';

    // Print the name
    Owner->Write (X, Y, Line, Attr);
}



/*****************************************************************************/
/*				     Code				     */
/*****************************************************************************/



static void EditDevConfig (DevConfig& Config, unsigned DevCount, int& Changed)
// Edit the given device configuration. Changed is set to 1 if anything has
// been changed, otherwise it is left untouched.
{
    // Settings name for the window position
    static const String StgPosName = "EditDevConfig.ConfigMenue.Position";

    // Menue constants
    const miDialCaps	  = 10;
    const miService	  = 20;
    const miReroute	  = 30;
    const miChargePulse   = 40;
    const miPIN		  = 50;
    const miTerminalMode  = 60;


    // Save the configuration
    DevConfig OldConfig = Config;

    // Load the menue
    Menue* M;
    if (FirmwareVersion < 1.92) {
	M = (Menue*) LoadResource ("@ICDEVS.ConfigMenue-1.90");
    } else if (FirmwareVersion < 1.93) {
	M = (Menue*) LoadResource ("@ICDEVS.ConfigMenue-1.92");
    } else {
	M = (Menue*) LoadResource ("@ICDEVS.ConfigMenue-1.93");
    }

    // If there is a stored window position, move the window to that position
    Point Pos = StgGetPoint (StgPosName, M->OuterBounds ().A);
    M->MoveAbs (Pos);

    // Set the menue header to show the device number
    M->SetHeader (FormatStr (LoadAppMsg (msDeviceHeader).GetStr (), Config.DevNum + 21));

    // Insert the values into the menue
    char Buf [10];
    M->SetToggleValue (miDialCaps, Config.DialCaps);
    M->SetToggleValue (miService, Config.Service);
    M->SetToggleValue (miChargePulse, Config.ChargePulse);
    M->SetStringValue (miPIN, FromBCD (Buf, Config.PIN, sizeof (Config.PIN)));
    if (Config.Reroute == 0x01) {
	// External
	char Buf [100];
	FromBCD (Buf, Config.ExtNum, sizeof (Config.ExtNum));
	M->SetStringValue (miReroute, Buf);
    } else if (Config.Reroute != 0x00) {
	// Internal
	M->SetStringValue (miReroute, U32Str (Config.Reroute));
    }
    if (M->ItemWithID (miTerminalMode) != NULL) {
	M->SetToggleValue (miTerminalMode, Config.TerminalMode);
    }

    // Display a new status line and activate the menue
    PushStatusLine (siAbort | siAccept);
    M->Activate ();

    // Allow editing
    int Done = 0;
    String Num;
    while (!Done) {

	// Get a choice from the user
	int Choice = M->GetChoice ();

	// Look what he wants...
	switch (Choice) {

	    case miDialCaps + dcKeine:
	    case miDialCaps + dcInland:
	    case miDialCaps + dcOrt:
	    case miDialCaps + dcHalbamt:
	    case miDialCaps + dcNichtamt:
		Config.DialCaps = Choice - miDialCaps;
		break;

	    case miService + svFernsprechen:
	    case miService + svFaxG3:
	    case miService + svDatenModem:
	    case miService + svDatexJModem:
	    case miService + svAnrufbeantworter:
	    case miService + svKombi:
		Config.Service = Choice - miService;
		break;

	    case miReroute:
		Num = M->GetStringValue (miReroute);
		if (Num.IsEmpty ()) {
		    Config.Reroute = 0x00;
		} else {
		    // Num does contain nothing than digits, so using atoi is
		    // save. Since the number is also range checked, I'll use
		    // an unsigned here
		    u32 Number = atoi (Num.GetStr ());
		    if (Num.Len () == 2 && Number >= 21 && Number < 21 + DevCount) {
			// Internal
			Config.Reroute = Number;
		    } else {
			// External
			Config.Reroute = 0x01;
			ToBCD (Num.GetStr (), Config.ExtNum, sizeof (Config.ExtNum));
		    }
		}
		break;

	    case miChargePulse+0:
	    case miChargePulse+1:
		Config.ChargePulse = Choice - miChargePulse;
		break;

	    case miPIN:
		ToBCD (M->GetStringValue (miPIN).GetStr (), Config.PIN, sizeof (Config.PIN));
		break;

	    case miTerminalMode:
		Config.TerminalMode = 0;
		break;

	    case miTerminalMode+1:
		Config.TerminalMode = 1;
		break;

	    case 0:
		if (M->GetAbortKey () == vkAbort) {
		    // Abort - ask if we have changes
		    if (Config != OldConfig) {
			// We have changes
			if (AskDiscardChanges () == 2) {
			    // Discard changes
			    Config = OldConfig;
			    Done = 1;
			}
		    } else {
			// No changes
			Done = 1;
		    }
		} else if (M->GetAbortKey () == vkAccept) {
		    // Accept the changes, set the Changed flag
		    if (Config != OldConfig) {
			// We had changes
			Changed = 1;
		    }
		    Done = 1;
		}
		break;

	}

    }

    // Save the current window position
    StgPutPoint (M->OuterBounds ().A, StgPosName);

    // Pop the status line, delete the menue
    PopStatusLine ();
    delete M;

}



void DeviceList (DevConfig* Devices, const IstecCharges& Charges,
		 unsigned Count, int& Changed)
// List all devices and there settings, including the charges. Allow editing
// the settings and charges.
{
    static const String StgPosName = "DeviceList.DeviceListWindow.Position";
    unsigned I;

    // Check the parameters
    PRECONDITION (Count > 0);

    // Load the window
    Menue* Win = (Menue*) LoadResource ("@ICDEVS.DeviceListWindow");

    // If there is a stored window position, move the window to that position
    Point Pos = StgGetPoint (StgPosName, Win->OuterBounds ().A);
    Win->MoveAbs (Pos);

    // Create a device collection and insert all devices
    DevInfoColl Coll;
    for (I = 0; I < Count; I++) {
	// Create an entry and insert the device info
	DevInfo* DI = new DevInfo (Devices [I], Charges [I], I);
	Coll.Insert (DI);
    }

    // Create a listbox inside the window
    Point Size;
    Size.X = Win->IXSize ();
    Size.Y = Win->IYSize () - 2;
    DevListBox* Box = new DevListBox (1, Size);
    Box->SetColl (&Coll);
    Win->AddItem (Box);
    Box->SetPos (0, 2);
    Box->Select ();
    Box->Draw ();
    Win->Activate ();

    // New status line
    PushStatusLine (siAbort | siSelectKeys | siChange | siAccept);

    // Allow choosing an entry
    int NewChanges = 0;
    int Done = 0;
    while (!Done) {

	// Get keyboard input
	Key K = ::KbdGet ();

	// Let the box look for a useful key
	Box->HandleKey (K);

	// Look what's left
	int Selected;
	switch (K) {

	    case kbEnter:
		Selected = Box->GetSelected ();
		if (Selected != -1) {
		    EditDevConfig (Coll [Selected]->DC, Count, NewChanges);
		    Box->Draw ();
		}
		break;

	    case vkAccept:
		// Copy the device infos back
		for (I = 0; I < Coll.GetCount (); I++) {
		    Devices [I] = Coll [I]->DC;
		}
		// If we had changes, tell the caller
		if (NewChanges) {
		    Changed = 1;
		}
		Done = 1;
		break;

	    case vkResize:
		Win->MoveResize ();
		break;

	    case vkAbort:
		if (NewChanges) {
		    if (AskDiscardChanges () == 2) {
			Done = 1;
		    }
		} else {
		    Done = 1;
		}
		break;

	}

    }

    // Restore the status line
    PopStatusLine ();

    // Set a new collection for the listbox (otherwise the box would try to
    // delete the collection)
    Box->SetColl (NULL);

    // Save the current window position
    StgPutPoint (Win->OuterBounds ().A, StgPosName);

    // Delete the window
    delete Win;

}



void ShowDevCharges (DevConfig* Devices, const IstecCharges& Charges,
		     unsigned Count)
// List all devices and there current charges.
{
    // Name of the settings resource
    static const String StgPosName = "ShowDevCharges.ChargesWindow.Position";

    // Check the parameters
    PRECONDITION (Count > 0);

    // Load the window
    Menue* Win = (Menue*) LoadResource ("@ICDEVS.ChargesWindow");

    // If there is a stored window position, move the window to that position
    Point Pos = StgGetPoint (StgPosName, Win->OuterBounds ().A);
    Win->MoveAbs (Pos);

    // Create a device collection and insert all devices
    DevInfoColl Coll;
    for (unsigned I = 0; I < Count; I++) {
	// Create an entry and insert the device info
	DevInfo* DI = new DevInfo (Devices [I], Charges [I], I);
	Coll.Insert (DI);
    }

    // Create a listbox inside the window
    Point Size;
    Size.X = Win->IXSize ();
    Size.Y = Win->IYSize () - 2;
    ChargesBox* Box = new ChargesBox (1, Size);
    Box->SetColl (&Coll);
    Win->AddItem (Box);
    Box->SetPos (0, 2);
    Box->Select ();
    Box->Draw ();
    Win->Activate ();

    // New status line
    PushStatusLine (siEnd);

    // Allow choosing an entry
    int Done = 0;
    while (!Done) {

	// Get keyboard input
	Key K = ::KbdGet ();

	// Let the box look for a useful key
	Box->HandleKey (K);

	// Look what's left
	switch (K) {

	    case vkResize:
		Win->MoveResize ();
		break;

	    case vkAccept:
	    case vkAbort:
		Done = 1;
		break;

	}

    }

    // Restore the status line
    PopStatusLine ();

    // Set a new collection for the listbox (otherwise the box would try to
    // delete the collection)
    Box->SetColl (NULL);

    // Save the current window position
    StgPutPoint (Win->OuterBounds ().A, StgPosName);

    // Delete the window
    delete Win;
}



