/*****************************************************************************/
/*									     */
/*				     ICCWIN.CC				     */
/*									     */
/* (C) 1996	Ullrich von Bassewitz					     */
/*		Zwehrenbuehlstrasse 33					     */
/*		D-72070 Tuebingen					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



// Call window



#include "coll.h"
#include "textitem.h"
#include "settings.h"
#include "progutil.h"

#include "icmsg.h"
#include "icobjid.h"
#include "istec.h"
#include "icwinmgr.h"
#include "iccwin.h"



/*****************************************************************************/
/*			       Message constants			     */
/*****************************************************************************/



const u16 msCallWindowTitle		= MSGBASE_ICCWIN + 0;
const u16 msCallWindowHeader1		= MSGBASE_ICCWIN + 1;
const u16 msCallWindowHeader2		= MSGBASE_ICCWIN + 2;



/*****************************************************************************/
/*			       class CallMsgColl			     */
/*****************************************************************************/



class CallMsgColl: public Collection<String> {

public:
    CallMsgColl ();
    // Create a CallMsgColl

    CallMsgColl (StreamableInit);
    // Create an empty CallMsgColl

    virtual u16 StreamableID () const;
    // return the object id

    static Streamable* Build ();
    // Return a new (empty) object

    virtual void Insert (String* S);
    // Insert a new string

    void Insert (const String& S);
    // Insert a new string

};



// Register the class
LINK (CallMsgColl, ID_CallMsgColl);



CallMsgColl::CallMsgColl ():
    Collection<String> (30, 0, 1)
// Create a CallMsgColl
{
}



CallMsgColl::CallMsgColl (StreamableInit):
    Collection<String> (Empty)
// Create an empty CallMsgColl
{
}



u16 CallMsgColl::StreamableID () const
// return the object id
{
    return ID_CallMsgColl;
}



Streamable* CallMsgColl::Build ()
// Return a new (empty) object
{
    return new CallMsgColl (Empty);
}



void CallMsgColl::Insert (String* S)
// Insert a new string
{
    if (Count == Limit) {
	// Collection is full, delete the first element
	AtDelete (0);
    }
    Collection<String>::Insert (S);
}



void CallMsgColl::Insert (const String& S)
// Insert a new string
{
    Insert (new String (S));
}



/*****************************************************************************/
/*			       class CallWindow				     */
/*****************************************************************************/



class CallWindow: public ItemWindow {

    friend void WriteCallWin (const String& S);


    static CallWindow*	Win;		// Pointer to the one and only window
    static unsigned WindowCount;	// Count of IMonWindows

    CallMsgColl*	CallMsg;	// List of messages
    Rect                ZoomSize;       // Small size of zooming


public:
    CallWindow (StreamableInit);
    // Build constructor

    CallWindow ();
    // Construct an CallWindow

    virtual ~CallWindow ();
    // Destruct an CallWindow

    virtual void Store (Stream &) const;
    virtual void Load (Stream &);
    virtual u16 StreamableID () const;
    static Streamable* Build ();
    // Make the window persistent

    virtual void DrawInterior ();
    // Draw the window contents

    virtual unsigned MinXSize () const;
    // Return the minimum X size of the window. Override this to limit resizing.

    virtual unsigned MinYSize () const;
    // Return the minumim Y size of the window. Override this to limit resizing.

    virtual void Zoom ();
    // Zoom the window

    void Write (const String& S);
    // Write a line to the window, advance the cursor

};



// Register the class
LINK (CallWindow, ID_CallWindow);



/*****************************************************************************/
/*				     Data				     */
/*****************************************************************************/



// Pointer to the one and only window
CallWindow* CallWindow::Win = 0;

// Count of CallWindows
unsigned CallWindow::WindowCount = 0;

// Name of the window size record in the settings file
static const String CallWindowBounds = "CallWindow.Bounds";



/*****************************************************************************/
/*			       class CallWindow				     */
/*****************************************************************************/



inline CallWindow::CallWindow (StreamableInit):
    ItemWindow (Empty)
// Build constructor
{
    // Be shure to check the window count
    CHECK (WindowCount++ == 0);

    // Disable the menue command to create more matrix windows
    ::DisableCommand (miCallWin);
}



CallWindow::CallWindow ():
    ItemWindow (Rect (0, 1, 80, 16),
		wfFramed | wfCanMove | wfCanResize | wfSaveVisible),
    CallMsg (new CallMsgColl),
    ZoomSize (OBounds)
// Construct an CallWindow
{
    // Count of windows must be zero, otherwise this is an error
    CHECK (WindowCount == 0);

    // Lock window output
    Lock ();

    // If there is a stored window size in the settings file, resize the
    // window to the stored rectangle.
    Rect StoredBounds = StgGetRect (CallWindowBounds, OBounds);
    if (StoredBounds != OBounds) {
	Resize (StoredBounds);
    }

    // Set the header
    SetHeader (::LoadAppMsg (msCallWindowTitle));

    // Create and insert the items for both header lines
    const String& Header1 = ::LoadAppMsg (msCallWindowHeader1);
    const String& Header2 = ::LoadAppMsg (msCallWindowHeader2);
    TextItem* H1 = new TextItem (Header1, 100, atTextNormal, NULL);
    TextItem* H2 = new TextItem (Header2, 101, atTextNormal, NULL);
    AddItem (H1);
    AddItem (H2);
    H1->SetWidth (Header1.Len ());
    H2->SetWidth (Header2.Len ());
    H1->SetPos (0, 0);
    H2->SetPos (0, 1);

    // Ok, we have the window now, make it available globally
    WindowCount++;
    Win = this;

    // Make an update of the window.
    DrawItems ();

    // Unlock the output
    Unlock ();

    // Disable the menue command to create more matrix windows
    ::DisableCommand (miCallWin);
}



CallWindow::~CallWindow ()
// Destruct an CallWindow
{
    // Store the current window position and size into the settings file
    StgPutRect (OBounds, CallWindowBounds);

    // Delete the string collection
    delete CallMsg;

    // Decrease the window count and invalidate the global pointer
    WindowCount--;
    Win = NULL;

    // Enable the menue command to create more call windows
    ::EnableCommand (miCallWin);
}



void CallWindow::Store (Stream& S) const
{
    // Call the inherited Store
    ItemWindow::Store (S);

    // Store the string collection
    S.Put (CallMsg);
    
    // Store the rest of the stuff
    S << ZoomSize;
}



void CallWindow::Load (Stream& S)
{
    // Call the inherited Load
    ItemWindow::Load (S);

    // Load the string collection
    CallMsg = (CallMsgColl*) S.Get ();

    // Load the rest of the stuff
    S >> ZoomSize;

    // Ok, the window is valid now. Do a check of the global pointer, then
    // replace it
    CHECK (Win == NULL);
    Win = this;
}



u16 CallWindow::StreamableID () const
{
    return ID_CallWindow;
}



Streamable* CallWindow::Build ()
// Make the window persistent
{
    return new CallWindow (Empty);
}



void CallWindow::DrawInterior ()
// Draw the window contents
{
    // Lock window output
    Lock ();

    // Call the inherited function to draw item stuff
    ItemWindow::DrawInterior ();

    // Draw window specifics
    unsigned YSize = IYSize () - 2;
    unsigned Count = CallMsg->GetCount ();
    unsigned Index = Count > YSize? Count - YSize : 0;

    // Write out the lines
    int Y = 2;
    while (Index < Count) {
	// Watcom compiler bug: WCC needs an override here
	ItemWindow::Write (0, Y++, *CallMsg->At (Index++));
    }

    // Unlock the window, allow output
    Unlock ();
}



unsigned CallWindow::MinXSize () const
// Return the minimum X size of the window. Override this to limit resizing.
{
    return 10;
}



unsigned CallWindow::MinYSize () const
// Return the minumim Y size of the window. Override this to limit resizing.
{
    return 2 + 3;	// One text line at least
}



void CallWindow::Zoom ()
// Zoom the window
{
    // Get the desktop bounds
    Rect Desktop = Background->GetDesktop ();

    // Check if we must zoom in or out
    if (OBounds != Desktop) {
	// Remember the old size, then zoom out
	ZoomSize = OBounds;
	Resize (Desktop);
    } else {
	// Zoom in
	Resize (ZoomSize);
    }
}



void CallWindow::Write (const String& S)
// Write a line to the window, advance the cursor
{
    // Remember the new line
    CallMsg->Insert (S);

    // Correct the cursor position, scroll if necessary
    if (++CursorPos.Y < 2) {
	CursorPos.Y = 2;
    }
    if (CursorPos.Y > MaxY ()) {
	CursorPos.Y = MaxY ();
	Lock ();
	Clear ();
	DrawInterior ();
	Unlock ();
    } else {
	// Watcom compiler bug: WCC needs an override here
	ItemWindow::Write (0, CursorPos.Y, S);
    }
}



/*****************************************************************************/
/*				     Code				     */
/*****************************************************************************/



void InitCallWin ()
// Create a call window
{
    WinMgr->AddWindow (new CallWindow);
}



void WriteCallWin (const String& S)
// Write a line to the call window or discard the string if no call window
// exists
{
    if (CallWindow::Win != 0) {
	CallWindow::Win->Write (S);
    }
}



