/*****************************************************************************/
/*                                                                           */
/*                                    USER.H                                 */
/*                                                                           */
/* (C) 1993,94  Michael Peschel                                              */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#ifndef __USER_H
#define __USER_H

#include "stream.h"
#include "str.h"



/******************************************************************************/
/*                       types and data declarations                          */
/******************************************************************************/

/* Index to ComErrorCounter */
enum {
    ceRXOverflow,               /* Puffer-berlauf beim Empfang */
    ceTXOverflow,               /* Puffer-berlauf beim Senden */
    ceOverrun,                  /* UART Overrun */
    ceBreak,                    /* Break empfangen */
    ceFrame,                    /* Framing-Fehler */
    ceParity                    /* Parity-Fehler */
};


/* an array which holds counters for the errors.
 */
typedef u32 ComErrorCounter [6];


//
/* bitmasks for checking the result of ComModemStatus */
enum {
    csDeltaCTS          = 0x01, /* Delta CTS                    */
    csDeltaDSR          = 0x02, /* Delta DSR                    */
    csDeltaCarrierDetect= 0x04, /* Delta CD                     */
    csRI                = 0x08, /* falling edge if RI           */
    csClearToSend       = 0x10, /* Clear To Send                */
    csDataSetReady      = 0x20, /* Data Set Ready               */
    csRingIndicator     = 0x40, /* Ring Indicator               */
    csCarrierDetect     = 0x80  /* Carrier Detect               */
};



/******************************************************************************/
/*                           class ComPort                                    */
/******************************************************************************/



// Make a forward for an implementation structure
struct _ComData;



class ComPort : public Streamable {

public:
    const   String  PortName;
    u32     Baudrate;
    char    DataBits;
    char    Parity;
    char    StopBits;
    char    Connection;
    char    XonXoff;

    ComPort ();

    ComPort (const  String &aPortName,
             u16    aRXBufSize,
             u16    aTXBufSize,
             u32    aBaudrate   = 9600,
             char   aDataBits   = 8,     /* 5..8 */
             char   aParity     = 'N',   /* <N>one, <O>dd, <E>ven, <S>pace, <M>ark */
             char   aStopBits   = 1,     /* 1, 2 */
             char   aConnection = 'M',   /* <D>irect, <M>odem */
             char   aXonXoff    = 'D'    /* <E>nabled, <D>isabled */
            );

    ComPort (StreamableInit);
    virtual ~ComPort ();


    virtual void ComError (u32 APICode);
    /* called for API Errors */

    u16 ComOpen ();
    /* opens the port, retuns 0 if success */

    void ComClose ();
    /* closes the port */

    u16 ComIsOpen ();
    /* returns 1 if the port is open, else 0. */

    void SetWriteTimeout (double timeout);
    void SetReadTimeout (double timeout);
    /* setting timeout in seconds for read and write operations */

    double GetWriteTimeout ();
    double GetReadTimeout ();
    /* actual timeout settings in seconds */

    void ComDTROn ();
    /* activates DTR */

    void ComDTROff ();
    /* deactivates DTR */

    u16 ComRXSize ();
    /* returns the size of the receive queue */

    u16 ComRXCount ();
    /* returns the number of characters in the receive queue */

    void ComRXFlush ();
    /* flushes the receive queue */

    u16 ComTXSize ();
    /* returns the size of the transmit queue */

    u16 ComTXCount ();
    /* returns the number of charcters in the transmit queue */

    u16 ComTXFree ();
    /* returns the free space in the transmit queue */

    void ComTXFlush ();
    /* flushes the transmit queue */

    i16 ComReceive ();
    /* reads and returns a character from the receive queue.
     * -1 is returned if the receive queue is empty.
     */

    i16 TimedReceive ();
    /* waits with timeout for a character. If a character is received within the
     * timeout-interval this character is returned, if not -1 is returned.
     */

    void TimedReceiveBlock (char Buffer[], u32 Count, u32 &ChRead);
    /* waits with timeout for Count characters. If the characters are received
     * within the timeout interval ChRead is equal Count. In case of timeout
     * ChRead holds the number of characters received.
     */

    void ComSend (unsigned char B);
    /* sends a character. If the transmit queue ist full the errorcounter
     * TXOverflow is incremented.
     */

    void TimedSend (unsigned char B);
    /* waits with timeout to send a character. If timeout occurs the errorcounter
     * TXOverflow is incremented. The character is ignored.
     */

    void TimedSendBlock  (const char Buffer[], u32 count, u32& ChWritten);
    /* waits with timeout to send the given block. If timout occurs the errorcounter
     * TXOverflow is incremented. ChWritten holds the number of characters sent.
     */

    void ComBreak (double Duration);
    /* sends break with the given duration */

    ComErrorCounter* ComErrors ();
    /* returns a pointer to the errorcounter array. */

    u16 ComModemStatus ();
    /* returns the state of the modem input signals. (see constants) */

    // Derived from class Streamable
    virtual void Load (Stream&);
    virtual void Store (Stream&) const;
    virtual u16 StreamableID () const;
    static Streamable *Build ();

private:

    ComErrorCounter ErrorCounter;

    // pointer to internal data structure
    struct _ComData* ComData;

    // simple interface for DosDevIOCtl call

    // async device control
    void ASYNC_IOCtl (u32 Function,
                      void *ParmList, u32 ParmLen,
                      void *DataList, u32 DataLen);

    // general device control
    void GENERAL_IOCtl (u32 Function,
                        void *ParmList, u32 ParmLen,
                        void *DataList, u32 DataLen);

    void UpdateErrorCounter (u16 ErrorWord);
    /* error counter update */

    void GetComError ();
    /* gets the error state of the port and updates the error counters */
};


// End of USER.H

#endif



