/*****************************************************************************/
/*                                                                           */
/*                                  SCREEN.H                                 */
/*                                                                           */
/* (C) 1993-95  Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#ifndef __SCREEN_H
#define __SCREEN_H



#include "machine.h"
#include "object.h"
#include "rect.h"



// Instance of the screen class to handle screen output. Must be initialized
// from outside (RootWindow)
class Screen;
extern Screen* TheScreen;



/*****************************************************************************/
/*                           Video mode constants                            */
/*****************************************************************************/


static const u16 vmBW40         = 0x0000;
static const u16 vmCO40         = 0x0001;
static const u16 vmBW80         = 0x0002;
static const u16 vmCO80         = 0x0003;
static const u16 vmMono         = 0x0007;

// Extended modes
static const u16 vmVGA_80x25    = vmCO80;
static const u16 vmVGA_80x30    = 0x0101;
static const u16 vmVGA_80x34    = 0x0102;
static const u16 vmVGA_80x43    = 0x0103;
static const u16 vmVGA_80x50    = 0x0104;
static const u16 vmVGA_80x60    = 0x0105;
static const u16 vmVGA_94x25    = 0x0110;
static const u16 vmVGA_94x30    = 0x0111;
static const u16 vmVGA_94x34    = 0x0112;
static const u16 vmVGA_94x43    = 0x0113;
static const u16 vmVGA_94x50    = 0x0114;
static const u16 vmVGA_94x60    = 0x0115;

// Card specific modes
static const u16 vmET4_100x40   = 0x012A;

// The "don't know" video mode
static const u16 vmInvalid      = 0xFFFF;



/*****************************************************************************/
/*                               class Screen                                */
/*****************************************************************************/


class Screen: public Object {

private:
    u16                 XSize;
    u16                 YSize;
    u16                 CurrentMode;    // Current video mode
    int                 Color;          // 1 if color mode, 0 if not
    int                 Console;        // 1 if console screen, 0 if not
    int                 CP437;          // 1 if CP437 should be used
    unsigned char*      TransTable;     // Translation table for output

    u16                 StartupMode;    // Mode before app was active
    u16                 StartupCursor;  // Cursor when app became active

    u16                 CF_HiddenCursor;
    u16                 CF_NormalCursor;
    u16                 CF_FatCursor;

    void TCInit ();
    // Initialize the termcap system. Used for *nixen only.

    char* GetIS (char* IS);
    char* GetRS (char* RS);
    // Return a replacement for the init strings IS and RS. Used for *nixen
    // only.

    void SetModeData ();
    // Internally called after setting a new mode, sets cursor data etc.

    unsigned char Translate (unsigned char C);
    // Translate the char via the translation table

    u16* Translate (u16* Target, u16* Source, unsigned Len);
    // Translate a complete buffer via the translation table

public:
    Screen ();
    ~Screen ();

    // Put a rectangular buffer region onto the screen
    void DisplayBuffer (const Rect& R, u16* Buf);

    // Video mode
    void SetMode (u16 Mode);
    u16 GetMode ();
    int IsColor ();
    int IsConsole ();

    // Cursor
    void SetCursorOn ();
    void SetCursorOff ();
    void SetCursorFat ();
    void SetCursorPos (const Point& Pos);

    // Screen size
    u16 GetXSize ();
    u16 GetYSize ();

    unsigned TerminalSpeed ();
    // Get some information on the terminal speed. This will return a value
    // between 0..10, where 10 is a very fast (direct access) screen and
    // 0 is a very slow (300 baud) serial line.
    // This may be used to change the amount of screen output, a program
    // produces.

    inline void SetTransTable (unsigned char* NewTable);
    // Set a new translation table. The old table is deleted. NewTable can
    // be NULL to clear the table.

};



inline unsigned char Screen::Translate (unsigned char C)
// Translate the char via the translation table
{
    return TransTable? TransTable [C] : C;
}



inline u16 Screen::GetXSize ()
{
    return XSize;
}



inline u16 Screen::GetYSize ()
{
    return YSize;
}



inline u16 Screen::GetMode ()
{
    return CurrentMode;
}



inline int Screen::IsColor ()
{
    return Color;
}



inline int Screen::IsConsole ()
{
    return Console;
}



inline void Screen::SetTransTable (unsigned char* NewTable)
// Set a new translation table. The old table is deleted. NewTable can be NULL
// to clear the table.
{
    delete [] TransTable;
    TransTable = NewTable;
}



// End of SCREEN.H

#endif

