/*****************************************************************************/
/*                                                                           */
/*                                 CIRCBUF.H                                 */
/*                                                                           */
/* (C) 1993,94  Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#ifndef __CIRCBUF_H
#define __CIRCBUF_H


#include "machine.h"
#include "check.h"
#include "object.h"
#include "strmable.h"
#include "stream.h"




template <class T>
class CircularBuffer : public Streamable {

protected:
    T*  Data;
    u16 In;
    u16 Out;
    u16 Limit;
    u16 Count;

protected:
    virtual void PutItem (Stream& S, T Item) const;
    virtual T GetItem (Stream& S);
    virtual void FreeItem (T&);

public:
    CircularBuffer (u16 Size);
    CircularBuffer (StreamableInit X);
    virtual ~CircularBuffer ();

    // Derived from class Streamable
    virtual void Load (Stream&);
    virtual void Store (Stream&) const;
    static Streamable* Build ();

    // New member functions
    int IsEmpty ();
    int IsFull ();
    void Put (const T& Item, int IgnoreOverflow = 1);
    void PutInFront (const T& Item, int IgnoreOverflow = 1);
    T Get ();
    const T& Peek ();
};



template <class T>
CircularBuffer<T>::CircularBuffer (u16 Size) :
    In (0),
    Out (0),
    Limit (Size),
    Count (0)
{
    // Check parameters
    PRECONDITION (Size > 0);

    // Allocate memory
    Data = new T [Limit];
}



template <class T>
inline CircularBuffer<T>::CircularBuffer (StreamableInit)
{
}



template <class T>
CircularBuffer<T>::~CircularBuffer ()
{
    // Free remaining items
    while (Count--) {
        FreeItem (Data [Out]);
        if (++Out >= Limit) {
            Out = 0;
        }
    }

    // Free the array
    delete [] Data;

}



template <class T>
void CircularBuffer<T>::Load (Stream& S)
{
    // Read data
    S >> In >> Out >> Limit >> Count;

    // Allocate memory
    Data = new T [Limit];

    // Now read the items
    u16 I = Count;
    u16 O = Out;
    while (I--) {
        Data [O] = GetItem (S);
        if (++O >= Limit) {
            O = 0;
        }
    }

}



template <class T>
void CircularBuffer<T>::Store (Stream& S) const
{
    // Store data
    S << In << Out << Limit << Count;

    // Store the items
    u16 I = Count;
    u16 O = Out;
    while (I--) {
        PutItem (S, Data [O]);
        if (++O >= Limit) {
            O = 0;
        }
    }
}



template <class T>
T CircularBuffer<T>::GetItem (Stream&)
{
    ABSTRACT ();
    return *((T*) NULL);
}



template <class T>
void CircularBuffer<T>::PutItem (Stream&, T) const
{
    ABSTRACT ();
}



template <class T>
void CircularBuffer<T>::FreeItem (T&)
{
    // Default is to do nothing
}



template <class T>
inline Streamable* CircularBuffer<T>::Build ()
{
    return new CircularBuffer<T> (Empty);
}



template <class T>
inline int CircularBuffer<T>::IsEmpty ()
{
    return (Count == 0);
}



template <class T>
inline int CircularBuffer<T>::IsFull ()
{
    return (Count == Limit);
}



template <class T>
void CircularBuffer<T>::Put (const T& Item, int IgnoreOverflow)
{
    if (Count > Limit) {
        if (IgnoreOverflow) {
            return;
        } else {
            FAIL ("CircularBuffer::Put: Overflow");
        }
    }

    Data [In] = Item;
    Count++;
    if (++In == Limit) {
        In = 0;
    }
}



template <class T>
void CircularBuffer<T>::PutInFront (const T& Item, int IgnoreOverflow)
{
    if (Count > Limit) {
        if (IgnoreOverflow) {
            return;
        } else {
            FAIL ("CircularBuffer::PutInFront: Overflow");
        }
    }

    Out = (u16) (Out ? Limit-1 : Out-1);
    Data [Out] = Item;
    Count++;
}



template <class T>
T CircularBuffer<T>::Get ()
{
    // There must be at least one element
    CHECK (Count > 0);

    u16 O = Out;
    if (++Out >= Limit) {
        Out = 0;
    }
    Count--;
    return Data [O];
}



template <class T>
const T & CircularBuffer<T>::Peek ()
{
    // There must be at least one element
    CHECK (Count >= 1);

    return Data [Out];
}



// End of CIRCBUF.H

#endif

