/*****************************************************************************/
/*                                                                           */
/*                                  ICIMON.CC                                */
/*                                                                           */
/* (C) 1995     Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include <stdio.h>
#include <sys/time.h>

#include "chartype.h"
#include "window.h"
#include "strparse.h"
#include "progutil.h"
#include "stdmsg.h"

#include "icmsg.h"
#include "istec.h"



/*****************************************************************************/
/*                             Message Constants                             */
/*****************************************************************************/



const u16 msIMonHeader1         = MSGBASE_ICIMON +  0;
const u16 msIMonHeader2         = MSGBASE_ICIMON +  1;
const u16 msIMonWinTitel        = MSGBASE_ICIMON +  2;
const u16 msIncoming            = MSGBASE_ICIMON +  3;
const u16 msOutgoing            = MSGBASE_ICIMON +  4;
const u16 msUsageNone           = MSGBASE_ICIMON +  5;
const u16 msUsageRaw            = MSGBASE_ICIMON +  6;
const u16 msUsageModem          = MSGBASE_ICIMON +  7;
const u16 msUsageNet            = MSGBASE_ICIMON +  8;
const u16 msUsageVoice          = MSGBASE_ICIMON +  9;
const u16 msUsageFax            = MSGBASE_ICIMON + 10;
const u16 msUsageUnknown        = MSGBASE_ICIMON + 11;
const u16 msNoInfoFile          = MSGBASE_ICIMON + 12;
const u16 msReadError           = MSGBASE_ICIMON + 13;



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



// Define the ISDN stuff
static const char InfoFileName [] = "/dev/isdninfo";

#ifdef CLEAN_IMON_SOURCE
#include "/usr/src/isdn/include/isdn.h"
#else
// Define the ISDN stuff
const unsigned ISDN_USAGE_NONE          = 0;
const unsigned ISDN_USAGE_RAW           = 1;
const unsigned ISDN_USAGE_MODEM         = 2;
const unsigned ISDN_USAGE_NET           = 3;
const unsigned ISDN_USAGE_VOICE         = 4;
const unsigned ISDN_USAGE_FAX           = 5;
const unsigned ISDN_USAGE_MASK          = 127;
const unsigned ISDN_USAGE_OUTGOING      = 128;
#endif



// Types of lines read from isdninfo
enum {
    li_idmap, li_chmap, li_drmap, li_usage, li_flags, li_phone, li_count
};



// The output window
static Window* IMonWin = NULL;



// Info file
static FILE* InfoFile = NULL;



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



void DoneIMonWin ()
// Delete the isdn4linux monitor window
{
    if (IMonWin) {
        delete IMonWin;
        IMonWin = NULL;
    }
    if (InfoFile) {
        fclose (InfoFile);
        InfoFile = NULL;
    }
}



void UpdateIMonWin ()
// Update the ISDN monitor window
{
    // If we have no window open, bail out now
    if (IMonWin == NULL) {
        return;
    }

    // Since isdn4linux 0.7beta, there will only be information available,
    // if something has changed. Use select() to make shure, we can read
    // from the device.
    // Timeout is zero
    timeval Timeout;
    Timeout.tv_usec = 0;
    Timeout.tv_sec  = 0;

    // Set the file descriptor
    fd_set Desc;
    FD_ZERO (&Desc);
    FD_SET (fileno (InfoFile), &Desc);

    // Check input status, return if no data available
    int Res = select (fileno (InfoFile) + 1, &Desc, NULL, NULL, &Timeout);
    if (Res <= 0) {
        // No info or some sort of error
        return;
    }

    // Read 6 lines from the file
    String Lines [li_count];
    for (unsigned I = 0; I < li_count; I++) {
        char Line [512];
        if (fgets (Line, sizeof (Line), InfoFile) == NULL) {
            // Error reading the file
            ErrorMsg (FormatStr (LoadAppMsg (msReadError).GetStr (), InfoFileName));
            DoneIMonWin ();
            // Menue entry is not correct after such an error. Since this is
            // just an oddity, don't do anything about it (setting the menue
            // entry would be possible but don't do it here. Returning an
            // error code would be a better solution.)
            return;
        }

        // Make a string and remove the trailing newline (if present)
        Lines [I] = &Line [7];
        int Len = Lines [I].Len ();
        while (Len >= 0 && IsSpace (Lines [I] [Len-1])) {
            Len--;
            Lines [I].Trunc (Len);
        }

    }

    // Lock the output window
    IMonWin->Lock ();

    // Clear the window
    IMonWin->Clear ();

    // Write the header lines to the window
    IMonWin->Write (0, 0, LoadAppMsg (msIMonHeader1));
    IMonWin->Write (0, 1, LoadAppMsg (msIMonHeader2));

    // Set up a parser object for each line
    StringParser SP_ChMap (Lines [li_chmap], StringParser::SkipWS);
    StringParser SP_DrMap (Lines [li_drmap], StringParser::SkipWS);
    StringParser SP_Usage (Lines [li_usage], StringParser::SkipWS);
    StringParser SP_Phone (Lines [li_phone], StringParser::SkipWS);

    // Write the lines to the window
    unsigned Y = 2;
    unsigned MaxY = IMonWin->MaxY ();
    while (Y <= MaxY) {

        // Get the infos
        i32 Driver;
        if (SP_DrMap.GetI32 (Driver) != 0) {
            // Error
            break;
        }
        i32 Channel;
        if (SP_ChMap.GetI32 (Channel) != 0) {
            // Error
            break;
        }
        u32 Usage;
        if (SP_Usage.GetU32 (Usage) != 0) {
            // Error
            break;
        }
        String Phone;
        if (SP_Phone.GetToken (Phone) != 0) {
            // Error
            break;
        }

        // This is the last entry if the driver number is -1
        if (Driver < 0) {
            break;
        }

        // Get the service message
        unsigned MsgNum;
        switch (Usage & ISDN_USAGE_MASK) {

            case ISDN_USAGE_NONE:
                MsgNum = msUsageNone;
                break;

            case ISDN_USAGE_RAW:
                MsgNum = msUsageRaw;
                break;

            case ISDN_USAGE_NET:
                MsgNum = msUsageNet;
                break;

            case ISDN_USAGE_MODEM:
                MsgNum = msUsageModem;
                break;

            case ISDN_USAGE_VOICE:
                MsgNum = msUsageVoice;
                break;

            case ISDN_USAGE_FAX:
                MsgNum = msUsageFax;
                break;

            default:
                MsgNum = msUsageUnknown;
                break;

        }
        String Service = LoadAppMsg (MsgNum);
        Service.Pad (String::Right, 10);

        // Get the direction message, but beware: If the channel is unused,
        // there is no in/out message
        String InOut;
        if ((Usage & ISDN_USAGE_MASK) != ISDN_USAGE_NONE) {
            if (Usage & ISDN_USAGE_OUTGOING) {
                InOut = LoadAppMsg (msOutgoing);
            } else {
                InOut = LoadAppMsg (msIncoming);
            }
        }
        InOut.Pad (String::Center, 7);

//      This is the header:      " Dr  Ch  Ein/Aus  Dienst      Nummer"
        String Line = FormatStr (" %2d  %2d  %s  %s  %s",
                                 Driver,
                                 Channel,
                                 InOut.GetStr (),
                                 Service.GetStr (),
                                 Phone.GetStr ());
        IMonWin->Write (0, Y, Line);
        Y++;
    }

    // Unlock the window
    IMonWin->Unlock ();

}



void InitIMonWin ()
// Create the isdn4linux monitor window
{
    // Delete an eventually existing old window
    DoneIMonWin ();

    // Open the info file
    InfoFile = fopen (InfoFileName, "r");
    if (InfoFile == NULL) {
        ErrorMsg (FormatStr (LoadAppMsg (msNoInfoFile).GetStr (), InfoFileName));
        return;
    }

    // Create a new window
    Rect Bounds = Background->GetDesktop ();
    Bounds.B.X -= 31;
    Bounds.A.Y = Bounds.B.Y - 11;
    IMonWin = new Window (Bounds, wfFramed);

    // Set the window header
    IMonWin->SetHeader (LoadAppMsg (msIMonWinTitel));

    // Display the current isdn4linux data
    UpdateIMonWin ();

    // Show the window
    IMonWin->Show ();
}



