/*****************************************************************************/
/*                                                                           */
/*                                 ICDIAG.CC                                 */
/*                                                                           */
/* (C) 1995     Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include <stdio.h>

#include "delay.h"
#include "strcvt.h"
#include "coll.h"
#include "datetime.h"
#include "menue.h"
#include "listbox.h"
#include "menuitem.h"
#include "progutil.h"

#include "icmsg.h"
#include "iccom.h"
#include "iclog.h"
#include "icdlog.h"
#include "icalias.h"



/*****************************************************************************/
/*                             Message constants                             */
/*****************************************************************************/



const u16 msMsgWinHeader        = MSGBASE_ICDIAG +  0;
const u16 msOn                  = MSGBASE_ICDIAG +  1;
const u16 msOff                 = MSGBASE_ICDIAG +  2;
const u16 msLEDState            = MSGBASE_ICDIAG +  3;
const u16 msChargeState         = MSGBASE_ICDIAG +  4;
const u16 msTFEState            = MSGBASE_ICDIAG +  5;
const u16 msDoorState           = MSGBASE_ICDIAG +  6;
const u16 msSwitchState         = MSGBASE_ICDIAG +  7;
const u16 msPulseDial           = MSGBASE_ICDIAG +  8;
const u16 msToneDial            = MSGBASE_ICDIAG +  9;
const u16 msCallWinHeader       = MSGBASE_ICDIAG + 10;
const u16 msCallStart           = MSGBASE_ICDIAG + 11;
const u16 msCallEnd             = MSGBASE_ICDIAG + 12;
const u16 msCallTimeFmt         = MSGBASE_ICDIAG + 13;



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



// Wait time after a call before requesting the charges
int DebugWaitAfterCall  = 500;

// If true, log calls with a chargecount of zero
int LogZeroCostCalls    = 1;



/*****************************************************************************/
/*                       Columns for the matrix window                       */
/*****************************************************************************/

//           1         2         3         4         5         6         7
// 01234567890123456789012345678901234567890123456789012345678901234567890
//
//  Nr. Amt1 Amt2 Int1 Int2 Int3 Ton WTon TFE Ruf

const unsigned colNr            =  1;
const unsigned colSchleife      =  3;
const unsigned colAmt1          =  6;
const unsigned colAmt2          = 11;
const unsigned colInt1          = 16;
const unsigned colInt2          = 21;
const unsigned colInt3          = 26;
const unsigned colTon           = 31;
const unsigned colWTon          = 35;
const unsigned colTFE           = 40;
const unsigned colRuf           = 44;



/*****************************************************************************/
/*               Constants for the State field in DevStateInfo               */
/*****************************************************************************/



const unsigned stSchleife       = 0x0001;
const unsigned stAmt1           = 0x0002;
const unsigned stAmt2           = 0x0004;
const unsigned stAmt            = stAmt1 | stAmt2;
const unsigned stInt1           = 0x0008;
const unsigned stInt2           = 0x0010;
const unsigned stInt3           = 0x0020;
const unsigned stInt            = stInt1 | stInt2 | stInt3;
const unsigned stTon            = 0x0040;
const unsigned stWTon           = 0x0080;
const unsigned stTFE            = 0x0100;
const unsigned stRuf            = 0x0200;



/*****************************************************************************/
/*                      Explicit template instantiation                      */
/*****************************************************************************/



#ifdef EXPLICIT_TEMPLATES
template class Collection<class DevStateInfo>;
template class SortedCollection<class DevStateInfo, unsigned char>;
template class ListBox<String>;
#endif



/*****************************************************************************/
/*                            class DevStateInfo                             */
/*****************************************************************************/



class DevStateInfo {

public:
    const unsigned char DevNum;         // Device number
    unsigned            State;          // Device state
    String              CallPhone;      // Phone number dialed
    String              CallPhoneTmp;   // Phone number while dialing
    Time                CallStart;      // Start of call
    TimeDiff            CallDuration;   // Duration of call
    u16                 StartCharges;   // Charges at start of call
    u16                 CallCharges;    // Charges for call
    unsigned            HasExt;         // True if device had an external call


    DevStateInfo (unsigned char Device);
    // Create a DevStateInfo object

    DevStateInfo (const DevStateInfo& X);
    // Copy constructor

    String LogMsg ();
    // Create a message for the logfile from the data

    int GetState (unsigned StateMask);
    void SetState (unsigned NewState);
    void ClrState (unsigned NewState);
    // Get/set/clear bits in State

};



DevStateInfo::DevStateInfo (unsigned char Device):
    DevNum (Device),
    State (0),
    HasExt (0)
{
}



DevStateInfo::DevStateInfo (const DevStateInfo& X):
    DevNum (X.DevNum),
    State (X.State),
    CallPhone (X.CallPhone),
    CallPhoneTmp (X.CallPhoneTmp),
    CallStart (X.CallStart),
    CallDuration (X.CallDuration),
    StartCharges (X.StartCharges),
    CallCharges (X.CallCharges),
    HasExt (X.HasExt)
{
}



String DevStateInfo::LogMsg ()
// Create a message for the logfile from the data
{
    // Format of the line:
    //           1         2         3         4         5         6         7         8
    // 012345678901234567890123456789012345678901234567890123456789012345678901234567890
    // NR  ALIAS_______  DATE____  START___  DURATION  CALLPHONE_______  EINH  DM____

    String StartDate = CallStart.DateTimeStr ("%d.%m.%y");
    String StartTime = CallStart.DateTimeStr ("%H:%M:%S");
    unsigned DurSec = CallDuration.GetSec ();
    unsigned DurMin = (DurSec % 3600) / 60;
    unsigned DurHour = DurSec / 3600;
    DurSec %= 60;
    String DevAlias = GetAlias (DevNum+21);
    DevAlias.Trunc (12);

    // Use the alias of the phone number if one is defined, otherwise use
    // the number
    String Phone = GetAlias (CallPhone);
    if (Phone.IsEmpty ()) {
        Phone = CallPhone;
    }
    Phone.Trunc (16);

    return FormatStr ("%d  %-12s  %s  %s  %02d:%02d:%02d  %-16s  %4d  %6.2f",
                      DevNum+21,
                      DevAlias.GetStr (),
                      StartDate.GetStr (),
                      StartTime.GetStr (),
                      DurHour, DurMin, DurSec,
                      Phone.GetStr (),
                      CallCharges,
                      CallCharges * PricePerUnit);
}



inline int DevStateInfo::GetState (unsigned StateMask)
{
    return State & StateMask;
}



inline void DevStateInfo::SetState (unsigned NewState)
{
    State |= NewState;
}



inline void DevStateInfo::ClrState (unsigned NewState)
{
    State &= ~NewState;
}



/*****************************************************************************/
/*                            class DevStateColl                             */
/*****************************************************************************/



class DevStateColl: public SortedCollection<DevStateInfo, unsigned char> {

protected:
    virtual int Compare (const unsigned char* Key1, const unsigned char* Key2);
    virtual const unsigned char* KeyOf (const DevStateInfo* Item);

public:
    DevStateColl ();
    // Create a DevStateColl

    void DeleteDev (unsigned char Num);
    // Delete the device with the given numer.

    DevStateInfo* NewDev (unsigned char Num);
    // Create and insert a device with the given numer.

    DevStateInfo* GetDevStateInfo (unsigned char Num);
    // Return a pointer to the entry. Calls FAIL if the entry does not exist

    void SetState (unsigned char Dev, unsigned NewState);
    void ClrState (unsigned char Dev, unsigned NewState);
    // Set/reset the device state bits

};



int DevStateColl::Compare (const unsigned char* Key1, const unsigned char* Key2)
{
    if (*Key1 < *Key2) {
        return -1;
    } else if (*Key1 > *Key2) {
        return 1;
    } else {
        return 0;
    }
}



const unsigned char* DevStateColl::KeyOf (const DevStateInfo* Item)
{
    return &Item->DevNum;
}



DevStateColl::DevStateColl ():
    SortedCollection <DevStateInfo, unsigned char> (10, 10, 1)
{
}



void DevStateColl::DeleteDev (unsigned char Num)
// Delete the device with the given numer.
{
    // Search for the device
    int Index;
    CHECK (Search (&Num, Index) != 0);

    // Delete the entry
    AtDelete (Index);
}



DevStateInfo* DevStateColl::NewDev (unsigned char Num)
// Create and insert a device with the given numer.
{
    // Create a new entry
    DevStateInfo* DI = new DevStateInfo (Num);

    // Insert the new entry
    Insert (DI);

    // And return it
    return DI;
}



DevStateInfo* DevStateColl::GetDevStateInfo (unsigned char Num)
// Return a pointer to the entry. Creates an entry if none exists.
{
    // Search for the entry
    int Index;
    if (Search (&Num, Index) == 0) {
        // No entry til now, create one
        return NewDev (Num);
    } else {
        // Found, return it
        return At (Index);
    }
}



void DevStateColl::SetState (unsigned char Dev, unsigned NewState)
// Set the device state bits
{
    GetDevStateInfo (Dev)->SetState (NewState);
}



void DevStateColl::ClrState (unsigned char Dev, unsigned NewState)
// Reset the device state bits
{
    GetDevStateInfo (Dev)->ClrState (NewState);
}



/*****************************************************************************/
/*                              class CallQueue                              */
/*****************************************************************************/



class CallQueue: public Collection<DevStateInfo> {

public:
    CallQueue ();
    // Create a CallQueue

};



CallQueue::CallQueue ():
    Collection <DevStateInfo> (10, 10, 1)
{
}



/*****************************************************************************/
/*                            class MatrixListBox                            */
/*****************************************************************************/



class MatrixListBox: public ListBox<String> {

protected:
    virtual void Print (int Index, int X, int Y, u16 Attr);
    // Display one of the listbox entries

public:
    MatrixListBox (i16 aID, const Point& aSize);

};



MatrixListBox::MatrixListBox (i16 aID, const Point& aSize):
    ListBox <String> ("", aID, aSize, atEditNormal, atEditBar, atEditNormal, NULL)
{
}



void MatrixListBox::Print (int Index, int X, int Y, u16 Attr)
{
    // Get the entry
    String* S = Coll->At (Index);

    // Write out the string
    Owner->Write (X, Y, *S, Attr);
}



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



// Some instances of classes that are needed when running in diag mode.
// All instances are created when diag mode starts
static Window* MsgWin = NULL;
static Window* CallWin = NULL;
static Menue* MatrixWin = NULL;
static Collection<String>* MatrixColl = NULL;
static MatrixListBox* MatrixBox = NULL;

// The following two must always exist
static DevStateColl DevState;
static CallQueue Queue;

// The diag data for tone dial is somewhat weird: When using pulse dial, the
// diag message from the istec contains the device number. When using tone
// dial, the number of the internal connection is transmitted instead. To
// show the device number in both cases, we hold the device that uses an
// internal connection in the array below
static unsigned IntCon [3];



/*****************************************************************************/
/*                          Matrix update functions                          */
/*****************************************************************************/



static void WriteDiagMsg (const String& DiagMsg)
{
    if (MsgWin) {
        MsgWin->FCWrite ("\r\n");
        MsgWin->FCWrite (DiagMsg);
    }
}



static void WriteCallMsg (const String& CallMsg)
{
    if (CallWin) {
        CallWin->FCWrite ("\r\n");
        CallWin->FCWrite (CallMsg);
    }
}



static void ReplaceMatrixCol (unsigned Dev, unsigned Col, char C)
// Replace a char in a matric column
{
    if (MatrixColl) {
        MatrixColl->At (Dev) -> Replace (Col, C);
    }
}



static void SetSchleife (unsigned Dev, int State)
{
    // Get the state struct
    DevStateInfo* DS = DevState.GetDevStateInfo (Dev);

    if (State) {

        // Mark the device in the matrix
        ReplaceMatrixCol (Dev, colSchleife, '*');

        // Set the state bit
        DS->SetState (stSchleife);

    } else {

        // Unmark the device
        ReplaceMatrixCol (Dev, colSchleife, ' ');

        // Reset the state bit
        DS->ClrState (stSchleife);

        // Clear the temporary phone number
        DS->CallPhoneTmp.Clear ();
    }
}



static void SetAmt (unsigned Dev, int State, unsigned Amt)
{
    // Map the line number to a column number in the matrix
    unsigned Column = 0;                // Make gcc happy
    unsigned Mask = 0;                  // Make gcc happy
    switch (Amt) {
        case 1:   Column = colAmt1; Mask = stAmt1;      break;
        case 2:   Column = colAmt2; Mask = stAmt2;      break;
        default:  FAIL ("SetAmt: Invalid column number");
    }

    // Get a pointer to the device state info
    DevStateInfo* DS = DevState.GetDevStateInfo (Dev);

    if (State) {

        // Update the matrix line
        ReplaceMatrixCol (Dev, Column, 'x');

        // Remember the line used
        DS->SetState (Mask);

    } else {

        // Update the matrix line
        ReplaceMatrixCol (Dev, Column, '-');

        // Clear the used line
        DS->ClrState (Mask);

        // If we had an external call, print the info
        if (DS->HasExt) {

            // Get current time
            Time Current = Now ();

            // Calculate the durcation of the call
            DS->CallDuration = Current - DS->CallStart;

            // Reset flag for external call
            DS->HasExt = 0;

            // Create an entry for the queue
            DevStateInfo* DI = new DevStateInfo (*DS);

            // Wait some time
            Delay (DebugWaitAfterCall);

            // Insert the entry into into the call queue
            Queue.Insert (DI);

            // Request a charge update from the istec
            IstecRequestCharges ();

        }
    }
}



static void SetInt (unsigned Dev, int State, unsigned Int)
{
    // Map the line number to a column number in the matrix
    unsigned Column = 0;                // Make gcc happy
    unsigned Mask = 0;                  // Make gcc happy
    switch (Int) {
        case 1:   Column = colInt1; Mask = stInt1;      break;
        case 2:   Column = colInt2; Mask = stInt2;      break;
        case 3:   Column = colInt3; Mask = stInt3;      break;
        default:  FAIL ("SetInt: Invalid column number");
    }

    // Get a pointer to the device state info
    DevStateInfo* DS = DevState.GetDevStateInfo (Dev);

    if (State) {

        // Show the change
        ReplaceMatrixCol (Dev, Column, 'x');

        // Set the state bit
        DS->SetState (Mask);

        // Remember which device uses this internal line
        IntCon [Int-1] = Dev;

    } else {

        // Show the change
        ReplaceMatrixCol (Dev, Column, '-');

        // Reset the state bit
        DS->ClrState (Mask);

        // If the internal connection goes off, but we have an external one,
        // we have an external call.
        if (DS->GetState (stAmt)) {

            // Remember the starting time
            DS->CallStart = Now ();

            // Remember the charge info on start
            DS->StartCharges = Charges [Dev];

            // Remember that we are connected externally
            DS->HasExt = 1;

            // Make the final phone number from the temporary one
            DS->CallPhone = DS->CallPhoneTmp;
            if (DS->CallPhone.IsEmpty () == 0 && DS->CallPhone [0] == '0') {
                DS->CallPhone.Del (0, 1);
            }

            // Log the facts
            String TS = DS->CallStart.DateTimeStr (LoadAppMsg (msCallTimeFmt));
            WriteCallMsg (FormatStr (LoadAppMsg (msCallStart).GetStr (),
                                     TS.GetStr (),
                                     Dev+21,
                                     DS->CallPhone.GetStr ()));

        }
    }
}



static void SetTon (unsigned Dev, int State)
{
    if (State) {
        ReplaceMatrixCol (Dev, colTon, 'x');
        DevState.SetState (Dev, stTon);
    } else {
        ReplaceMatrixCol (Dev, colTon, '-');
        DevState.ClrState (Dev, stTon);
    }
}



static void SetWTon (unsigned Dev, int State)
{
    if (State) {
        ReplaceMatrixCol (Dev, colWTon, 'x');
        DevState.SetState (Dev, stWTon);
    } else {
        ReplaceMatrixCol (Dev, colWTon, '-');
        DevState.ClrState (Dev, stWTon);
    }
}



static void SetTFE (unsigned Dev, int State)
{
    if (State) {
        ReplaceMatrixCol (Dev, colTFE, 'x');
        DevState.SetState (Dev, stTFE);
    } else {
        ReplaceMatrixCol (Dev, colTFE, '-');
        DevState.ClrState (Dev, stTFE);
    }
}



static void SetRuf (unsigned Dev, int State)
{
    if (State) {
        ReplaceMatrixCol (Dev, colRuf, 'x');
        DevState.SetState (Dev, stRuf);
    } else {
        ReplaceMatrixCol (Dev, colRuf, '-');
        DevState.ClrState (Dev, stRuf);
    }
}



static void RedrawMatrix ()
{
    // Update the matrix column
    if (MatrixBox) {
        MatrixBox->Draw ();
    }
}



static void WriteDialStatus (unsigned MsgNum, unsigned Dev, unsigned Digit)
{
    // Remap the digit code to a real digit
    switch (Digit) {
        case  1:        Digit = '1';    break;
        case  2:        Digit = '2';    break;
        case  3:        Digit = '3';    break;
        case  4:        Digit = '4';    break;
        case  5:        Digit = '5';    break;
        case  6:        Digit = '6';    break;
        case  7:        Digit = '7';    break;
        case  8:        Digit = '8';    break;
        case  9:        Digit = '9';    break;
        case 10:        Digit = '0';    break;
        case 16:        Digit = 'R';    break;
        default:        Digit = '?';    break;
    }

    // Assemble the message and print it
    WriteDiagMsg (FormatStr (LoadAppMsg (MsgNum).GetStr (), Dev+21, Digit));

    // Remember the digit, but only if we don't have already a connection
    DevStateInfo* DS = DevState.GetDevStateInfo (Dev);
    if (Digit != 'R' && ((DS->GetState (stAmt)) == 0 || DS->HasExt == 0)) {
        DS->CallPhoneTmp += (char) Digit;
    }
}



static void WriteDiagStatus (unsigned MsgNum, int State)
{
    String OnOff = LoadAppMsg (State? msOn : msOff);
    WriteDiagMsg (FormatStr (LoadAppMsg (MsgNum).GetStr (), OnOff.GetStr ()));
}



static void SetMatrix (unsigned Col, unsigned Dev, int State)
{
    // Check which matrix column to set
    switch (Col) {

        case 0:
            SetAmt (Dev, State, 1);
            break;

        case 1:
            SetAmt (Dev, State, 2);
            break;

        case 2:
            SetInt (Dev, State, 1);
            break;

        case 3:
            SetInt (Dev, State, 2);
            break;

        case 4:
            SetInt (Dev, State, 3);
            break;

        case 5:
            SetTon (Dev, State);
            break;

        case 6:
            SetWTon (Dev, State);
            break;

        case 7:
            SetTFE (Dev, State);
            break;

        default:
            WriteDiagMsg (FormatStr ("Unknown matrix col: %d", Col));
            break;

    }

    // Update the matrix column
    RedrawMatrix ();
}



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



void DoneDiagWin ()
// Delete the istec diag window
{
    delete MatrixWin;
    MatrixWin = NULL;
    MatrixColl = NULL;
    MatrixBox = NULL;
}



void InitDiagWin (unsigned DevCount)
// Show the istec diag window
{
    // Delete old diag window
    DoneDiagWin ();

    // Load the matrix window from the resource, get the width
    MatrixWin = (Menue*) LoadResource ("@ICDIAG.MatrixWin");
    unsigned Width = MatrixWin->IXSize ();

    // Position the window to the lower left corner of the screen
    Rect Bounds = Background->GetDesktop ();
    Rect C = MatrixWin->OuterBounds ();
    MatrixWin->MoveRel (Point (Bounds.A.X, Bounds.B.Y) - Point (C.A.X, C.B.Y));

    // Create a string collection to hold the entries for the listbox
    MatrixColl = new Collection<String> (DevCount, 0, 1);

    // Insert an entry for every device
    for (unsigned Dev = 0; Dev < DevCount; Dev++) {

        // Get a device entry
        DevStateInfo* DS = DevState.GetDevStateInfo (Dev);

        // Create an empty string for the matrix
        String* DevStr = new String (Width);
        DevStr->Set (0, Width, ' ');
        DevStr->Replace (1, U32Str (Dev + 21));
        DevStr->Replace (colSchleife, DS->GetState (stSchleife) ? '*' : ' ');
        DevStr->Replace (colAmt1, DS->GetState (stAmt1) ? 'x' : '-');
        DevStr->Replace (colAmt2, DS->GetState (stAmt2) ? 'x' : '-');
        DevStr->Replace (colInt1, DS->GetState (stInt1) ? 'x' : '-');
        DevStr->Replace (colInt2, DS->GetState (stInt2) ? 'x' : '-');
        DevStr->Replace (colInt3, DS->GetState (stInt3) ? 'x' : '-');
        DevStr->Replace (colTon, DS->GetState (stTon) ? 'x' : '-');
        DevStr->Replace (colWTon, DS->GetState (stWTon) ? 'x' : '-');
        DevStr->Replace (colTFE, DS->GetState (stTFE) ? 'x' : '-');
        DevStr->Replace (colRuf, DS->GetState (stRuf) ? 'x' : '-');

        // Insert the new string
        MatrixColl->Insert (DevStr);

    }

    // Create a listbox inside the window
    Point Size;
    Size.X = Width;
    Size.Y = MatrixWin->IYSize () - 1;
    MatrixBox = new MatrixListBox (1, Size);
    MatrixBox->SetColl (MatrixColl);
    MatrixWin->AddItem (MatrixBox);
    MatrixBox->SetPos (0, 1);
    MatrixBox->Draw ();

    // Show the matrix window
    MatrixWin->Show ();
}



void DoneStatusWin ()
// Delete the istec status window
{
    delete MsgWin;
    MsgWin = NULL;
}



void InitStatusWin ()
// Show the istec status window
{
    // Delete old diag status window
    DoneStatusWin ();

    // Create a new debug window
    Rect Bounds = Background->GetDesktop ();
    Bounds.A.X = Bounds.B.X - 30;
    MsgWin = new Window (Bounds, wfFramed);

    // Set the window header and show the window
    MsgWin->SetHeader (LoadAppMsg (msMsgWinHeader));
    MsgWin->Show ();
}



void DoneCallWin ()
// Delete the window for outgoing calls
{
    delete CallWin;
    CallWin = NULL;
}



void InitCallWin ()
// Show the window for outgoing calls
{
    // Get the size of the matrix window
    unsigned Height;
    if (MatrixWin) {
        Height = MatrixWin->OYSize ();
    } else {
        // OOPS - load it temporarily
        Menue* Win = (Menue*) LoadResource ("@ICDIAG.MatrixWin");
        Height = Win->OYSize ();
        delete Win;
    }

    // Create a call window
    Rect Bounds = Background->GetDesktop ();
    Bounds.B.X -= 31;
    Bounds.B.Y -= Height + 1;
    CallWin = new Window (Bounds, wfFramed);

    // Set the window header and show the window
    CallWin->SetHeader (LoadAppMsg (msCallWinHeader));
    CallWin->Show ();
}



void HandleDiagMsg (const unsigned char* Msg)
// Handle a diagnostic message from the istec
{
    // Check the event type
    String S;
    switch (Msg [1]) {

        case 0x02:
            // connection matrix
            SetMatrix (Msg [2], Msg [3], Msg [4]);
            break;

        case 0x03:
            // call
            SetRuf (Msg [2], Msg [4]);
            RedrawMatrix ();
            break;

        case 0x04:
            // Schleifenzustand
            SetSchleife (Msg [2], Msg [4]);
            RedrawMatrix ();
            break;

        case 0x05:
            // Pulse dial
            WriteDialStatus (msPulseDial, Msg [2], Msg [4]);
            break;

        case 0x06:
            // Tone dial
            WriteDialStatus (msToneDial, IntCon [Msg [2]], Msg [4]);
            break;

        case 0x07:
            // LED state
            WriteDiagStatus (msLEDState, Msg [4]);
            break;

        case 0x08:
            // Charges
            WriteDiagStatus (msChargeState, Msg [4]);
            break;

        case 0x09:
            // TFE amplifier
            WriteDiagStatus (msTFEState, Msg [4]);
            break;

        case 0x0A:
            // Door opener
            WriteDiagStatus (msDoorState, Msg [4]);
            break;

        case 0x0D:
            // Switch
            WriteDiagStatus (msSwitchState, Msg [4]);
            break;

        default:
            // Unknown debug message!
            S = FormatStr ("%02x %02x %02x %02x %02x",
                           Msg [1], Msg [2], Msg [3], Msg [4], Msg [5]);
            WriteDiagMsg ("OOPS: " + S);
            WriteDebugLog ("Warning: Got unknown diagnostic message: " + S);
            break;

    }
}



void BrowseMatrixWin ()
// Allow browsing and moving for the matrix window
{
    // The matrix window must exist
    CHECK (MatrixWin != NULL);

    // New status line
    PushStatusLine (siEnd);

    // Activate the matrix window
    MatrixBox->Select ();
    MatrixWin->Activate ();

    // Allow choosing an entry
    int Done = 0;
    while (!Done) {

        // Get keyboard input
        Key K = ::KbdGet ();

        // Let the box look for a useful key
        MatrixBox->HandleKey (K);

        // Look what's left
        switch (K) {

            case vkResize:
                MatrixWin->MoveResize ();
                break;

            case vkAccept:
            case vkAbort:
                Done = 1;
                break;

        }

    }

    // Deactivate the matrix window
    MatrixBox->Deselect ();
    MatrixWin->Deactivate ();

    // Restore the status line
    PopStatusLine ();
}



void HandleCallQueue ()
// Handle messages in the call queue
{
    if (Queue.GetCount () > 0) {

        // We have a queue and there's something in it. Get first entry.
        DevStateInfo* DS = Queue.At (0);

        // Update the charges
        DS->CallCharges = Charges [DS->DevNum] - DS->StartCharges;

        // Print out a message in the window
        Time T = (DS->CallStart + DS->CallDuration);
        String TS = T.DateTimeStr (LoadAppMsg (msCallTimeFmt));
        WriteCallMsg (FormatStr (LoadAppMsg (msCallEnd).GetStr (),
                                 TS.GetStr (),
                                 DS->DevNum + 21,
                                 DS->CallPhone.GetStr (),
                                 DS->CallDuration.GetSec (),
                                 DS->CallCharges));

        // Log the message into the logfiles
        if (LogZeroCostCalls || DS->CallCharges > 0) {
            LogCall (DS->LogMsg (), DS->CallStart);
        }

        // Delete the entry
        Queue.AtDelete (0);

    }
}



