/*****************************************************************************/
/*                                                                           */
/*                                ICCONFIG.CC                                */
/*                                                                           */
/* (C) 1995     Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



// Please notice: All structs defined in this module must be char aligned
// internally. See isteccfg.txt (author unknown).



#include <string.h>

#include "check.h"

#include "icerror.h"
#include "icconfig.h"



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



// Version number of the ISTEC firmware
double FirmwareVersion  = 1.7;



/*****************************************************************************/
/*                          struct IstecBaseConfig                           */
/*****************************************************************************/



unsigned char* IstecBaseConfig::Pack (unsigned char* Buf) const
// Pack the data of struct IstecBaseConfig into an array ready for
// transmission. The function returns Buf.
{
    unsigned char* B = Buf;

    *B++ = Connection;
    *B++ = AB_InterfaceCount;
    *B++ = Protocol;
    *B++ = VersionHigh;
    *B++ = VersionLow;
    *B++ = TFEAssignment;
    for (unsigned I = 0; I < bcMSNCount; I++) {
        memcpy (B, MSN [I], sizeof (MSN [I]));
        B += sizeof (MSN [I]);
    }
    *B++ = Music;
    *B++ = IntS0;
    *B++ = ExtS0;
    *B++ = QueryLoc1;
    *B++ = QueryLoc2;
    memcpy (B, Number1, sizeof (Number1));
    B += sizeof (Number1);
    memcpy (B, Number2, sizeof (Number2));
    B += sizeof (Number2);
    memcpy (B, MSNGroups, sizeof (MSNGroups));

    // Return the prepared buffer
    return Buf;
}



IstecBaseConfig& IstecBaseConfig::Unpack (const unsigned char* Buf)
// Unpack an array of char that contains data for an IstecBaseConfig struct.
// The function returns this.
{
    Connection        = *Buf++;
    AB_InterfaceCount = *Buf++;
    Protocol          = *Buf++;
    VersionHigh       = *Buf++;
    VersionLow        = *Buf++;
    TFEAssignment     = *Buf++;
    for (unsigned I = 0; I < bcMSNCount; I++) {
        memcpy (MSN [I], Buf, sizeof (MSN [I]));
        Buf += sizeof (MSN [I]);
    }
    Music             = *Buf++;
    IntS0             = *Buf++;
    ExtS0             = *Buf++;
    QueryLoc1         = *Buf++;
    QueryLoc2         = *Buf++;
    memcpy (Number1, Buf, sizeof (Number1));
    Buf += sizeof (Number1);
    memcpy (Number2, Buf, sizeof (Number2));
    Buf += sizeof (Number2);
    memcpy (MSNGroups, Buf, sizeof (MSNGroups));

    // Return the prepared struct
    return *this;
}



int operator == (const IstecBaseConfig& lhs, const IstecBaseConfig& rhs)
{
    // For simplicity, convert to an array, then do a memcmp
    unsigned char Left [BaseConfigSize], Right [BaseConfigSize];
    return memcmp (lhs.Pack (Left), rhs.Pack (Right), BaseConfigSize) == 0;
}



int operator != (const IstecBaseConfig& lhs, const IstecBaseConfig& rhs)
{
    // For simplicity, convert to an array, then do a memcmp
    unsigned char Left [BaseConfigSize], Right [BaseConfigSize];
    return memcmp (lhs.Pack (Left), rhs.Pack (Right), BaseConfigSize) != 0;
}



/*****************************************************************************/
/*                              struct DevConfig                             */
/*****************************************************************************/



unsigned char* DevConfig::Pack (unsigned char* Buf) const
// Pack the data of struct DevConfig into an array ready for transmission.
// The function returns Buf.
{
    unsigned char* B = Buf;

    *B++ = DevNum;
    *B++ = DialCaps;
    *B++ = Service;
    *B++ = Reroute;
    *B++ = ChargePulse;
    *B++ = PIN [0];
    *B++ = PIN [1];
    memcpy (B, ExtNum, sizeof (ExtNum));

    // Return the prepared buffer
    return Buf;
}



DevConfig& DevConfig::Unpack (const unsigned char* Buf)
// Unpack an array of char that contains data for a DevConfig struct.
// The function returns this.
{
    DevNum          = *Buf++;
    DialCaps        = *Buf++;
    Service         = *Buf++;
    Reroute         = *Buf++;
    ChargePulse     = *Buf++;
    PIN [0]         = *Buf++;
    PIN [1]         = *Buf++;
    memcpy (ExtNum, Buf, sizeof (ExtNum));

    // Return the prepared struct
    return *this;
}



int operator == (const DevConfig& lhs, const DevConfig& rhs)
{
    // For simplicity, convert to an array, then do a memcmp
    unsigned char Left [DevConfigSize], Right [DevConfigSize];
    return memcmp (lhs.Pack (Left), rhs.Pack (Right), DevConfigSize) == 0;
}



int operator != (const DevConfig& lhs, const DevConfig& rhs)
{
    // For simplicity, convert to an array, then do a memcmp
    unsigned char Left [DevConfigSize], Right [DevConfigSize];
    return memcmp (lhs.Pack (Left), rhs.Pack (Right), DevConfigSize) != 0;
}



/*****************************************************************************/
/*                             class IstecCharges                            */
/*****************************************************************************/



void IstecCharges::Clear ()
// Clear the charges
{
    for (unsigned I = 0; I < IstecDevCount; I++) {
        Charges [I] = 0;
    }
}



unsigned& IstecCharges::operator [] (unsigned Device)
// Return a reference to the charges of a specific device
{
    PRECONDITION (Device < IstecDevCount);
    return Charges [Device];
}



const unsigned& IstecCharges::operator [] (unsigned Device) const
// Return a reference to the charges of a specific device
{
    PRECONDITION (Device < IstecDevCount);
    return Charges [Device];
}



unsigned char* IstecCharges::Pack (unsigned char* Buf) const
// Pack the data of IstecCharges into an array ready for transmission.
// The function returns Buf.
{
    unsigned char* B = Buf;

    for (unsigned I = 0; I < IstecDevCount; I++) {

        // Check range
        CHECK (Charges [I] <= 0xFFFF);

        // Get the value
        u16 Val = (u16) Charges [I];

        // Store the value into the array in little endian format
        *B++ = (unsigned char) Val;
        *B++ = (unsigned char) (Val / 256);

    }

    // Return the prepared buffer
    return Buf;
}



IstecCharges& IstecCharges::Unpack (const unsigned char* Buf)
// Unpack an array of char that contains data for an IstecCharges struct.
// The function returns this.
{
    for (unsigned I = 0; I < IstecDevCount; I++) {

        // Convert the value from little endian format
        Charges [I] = Buf [0] + Buf [1] * 256;

        // Next one
        Buf += 2;
    }

    // Return the prepared struct
    return *this;
}



int operator == (const IstecCharges& lhs, const IstecCharges& rhs)
{
    for (unsigned I = 0; I < IstecDevCount; I++) {
        if (lhs.Charges [I] != rhs.Charges [I]) {
            return 0;
        }
    }
    return 1;
}



int operator != (const IstecCharges& lhs, const IstecCharges& rhs)
{
    for (unsigned I = 0; I < IstecDevCount; I++) {
        if (lhs.Charges [I] == rhs.Charges [I]) {
            return 0;
        }
    }
    return 1;
}



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



unsigned char* ToBCD (const char* S, unsigned char* T, unsigned TSize)
// Convert the number string in S to a BCD representation in T, filling unused
// digits with 'F'. The function returns T.
{
    // Check the parameters
    PRECONDITION (S != NULL && T != NULL && TSize > 0);

    // Calculate the maximum digit count and make shure, T is big enough
    unsigned Count = TSize * 2;
    unsigned Len   = strlen (S);
    if (Len > Count) {
        // OOPS - S is too long, what now?
        IstecError (msErrorIntIgnored);
        Len = Count;
    }

    // Pre-fill the target string with FF
    memset (T, 0xFF, TSize);

    // Convert to BCD and store
    int LowNibble = 1;
    unsigned char* P = T;
    while (Len--) {

        if (LowNibble) {
            *P = (*S - '0') | 0xF0;
            LowNibble = 0;
        } else {
            *P = ((*S - '0') << 4) | (*P & 0x0F);
            P++;
            LowNibble = 1;
        }
        S++;
    }

    // Return the target buffer
    return T;
}



char* FromBCD (char* S, const unsigned char* T, unsigned TSize)
// Convert the BCD string in T into an ASCII representation in S. Conversion
// stops if an invalid BCD char or TSize is reached. A trailing zero is added
// to S. It is assumed that S is big anough to hold the resulting string (S
// must be TSize*2+1 chars in size).
// The function returns S.
{
    // Check the parameters
    PRECONDITION (S != NULL && T != NULL && TSize > 0);

    // Calculate the nibble count
    TSize *= 2;

    // Conversion loop
    char* Q = S;
    unsigned LowNibble = 1;
    while (TSize--) {

        int C;
        if (LowNibble) {
            C = *T & 0x0F;
            LowNibble = 0;
        } else {
            C = (*T & 0xF0) >> 4;
            T++;
            LowNibble = 1;
        }

        if (C < 10) {
            *Q++ = '0' + C;
        } else {
            // Invalid char, end reached
            break;
        }
    }

    // Add a trailing zero
    *Q = '\0';

    // return the result
    return S;
}



unsigned char* ToPascal (const char* S, unsigned char* T, unsigned TSize)
// Convert the C style string in S to the pascal string in T and return T.
{
    // Check parameters
    PRECONDITION (S != NULL && T != NULL && TSize > 0);

    // Copy the string
    unsigned char* Q = T + 1;
    unsigned Length = 0;
    while (*S && TSize--) {
        *Q++ = *S++;
        Length++;
    }

    // Pad the target string with zeros
    while (TSize--) {
        *Q++ = '\0';
    }

    // Set the string length
    *T = Length;

    // Return the result
    return T;
}



char* FromPascal (char* S, const unsigned char* T, unsigned SSize)
// Convert the pascal style string in T into a C like string in S and return S.
{
    // Check parameters
    PRECONDITION (S != NULL && T != NULL && SSize > 0);

    // Get the string length and restrict it if S is not big enough
    unsigned Length = *T++;
    if (Length > SSize-1) {
        Length = SSize-1;
    }

    // Copy the string
    char* Q = S;
    while (Length--) {
        *Q++ = *T++;
    }

    // Add a trailing zero
    *Q = '\0';

    // Return the result
    return S;
}



