/*****************************************************************************/
/*                                                                           */
/*                                ICBASEED.CC                                */
/*                                                                           */
/* (C) 1995     Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include "cont.h"
#include "progutil.h"
#include "menue.h"
#include "menuitem.h"
#include "statline.h"
#include "stdmenue.h"
#include "stdmsg.h"

#include "icmsg.h"
#include "icerror.h"
#include "icconfig.h"
#include "icident.h"



/*****************************************************************************/
/*                             Message constants                             */
/*****************************************************************************/



const u16 msExtLevel            = MSGBASE_ICBASEED +  0;
const u16 msNumberTooLong       = MSGBASE_ICBASEED +  1;



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



static void EditQueryLoc (IstecBaseConfig& Cfg, int& Changed)
{
    // Remember the old config
    IstecBaseConfig OldCfg = Cfg;

    // Load the menue
    Menue* Win = (Menue*) LoadResource ("@ICBASEED.Abfragestellen-Menue");

    // Get pointers to the items
    LongItem* QL1 = (LongItem*) Win->ForcedItemWithID (1);
    LongItem* QL2 = (LongItem*) Win->ForcedItemWithID (2);

    // Transfer data into the menue
    QL1->SetValue (Cfg.QueryLoc1);
    QL2->SetValue (Cfg.QueryLoc2);

    // Set the limits
    QL1->SetMinMax (21, 20 + Cfg.AB_InterfaceCount);
    QL2->SetMinMax (21, 20 + Cfg.AB_InterfaceCount);

    // Activate the menue
    Win->Activate ();

    // New status line
    PushStatusLine (siAbort | siSelectKeys | siChange | siAccept);

    // Edit the menue
    int Done = 0;
    while (!Done) {

        // Get a selection
        switch (Win->GetChoice ()) {

            case 1:
                Cfg.QueryLoc1 = QL1->GetValue ();
                break;

            case 2:
                Cfg.QueryLoc2 = QL2->GetValue ();
                break;

            case 0:
                // Check the abort key
                if (Win->GetAbortKey () == vkAbort) {
                    // Abort - ask if we have changes
                    if (Cfg != OldCfg) {
                        // We have changes
                        if (AskDiscardChanges () == 2) {
                            // Discard changes
                            Cfg = OldCfg;
                            Done = 1;
                        }
                    } else {
                        // No changes
                        Done = 1;
                    }
                } else if (Win->GetAbortKey () == vkAccept) {
                    // Accept the changes, set the Changed flag
                    if (Cfg != OldCfg) {
                        // We had changes
                        Changed = 1;
                    }
                    Done = 1;
                }
                break;

        }
    }

    // Pop statusline, delete the menue
    PopStatusLine ();
    delete Win;
}



static void EditMSN (IstecBaseConfig& Config, int& Changed)
// Edit the MSN's
{
    // Remember the old config
    IstecBaseConfig OldConfig = Config;

    // Load the menue
    Menue* Win = (Menue*) LoadResource ("@ICBASEED.MSN-Eingabemenue");

    // Transfer data into the menue
    char Buf [100];
    for (unsigned I = 0; I < 10; I++) {
        Win->SetStringValue (I+1, FromBCD (Buf, Config.MSN [I], sizeof (Config.MSN [I])));
    }

    // Activate the menue
    Win->Activate ();

    // New status line
    PushStatusLine (siAbort | siSelectKeys | siChange | siAccept);

    // Edit the menue
    int Done = 0;
    while (!Done) {

        // Get a selection
        int Choice = Win->GetChoice ();

        //
        const char* S;
        switch (Choice) {

            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
            case 9:
            case 10:
                S = Win->GetStringValue (Choice).GetStr ();
                ToBCD (S, Config.MSN [Choice-1], sizeof (Config.MSN [0]));
                break;

            case 0:
                if (Win->GetAbortKey () == vkAbort) {
                    // Abort - ask if we have changes
                    if (Config != OldConfig) {
                        // We have changes
                        if (AskDiscardChanges () == 2) {
                            // Discard changes
                            Config = OldConfig;
                            Done = 1;
                        }
                    } else {
                        // No changes
                        Done = 1;
                    }
                } else if (Win->GetAbortKey () == vkAccept) {
                    // Accept the changes, set the Changed flag
                    if (Config != OldConfig) {
                        // We had changes
                        Changed = 1;
                    }
                    Done = 1;
                }
                break;

        }

    }

    // Pop statusline, delete the menue
    PopStatusLine ();
    delete Win;
}



static void EditMSNGroups (IstecBaseConfig& Cfg, int& Changed, const String& ResName)
// Edit the MSN or EAZ grouping
{
    // Load the menue
    Menue* M = (Menue*) LoadResource (ResName);

    // Make a copy of the MSN groups
    unsigned char OldGroups [sizeof (Cfg.MSNGroups)];
    memcpy (OldGroups, Cfg.MSNGroups, sizeof (OldGroups));

    // Set up the menue
    char Buf [50];
    for (unsigned I = 0; I < 10; I++) {
        // Set the MSN if the item for the MSN is available (otherwise it is
        // the EAZ grouping menue which has no MS numbers to show)
        StringItem* Item = (StringItem*) M->ItemWithID (1000 + I);
        if (Item != NULL) {
            Item->SetValue (FromBCD (Buf, Cfg.MSN [I], sizeof (Cfg.MSN [I])));
        }

        // Set the grouping
        unsigned Mask = 0x01;
        for (unsigned J = 0; J < 8; J++) {
            M->SetToggleValue (I*16+J*2+1, (Cfg.MSNGroups [I] & Mask) != 0);
            Mask <<= 1;
        }
    }

    // New statusline
    PushStatusLine (siEnd | siSelectKeys | siChange);

    // Activate the menue
    M->Activate ();

    // Allow editing
    int Done = 0;
    while (!Done) {

        // Get a selection
        unsigned Sel = M->GetChoice ();

        // Check for abort
        if (Sel == 0) {
            if (M->GetAbortKey () == vkAbort) {
                // Abort - ask if we have changes
                if (memcmp (OldGroups, Cfg.MSNGroups, sizeof (OldGroups)) != 0) {
                    // We have changes
                    if (AskDiscardChanges () == 2) {
                        // Discard changes
                        memcpy (Cfg.MSNGroups, OldGroups, sizeof (OldGroups));
                        Done = 1;
                    }
                } else {
                    // No changes
                    Done = 1;
                }
            } else if (M->GetAbortKey () == vkAccept) {
                // Accept the changes, set the Changed flag
                if (memcmp (OldGroups, Cfg.MSNGroups, sizeof (OldGroups)) != 0) {
                    // We have changes
                    Changed = 1;
                }
                Done = 1;
            }
            break;

        }

        unsigned MSN = (Sel - 1) / 16;
        unsigned Mask = 0x01 << (((Sel - 1) % 16) / 2);

        if (Sel & 0x0001)  {
            // Switch off
            Cfg.MSNGroups [MSN] &= ~Mask;
        } else {
            // Switch on
            Cfg.MSNGroups [MSN] |= Mask;
        }

    }

    // Old status line
    PopStatusLine ();

    // Discard the menue
    delete M;
}



void LoadConfigDefault (IstecBaseConfig& BaseConfig,
                        DevConfig* DevCfg,
                        unsigned IstecType)
// Load the default parameter settings for the given istec type.
{
    // Construct the resource name
    String ResName = FormatStr ("@ICBASEED.Default-%u", IstecType);

    // Load the container with the default data
    Container* C = (Container*) LoadResource (ResName);

    // Get a pointer to the data. Cast to char*.
    const char* Data = (const char*) C->GetData ();

    // Copy the data for both config structs
    memcpy (&BaseConfig, Data, sizeof (IstecBaseConfig));
    Data += sizeof (BaseConfig);
    memcpy (DevCfg, Data, sizeof (IstecDevConfig));

    // Discard the container
    delete C;
}



static unsigned NewIstecType (WindowItem* Item)
// Get a new istec type
{
    // Load the menue
    Menue* M = (Menue*) LoadResource ("@ICBASEED.TypeMenue");

    // Place it near the item
    M->PlaceNear (Item);

    // Get a choice
    unsigned Sel = M->GetChoice ();

    // Delete the menue
    delete M;

    // Return the selection
    return Sel;
}



void EditBaseConfig (IstecBaseConfig& BaseCfg,
                     DevConfig* DevCfg,
                     int IstecPresent,
                     int& Changed)
// Allow editing the istec base configuration. If the istec is not present,
// the type of the istec can be changed, resulting in a load of the default
// for this istec type. If Config has been changed, Changed is set to 1,
// otherwise, Changed is left untouched.
{
    // ID's of menue items for EditBaseConfig
    const miIstecType           =  10;
    const miProtocol            =  20;
    const miExtLevel            =  30;
    const miMusic               =  40;
    const miConnection          =  50;
    const miNumber1             =  60;
    const miNumber2             =  70;
    const miTFEAssignment       =  80;
    const miQueryLoc            =  90;
    const miMSN                 = 100;
    const miMSNGroups           = 110;
    const miEAZGroups           = 120;



    // Remember the configuration
    IstecBaseConfig OldBaseCfg = BaseCfg;
    IstecDevConfig OldDevCfg;
    memcpy (OldDevCfg, DevCfg, sizeof (OldDevCfg));

    // New status line
    PushStatusLine (siAbort | siSelectKeys | siChange | siAccept);

    // Loop ...
    int Done = 0;
    int NeedConfig = 1;
    unsigned IstecType = 0;     // Initialize it to make gcc happy
    Menue* Win = NULL;
    while (!Done) {

        if (NeedConfig) {

            // Delete the menue if present and reload it
            delete Win;
            Win = (Menue*) LoadResource ("@ICBASEED.BaseConfigMenue");

            // Get the type of the istec
            IstecType = IdentifyIstec (BaseCfg.AB_InterfaceCount,
                                       BaseCfg.ExtS0,
                                       BaseCfg.IntS0);

            // Transfer data into the menue
            char Buf [100];
            Win->SetStringValue (miIstecType, GetIstecName (IstecType));
            Win->SetToggleValue (miProtocol, BaseCfg.Protocol != pr1TR6);
            Win->SetToggleValue (miMusic, BaseCfg.Music != 0);
            Win->SetToggleValue (miConnection, BaseCfg.Connection != coMehrgeraete);
            Win->SetStringValue (miNumber1, FromPascal (Buf, BaseCfg.Number1, sizeof (Buf)));
            Win->SetStringValue (miNumber2, FromPascal (Buf, BaseCfg.Number2, sizeof (Buf)));
            Win->SetLongValue   (miTFEAssignment, BaseCfg.TFEAssignment);

            // Enable/disable menue items dependent on the system
            if (IstecPresent) {
                Win->GrayItem (miIstecType);
            }
            if (BaseCfg.Protocol == pr1TR6) {
                Win->GrayItem (miMSN);
                Win->GrayItem (miMSNGroups);
            } else if (BaseCfg.Protocol == prDSS1) {
                Win->GrayItem (miEAZGroups);
            }

            switch (IstecType) {

                case 1003:
                case 1008:
                    if (IstecPresent) {
                        // Protocol is hard wired in the istec
                        Win->GrayItem (miProtocol);
                    }
                    Win->GrayItem (miExtLevel);
                    if (BaseCfg.Connection == coMehrgeraete) {
                        Win->GrayItem (miNumber1);
                    }
                    Win->GrayItem (miNumber2);
                    Win->GrayItem (miQueryLoc);
                    break;

                case 1016:
                    Win->GrayItem (miConnection);
                    Win->GrayItem (miNumber2);
                    break;

                case 1024:
                    Win->GrayItem (miExtLevel);
                    Win->GrayItem (miConnection);
                    Win->GrayItem (miNumber2);
                    break;

                case 2016:
                    Win->GrayItem (miConnection);
                    break;

                case 2024:
                    Win->GrayItem (miExtLevel);
                    Win->GrayItem (miConnection);
                    break;

                case 2400:
                    Win->GrayItem (miExtLevel);
                    if (BaseCfg.Connection == coMehrgeraete) {
                        Win->GrayItem (miNumber1);
                        Win->GrayItem (miNumber2);
                    }
                    break;

                case 2416:
                    Win->GrayItem (miConnection);
                    break;

                case 2424:
                    Win->GrayItem (miExtLevel);
                    Win->GrayItem (miConnection);
                    break;

            }

            // Activate the menue
            Win->Activate ();

            // Config done
            NeedConfig = 0;

        }

        // Get a selection
        int Choice = Win->GetChoice ();

        //
        String Num;
        unsigned NewType;
        switch (Choice) {

            case miIstecType:
                NewType = NewIstecType (Win->ItemWithID (miIstecType));
                if (NewType != 0 && NewType != IstecType) {
                    // Got a new selection, load the default
                    NeedConfig = 1;
                    LoadConfigDefault (BaseCfg, DevCfg, NewType);
                }
                break;

            case miProtocol:
                BaseCfg.Protocol = pr1TR6;
                Win->GrayItem (miMSN);
                Win->GrayItem (miMSNGroups);
                Win->ActivateItem (miEAZGroups);
                break;

            case miProtocol+1:
                BaseCfg.Protocol = prDSS1;
                Win->ActivateItem (miMSN);
                Win->ActivateItem (miMSNGroups);
                Win->GrayItem (miEAZGroups);
                break;

            case miExtLevel:
                InformationMsg (LoadAppMsg (msExtLevel));
                break;

            case miMusic:
                BaseCfg.Music = 0;
                break;

            case miMusic+1:
                BaseCfg.Music = 1;
                break;

            case miConnection:
                BaseCfg.Connection = coMehrgeraete;
                Win->GrayItem (miNumber1);
                Win->GrayItem (miNumber1);
                break;

            case miConnection+1:
                BaseCfg.Connection = coAnlagen;
                Win->ActivateItem (miNumber1);
                if (BaseCfg.ExtS0 > 1) {
                    Win->ActivateItem (miNumber2);
                }
                break;

            case miNumber1:
                Num = Win->GetStringValue (miNumber1);
                ToPascal (Num.GetStr (), BaseCfg.Number1, sizeof (BaseCfg.Number1));
                break;

            case miNumber2:
                Num = Win->GetStringValue (miNumber2);
                ToPascal (Num.GetStr (), BaseCfg.Number2, sizeof (BaseCfg.Number2));
                break;

            case miTFEAssignment:
                BaseCfg.TFEAssignment = Win->GetLongValue (miTFEAssignment);
                break;

            case miQueryLoc:
                EditQueryLoc (BaseCfg, Changed);
                break;

            case miMSN:
                EditMSN (BaseCfg, Changed);
                break;

            case miMSNGroups:
                EditMSNGroups (BaseCfg, Changed, "@ICBASEED.MSN-GruppenMenue");
                break;

            case miEAZGroups:
                EditMSNGroups (BaseCfg, Changed, "@ICBASEED.EAZ-GruppenMenue");
                break;

            case 0:
                if (Win->GetAbortKey () == vkAbort) {
                    // Abort - ask if we have changes
                    if (BaseCfg != OldBaseCfg ||
                        memcmp (OldDevCfg, DevCfg, sizeof (OldDevCfg)) != 0) {
                        // We have changes
                        if (AskDiscardChanges () == 2) {
                            // Discard changes
                            BaseCfg = OldBaseCfg;
                            memcpy (DevCfg, OldDevCfg, sizeof (OldDevCfg));
                            Done = 1;
                        }
                    } else {
                        // No changes
                        Done = 1;
                    }
                } else if (Win->GetAbortKey () == vkAccept) {
                    // Accept the changes, set the Changed flag
                    if (BaseCfg != OldBaseCfg ||
                        memcmp (OldDevCfg, DevCfg, sizeof (OldDevCfg)) != 0) {
                        // We have changes
                        Changed = 1;
                    }
                    Done = 1;
                }
                break;

        }

    }

    // Delete the menue, restore old statusline
    delete Win;
    PopStatusLine ();

}

