// ===================================================================
// sphflake.cpp
//	Implementation file for SphereFlake class.
//	The code for this file is derived from balls.c in Eric
//	Haines's public domain Standard Procedural Database (SPD).
//	For more information on the SPD, see his article "A Proposal 
//	For Standard Graphics Environments" in the November 1987 
//	issue of IEEE Computer Graphics & Applications.
//
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "utils.h"

// Declare some globals that are static within the class.
Vector3D SphereFlake::objset[9];
int SphereFlake::objset_computed = 0;

void
SphereFlake::CreateObjSet(Vector3D passbk[9])
{
    int num_set, num_vert;
    Vector3D axis, temp_pt, trio_dir[3];
    Matrix mx;

    double dist = 1 / sqrt(2);
    trio_dir[0] = Vector3D(dist, dist, 0);
    trio_dir[1] = Vector3D(dist, 0, -dist);
    trio_dir[2] = Vector3D(0, dist, -dist);
    axis = Normalize(Vector3D(1, -1, 0));
    mx = RotationAxisMatrix(axis, asin(2.0 / sqrt(6)));
    for (num_vert = 0; num_vert < 3; num_vert++)
        trio_dir[num_vert] = mx * trio_dir[num_vert];
    for (num_set = 0; num_set < 3; num_set++) {
        mx = RotationZMatrix(num_set*2*M_PI/3);
        for (num_vert = 0; num_vert < 3; num_vert++)
            objset[3*num_set+num_vert] = mx * trio_dir[num_vert];
    }
}

void
SphereFlake::AddSphere(const Vector3D& center, float rad, const Vector3D& dir, int depth, Surface *surf)
{
    if (! depth--)
        return;
    AddObject(new Sphere(center, rad, surf));
    Matrix mulby;

    if (dir.z >= 1)
        mulby = IdentityMatrix();
    else if (dir.z <= -1)
        mulby = RotationYMatrix(M_PI);
    else {
	Vector3D axis = Normalize(CrossProd(Vector3D(0, 0, 1), dir));
	float ang = acos(DotProd(axis, Vector3D(0, 0, 1)));
        mulby = RotationAxisMatrix(axis, ang);
    }
    double scale = rad * (1.0 + 1.0/3.0);
    for (int num_vert = 0; num_vert < 9; num_vert++) {
	Vector3D child_pt = mulby * objset[num_vert] * scale + center;
	Vector3D child_dir = (child_pt - center) / scale;
        AddSphere(child_pt, rad / 3, child_dir, depth, surf);
    }
}

SphereFlake::SphereFlake(const Vector3D& center, float rad, const Vector3D& dir, int depth, Surface *surf)
{
    if (! objset_computed) {
        CreateObjSet(objset);
        objset_computed = 1;
    }
    AddSphere(center, rad, dir, depth, surf);
}
