// ===================================================================
// sphere.cpp
//	Sphere support routines for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

// Sphere constructor.	Takes center, radius, color.
Sphere::Sphere(const Vector3D& C, float R, Surface *clr):
  Object3D(clr, 0), center(C), radius(R)
{
    radiussq = radius*radius;
}

// Sphere intersection routine.
int
Sphere::NearestInt(Ray& ray, float& t, float maxt)
{
    float b, c, d;
    float t1, t2;
    Vector3D diff;

    Statistics::Intersections::Sphere++;

    diff = center - ray.loc;
    c = (float) DotProd(diff, diff) - radiussq;
    b = -2 * DotProd(diff, ray.dir);
    d = b*b - 4*c;

    if (d <= 0)
	return 0;

    d = sqrt(d);
    t1 = (-b+d)/2.0;
    t2 = (-b-d)/2.0;

    if (t1 < Limits::Threshold || t1 >= maxt)
	t1 = -1;
    if (t2 < Limits::Threshold || t2 >= maxt)
	t2 = -1;
    if (t1 < 0) {
	if (t2 < 0)
	    return 0;
	t = t2;
    }
    else {
	if (t2 < 0)
	    t = t1;
	else
	    t = (t1 < t2) ? t1 : t2;
    }
    if (t2 < t1) {
	float temp = t1;
	t1 = t2;
	t2 = temp;
    }
    return t > Limits::Threshold;
}

SpanList *
Sphere::FindAllIntersections(Ray& ray)
{
    float b, c, d;
    float t1, t2;
    Vector3D diff;

    Statistics::Intersections::Sphere++;

    diff = center - ray.loc;
    c = (float) DotProd(diff, diff) - radiussq;
    b = -2 * DotProd(diff, ray.dir);
    d = b*b - 4*c;

    if (d <= 0)
	return 0;

    d = sqrt(d);
    t1 = (-b+d)/2.0;
    t2 = (-b-d)/2.0;

    // If both parameters are less than threshold, return no-intersection.
    if (t1 < Limits::Threshold && t2 < Limits::Threshold)
	return 0;

    if (t1 > t2)
	Swap(t1, t2);
    if (t1 < 0) {
	if (IsInside(ray.loc))
	    return new SpanList(Span(t1, this, t2, this));
	else
	    return new SpanList(Span(t2, this, Limits::Infinity, this));
    }
    return new SpanList(Span(t1, this, t2, this));
}

// Sphere normal-finding routine.
Vector3D
Sphere::FindNormal(const Vector3D& intersection)
{
    return(Normalize(intersection - center));
}

// Apply transform to sphere.  Only center point is affected.
void
Sphere::ApplyTransform(const Matrix& tform)
{
    center = tform * center;
}

// Duplicate sphere.
Object3D *
Sphere::Dup() const
{
    return(new Sphere(center, radius, surf));
}

int
Sphere::IsInside(Vector3D& v)
{
    Vector3D diff = v - center;
    return Magnitude(diff) < radiussq;
}

float
Sphere::PtDistance(Vector3D& v)
{
    Vector3D diff = v - center;
    return DotProd(diff, diff) - radiussq;
}

// Describe sphere.  Indent spaces first.
void
Sphere::Describe(int ind) const
{
    indent(ind);
    cout << "Sphere: center at " << center << ", radius " << radius << ".\n";
}

AxisAlignedBox
Sphere::BBox() const
{
    return AxisAlignedBox(Vector3D(center - Vector3D(radius)),
			  Vector3D(center + Vector3D(radius)));
}
