// ===================================================================
// noise.cpp
//	Solid noise implementations.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

// -------------------------------------------------------------------
// NoiseMakers:
//	PerlinNoise
// -------------------------------------------------------------------

// DNoise (defined in header), Turbulence and DTurbulence are
// invariant with the type of noise.

// Turbulence: Sum noise values scaled by progressively smaller
//	       values until the scale value goes under 0.1.
float
NoiseMaker::Turbulence(const Vector3D& p)
{
    float scale;
    float value;
    float ret;

    ret = 0;
    for (scale = 1.0; scale > 0.1; scale *= 0.5) {
	value = Noise(p / scale);
	ret += fabs(value) * scale;
    }
    return ret;
}

// DTurbulence: Vector3D-valued turbulence function.
Vector3D
NoiseMaker::DTurbulence(const Vector3D& p)
{
    Vector3D value;
    Vector3D ret;
    float scale;

    ret = Vector3D(0);
    for (scale = 1.0; scale >= 0.01; scale *= 0.5) {
	value = DNoise(p / scale);
	ret += value * scale;
    }
    return ret;
}


// -------------------------------------------------------------------
// WhiteNoise: simple linear-interpolating
// -------------------------------------------------------------------

WhiteNoise::WhiteNoise(int NumValues)
{
    n = NumValues;
    P = new int[n];
    R = new float[n];
    for (int i = 0; i < n; i++) {
	P[i] = i;
	R[i] = RandomValue(0, 1);
    }
    for (i = 0; i < n; i++)
	Swap(P[i], P[rand() % n]);
}

WhiteNoise::~WhiteNoise()
{
    delete[] P;
    delete[] R;
}

float
WhiteNoise::Noise(const Vector3D& p)
{
    long fx, fy, fz;
    float xfact, yfact, zfact;
    float ret = 0;

    fx = p.x;
    fy = p.y;
    fz = p.z;
    xfact = fabs(p.x - fx);
    yfact = fabs(p.y - fy);
    zfact = fabs(p.z - fz);
    for (long i = fx; i <= fx + 1; i++, xfact = 1 - xfact) {
    	for (long j = fy; j <= fy + 1; j++, yfact = 1 - yfact) {
	    for (long k = fz; k <= fz + 1; k++, zfact = 1 - zfact) {
		float noiseval = R[Phi(i+Phi(j+Phi(k)))];
		ret += noiseval * xfact * yfact * zfact;
	    }
	}
    }
    return ret;
}

// -------------------------------------------------------------------
// The PerlinNoise class implements the noise function defined in
// Ken Perlin's article "Hypertexture" on pg. 253 of the SIGGRAPH '89
// Proceedings.
// -------------------------------------------------------------------

void
PerlinNoise::Initialize()
{
    int i, j;

    P = new int[NUM_NOISE_POINTS];
    G = new Vector3D[NUM_NOISE_POINTS];
    for (i = 0; i < NUM_NOISE_POINTS; i++)
	P[i] = i;
    for (i = 0; i < NUM_NOISE_POINTS; i++) {
	int inx = rand() % NUM_NOISE_POINTS;
	int temp = P[inx];
	P[inx] = P[i];
	P[i] = temp;
    }
    for (i = 0; i < NUM_NOISE_POINTS; i++) {
	G[i] = Vector3D(1);
	while (Magnitude(G[i]) > 1) {
	    for (j = 0; j < 3; j++) {
		int num = rand();
		if (num & (RAND_MAX+1 >> 1))
		    num |= RAND_MAX + 1;
		G[i][j] = (float) num / RAND_MAX;
	    }
	    G[i] = Normalize(G[i]);
	}
    }
}

PerlinNoise::PerlinNoise()
{
    Initialize();
}

PerlinNoise::PerlinNoise(int seed)
{
    srand(seed);
    Initialize();
}

PerlinNoise::~PerlinNoise()
{
    delete P;
    delete G;
}

float
PerlinNoise::Noise(const Vector3D& p)
{
    long fx, fy, fz;
    long i, j, k;
    float subt;

    fx = p.x;
    fy = p.y;
    fz = p.z;

    subt = 0;
    for (i = fx; i <= fx + 1; i++)
	for (j = fy; j <= fy + 1; j++)
	    for (k = fz; k <= fz + 1; k++)
		subt += Omega(p - Vector3D(i, j, k), i, j, k);

    subt += 0.5;
    if (subt <= 0)
	return 0;
    else if (subt > 1)
	return 1;
    return subt;
}
