// ===================================================================
// matrix.cpp
//	Matrix routines for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include <math.h>
#include <iomanip.h>
#include "matrix.h"
#include "vector.h"

// Invert the matrix using Gaussian elimination.  Not efficient,
// nor does it check for singular matrices.
void
Matrix::Invert()
{
    int i, j, k;
    Matrix Out = IdentityMatrix();

    for (i = 0; i < 4; i++) {
	if (x[i][i] != 1.0) {
	    float divby = x[i][i];
	    for (j = 0; j < 4; j++) {
		Out.x[i][j] /= divby;
		x[i][j] /= divby;
	    }
	}
	for (j = 0; j < 4; j++) {
	    if (j != i) {
		if (x[j][i] != 0.0) {
		    float mulby = x[j][i];
		      for (k = 0; k < 4; k++) {
			  x[j][k] -= mulby * x[i][k];
			  Out.x[j][k] -= mulby * Out.x[i][k];
		      }
		}
	    }
	}
    }
    *this = Out;
}

// Invert the given matrix using the above inversion routine.
Matrix
Invert(const Matrix& M)
{
    Matrix InvertMe = M;
    InvertMe.Invert();
    return InvertMe;
}

// Transpose the matrix.
void
Matrix::Transpose()
{
    for (int i = 0; i < 4; i++)
	for (int j = i; j < 4; j++)
	    if (i != j) {
		float temp = x[i][j];
		x[i][j] = x[j][i];
		x[j][i] = temp;
	    }
}

// Transpose the given matrix using the transpose routine above.
Matrix
Transpose(const Matrix& M)
{
    Matrix TransposeMe = M;
    TransposeMe.Transpose();
    return TransposeMe;
}

// Construct an identity matrix.
Matrix
IdentityMatrix()
{
    Matrix M;

    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    M.x[i][j] = (i == j) ? 1.0 : 0.0;
    return M;
}

// Construct a zero matrix.
Matrix
ZeroMatrix()
{
    Matrix M;
    for (int i = 0; i < 4; i++)
        for (int j = 0; j < 4; j++)
            M.x[i][j] = 0;
    return M;
}

// Construct a translation matrix given the location to translate to.
Matrix
TranslationMatrix(const Vector3D& Location)
{
    Matrix M = IdentityMatrix();
    M.x[3][0] = Location.x;
    M.x[3][1] = Location.y;
    M.x[3][2] = Location.z;
    return M;
}

// Construct a rotation matrix.  Rotates Angle radians about the
// X axis.
Matrix
RotationXMatrix(float Angle)
{
    Matrix M = IdentityMatrix();
    float Cosine = cos(Angle);
    float Sine = sin(Angle);
    M.x[1][1] = Cosine;
    M.x[2][1] = -Sine;
    M.x[1][2] = Sine;
    M.x[2][2] = Cosine;
    return M;
}

// Construct a rotation matrix.  Rotates Angle radians about the
// Y axis.
Matrix
RotationYMatrix(float Angle)
{
    Matrix M = IdentityMatrix();
    float Cosine = cos(Angle);
    float Sine = sin(Angle);
    M.x[0][0] = Cosine;
    M.x[2][0] = -Sine;
    M.x[0][2] = Sine;
    M.x[2][2] = Cosine;
    return M;
}

// Construct a rotation matrix.  Rotates Angle radians about the
// Z axis.
Matrix
RotationZMatrix(float Angle)
{
    Matrix M = IdentityMatrix();
    float Cosine = cos(Angle);
    float Sine = sin(Angle);
    M.x[0][0] = Cosine;
    M.x[1][0] = -Sine;
    M.x[0][1] = Sine;
    M.x[1][1] = Cosine;
    return M;
}

// Construct a yaw-pitch-roll rotation matrix.	Rotate Yaw
// radians about the XY axis, rotate Pitch radians in the
// plane defined by the Yaw rotation, and rotate Roll radians
// about the axis defined by the previous two angles.
Matrix
RotationYPRMatrix(float Yaw, float Pitch, float Roll)
{
    Matrix M;
    float ch = cos(Yaw);
    float sh = sin(Yaw);
    float cp = cos(Pitch);
    float sp = sin(Pitch);
    float cr = cos(Roll);
    float sr = sin(Roll);

    M.x[0][0] = ch*cr + sh*sp*sr;
    M.x[1][0] = -ch*sr + sh*sp*cr;
    M.x[2][0] = sh*cp;
    M.x[0][1] = sr*cp;
    M.x[1][1] = cr*cp;
    M.x[2][1] = -sp;
    M.x[0][2] = -sh*cr - ch*sp*sr;
    M.x[1][2] = sr*sh + ch*sp*cr;
    M.x[2][2] = ch*cp;
    for (int i = 0; i < 4; i++)
	M.x[3][i] = M.x[i][3] = 0;
    M.x[3][3] = 1;

    return M;
}

// Construct a rotation of a given angle about a given axis.
// Derived from Eric Haines's SPD (Standard Procedural 
// Database).
Matrix
RotationAxisMatrix(const Vector3D& axis, float angle)
{
    Matrix M;
    double cosine = cos(angle);
    double sine = sin(angle);
    double one_minus_cosine = 1 - cosine;

    M.x[0][0] = axis.x * axis.x + (1.0 - axis.x * axis.x) * cosine;
    M.x[0][1] = axis.x * axis.y * one_minus_cosine + axis.z * sine;
    M.x[0][2] = axis.x * axis.z * one_minus_cosine - axis.y * sine;
    M.x[0][3] = 0;

    M.x[1][0] = axis.x * axis.y * one_minus_cosine - axis.z * sine;
    M.x[1][1] = axis.y * axis.y + (1.0 - axis.y * axis.y) * cosine;
    M.x[1][2] = axis.y * axis.z * one_minus_cosine + axis.x * sine;
    M.x[1][3] = 0;

    M.x[2][0] = axis.x * axis.z * one_minus_cosine + axis.y * sine;
    M.x[2][1] = axis.y * axis.z * one_minus_cosine - axis.x * sine;
    M.x[2][2] = axis.z * axis.z + (1.0 - axis.z * axis.z) * cosine;
    M.x[2][3] = 0;

    M.x[3][0] = 0;
    M.x[3][1] = 0;
    M.x[3][2] = 0;
    M.x[3][3] = 1;

    return M;
}
    

// Construct a scale matrix given the X, Y, and Z parameters
// to scale by.  To scale uniformly, let X==Y==Z.
Matrix
ScaleMatrix(float X, float Y, float Z)
{
    Matrix M = IdentityMatrix();

    M.x[0][0] = X;
    M.x[1][1] = Y;
    M.x[2][2] = Z;

    return M;
}

// Construct a rotation matrix that makes the x, y, z axes
// correspond to the vectors given.
Matrix
GenRotation(const Vector3D& x, const Vector3D& y, const Vector3D& z)
{
    Matrix M = IdentityMatrix();

    M.x[0][0] = x.x;
    M.x[1][0] = x.y;
    M.x[2][0] = x.z;
    M.x[0][1] = y.x;
    M.x[1][1] = y.y;
    M.x[2][1] = y.z;
    M.x[0][2] = z.x;
    M.x[1][2] = z.y;
    M.x[2][2] = z.z;

    return M;
}

// Construct a view matrix to rotate and translate the world
// given that the viewer is located at Viewer, looking at
// LookAt, and the up vector is UpL.  The transformation
// changes things around so the viewer is at the origin and
// looking down the -Z axis.
Matrix
ViewMatrix(const Vector3D& LookAt, const Vector3D& Viewer, const Vector3D& UpL)
{
    Matrix M = IdentityMatrix();
    Vector3D U, V, N;
    Vector3D Up = Normalize(UpL);

    N = Normalize(Viewer - LookAt);

    V = Up - LookAt;
    V -= N * DotProd(V, N);
    V = Normalize(V);
//    V = Normalize(Up - N * DotProd(Up, N));

    U = CrossProd(V, N);

    M.x[0][0] = U.x;
    M.x[1][0] = U.y;
    M.x[2][0] = U.z;
    M.x[0][1] = V.x;
    M.x[1][1] = V.y;
    M.x[2][1] = V.z;
    M.x[0][2] = N.x;
    M.x[1][2] = N.y;
    M.x[2][2] = N.z;

    M.x[3][0] = -DotProd(U, Viewer);
    M.x[3][1] = -DotProd(V, Viewer);
    M.x[3][2] = -DotProd(N, Viewer);

    return M;
}

// Construct a quadric matrix.	After Foley et al. pp. 528-529.
Matrix
QuadricMatrix(float a, float b, float c, float d, float e,
	      float f, float g, float h, float j, float k)
{
    Matrix M;

    M.x[0][0] = a;  M.x[0][1] = d;  M.x[0][2] = f;  M.x[0][3] = g;
    M.x[1][0] = d;  M.x[1][1] = b;  M.x[1][2] = e;  M.x[1][3] = h;
    M.x[2][0] = f;  M.x[2][1] = e;  M.x[2][2] = c;  M.x[2][3] = j;
    M.x[3][0] = g;  M.x[3][1] = h;  M.x[3][2] = j;  M.x[3][3] = k;

    return M;
}

// Construct various "mirror" matrices, which flip coordinate
// signs in the various axes specified.
Matrix
MirrorX()
{
    Matrix M = IdentityMatrix();
    M.x[0][0] = -1;
    return M;
}

Matrix
MirrorY()
{
    Matrix M = IdentityMatrix();
    M.x[1][1] = -1;
    return M;
}

Matrix
MirrorZ()
{
    Matrix M = IdentityMatrix();
    M.x[2][2] = -1;
    return M;
}

Matrix
RotationOnly(const Matrix& x)
{
    Matrix M = x;
    M.x[3][0] = M.x[3][1] = M.x[3][2] = 0;
    return M;
}

// Add corresponding elements of the two matrices.
Matrix&
Matrix::operator+= (const Matrix& A)
{
    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    x[i][j] += A.x[i][j];
    return *this;
}

// Subtract corresponding elements of the matrices.
Matrix&
Matrix::operator-= (const Matrix& A)
{
    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    x[i][j] -= A.x[i][j];
    return *this;
}

// Scale each element of the matrix by A.
Matrix&
Matrix::operator*= (float A)
{
    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    x[i][j] *= A;
    return *this;
}

// Multiply two matrices.
Matrix&
Matrix::operator*= (const Matrix& A)
{
    Matrix ret = *this;

    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++) {
	    float subt = 0;
	    for (int k = 0; k < 4; k++)
		subt += ret.x[i][k] * A.x[k][j];
	    x[i][j] = subt;
	}
    return *this;
}

// Add corresponding elements of the matrices.
Matrix
operator+ (const Matrix& A, const Matrix& B)
{
    Matrix ret;

    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    ret.x[i][j] = A.x[i][j] + B.x[i][j];
    return ret;
}

// Subtract corresponding elements of the matrices.
Matrix
operator- (const Matrix& A, const Matrix& B)
{
    Matrix ret;

    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    ret.x[i][j] = A.x[i][j] - B.x[i][j];
    return ret;
}

// Multiply matrices.
Matrix
operator* (const Matrix& A, const Matrix& B)
{
    Matrix ret;

    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++) {
	    float subt = 0;
	    for (int k = 0; k < 4; k++)
		subt += A.x[i][k] * B.x[k][j];
	    ret.x[i][j] = subt;
	}
    return ret;
}

// Transform a vector by a matrix.
Vector3D
operator* (const Matrix& M, const Vector3D& v)
{
    Vector3D ret;
    float denom;

    ret.x = v.x * M.x[0][0] + v.y * M.x[1][0] + v.z * M.x[2][0] + M.x[3][0];
    ret.y = v.x * M.x[0][1] + v.y * M.x[1][1] + v.z * M.x[2][1] + M.x[3][1];
    ret.z = v.x * M.x[0][2] + v.y * M.x[1][2] + v.z * M.x[2][2] + M.x[3][2];
    denom = M.x[0][3] + M.x[1][3] + M.x[2][3] + M.x[3][3];
    if (denom != 1.0)
	ret /= denom;
    return ret;
}

// Apply the rotation portion of a matrix to a vector.
Vector3D
RotateOnly(const Matrix& M, const Vector3D& v)
{
    Vector3D ret;
    float denom;

    ret.x = v.x * M.x[0][0] + v.y * M.x[1][0] + v.z * M.x[2][0];
    ret.y = v.x * M.x[0][1] + v.y * M.x[1][1] + v.z * M.x[2][1];
    ret.z = v.x * M.x[0][2] + v.y * M.x[1][2] + v.z * M.x[2][2];
    denom = M.x[0][3] + M.x[1][3] + M.x[2][3] + M.x[3][3];
    if (denom != 1.0)
	ret /= denom;
    return ret;
}

// Scale each element of the matrix by B.
Matrix
operator* (const Matrix& A, float B)
{
    Matrix ret;

    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    ret.x[i][j] = A.x[i][j] * B;
    return ret;
}

// Overloaded << for C++-style output.
ostream&
operator<< (ostream& s, const Matrix& M)
{
    for (int i = 0; i < 4; i++) {
	for (int j = 0; j < 4; j++)
	    s << setprecision(2) << setw(10) << M.x[i][j];
	s << '\n';
    }
    return s;
}

// Rotate a direction vector...
Vector3D
PlaneRotate(const Matrix& tform, const Vector3D& p)
{
    // I sure hope that matrix is invertible...
    Matrix use = Transpose(Invert(tform));

    return RotateOnly(use, p);
}

