// ===================================================================
// main.cpp
//	This file contains declarations of static member functions,
//	plus some miscellaneous functions that don't make sense to
//	put elsewhere (such as Report(), which reports on the ray
//	tracer's statistics, and Usage(), which tells the user
//	the valid command-line options).
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "world.h"

NoiseMaker *GlobalNoise::Noise = 0;
WaveSource *GlobalNoise::Waves = 0;
FreqSource *GlobalNoise::Freqs = 0;

char *Statistics::AppendTo = 0;

long Statistics::PrimaryRays = 0L;
long Statistics::ShadowRays = 0L;
long Statistics::ReflectedRays = 0L;
long Statistics::TransmittedRays = 0L;
long Statistics::ShadowTotal = 0L;
long Statistics::ShadowHits = 0L;
long Statistics::NearestTotal = 0L;
long Statistics::NearestHits = 0L;
float Statistics::RestrictedVolume = 0;
float Statistics::TotalVolume = 0;
long Statistics::PrepTime;
long Statistics::RenderTime;
long Statistics::EpsilonTotal = 0;
long Statistics::EpsilonNeeded = 0;

float Statistics::HierarchyHeuristic;

unsigned long Statistics::RayPixHist[16] = {0};

long Statistics::Intersections::BBox = 0;
long Statistics::Intersections::Plane = 0;
long Statistics::Intersections::Polygon = 0;
long Statistics::Intersections::Quadric = 0;
long Statistics::Intersections::Sphere = 0;
long Statistics::Intersections::Ellipsoid = 0;
long Statistics::Intersections::Ring = 0;
long Statistics::Intersections::Algebraic = 0;
long Statistics::Intersections::Triangle = 0;
long Statistics::Intersections::CSGUnion = 0;
long Statistics::Intersections::CSGIntersection = 0;
long Statistics::Intersections::CSGDifference = 0;

long Statistics::Objects::BBox;
long Statistics::Objects::Plane;
long Statistics::Objects::Polygon;
long Statistics::Objects::Quadric;
long Statistics::Objects::Sphere;
long Statistics::Objects::Ellipsoid;
long Statistics::Objects::Ring;
long Statistics::Objects::Algebraic;
long Statistics::Objects::Triangle;
long Statistics::Objects::CSGUnion = 0;
long Statistics::Objects::CSGIntersection = 0;
long Statistics::Objects::CSGDifference = 0;

float Limits::Threshold = 1.0;
float Limits::Small = 0.1;
float Limits::Infinity = 100000;

#if __BCPLUSPLUS__
extern unsigned _stklen = 20000;
#endif

// Puts the specified number of spaces out to cout.  Used for
// indentation during Object3D::Describe().
void
indent(int ind)
{
    int i;

    for (i = 0; i < ind; i++)
    	cout << ' ';
}

// -------------------------------------------------------------------
// Report function.
//	Reads Statistics static members, interprets them and appends
//	them to the given stream.
// -------------------------------------------------------------------
void
Report(ostream& app)
{
    app << "Database statistics:\n";
    if (Statistics::Objects::BBox)
	app << "\tBounding boxes: " << Statistics::Objects::BBox << '\n';
    if (Statistics::Objects::Plane)
	app << "\tPlanes: " << Statistics::Objects::Plane << '\n';
    if (Statistics::Objects::Polygon)
	app << "\tPolygons: " << Statistics::Objects::Polygon << '\n';
    if (Statistics::Objects::Quadric)
	app << "\tQuadrics: " << Statistics::Objects::Quadric << '\n';
    if (Statistics::Objects::Sphere)
	app << "\tSpheres: " << Statistics::Objects::Sphere << '\n';
    if (Statistics::Objects::Ellipsoid)
	app << "\tEllipsoids: " << Statistics::Objects::Ellipsoid << '\n';
    if (Statistics::Objects::Ring)
	app << "\tRings: " << Statistics::Objects::Ring << '\n';
    if (Statistics::Objects::Algebraic)
	app << "\tAlgebraic surfaces: " << Statistics::Objects::Algebraic << '\n';
    if (Statistics::Objects::Triangle)
	app << "\tTriangles: " << Statistics::Objects::Triangle << '\n';
    if (Statistics::Objects::CSGUnion)
	app << "\tCSG unions: " << Statistics::Objects::CSGUnion << '\n';
    if (Statistics::Objects::CSGIntersection)
	app << "\tCSG intersections: " << Statistics::Objects::CSGIntersection << '\n';
    if (Statistics::Objects::CSGDifference)
	app << "\tCSG differences: " << Statistics::Objects::CSGDifference << '\n';
    app << "Cost of hierarchy: " << Statistics::HierarchyHeuristic << '\n';

    app << "Primary rays traced: " << Statistics::PrimaryRays << '\n';
    app << "Shadow rays traced: " << Statistics::ShadowRays << '\n';
    app << "Reflected rays traced: " << Statistics::ReflectedRays << '\n';
    app << "Transmitted rays traced: " << Statistics::TransmittedRays << '\n';

    app << Statistics::PrepTime << " seconds preparing database\n";
    app << Statistics::RenderTime << " seconds rendering\n";

    app << setprecision(1);
    if (Statistics::NearestTotal) {
	app << Statistics::NearestTotal << " object-cached intersection tests, ";
	app << Statistics::NearestHits * 100.0 / Statistics::NearestTotal;
	app << "% successful\n";
    }
    if (Statistics::ShadowTotal) {
	app << Statistics::ShadowTotal << " shadow cache intersection tests, ";
	app << Statistics::ShadowHits * 100.0 / Statistics::ShadowTotal;
	app << "% successful\n";
    }
    if (Statistics::TotalVolume > Limits::Small &&
	Statistics::RestrictedVolume > Limits::Small) {
	app << "Restricted bbox volume ";
	app << Statistics::TotalVolume / Statistics::RestrictedVolume;
	app << " times smaller than unrestricted volume\n";
    }
    if (Statistics::EpsilonTotal) {
	app << Statistics::EpsilonTotal << " less-than-within-epsilon tests, ";
	app << Statistics::EpsilonNeeded << " needed comparisons w/ epsilon.\n";
    }

    if (Statistics::Intersections::BBox)
	app << Statistics::Intersections::BBox << " bounding box intersections\n";
    if (Statistics::Intersections::Plane)
	app << Statistics::Intersections::Plane << " plane intersections\n";
    if (Statistics::Intersections::Polygon)
	app << Statistics::Intersections::Polygon << " polygon intersections\n";
    if (Statistics::Intersections::Quadric)
	app << Statistics::Intersections::Quadric << " quadric intersections\n";
    if (Statistics::Intersections::Sphere)
	app << Statistics::Intersections::Sphere << " sphere intersections\n";
    if (Statistics::Intersections::Ellipsoid)
	app << Statistics::Intersections::Ellipsoid << " ellipsoid intersections\n";
    if (Statistics::Intersections::Ring)
	app << Statistics::Intersections::Ring << " ring intersections\n";
    if (Statistics::Intersections::Algebraic)
	app << Statistics::Intersections::Algebraic << " algebraic surface intersections\n";
    if (Statistics::Intersections::Triangle)
	app << Statistics::Intersections::Triangle << " triangle intersections\n";
    if (Statistics::Intersections::CSGUnion)
	app << Statistics::Intersections::CSGUnion << " CSG union intersections\n";
    if (Statistics::Intersections::CSGIntersection)
	app << Statistics::Intersections::CSGIntersection << " CSG intersection intersections\n";
    if (Statistics::Intersections::CSGDifference)
	app << Statistics::Intersections::CSGDifference << " CSG difference intersections\n";
    int prhist = 0;
    for (int i = 0; i < 16; i++) {
	if (! prhist)
	    app << "Rays/pixel histogram:\n";
	prhist = 1;
	if (Statistics::RayPixHist[i])
	    app << i + 1 << " rays: " << Statistics::RayPixHist[i] << " pixels\n";
    }
}

void
Usage()
{
    cout << "OORT command line options are as follows:\n\n";
    cout << "\t-D[limit]\tSet depth limit of recursion\n";
    cout << "\t-S[file]\tAppend statistics to given file\n";
    cout << "\t-o[outfile]\tSet output file (default extension .RAW)\n";
    cout << "\t-w[width]\tSet width of output file\n";
    cout << "\t-h[height]\tSet height of output file\n";
    cout << "\t-interactive[+|-]\tTurn interactive ray tracing on/off\n";
    cout << "\t-s[start]\tStart at given scanline (range [0, height-1])\n";
    cout << "\t-e[end]\t\tEnd at given scanline (range [0, height-1])\n";
    cout << "\t-t[num]\t\tSet threshold of no-intersection (default 1.0)\n";
    cout << "\t-q\t\tSuppress printing period after each scanline computed\n";
    cout << "\n\n\t-hierarchy\tPrint hierarchy before beginning trace\n";
    exit(0);
}

