// ===================================================================
// hier.cpp
//	Bounding volume hierarchy-related routines, including
//	automatic bounding volume hierarchy generation.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include <alloc.h>
#include "oort.h"
#include "world.h"

void
World::RemoveUnboundeds()
{
    ObjectList::Manipulator olist(Objects);
    AxisAlignedBox bbox;

    while (olist.Valid()) {
	bbox = olist.Contents()->BBox();
	if (bbox.Unbounded()) {
	    Unboundeds.AddToList(olist.Contents());
	    olist.DeleteNode();
	}
	else	// DeleteNode makes it point to the next anyway.
	    olist.GotoNext();
    }
}

void
World::CountObjectsRec(ObjectList& list)
{
    for (ObjectList::Iterator sc = list; sc.Valid(); sc.GotoNext()) {
	sc.Contents()->CountMe();
	if (sc.Contents()->Flags() & BOUNDING)
	    CountObjectsRec( ((BoundingVolume *) sc.Contents())->contents);
    }
}

void
World::CountObjects()
{
    Statistics::Objects::BBox =
    Statistics::Objects::Plane =
    Statistics::Objects::Polygon =
    Statistics::Objects::Quadric =
    Statistics::Objects::Sphere = 
    Statistics::Objects::Ellipsoid =
    Statistics::Objects::Ring =
    Statistics::Objects::Algebraic = 
    Statistics::Objects::CSGUnion =
    Statistics::Objects::CSGIntersection =
    Statistics::Objects::CSGDifference = 0;
    CountObjectsRec(Unboundeds);
    CountObjectsRec(Objects);
}

void
World::HierarchyCostRec(BoundingVolume *root, float& subt, float rootsa)
{
    subt += root->contents.ListCount() * root->SurfaceArea() / rootsa;
    for (ObjectList::Iterator sc(root->contents); sc.Valid(); sc.GotoNext())
	if (sc.Contents()->Flags() & BOUNDING)
	    HierarchyCostRec( (BoundingVolume *) sc.Contents(), subt, rootsa);
}

float
World::HierarchyCost(const ObjectList& Objects)
{
    if (! Objects.ListCount())
	return 0;

    if (! ((*Objects.Head())->Flags() & BOUNDING))
	return 0;

    float rootsa = ((Object3D *) *Objects.Head())->BBox().SurfaceArea();
    float ret = 0;

    HierarchyCostRec( (BoundingVolume *) *Objects.Head(), ret, rootsa);

    return ret;
}

void
World::BuildOneHierarchy(ObjectList& list, Object3D **objs, int n)
{
    int i;
    for (i = 0; i < n; i++) {
	int inx = rand() % n;
	Object3D *temp = objs[inx];
	objs[inx] = objs[i];
	objs[i] = temp;
    }
    list.AddToList(objs[0]);

    int pers = 79;

    for (i = 1; i < n; i++) {
	AddNewObject(&list, objs[i]);
	if (n > 80) {
	    if (! (i % (n / 80)) && pers--)
		cout << '.';
	}
	else {
	    int dec = 80 / n;
	    while (dec--)
		if (pers--)
		    cout << '.';
	}
    }
    cout << '\n';
}

void
World::BuildHierarchy(int tries)
{
    int numobjs = Objects.ListCount();

    if (numobjs <= 1)
	return;

    cout << "Building hierarchy...\n";

    Object3D **objs = new Object3D*[numobjs];
    int i;

    // Set objs array to point to each object in the list
    for (i = 0; Objects.ListCount(); i++)
	objs[i] = Objects.ExtractHead();

    BuildOneHierarchy(Objects, objs, numobjs);
    float currcost = HierarchyCost(Objects);
    for (i = 1; i < tries; i++) {
	ObjectList candidate;
	BuildOneHierarchy(candidate, objs, numobjs);
	float tempcost = HierarchyCost(candidate);
	if (tempcost < currcost) {
	    currcost = tempcost;
	    Objects = candidate;
	}
    }
    delete[] objs;
}

// Remove from the hierarchy bounding boxes with only one child.
void
World::CullSingletons(Object3D *bv, ObjectList *List)
{
    if (bv->Flags() & BOUNDING) {
	int numchildren = ((BoundingVolume *) bv)->contents.ListCount();
	Object3D **children = new Object3D*[numchildren];
	int i;

	if (! children) {
	    cerr << "Out of memory in World::CullSingletons\n";
	    exit(2);
	}
	ObjectList::Iterator sc( ((BoundingVolume *) bv)->contents);
	i = 0;
	while (sc.Valid()) {
	    children[i++] = sc.Contents();
	    sc.GotoNext();
	}

	for (i = 0; i < numchildren; i++)
	    CullSingletons(children[i], &((BoundingBox *) bv)->contents);

	if (((BoundingVolume *) bv)->contents.SingleObject())
	    *List = ((BoundingVolume *) bv)->contents;
	delete[] children;
    }
}


// -------------------------------------------------------------------
//	AUTOMATIC BOUNDING VOLUME HIERARCHY GENERATION ROUTINES FOLLOW
// -------------------------------------------------------------------


// Compute increase in bounding box volume if the bounding box of
// an object and another bounding box are union'ed.
// If the bounding boxes are disjoint, the increase should be 0.
static float
BBoxIncrease(const Object3D *candidate, const AxisAlignedBox& bbox)
{
    AxisAlignedBox cbbox = candidate->BBox();
    return Union(cbbox, bbox).SurfaceArea() - bbox.SurfaceArea();
}

// Given a list of objects and a bounding box, return a manipulator
// pointing at the element whose bounding box increases least in
// volume when the bounding box is union'ed with it.
static ObjectList::Manipulator *
MinVolumeIncrease(ObjectList *x, const AxisAlignedBox& bbox)
{
    ObjectList::Manipulator sc(*x);
    ObjectList::Manipulator ret(*x);
    float minincrease = BBoxIncrease(ret.Contents(), bbox);
    while (sc.Valid()) {
	float tempincrease = BBoxIncrease(sc.Contents(), bbox);
	if (tempincrease < minincrease) {
	    minincrease = tempincrease;
	    ret = sc;
	}
	sc.GotoNext();
    }
    return new ObjectList::Manipulator(ret);
}

static int
HeuristicIncrease(Object3D *candidate,
		  const AxisAlignedBox& bbox,
		  float *increase,
		  const SimpleStack<ObjectList::Manipulator *>& stk)
{
    // Precompute some important stuff and assign to local vars.
    int k = candidate->NumChildren();
    AxisAlignedBox cbbox = candidate->BBox();

    // Compute surface area of the union of the candidate's bbox
    // and the new object's bbox.
    float newsa = Union(bbox, cbbox).SurfaceArea();

    // Compute the new heuristic value if you add the object
    // directly to the bbox.
    float addbbox = newsa * (k + 1);

    // Compute the new heuristic value if you replace the existing
    // bbox with one that contains the old one plus the object
    // being added.
    float replacebbox = 2 * newsa + k * cbbox.SurfaceArea();

    // Declare the integer return value.  It will be set in the
    // following if statement.
    int ret;

    // Decide whether to add the object directly to the bbox, or
    // keep the old bbox and enclose it in a new bounding volume
    // that also encloses the object being added.
    if (addbbox < replacebbox) {
	// Keep the old bounding box, add an object to it.
	*increase = addbbox;
	ret = 0;
    }
    else {
	// Replace the old bounding box.
	*increase = replacebbox;
	ret = 1;
    }

    // Declare a stack iterator to nondestructively examine the
    // contents of the stack.
    SimpleStack<ObjectList::Manipulator *>::Iterator stksc(stk);

    stksc.GotoNext();	// (we've already taken care of TOS)

    while (stksc.Valid()) {
	int numchildren = stksc.Contents()->Contents()->NumChildren();
	*increase += numchildren * Union(stksc.Contents()->Contents()->BBox(),
					 bbox).SurfaceArea();
	stksc.GotoNext();
    }
    return ret;
}

// Add an object to a hierarchy using Goldsmith and Salmon's
// technique of automatic bounding volume hierarchy generation.
void
AddNewObject(ObjectList *root, Object3D *addme)
{
    SimpleStack<ObjectList::Manipulator *> stk;
    AxisAlignedBox bbox = addme->BBox();
    ObjectList *curlist = root;
    int flags;

    do {
	stk.Push(MinVolumeIncrease(curlist, bbox));
	flags = stk.Top()->Contents()->Flags();
	if (flags & BOUNDING)
	    curlist = ((BoundingVolume *) stk.Top()->Contents())->List();
    } while (flags & BOUNDING);

    // Make a copy of the stack so we can compute the bounds later.
    SimpleStack<ObjectList::Manipulator *> cpystk = stk;
    ObjectList::Manipulator *min;
    int minmethod, tempmethod;
    float minincrease, tempincrease;
    min = stk.Top();
    minmethod = HeuristicIncrease(stk.Top()->Contents(), bbox, &minincrease, stk);
    while (! stk.Empty()) {
	tempmethod = HeuristicIncrease(stk.Top()->Contents(), bbox, &tempincrease, stk);
	if (tempincrease < minincrease) {
	    min = stk.Top();
	    minincrease = tempincrease;
	    minmethod = tempmethod;
	}
	stk.Pop();
    }
    if (minmethod || (! (min->Contents()->Flags() & BOUNDING))) {
	// Replace old node in the tree with a bounding box.
	BoundingBox *newbbox = new BoundingBox;
	newbbox->AddObject(min->Contents());
	newbbox->AddObject(addme);
	min->ReplaceContents(newbbox);
    }
    else {
	// Just add the new object to the bounding box.
	((BoundingVolume *) min->Contents())->AddObject(addme);
    }
    while (! cpystk.Empty()) {
	if (cpystk.Top()->Contents()->Flags() & BOUNDING) {
	    BoundingBox *innernode = (BoundingBox *) cpystk.Top()->Contents();
	    innernode->Include(bbox);
	}
	cpystk.Pop();
    }
}
