// ================================================================
// bump.cpp
//	Implementation file for bump maps.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ================================================================

#include "oort.h"

// =========================================================
// BumpMapReference
//	Encapsulates a bump map that also takes a reference
//	matrix.  Reference matrices are applied to input
// 	vectors before using noise & other tools to decide
//	how to perturb the normal.
// =========================================================
BumpMapReference::BumpMapReference(const Matrix& _reference)
{
    reference = _reference;
}

Vector3D
BumpMapReference::PreprocessVector(const Vector3D& v) const
{
    return reference * v;
}

void
BumpMapReference::ApplyTransform(const Matrix& tform)
{
    reference *= tform;
}

void
BumpMapReference::PreMulTransform(const Matrix& tform)
{
    reference = tform * reference;
}

// =========================================================
// ReverseNormal
//	Simply reverses the input normal.
// =========================================================
void
ReverseNormal::PerturbNormal(ShadingInfo& shade)
{
    if (bump)
	bump->PerturbNormal(shade);
    shade.normal = -shade.normal;
}

// =========================================================
// TwoSided
//	Reverses the normal, if necessary, so it faces
//	toward the incident direction.
// =========================================================
void
TwoSided::PerturbNormal(ShadingInfo& shade)
{
    if (bump)
	bump->PerturbNormal(shade);
    if (DotProd(shade.incident, shade.normal) > 0)
	shade.normal = -shade.normal;
}

// =========================================================
// WaveSource
//	Precomputes a set of vector noise values.
// =========================================================
WaveSource::WaveSource(int N, NoiseMaker *nm)
{
    waves = new Vector3D[n = N];
    for (int i = 0; i < n; i++)
	waves[i] = Normalize(nm->DNoise(Vector3D(i, 0, 0)));
}

// =========================================================
// FreqSource
//	Precomputes a set of solid noise values.
// =========================================================
FreqSource::FreqSource(int N, NoiseMaker *nm)
{
    freqs = new float[n = N];
    for (int i = 0; i < n; i++)
	freqs[i] = (rand() & (RAND_MAX - 1)) / RAND_MAX + 0.01;
}

// =========================================================
// Ripples
//	Makes a surface appear rippled.
// =========================================================
void
Ripples::PerturbNormal(ShadingInfo& shade)
{
    Vector3D ret, point;
    double length, scalar, index;

    ret = shade.normal;
    for (int i = 0; i < ws->N(); i++) {
	point = shade.p - ws->GetWave(i);
	length = DotProd(point, point);
	if (length < Limits::Small)
	    length = 1 - length;
	length = sqrt(length);
	index = length * freq + phase;
	scalar = cycloidal(index) * bumpamt;
	point *= scalar / length / ws->N();
	ret += point;
    }
    shade.normal = Normalize(ret);
}

BumpMap *
Ripples::Dup() const
{
    return new Ripples(ws, fs, freq, phase, bumpamt, reference);
}

// =========================================================
// Dented
//	Displaces the intersection point to make the object
//	look dented.  Not sure this can work correctly,
//	since the normal is not recomputed correctly.
// =========================================================
Dented::Dented(float _Km, const Matrix& mat): 
    BumpMapReference(mat)
{
    Km = _Km;
}

void
Dented::PerturbNormal(ShadingInfo& shade)
{
    float size = 1, mag = 0;
    Vector3D p = PreprocessVector(shade.p);
    for (int i = 0; i < 6; i++) {
	mag += fabs(0.5 - GlobalNoise::Noise->Noise(p * size)) / size;
	size *= 2;
    }
    shade.p -= shade.normal * mag * mag * mag * Km;
    shade.normal = shade.obj->FindNormal(p);
}

BumpMap *
Dented::Dup() const
{
    return new Dented(Km, reference);
}

