// ===================================================================
// alg.cpp
//	Implementation file for algebraic surfaces.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "poly.h"

Ray
InverseRayObject::InvertRay(Ray& ray)
{
    Ray ret = ray;
    ret.ApplyTransform(inv);
    return ret;
}

Vector3D
InverseRayObject::InvertPt(const Vector3D& v)
{
    return inv * v;
}

Vector3D
InverseRayObject::InvRotatePt(const Vector3D& v)
{
    return RotateOnly(inv, v);
}

Ray
InverseRayObject::TransformRay(Ray& ray)
{
    Ray ret = ray;
    ret.ApplyTransform(mat);
    return ret;
}

Vector3D
InverseRayObject::TransformPt(const Vector3D& v)
{
    return mat * v;
}

Vector3D
InverseRayObject::RotatePt(const Vector3D& v)
{
    return RotateOnly(mat, v);
}

// Given a list of spans in the object's canonical space, transform
// it into a list of spans in world space.
void
InverseRayObject::ApplyTransform(const Matrix& tform)
{
    mat *= tform;
    inv = Invert(tform) * inv;
}

Algebraic::Algebraic(const Quadric& q): InverseRayObject(q.ReturnColor())
{
    float eps = 1e-8;
    if (fabs(q.mat.x[0][0]) > eps) AddTerm(q.mat.x[0][0], 2, 0, 0);
    if (fabs(q.mat.x[1][1]) > eps) AddTerm(q.mat.x[1][1], 0, 2, 0);
    if (fabs(q.mat.x[2][2]) > eps) AddTerm(q.mat.x[2][2], 0, 0, 2);
    if (fabs(q.mat.x[3][3]) > eps) AddTerm(q.mat.x[3][3], 0, 0, 0);
    if (fabs(q.D) > eps) AddTerm(q.D, 1, 1, 0);
    if (fabs(q.E) > eps) AddTerm(q.E, 0, 1, 1);
    if (fabs(q.F) > eps) AddTerm(q.F, 1, 0, 1);
    if (fabs(q.G) > eps) AddTerm(q.G, 1, 0, 0);
    if (fabs(q.H) > eps) AddTerm(q.H, 0, 1, 0);
    if (fabs(q.J) > eps) AddTerm(q.J, 0, 0, 1);
}

// Returns x^y.  Should be faster than calling pow(), since
// this function only works when y is an integer.
double
intpow(double x, unsigned int y)
{
    double ret = 1;
    double mulby = x;
    while (y) {
	if (y & 1)
	    ret *= mulby;
	mulby *= x;
	y >>= 1;
    }
    return ret;
}

void
Algebraic::AddTerm(double c, int i, int j, int k)
{
    for (TermList::Iterator sc(terms); sc.Valid(); sc.GotoNext()) {
	if (*sc.Contents() == Term(c, i, j, k)) {
	    sc.Contents()->c += c;
	    return;
	}
	sc.GotoNext();
    }
    // Detach iterator from list
    sc.Detach();
    terms.AddToList(new Term(c, i, j, k));
}

static int
compare_doubles(const void *A, const void *B)
{
    const double *a = (const double *) A;
    const double *b = (const double *) B;
    if (*a < *b)
	return -1;
    else if (*b < *a)
	return 1;
    return 0;
}

Polynomial
Algebraic::RayToPoly(Ray& ray)
{
    Polynomial ret;
    for (TermList::Iterator sc(terms); sc.Valid(); sc.GotoNext()) {
	Term *t = sc.Contents();
	Polynomial addme;
	addme.AddElm(0, 1);
	if (t->i) addme *= Triangle(ray.dir.x, ray.loc.x, t->i);
	if (t->j) addme *= Triangle(ray.dir.y, ray.loc.y, t->j);
	if (t->k) addme *= Triangle(ray.dir.z, ray.loc.z, t->k);
	ret += t->c * addme;
    }
    return ret;
}

int
Algebraic::NearestInt(Ray& cpyme, float& t, float maxt)
{
    float tempt = maxt;
    int ret = 0;
    Ray ray = InvertRay(cpyme);

    Statistics::Intersections::Algebraic++;

    // Compute polynomial
    Polynomial solveme = RayToPoly(ray);

    // Find the roots and sort them in ascending order
    int rootcount;
    double *roots;
    rootcount = solveme.FindRealRoots(&roots);

    if (rootcount) {
	for (int i = 0; i < rootcount; i++) {
	    // Transform each parameter back into world space
	    if (roots[i] > Limits::Threshold && roots[i] < tempt) {
		tempt = roots[i];
		ret = 1;
	    }
	}
	t = tempt;
	delete[] roots;
    }
    return ret;
}

// Compute all intersections with the algebraic surface.
SpanList *
Algebraic::FindAllIntersections(Ray& cpyme)
{
    Ray ray = InvertRay(cpyme);
    SpanList *ret = 0;

    Statistics::Intersections::Algebraic++;

    Polynomial solveme = RayToPoly(ray);

    int rootcount;
    double *roots;
    rootcount = solveme.FindRealRoots(&roots);

    if (rootcount) {
	float tmin, tmax;
	Object3D *omin, *omax;
	int i;

	qsort(roots, rootcount, sizeof(double), compare_doubles);
	omin = omax = 0;
	for (i = 0; i < rootcount; i++) {
	    if (fabs(roots[i]) < Limits::Threshold)
		return 0;
	    if (roots[i] > Limits::Threshold)
		break;
	}
	// If every parameter is negative, return no-intersection.
	if (i == rootcount)
	    return 0;
	omin = omax = this;
	if (IsInside(cpyme.loc)) {
	    if (i != 0) {
		tmin = roots[i - 1];
		tmax = roots[i++];
	    }
	    else {
		tmin = -Limits::Infinity;
		tmax = roots[i++];
	    }
	}
	else {
	    tmin = roots[i++];
	    if (i == rootcount)
	        tmax = Limits::Infinity;
	    else
	        tmax = roots[i++];
	}
	ret = new SpanList;
	while (i < rootcount) {
	    if (omin && omax) {
		ret->AddSpan(Span(tmin, omin, tmax, omax));
		omin = omax = 0;
	    }
	    if (omin) {
		tmax = roots[i++];
		omax = this;
	    }
	    else {
		tmin = roots[i++];
		omin = this;
	    }
	}
	if (omin && omax)
	    ret->AddSpan(Span(tmin, omin, tmax, omax));
	else if (omin)
	    ret->AddSpan(Span(tmin, omin, Limits::Infinity, this));
    }
    if (ret) {
	if (ret->NumSpans())
	    return ret;
	delete ret;
    }
    return 0;
}

Vector3D
Algebraic::FindNormal(const Vector3D& xformme)
{
    Vector3D x = InvertPt(xformme);
    Vector3D ret(0);

    for (TermList::Iterator sc(terms); sc.Valid(); sc.GotoNext()) {
	Term *t = sc.Contents();
	Vector3D addme(0);
	if (t->i) {
	    float subt = intpow(x.x, t->i - 1) *
			 intpow(x.y, t->j) *
			 intpow(x.z, t->k);
	    addme += Vector3D(t->i * subt, 0, 0);
	}
	if (t->j) {
	    float subt = intpow(x.y, t->j - 1) *
			 intpow(x.x, t->i) *
			 intpow(x.z, t->k);
	    addme += Vector3D(0, t->j * subt, 0);
	}
	if (t->k) {
	    float subt = intpow(x.z, t->k - 1) *
			 intpow(x.x, t->i) *
			 intpow(x.y, t->j);
	    addme += Vector3D(0, 0, t->k * subt);
	}
	ret += t->c * addme;
    }
//    ret = Normalize(PlaneRotate(mat, ret));
    return Normalize(RotateOnly(mat, ret));
}

Object3D *
Algebraic::Dup() const
{
    Algebraic *ret = new Algebraic;
    for (TermList::Iterator sc(terms); sc.Valid(); sc.GotoNext()) {
	Term *t = sc.Contents();
	ret->AddTerm(t->c, t->i, t->j, t->k);
    }
    return (Object3D *) ret;
}

void
Algebraic::Describe(int ind) const
{
    for (TermList::Iterator sc(terms); sc.Valid(); sc.GotoNext()) {
	Term *t = sc.Contents();
	cout << t->c << " * x^" << t->i << "y^" << t->j << "z^" << t->k << " + ";
    }
    Term *t = sc.Contents();
    if (t)
	cout << t->c << " * x^" << t->i << "y^" << t->j << "z^" << t->k << '\n';
}

float
Algebraic::PtDistance(Vector3D& v)
{
    Vector3D x = InvertPt(v);
    float ret = 0;

    for (TermList::Iterator sc(terms); sc.Valid(); sc.GotoNext()) {
	Term *t = sc.Contents();
	ret += t->c * intpow(x.x, t->i) *
		      intpow(x.y, t->j) *
		      intpow(x.z, t->k);
    }
    return ret;
}

void
Torus::ApplyTransform(const Matrix& tform)
{
    bbox = Transform(bbox, tform);
    Algebraic::ApplyTransform(tform);
}

AxisAlignedBox
Torus::BBox() const
{
    return Transform(bbox, inv);
}


// Torus constructor.
Torus::Torus(float a, float b, float r, Surface *clr) :
      Algebraic(clr),
      bbox(Vector3D(-r-a, -b, -r-a), Vector3D(r+a, b, r+a))
{
    float p = (a*a) / (b*b);
    float a0 = 4 * r * r;
    float b0 = r*r - a*a;
    AddTerm(1, 4, 0, 0);		// x^4
    AddTerm(1, 2, 0, 2);		// x^2z^2
    AddTerm(p, 2, 2, 0);		// px^2y^2
    AddTerm(b0, 2, 0, 0);		// b0x^2

    AddTerm(1, 2, 0, 2);		// x^2z^2
    AddTerm(1, 0, 0, 4);		// z^4
    AddTerm(p, 0, 2, 2);		// py^2z^2
    AddTerm(b0, 0, 0, 2);		// b0z^2

    AddTerm(p, 2, 2, 0);		// px^2y^2
    AddTerm(p, 0, 2, 2);		// py^2z^2
    AddTerm(p*p, 0, 4, 0);		// p*p*y^4
    AddTerm(b0*p, 0, 2, 0); 		// b0*p*y^4

    AddTerm(b0, 2, 0, 0);		// b0*x^2
    AddTerm(b0, 0, 0, 2);		// b0*z^2
    AddTerm(b0*p, 0, 2, 0); 		// b0*p*y^2
    AddTerm(b0*b0, 0, 0, 0);		// b0^2

    AddTerm(-a0, 2, 0, 0);		// -a0*x^2
    AddTerm(-a0, 0, 0, 2);		// -a0*z^2
}

// AlgQuadric: a class solely intended to help debug Algebraics.
// It's a second-degree Algebraic that uses Algebraic code to
// do everything Quadrics can do right now.

// I put this in when developing algebraics, and didn't feel
// like deleting it (too useful a debugging tool). I might
// delete it after algebraics have matured a little.

AlgQuadric::AlgQuadric(float a, float b, float c, float d, float e,
		       float f, float g, float h, float j, float k,
		       Surface *surf): q(a, b, c, d, e, f, g, h, j, k, surf),
			Algebraic(Quadric(a, b, c, d, e, f, g, h, j, k, surf))
{
}

void
AlgQuadric::ApplyTransform(const Matrix& tform)
{
    Algebraic::ApplyTransform(tform);
    q.ApplyTransform(tform);
}


SpanList *
AlgQuadric::FindAllIntersections(Ray& ray)
{
    float eps = 1e-3;
    SpanList *me = Algebraic::FindAllIntersections(ray);
    SpanList *you = q.FindAllIntersections(ray);

    if ((! me) || (! you)) {
	if (me) {
	    me = new SpanList;
	}
	else if (you) {
	    you = new SpanList;
	}
	else return 0;
    }

    if (me->NumSpans() == you->NumSpans()) {
	SpanList *ret = new SpanList;
	while (me->NumSpans()) {
	    Span memin = me->ExtractMinSpan();
	    Span youmin = you->ExtractMinSpan();
	    if (fabs(memin.tmin - youmin.tmin) > eps ||
		fabs(memin.tmax - youmin.tmax) > eps) {
		ret->AddSpan(youmin);
	    }
	    ret->AddSpan(memin);
	}
	delete me;
	delete you;
	return ret;
    }
    else {
	return 0;
    }
}

