// ==================================================================
// poly.h
//	Header file for Polynomial class.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ==================================================================


// ---------------------------------------------------------
// Polynomial
//	Implements an arbitrary-degree polynomial.
// ---------------------------------------------------------
class Polynomial {
public:
    // Maximum power of 10 to search for.
    // Default is 32.
    static int MAXPOW;

    // A coefficient smaller than SMALL_ENOUGH is
    // considered to be zero.  Default is 1e-12.
    static double SMALL_ENOUGH;

    // Smallest relative error we want.
    // Default is 1e-8.
    static double RELERROR;

    // Maximum number of iterations.
    // Default is 800.
    static int MAXIT;

    // Default constructor doesn't allocate any memory.
    Polynomial() { ord = 0; coeff = 0; }

    // This constructor allocates memory and copies the
    // coefficients given.
    Polynomial(int Order, double *Coeff);

    // Copy constructor.
    Polynomial(const Polynomial&);

    // Assignment operator.
    Polynomial& operator=(const Polynomial&);

    // Destructor.
    ~Polynomial();

    // Add an element to the polynomial
    void AddElm(int deg, double x);

    // Evaluate polynomial in terms of given x
    double Evaluate(double x) const;

    int SturmSequence(Polynomial **);

    // Root-finding

    // Finds the number of roots given the Sturm sequence
    friend int NumRoots(Polynomial *sseq, int, int *, int *);
    friend int NumChanges(Polynomial *sseq, int, double);
    friend void SBisect(Polynomial *, int, double, double, int, int, double *);

    // Returns all real roots (allocates passbk using operator new,
    // then returns the root count).  No memory is allocated if there
    // are no real roots.
    int FindRealRoots(double **passbk);

    // Unary operators
    Polynomial operator+ () { return *this; }	// unary plus
    Polynomial operator- ();			// unary minus
    friend Polynomial Normalize(const Polynomial&);  // Make coeff[ord-1]==1
    friend Polynomial Derivative(const Polynomial&); // Manufactures derivative

    Polynomial& operator+= (const Polynomial&);	// Add another polynomial
    Polynomial& operator*= (const Polynomial&);	// Multiply by a polynomial
    Polynomial& operator*= (double);		// Scale by double

    // Binary operators
    friend Polynomial operator+ (const Polynomial&, const Polynomial&);
    friend Polynomial operator- (const Polynomial&, const Polynomial&);
    friend Polynomial operator* (const Polynomial&, const Polynomial&);
    friend Polynomial operator* (const Polynomial&, double&);
    friend Polynomial operator* (double, const Polynomial&);
    friend Polynomial operator% (const Polynomial&, const Polynomial&);

    friend ostream& operator<< (ostream& x, const Polynomial& y);
protected:
    int ord;		// Order of polynomial
    double *coeff;	// Coefficients.

    int ModRF(double a, double b, double *val);

};

// Returns the polynomial (ax+b)^t.
Polynomial Triangle(double a, double b, int t);

extern "C" int SolveQuadric(double[], double[]);
extern "C" int SolveCubic(double[], double[]);
extern "C" int SolveQuartic(double[], double[]);


