// ===============================================================
// mapping.h
//      Header file for inverse mappings.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===============================================================

// ---------------------------------------------------------------
// Mapping 
//	Implements various methods of taking 3D points and 
//	turning them into 2D points.
// ---------------------------------------------------------------

class Mapping {
public:
    Mapping() { }
    virtual ~Mapping() { }
    virtual Mapping *Dup() const = 0;
    virtual Vector3D Map3DTo2D(const Vector3D& v) const = 0;
};

// ---------------------------------------------------------------
// CircularMapping
//	Maps onto a circle of arbitrary radius centered 
//	about the origin.
// ---------------------------------------------------------------
class CircularMapping : public Mapping {
public:
    CircularMapping(float Radius = 1);
    Mapping *Dup() const;
    Vector3D Map3DTo2D(const Vector3D& v) const;
private:
    float radius;
};


// ---------------------------------------------------------------
// ConicalMapping
//	Maps onto a cone.
// ---------------------------------------------------------------
class ConicalMapping : public Mapping {
public:
    ConicalMapping(float rad0, float radh, float height);
    Mapping *Dup() const;
    Vector3D Map3DTo2D(const Vector3D& v) const;
private:
    float rad0, radh, height;
};


// ---------------------------------------------------------------
// CylindricalMapping
//	Maps onto a cylinder centered on the Y axis.
// ---------------------------------------------------------------
class CylindricalMapping : public Mapping {
public:
    CylindricalMapping(float radius, float height);
    Mapping *Dup() const;
    Vector3D Map3DTo2D(const Vector3D& v) const;
private:
    float radius, height;
};


// ---------------------------------------------------------------
// ProjectionMapping
//	Projects the input vector onto the XY coordinate plane.
// ---------------------------------------------------------------
class ProjectionMapping : public Mapping {
public:
    Mapping *Dup() const;
    Vector3D Map3DTo2D(const Vector3D& v) const;
};


// ---------------------------------------------------------------
// QuadrilateralMapping
//	Maps points onto a quadrilateral.
// ---------------------------------------------------------------
class QuadrilateralMapping : public Mapping {
public:
    QuadrilateralMapping(const Vector3D *Pts);
    Mapping *Dup() const;
    Vector3D Map3DTo2D(const Vector3D& v) const;
private:
    Vector3D pts[4];
    Vector3D Na, Nb, Nc, Nd;
    float Du0, Du1, Du2, Dv0, Dv1, Dv2;
    int uaxespar, vaxespar;
    Vector3D Qux, Quy, Qvx, Qvy;
    float Dux, Duy, Dvx, Dvy;
};


// ---------------------------------------------------------------
// SphericalMapping
//	Maps points onto a sphere centered about an arbitrary point.
// ---------------------------------------------------------------
class SphericalMapping : public Mapping {
public:
    SphericalMapping(Vector3D _center = Vector3D(0));
    Mapping *Dup() const;
    Vector3D Map3DTo2D(const Vector3D& v) const;
private:
    Vector3D center;
};

