// ===================================================================
// light.h
//	Header file for Light abstract class and its derivatives.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================


// -------------------------------------------------------------------
// Light
//	Abstract class encapsulates all types of light source.
//	Color is the color of light emitted by the light source.
//	c1, c2 and c3 define the attention of the light due to
//	distance.  The attenuatin is 1 / (c1 + c2*d + c3*d^2) where d
//	is the distance from the light source.	Thus, c1 == 1 means no
//	attenuation, while if c3 is nonzero the attenuation is
//	proportional to the square of the distance (the physically
//	correct model, which does not work well in practice).
// -------------------------------------------------------------------
class Light {
public:
    Light(const RGBColor& clr, float C1, float C2, float C3);

    float DistAtten(float t);	  // Return attenuation due to distance

    // -----------------------------------------------------------------
    // Return nonzero if the given ray hits LastOpaque.  If it misses,
    // LastOpaque is reset to 0.
    // -----------------------------------------------------------------
    int HitCachedObject(Ray& ray, float maxt);

    // -----------------------------------------------------------------
    // Fires a collection of rays according to the type of light source
    // and number of rays authorized by the user.  Returns the color of
    // the light source as seen from the intersection point.
    // -----------------------------------------------------------------
    virtual RGBColor ShadowLight(ShadingInfo& shade, Surface *surf) = 0;
    virtual void ApplyTransform(const Matrix& tform) = 0;

protected:
    RGBColor Color;		// Color of light's light
    float c1, c2, c3;		// Light's attenuation by distance
    Object3D *LastOpaque;	// Cache for last object which
				// shadowed this light

    RGBColor ShadowRay(ShadingInfo& shade,
		       Vector3D& target,
		       Surface *surf);

};


// ---------------------------------------------------------
// Derive list of lights from SimpleList class template.
// ---------------------------------------------------------
typedef SimpleList<Light *> LightList;


// -------------------------------------------------------------------
// PointLight
//	Encapsulates a point light source.
//	One shadow ray is fired at the location, and if it hits an
//	opaque object between the intersection point and the point,
//	the light is in shadow.
// -------------------------------------------------------------------
class PointLight : public Light {
public:
    PointLight(const Vector3D& loc, const RGBColor& clr, float C1 = 1, float C2 = 0, float C3 = 0);
    RGBColor ShadowLight(ShadingInfo& shade, Surface *surf);
    void ApplyTransform(const Matrix& tform);
private:
    Vector3D Location;		// Where light is coming from
};

// -------------------------------------------------------------------
// DirectLight
//	Encapsulates a directional light source.  Light comes from
//	a fixed direction (the direction vector points _toward_ the
//	direction the light is coming _from_).	Directional light
//	sources do not attenuate their light by distance, since they
//	are infinitely far away.
// -------------------------------------------------------------------
class DirectLight : public Light {
public:
    DirectLight(const Vector3D& loc, const RGBColor& clr);

    RGBColor ShadowLight(ShadingInfo& shade, Surface *surf);
    void ApplyTransform(const Matrix& tform);
private:
    Vector3D Direction;		  // Where light is coming from
};

// -------------------------------------------------------------------
// RectangularLight
//	Encapsulates a rectangular light source.
//	Instead of specifying the coordinates of the rectangle, the
//	width and height of the rectangle are specified and it is
//	assumed to begin in the XY plane, centered at the origin.
//	Then transformations applied to it are concatenated into the
//	"applied" matrix, and this matrix is applied to the sample
//	points used for stochastic sampling of the light source.
// -------------------------------------------------------------------
class RectangularLight : public Light {
public:
    RectangularLight(float width, float height,
		     const RGBColor& clr = RGBColor(1, 1, 1),
		     int min = 4, int max = 16, int numcandidates = 10,
		     const RGBColor& cthreshold = RGBColor(0.4, 0.3, 0.6),
		     float C1 = 1, float C2 = 0, float C3 = 0);
    RGBColor ShadowLight(ShadingInfo& shade, Surface *surf);
    void ApplyTransform(const Matrix& tform);
protected:
    virtual Vector3D NewSample(Vector3D *, int);
private:

    // Size of rectangle
    float width, height;

    // Distribution parmeters
    int MinSamples, MaxSamples;	// # samples allowed
    int NumCandidates;		// # candidates for each sample
    RGBColor ContrastThreshold;	// Threshold for contrast

    // Matrix applied to rectangle
    Matrix applied;
};
