// ===================================================================
// colormap.h
//	Declares ColorMap and ColorMapEntry classes.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

// ---------------------------------------------------------
// ColorMap
//	Provides a way to map a value in the range [0, 1] to
//	an RGB triple.  The ColorMapEntries specify ranges
//	within [0, 1]; given a value in its range, a given
//	ColorMapEntry uses the value to interpolate between 
//	the colors in the ColorMapEntry.
// ---------------------------------------------------------

// Forward declaration.
class ColorMapEntry;

class ColorMap {
public:
    ColorMap() { 
        num_entries = 0; 
        entries = new ColorMapEntry*[max_entries = 2]; 
    }
    ~ColorMap();

    void AddEntry(ColorMapEntry *);
    RGBColor ComputeColor(float value);
private:
    int num_entries, max_entries;
    ColorMapEntry **entries;
};

// ---------------------------------------------------------
// ColorMapEntry
//	Encapsulates an entry in the color map.  A
//	ColorMapEntry has to provide two basic operations:
//	    - Tell whether a given value is within the
//	      bounds of the color map entry.  This is
//	      implemented by the IsInside member function.
//	    - Compute a color given that value.  This is
//	      implemented by the ComputeColor member function.
// ---------------------------------------------------------
class ColorMapEntry {
public:
    ColorMapEntry() { }
    ColorMapEntry(float s, float e): start(s), end(e) { }
    virtual ~ColorMapEntry() { }

    // Compute color inside this entry given the value.
    virtual RGBColor ComputeColor(float value) const = 0;

    // Duplicate a color map entry.
    virtual ColorMapEntry *Dup() const = 0;

    // Given a value, return nonzero if it's within this entry's bounds.
    int IsInside(float value) {
        return value >= start && value <= end;
    }
protected:
    float start, end;
    float fract(float value) {
        return (value - start) / (end - start);
    }
};

// ---------------------------------------------------------
// TwoColorMapEntry
//	Encapsulates color map entries with two endpoints
//	and two colors.  LinearColorMapEntry and
//	HermiteColorMapEntry both derive from this.
// ---------------------------------------------------------
class TwoColorMapEntry : public ColorMapEntry {
public:

    TwoColorMapEntry(float Start, float End, 
                     const RGBColor& vst, const RGBColor& vend):
        ColorMapEntry(Start, End), start_clr(vst), end_clr(vend) { }
    virtual RGBColor ComputeColor(float value) const = 0;
    virtual ColorMapEntry *Dup() const = 0;
protected:
    RGBColor start_clr, end_clr;
};

// ---------------------------------------------------------
// OneColorMapEntry
//	This color map entry just returns one color, no
//	matter where in the interval the value falls.
// ---------------------------------------------------------
class OneColorMapEntry : public ColorMapEntry {
public:
    OneColorMapEntry(float Start, float End, const RGBColor& Clr):
	ColorMapEntry(Start, End), clr(Clr) { }
    virtual RGBColor ComputeColor(float value) const { return clr; }
    virtual ColorMapEntry *Dup() const;
protected:
    RGBColor clr;
};

inline ColorMapEntry *
OneColorMapEntry::Dup() const
{
    return new OneColorMapEntry(start, end, clr);
}

// ---------------------------------------------------------
// LinearColorMapEntry
//	Linearly interpolates between two colors at the
//	endpoints of the interval.
// ---------------------------------------------------------
class LinearColorMapEntry : public TwoColorMapEntry {
public:
    LinearColorMapEntry(float Start, 
                        float End, 
                        const RGBColor& StartClr, 
                        const RGBColor& EndClr):
        TwoColorMapEntry(Start, End, StartClr, EndClr) { }
    virtual RGBColor ComputeColor(float value) const {
        float fraction = (value - start) / (end - start);
		return start_clr + fraction * (end_clr - start_clr);
    }
    virtual ColorMapEntry *Dup() const {
        return new LinearColorMapEntry(start, end, start_clr, end_clr);
    }
};

// ---------------------------------------------------------
// HermiteColorMapEntry
//	Uses a Hermite interpolation to compute colors in
//	the interval.
// ---------------------------------------------------------
class HermiteColorMapEntry : public TwoColorMapEntry {
public:
    HermiteColorMapEntry(float Start, float End, 
			 const RGBColor& vst, 
			 const RGBColor& vend):
        TwoColorMapEntry(Start, End, vst, vend) { }
    virtual RGBColor ComputeColor(float value) const {
        float t = (value - start) / (end - start);
        float fraction = t*t*(-3 + 2 * t) + 1;
		return start_clr + fraction * (end_clr - start_clr);
    }
    virtual ColorMapEntry *Dup() const {
        return new HermiteColorMapEntry(start, end, start_clr, end_clr);
    }
};


