// ===================================================================
// tetra.cpp
//	Implementation file for Tetrahedron example.
//	The code for this file is derived from tetra.c in Eric
//	Haines's public domain Standard Procedural Database (SPD).
//	For more information on the SPD, see his article "A Proposal 
//	For Standard Graphics Environments" in the November 1987 
//	issue of IEEE Computer Graphics & Applications.
//
//	Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================

#include "oort.h"
#include "world.h"
#include "utils.h"

void
PopulateWorld(World& world)
{
    world.SetDepthLimit(3);
    world.SetOutputFile("tetra.raw");

    Vector3D LookAt(-0.004103, -0.004103, 0.216539);
    Vector3D Viewer = 1.5 * Vector3D(1.02285, -3.17715, -2.17451);
    Vector3D Up(-0.816497, -0.816497, 0.816497);
    world.SetViewerParameters(LookAt, Viewer, Up);

    world.SetScreenWidth(asin(45*M_PI/180));
    world.SetScreenHeight(asin(45*M_PI/180));
    world.SetScreenDistance(1);

    // Important: set threshold of no-intersection to a
    // much smaller value than the default.
    Limits::Threshold = 0.01;
    Limits::Small = 0.001;

    world.SetBackgroundColor(RGBColor(0));
    world.AddLight(new PointLight(Vector3D(1.87607, -18.1239, -5.00042), Vector3D(0.5), 1, 0, 0));
    world.AddLight(new PointLight(Vector3D(-5, -6, -5), Vector3D(0.5), 1, 0, 0));

    HallSurface *tetracolor = new HallSurface;
    Texture *tetratext = new PureColor(1, 0.2, 0.2);
    tetracolor->SetDiffuse(tetratext);

    Object3D *tetra = new STetrahedron(5, 1, tetracolor);
    tetra->ApplyTransform(RotationAxisMatrix(Normalize(Vector3D(-1, -1, 1)), 45*M_PI/180));
    world.AddObject(tetra);

    HallSurface *checks = new HallSurface;

    Matrix ref = Invert(ViewMatrix(LookAt, Viewer, Up));
    checks->SetBumpMap(new TwoSided);//(new Waves(2000, 0, 0.025, ref * ScaleMatrix(1/1000., 1/100., 1/1000.) * Invert(ref))));

    checks->SetDiffuse(new PureColor(0, 0.4, 0.8));

    Plane *facing = new Plane(-Up, -7, checks);
    world.AddObject(facing);
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}

