// ===================================================================
// SPECULAR.CPP
//	Scene derived from Figure 16.10 in Foley, van Dam, Feiner 
//	and Hughes, figure by David Kurlander.  Used by permission.
// Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================

#include "oort.h"
#include "world.h"

void
PopulateWorld(World& world)
{
    float specularities[5] = {3.0, 5.0, 10.0, 27.0, 200.0};
    float sattens[3] = {0.1, 0.25, 0.5}; 

    world.SetDepthLimit(3);
    world.SetOutputFile("specular.raw");

    world.SetViewerParameters(Vector3D(0), Vector3D(0, 0, 50), Vector3D(0, 1, 0));
    world.SetScreenWidth(8);
    world.SetScreenHeight(4);
    world.SetScreenDistance(10);

    world.SetAmbientLight(RGBColor(1));
    world.SetBackgroundColor(RGBColor(0));

    world.AddLight(new PointLight(Vector3D(-20, 40, 80), RGBColor(1), 1, 0, 0));

    for (int i = 0; i < 5; i++) {
	for (int j = 0; j < 3; j++) {
	    HallSurface *sphsurf = new HallSurface;
	    sphsurf->SetAmbient(new PureColor(RGBColor(0.1)));
	    sphsurf->SetDiffuse(new PureColor(RGBColor(0.45)));
            sphsurf->SetSpecular(new PureColor(RGBColor(sattens[j])), specularities[i]); 
	    Sphere *addme = new Sphere(Vector3D( (i-2) * 6.0, (j-1) * 6.0, 0), 2.0, sphsurf);
            world.AddObject(addme);
	}
    }
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}
