// ===================================================================
// quadrics.cpp
//	Illustrate the various quadric surfaces.
// Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================

#include "oort.h"
#include "world.h"
#include "colors.h"
#include "utils.h"

void
PopulateWorld(World& world)
{
    world.SetDepthLimit(3);
    world.SetOutputFile("quadrics.raw");

    world.SetViewerParameters(Vector3D(0), Vector3D(0, 0, 10), Vector3D(0, 1, 0));
    world.SetScreenWidth(40);
    world.SetScreenHeight(30);
    world.SetScreenDistance(10);
    world.SetAmbientLight(RGBColor(0.3, 0.3, 0.3));

    world.SetBackgroundColor(RGBColor(0));

    Limits::Threshold = 0.01;
    Limits::Small = 0.001;

    world.AddLight(new PointLight(Vector3D(0, 5, 10), RGBColor(0.8), 1, 0, 0));

    Texture *ChessBoard = new Checkerboard(5, 5,
					   new PureColor(58623/65535., 31092/65535., 2921/65535.),
    					   new PureColor(58623/65535., 53948/65535., 12894/65535.),
					   Matrix(RotationXMatrix(-90)));

    HallSurface *surf = new HallSurface;
    surf->SetBumpMap(new TwoSided);
    surf->SetDiffuse(new PureColor(Blue));
    surf->SetAmbient(new PureColor(Blue));

    Object3D *addme = new Algebraic(Ellipsoid(4, 2, 1, surf));
    addme->ApplyTransform(TranslationMatrix(Vector3D(10, 0, 0)));
    addme->ApplyTransform(RotationZMatrix(0*2*M_PI/5));
    world.AddObject(addme);

    addme = new CSGIntersection(new Plane(Vector3D(0, 1, 0), 0, surf), MakeCone(0.5, 0.5, surf));
    addme = new CSGIntersection(addme, new Plane(Vector3D(0, -1, 0), -6, surf));
    addme->ApplyTransform(RotationXMatrix(60*M_PI/180) * TranslationMatrix(Vector3D(10, 0, 0)));
    addme->ApplyTransform(RotationZMatrix(1*2*M_PI/5));
    world.AddObject(addme);

    addme = new CSGIntersection(new Plane(Vector3D(0, 1, 0), -3, surf), new Plane(Vector3D(0, -1, 0), -3, surf));
    addme = new CSGIntersection(addme, MakeCylinder(1, 1, surf));
    addme->ApplyTransform(RotationXMatrix(-30*M_PI/180) * RotationZMatrix(-20*M_PI/180));
    addme->ApplyTransform(TranslationMatrix(Vector3D(5, 0, 0)));
    addme->ApplyTransform(RotationZMatrix(2*2*M_PI/5));
    world.AddObject(addme);                              

    addme = new CSGIntersection(new Plane(Vector3D(0, 1, 0), -2, surf), new Plane(Vector3D(0, -1, 0), -2, surf));
    addme = new CSGIntersection(addme, MakeHyperboloid(1, 1, 0.5, surf));
    addme->ApplyTransform(RotationZMatrix(90*M_PI/180));
    addme->ApplyTransform(TranslationMatrix(Vector3D(-10, 0, 0)));
    world.AddObject(addme);

    addme = MakeParaboloid(1, 1, 0.2, surf);
    addme->ApplyTransform(MirrorY());
    addme = new CSGIntersection(new Plane(Vector3D(0, -1, 0), -5, surf), addme);
    addme->ApplyTransform(TranslationMatrix(Vector3D(0, 2, 0)));
    world.AddObject(addme);

    HallSurface *checks = new HallSurface;
    checks->SetDiffuse(ChessBoard);
    checks->SetDiffuse(new PureColor(0.2, 0.6, 0.7));
    checks->SetAmbient(new PureColor(0.2, 0.6, 0.7));
    world.AddObject(new Plane(Vector3D(0, 1, 0), 11.0, checks));
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}
