/******************************************************************************
* Bzr_Wrt.c - Bezier handling routines - write to file.			      *
*******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                 *
*******************************************************************************
* Written by Gershon Elber, Mar. 90.					      *
******************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "prsr_loc.h"

/*****************************************************************************
* DESCRIPTION:                                                               M
* Writes Bezier curve(s) list into file. Returns TRUE if succesful, FALSE    M
* otherwise.								     M
*   If Comment is NULL, no comment is wrriten, if "" only internal comment   M
* is written.								     M
*                                                                            *
* PARAMETERS:                                                                M
*   Crvs:       To write to file FileName.                                   M
*   FileName:   Name of file to open so we can write Crvs in.                M
*   Indent:     Primary indentation. All information will be written         M
*               from the column specified by Indent.                         M
*   Comment:    Optional, to describe the geometry.                          M
*   ErrStr:     If an error occurs, to describe the error.                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   BzrCrvWriteToFile, files, write                                          M
*****************************************************************************/
int BzrCrvWriteToFile(CagdCrvStruct *Crvs,
		      char *FileName,
		      int Indent,
		      char *Comment,
		      char **ErrStr)
{
    int i, Handler;
    FILE *f;

    if ((f = fopen(FileName, "w")) == NULL) {
	*ErrStr = "Fail to open file";
	return FALSE;
    }
    Handler = IritPrsrOpenStreamFromFile(f, FALSE,
					 IritPrsrSenseBinaryFile(FileName),
					 FALSE);

    i = BzrCrvWriteToFile2(Crvs, Handler, Indent, Comment, ErrStr);

    IritPrsrCloseStream(Handler, TRUE);

    return i;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Writes Bezier curve(s) list into file. Returns TRUE if succesful, FALSE    M
* otherwise. The file descriptor is not closed.				     M
*   If Comment is NULL, no comment is wrriten, if "" only internal comment   M
* is written.								     M
*                                                                            *
* PARAMETERS:                                                                M
*   Crvs:       To write to open stream.                                     M
*   Handler:    A handler to the open stream.				     M
*   Indent:     Primary indentation. All information will be written         M
*               from the column specified by Indent.                         M
*   Comment:    Optional, to describe the geometry.                          M
*   ErrStr:     If an error occurs, to describe the error.                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   BzrCrvWriteToFile2, files, write                                         M
*****************************************************************************/
int BzrCrvWriteToFile2(CagdCrvStruct *Crvs,
		       int Handler,
		       int Indent,
		       char *Comment,
		       char **ErrStr)
{
    int i, j, MaxCoord;

    if (Comment != NULL) {
	_IPFprintf(Handler, Indent, "#\n");
	_IPFprintf(Handler, Indent, "# cagd_lib - bezier curve(s) dump.\n");
	_IPFprintf(Handler, Indent, "#\n");
	_IPFprintf(Handler, Indent, "# %s\n", Comment);
	_IPFprintf(Handler, Indent, "#\n");
    }

    *ErrStr = NULL;

    while (Crvs) {
	MaxCoord = CAGD_NUM_OF_PT_COORD(Crvs -> PType);

	if (Crvs -> GType != CAGD_CBEZIER_TYPE) {
	    *ErrStr = "Given curve(s) is (are) not BEZIER curve(s)";
	    break;
	}
	_IPFprintf(Handler, Indent, "[CURVE BEZIER %d %c%c\n",
		Crvs -> Length,
		CAGD_IS_RATIONAL_PT(Crvs -> PType) ? 'P' : 'E',
		MaxCoord + '0');
	Indent += 4;

	for (i = 0; i < Crvs -> Length; i++) {
	    _IPFprintf(Handler, Indent, "[");
	    if (CAGD_IS_RATIONAL_PT(Crvs -> PType))
		_IPFprintf(Handler, 0, "%s ", _IPReal2Str(Crvs -> Points[0][i]));
	    for (j = 1; j <= MaxCoord; j++) {
		_IPFprintf(Handler, 0, "%s", _IPReal2Str(Crvs -> Points[j][i]));
		if (j < MaxCoord) _IPFprintf(Handler, 0, " ");
	    }
	    _IPFprintf(Handler, 0, "]\n");
	}

	Indent -= 4;
	_IPFprintf(Handler, Indent, "]\n");

	Crvs = Crvs -> Pnext;
    }

    return *ErrStr == NULL;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Writes Bezier surface(s) list into file. Returns TRUE if succesful, FALSE  M
* otherwise.								     M
*   If Comment is NULL, no comment is wrriten, if "" only internal comment   M
* is written.								     M
*                                                                            *
* PARAMETERS:                                                                M
*   Srfs:       To write to file FileName.                                   M
*   FileName:   Name of file to open so we can write Srfs in.                M
*   Indent:     Primary indentation. All information will be written         M
*               from the column specified by Indent.                         M
*   Comment:    Optional, to describe the geometry.                          M
*   ErrStr:     If an error occurs, to describe the error.                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   BzrSrfWriteToFile, files, write                                          M
*****************************************************************************/
int BzrSrfWriteToFile(CagdSrfStruct *Srfs,
		      char *FileName,
		      int Indent,
		      char *Comment,
		      char **ErrStr)
{
    int i, Handler;
    FILE *f;

    if ((f = fopen(FileName, "w")) == NULL) {
	*ErrStr = "Fail to open file";
	return FALSE;
    }
    Handler = IritPrsrOpenStreamFromFile(f, FALSE,
					 IritPrsrSenseBinaryFile(FileName),
					 FALSE);

    i = BzrSrfWriteToFile2(Srfs, Handler, Indent, Comment, ErrStr);

    IritPrsrCloseStream(Handler, TRUE);

    return i;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Writes Bezier surface(s) list into file. Returns TRUE if succesful, FALSE  M
* otherwise. The file descriptor is not closed.				     M
*   If Comment is NULL, no comment is wrriten, if "" only internal comment   M
* is written.								     M
*                                                                            *
* PARAMETERS:                                                                M
*   Srfs:       To write to open stream.                                     M
*   Handler:    A handler to the open stream.				     M
*   Indent:     Primary indentation. All information will be written         M
*               from the column specified by Indent.                         M
*   Comment:    Optional, to describe the geometry.                          M
*   ErrStr:     If an error occurs, to describe the error.                   M
*                                                                            *
* RETURN VALUE:                                                              M
*   int:        TRUE if succesful, FALSE otherwise.                          M
*                                                                            *
* KEYWORDS:                                                                  M
*   BzrSrfWriteToFile2, files, write                                         M
*****************************************************************************/
int BzrSrfWriteToFile2(CagdSrfStruct *Srfs, 
		       int Handler,
		       int Indent,
		       char *Comment,
		       char **ErrStr)
{
    int i, j, MaxCoord;

    if (Comment != NULL) {
	_IPFprintf(Handler, Indent, "#\n");
	_IPFprintf(Handler, Indent, "# cagd_lib - bezier srf(s) dump.\n");
	_IPFprintf(Handler, Indent, "#\n");
	_IPFprintf(Handler, Indent, "# %s\n", Comment);
	_IPFprintf(Handler, Indent, "#\n");
    }

    *ErrStr = NULL;

    while (Srfs) {
	MaxCoord = CAGD_NUM_OF_PT_COORD(Srfs -> PType);

	if (Srfs -> GType != CAGD_SBEZIER_TYPE) {
	    *ErrStr = "Given surface(s) is (are) not BEZIER surface(s)";
	    break;
	}
	_IPFprintf(Handler, Indent, "[SURFACE BEZIER %d %d %c%c\n",
		Srfs -> ULength, Srfs -> VLength,
		CAGD_IS_RATIONAL_PT(Srfs -> PType) ? 'P' : 'E',
		MaxCoord + '0');
	Indent += 4;

	for (i = 0; i < Srfs -> VLength * Srfs -> ULength; i++) {
	    if (i && i % Srfs -> ULength == 0)
		_IPFprintf(Handler, 0, "\n");    /* Put empty lines between raws. */

	    _IPFprintf(Handler, Indent, "[");
	    if (CAGD_IS_RATIONAL_PT(Srfs -> PType))
		_IPFprintf(Handler, 0, "%s ", _IPReal2Str(Srfs -> Points[0][i]));
	    for (j = 1; j <= MaxCoord; j++) {
		_IPFprintf(Handler, 0, "%s", _IPReal2Str(Srfs -> Points[j][i]));
		if (j < MaxCoord) _IPFprintf(Handler, 0, " ");
	    }
	    _IPFprintf(Handler, 0, "]\n");
	}

	Indent -= 4;
	_IPFprintf(Handler, Indent, "]\n");

	Srfs = Srfs -> Pnext;
    }

    return *ErrStr == NULL;
}
