/*****************************************************************************
*   Default trimmed surface drawing routine common to graphics drivers.	     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Gershon Elber				Ver 0.1, June 1993.  *
*****************************************************************************/

#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "cagd_lib.h"
#include "symb_lib.h"
#include "trim_lib.h"
#include "ip_cnvrt.h"
#include "iritgrap.h"

/*****************************************************************************
* DESCRIPTION:                                                               M
* Draw a single trimmed surface object using current modes and		     M
* transformations.							     M
*   Piecewise linear approximation is cashed under "_isoline" and "_ctlmesh" M
* attributes of PObj. Polygonal approximation is saved under "_polygons".    M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:     A trimmed surface object to draw.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IGDrawTrimSrf                                                            M
*****************************************************************************/
void IGDrawTrimSrf(IPObjectStruct *PObj)
{
    IPObjectStruct *PObjPolylines, *PObjCtlMesh, *PObjPolygons;
    IPPolygonStruct *PPolylines, *PCtlMesh, *PPolygons, *PPolygonTemp;
    RealType RelativeFineNess;

    if ((RelativeFineNess = AttrGetObjectRealAttrib(PObj,
					   "resolution")) >= IP_ATTR_BAD_REAL)
        RelativeFineNess = 1.0;
		
    if (IGGlblDrawSurfacePoly || IGGlblDrawSolid) {
	if ((PObjPolygons = AttrGetObjectObjAttrib(PObj, "_polygons"))
								    == NULL) {
	    TrimSrfStruct *TSrf,
	        *TSrfs = PObj -> U.TrimSrfs;

	    PObjPolygons = IPAllocObject("", IP_OBJ_POLY, NULL);
	    PObjPolygons -> Attrs = AttrCopyAttributes(PObj -> Attrs);
	    IP_SET_POLYGON_OBJ(PObjPolygons);

	    for (TSrf = TSrfs; TSrf != NULL; TSrf = TSrf -> Pnext) {
		RealType t;

		if ((t = AttrGetObjectRealAttrib(PObj, "u_resolution"))
							    < IP_ATTR_BAD_REAL)
		    AttrSetRealAttrib(&TSrf -> Attr, "u_resolution", t);
		if ((t = AttrGetObjectRealAttrib(PObj, "v_resolution"))
							    < IP_ATTR_BAD_REAL)
		    AttrSetRealAttrib(&TSrf -> Attr, "v_resolution", t);

		PPolygons = IritTrimSrf2Polygons(TSrf, IGGlblFourPerFlat,
						 IGGlblFineNess *
						     RelativeFineNess,
						 FALSE, TRUE,
						 IGGlblPolygonOptiApprox);
		if (PPolygons != NULL) {
		    if (PPolygons) {
			for (PPolygonTemp = PPolygons;
			     PPolygonTemp -> Pnext;
			     PPolygonTemp = PPolygonTemp -> Pnext);
			PPolygonTemp -> Pnext = PObjPolygons -> U.Pl;
			PObjPolygons -> U.Pl = PPolygons;
		    }
		}
	    }
	    if (IGGlblCacheGeom)
		AttrSetObjectObjAttrib(PObj, "_polygons", PObjPolygons, FALSE);
	}

        if (PObjPolygons != NULL) {
	    IGDrawPoly(PObjPolygons);

	    if (!IGGlblCacheGeom)
		IPFreeObject(PObjPolygons);
	}
    }
    else {
	if ((PObjPolylines = AttrGetObjectObjAttrib(PObj, "_isoline")) == NULL &&
	    IGGlblNumOfIsolines > 0) {
	    TrimSrfStruct *TrimSrf,
	        *TrimSrfs = PObj -> U.TrimSrfs;

	    PObjPolylines = IPAllocObject("", IP_OBJ_POLY, NULL);
	    PObjPolylines -> Attrs = AttrCopyAttributes(PObj -> Attrs);
	    IP_SET_POLYLINE_OBJ(PObjPolylines);
	    for (TrimSrf = TrimSrfs; TrimSrf != NULL; TrimSrf = TrimSrf -> Pnext) {
		int NumOfIso[2];

		NumOfIso[0] = -IGGlblNumOfIsolines;
		NumOfIso[1] = -IGGlblNumOfIsolines;
		PPolylines = IritTrimSrf2Polylines(TrimSrf, NumOfIso,
						   IGGlblSamplesPerCurve,
						   IGGlblPolylineOptiApprox,
						   TRUE, TRUE);

		if (PPolylines != NULL) {
		    for (PPolygonTemp = PPolylines;
			 PPolygonTemp -> Pnext;
			 PPolygonTemp = PPolygonTemp -> Pnext);
		    PPolygonTemp -> Pnext = PObjPolylines -> U.Pl;
		    PObjPolylines -> U.Pl = PPolylines;
		}
	    }
	    if (IGGlblCacheGeom)
	        AttrSetObjectObjAttrib(PObj, "_isoline", PObjPolylines, FALSE);
	}

	if (PObjPolylines != NULL) {
	    IGDrawPoly(PObjPolylines);

	    if (!IGGlblCacheGeom)
		IPFreeObject(PObjPolylines);
	}
    }

    if (IGGlblDrawSurfaceMesh) {
	if ((PObjPolylines = AttrGetObjectObjAttrib(PObj, "_ctlmesh"))
								== NULL) {
	    TrimSrfStruct *TrimSrf,
		*TrimSrfs = PObj -> U.TrimSrfs;

	    PObjCtlMesh = IPAllocObject("", IP_OBJ_POLY, NULL);
	    PObjCtlMesh -> Attrs = AttrCopyAttributes(PObj -> Attrs);
	    IP_SET_POLYLINE_OBJ(PObjCtlMesh);
	    for (TrimSrf = TrimSrfs;
		 TrimSrf != NULL;
		 TrimSrf = TrimSrf -> Pnext) {
		PCtlMesh = IritTrimSrf2CtlMesh(TrimSrf);

		for (PPolygonTemp = PCtlMesh;
		     PPolygonTemp -> Pnext;
		     PPolygonTemp = PPolygonTemp -> Pnext);
		PPolygonTemp -> Pnext = PObjCtlMesh -> U.Pl;
		PObjCtlMesh -> U.Pl = PCtlMesh;
	    }
	    AttrSetObjectObjAttrib(PObj, "_ctlmesh", PObjCtlMesh, FALSE);
	}

	IGDrawPoly(AttrGetObjectObjAttrib(PObj, "_ctlmesh"));
    }
}
