/*****************************************************************************
*   Default triangular surface drawing routine common to graphics drivers.   *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Gershon Elber				Ver 0.1, Aug. 1996.  *
*****************************************************************************/

#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "cagd_lib.h"
#include "symb_lib.h"
#include "trng_lib.h"
#include "ip_cnvrt.h"
#include "iritgrap.h"

/*****************************************************************************
* DESCRIPTION:                                                               M
* Draw a single triangular surface object using current modes and	     M
* transformations.							     M
*   Piecewise linear approximation is cashed under "_isoline" and "_ctlmesh" M
* attributes of PObj. Polygonal approximation is saved under "_polygons".    M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:     A triangular surface object to draw.                           M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IGDrawTriangSrf                                                          M
*****************************************************************************/
void IGDrawTriangSrf(IPObjectStruct *PObj)
{
    IPObjectStruct *PObjPolylines, *PObjCtlMesh, *PObjPolygons;
    IPPolygonStruct *PPolylines, *PCtlMesh, *PPolygons, *PPolygonTemp;
    RealType RelativeFineNess;

    if ((RelativeFineNess = AttrGetObjectRealAttrib(PObj,
					   "resolution")) >= IP_ATTR_BAD_REAL)
	RelativeFineNess = 1.0;

    if (IGGlblDrawSurfacePoly || IGGlblDrawSolid) {
	if ((PObjPolygons = AttrGetObjectObjAttrib(PObj, "_polygons"))
								    == NULL) {
	    TrngTriangSrfStruct *TriSrf,
	        *TriSrfs = PObj -> U.TriSrfs;

	    PObjPolygons = IPAllocObject("", IP_OBJ_POLY, NULL);
	    PObjPolygons -> Attrs = AttrCopyAttributes(PObj -> Attrs);
	    IP_SET_POLYGON_OBJ(PObjPolygons);

	    for (TriSrf = TriSrfs; TriSrf != NULL; TriSrf = TriSrf -> Pnext) {
		PPolygons = IritTriSrf2Polygons(TriSrf,	IGGlblFineNess *
						            RelativeFineNess,
						FALSE, TRUE,
						IGGlblPolygonOptiApprox);

		if (PPolygons != NULL) {
		    if (PPolygons) {
			for (PPolygonTemp = PPolygons;
			     PPolygonTemp -> Pnext;
			     PPolygonTemp = PPolygonTemp -> Pnext);
			PPolygonTemp -> Pnext = PObjPolygons -> U.Pl;
			PObjPolygons -> U.Pl = PPolygons;
		    }
		}
	    }
	    if (IGGlblCacheGeom)
		AttrSetObjectObjAttrib(PObj, "_polygons", PObjPolygons, FALSE);
	}

        if (PObjPolygons != NULL) {
	    IGDrawPoly(PObjPolygons);

	    if (!IGGlblCacheGeom)
		IPFreeObject(PObjPolygons);
	}
    }
    else {
	if ((PObjPolylines = AttrGetObjectObjAttrib(PObj, "_isoline")) == NULL &&
	    IGGlblNumOfIsolines > 0) {
	    TrngTriangSrfStruct *TriSrf,
		*TriSrfs = PObj -> U.TriSrfs;

	    PObjPolylines = IPAllocObject("", IP_OBJ_POLY, NULL);
	    PObjPolylines -> Attrs = AttrCopyAttributes(PObj -> Attrs);
	    IP_SET_POLYLINE_OBJ(PObjPolylines);
	    for (TriSrf = TriSrfs; TriSrf != NULL; TriSrf = TriSrf -> Pnext) {
		int NumOfIso[3];

		NumOfIso[0] = -IGGlblNumOfIsolines;
		NumOfIso[1] = -IGGlblNumOfIsolines;
		NumOfIso[2] = -IGGlblNumOfIsolines;
		PPolylines = IritTriSrf2Polylines(TriSrf, NumOfIso,
						  IGGlblSamplesPerCurve,
						  IGGlblPolylineOptiApprox);

		if (PPolylines != NULL) {
		    for (PPolygonTemp = PPolylines;
			 PPolygonTemp -> Pnext;
			 PPolygonTemp = PPolygonTemp -> Pnext);
		    PPolygonTemp -> Pnext = PObjPolylines -> U.Pl;
		    PObjPolylines -> U.Pl = PPolylines;
		}
	    }
	    if (IGGlblCacheGeom)
		AttrSetObjectObjAttrib(PObj, "_isoline", PObjPolylines, FALSE);
	}

        if (PObjPolylines != NULL) {
	    IGDrawPoly(PObjPolylines);

	    if (!IGGlblCacheGeom)
		IPFreeObject(PObjPolylines);
	}
    }

    if (IGGlblDrawSurfaceMesh) {
	if ((PObjCtlMesh = AttrGetObjectObjAttrib(PObj, "_ctlmesh"))
								== NULL) {
	    TrngTriangSrfStruct *TriSrf,
		*TriSrfs = PObj -> U.TriSrfs;

	    PObjCtlMesh = IPAllocObject("", IP_OBJ_POLY, NULL);
	    PObjCtlMesh -> Attrs = AttrCopyAttributes(PObj -> Attrs);
	    IP_SET_POLYLINE_OBJ(PObjCtlMesh);
	    for (TriSrf = TriSrfs; TriSrf != NULL; TriSrf = TriSrf -> Pnext) {
		PCtlMesh = IritTriSrf2CtlMesh(TriSrf);

		for (PPolygonTemp = PCtlMesh;
		     PPolygonTemp -> Pnext;
		     PPolygonTemp = PPolygonTemp -> Pnext);
		PPolygonTemp -> Pnext = PObjCtlMesh -> U.Pl;
		PObjCtlMesh -> U.Pl = PCtlMesh;
	    }
	    if (IGGlblCacheGeom)
		AttrSetObjectObjAttrib(PObj, "_ctlmesh", PObjCtlMesh, FALSE);
	}

        if (PObjCtlMesh != NULL) {
	    IGDrawPoly(PObjCtlMesh);

	    if (!IGGlblCacheGeom)
		IPFreeObject(PObjCtlMesh);
	}
    }
}
