/*****************************************************************************
* Filter to convert IRIT data files to SCN format.			     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Made by:  Gershon Elber 						     *
* Modified by:  Antonio Costa				Ver 1.0, Apr 1992    *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "iritgrap.h"
#include "getarg.h"
#include "genmat.h"
#include "ffcnvrt.h"
#include "ip_cnvrt.h"

#define DIST_IRIT_EPS	2e-4
#define SIZE_IRIT_EPS	1e-5

#ifdef NO_CONCAT_STR
static char *VersionStr =
	"Irit2Scn		Version 7.0,	Antonio Costa,\n\
	 (C) Copyright 1989/90-97 Gershon Elber, Non commercial use only.";
#else
static char *VersionStr = "Irit2Scn	" IRIT_VERSION ",	Antonio Costa,	"
	__DATE__ ",   " __TIME__ "\n" IRIT_COPYRIGHT	", Non commercial use only.";
#endif /* NO_CONCAT_STR */

static char
#ifdef IRIT_DOUBLE
    *CtrlStr = "irit2scn l%- 4%- F%-PolyOpti|FineNess!d!F o%-OutName!s g%- T%-  t%-AnimTime!F z%- DFiles!*s";
#else
    *CtrlStr = "irit2scn l%- 4%- F%-PolyOpti|FineNess!d!f o%-OutName!s g%- T%-  t%-AnimTime!f z%- DFiles!*s";
#endif /* IRIT_DOUBLE */

static char
    *OutFileName = "irit2scn";

static FILE
    *OutputFileGeom = NULL,
    *OutputFileScn = NULL;

static int
    GlblTotalPolys = 0,
    GlblDumpOnlyGeometry = FALSE;

static MatrixType CrntViewMat;			/* This is the current view! */

static int TransColorTable[][4] = {
    { /* BLACK		*/ 0,    0,   0,   0 },
    { /* BLUE		*/ 1,    0,   0, 255 },
    { /* GREEN		*/ 2,    0, 255,   0 },
    { /* CYAN		*/ 3,    0, 255, 255 },
    { /* RED		*/ 4,  255,   0,   0 },
    { /* MAGENTA 	*/ 5,  255,   0, 255 },
    { /* BROWN		*/ 6,   50,   0,   0 },
    { /* LIGHTGRAY	*/ 7,  127, 127, 127 },
    { /* DARKGRAY	*/ 8,   63,  63,  63 },
    { /* LIGHTBLUE	*/ 9,    0,   0, 255 },
    { /* LIGHTGREEN	*/ 10,   0, 255,   0 },
    { /* LIGHTCYAN	*/ 11,   0, 255, 255 },
    { /* LIGHTRED	*/ 12, 255,   0,   0 },
    { /* LIGHTMAGENTA	*/ 13, 255,   0, 255 },
    { /* YELLOW		*/ 14, 255, 255,   0 },
    { /* WHITE		*/ 15, 255, 255, 255 },
    { /* BROWN		*/ 20,  50,   0,   0 },
    { /* DARKGRAY	*/ 56,  63,  63,  63 },
    { /* LIGHTBLUE	*/ 57,   0,   0, 255 },
    { /* LIGHTGREEN	*/ 58,   0, 255,   0 },
    { /* LIGHTCYAN	*/ 59,   0, 255, 255 },
    { /* LIGHTRED	*/ 60, 255,   0,   0 },
    { /* LIGHTMAGENTA	*/ 61, 255,   0, 255 },
    { /* YELLOW		*/ 62, 255, 255,   0 },
    { /* WHITE		*/ 63, 255, 255, 255 },
    {			   -1,   0,   0,   0 }
};

static void DumpDataForSCN(IPObjectStruct *PObjects);
static void DumpOneTraversedObject(IPObjectStruct *PObj, MatrixType Mat);
static int DumpOneObject(FILE *FScn, FILE *FGeom, IPObjectStruct *PObject);
static int DumpOnePolygon(FILE *f,
			  IPPolygonStruct *PPolygon,
			  int IsPolygon);
static RealType *MapPoint(RealType *Pt);
static RealType *MapVector(RealType *Pt, RealType *Vec);
static void Irit2ScnExit(int ExitCode);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of irit2scn - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    int Error,
	HasTime = FALSE,
	FineNessFlag = FALSE,
	VerFlag = FALSE,
	OutFileFlag = FALSE,
	NumFiles = 0;
    char Line[LINE_LEN_LONG], *p,
	**FileNames = NULL;
    RealType CurrentTime;
    IPObjectStruct *PObjects;

    if ((Error = GAGetArgs(argc, argv, CtrlStr,
			   &FFCState.LinearOnePolyFlag,
			   &FFCState.FourPerFlat, &FineNessFlag,
			   &FFCState.OptimalPolygons,
			   &FFCState.FineNess, &OutFileFlag,
			   &OutFileName, &GlblDumpOnlyGeometry,
			   &FFCState.Talkative, &HasTime, &CurrentTime,
			   &VerFlag, &NumFiles, &FileNames)) != 0) {
	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	Irit2ScnExit(1);
    }

    if (VerFlag) {
	fprintf(stderr, "\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	Irit2ScnExit(0);
    }

    if (FFCState.LinearOnePolyFlag) {
	CagdSetLinear2Poly(CAGD_ONE_POLY_PER_COLIN);
    }
    else
        CagdSetLinear2Poly(CAGD_REG_POLY_PER_LIN);

    if (!NumFiles) {
	fprintf(stderr, "No data file names were given, exit.\n");
	GAPrintHowTo(CtrlStr);
	Irit2ScnExit(1);
    }

    if (!OutFileFlag) {		/* Pick the first input name as output name. */
	strcpy(Line, FileNames[0]);
	if ((p = strstr(Line, ".dat")) != NULL)	    /* Remove old file type. */
	    *p = 0;
	OutFileName = IritStrdup(Line);
    }

    /* Get the data files: */
    IritPrsrSetFlattenObjects(FALSE);
    if ((PObjects = IritPrsrGetDataFiles(FileNames, NumFiles, TRUE, FALSE)) ==
									NULL)
	Irit2ScnExit(1);
    PObjects = IritPrsrResolveInstances(PObjects);
    if (HasTime)
	AnimEvalAnimation(CurrentTime, PObjects);

    if (IritPrsrWasPrspMat)
	MatMultTwo4by4(CrntViewMat, IritPrsrViewMat, IritPrsrPrspMat);
    else
	GEN_COPY(CrntViewMat, IritPrsrViewMat, sizeof(MatrixType));

    DumpDataForSCN(PObjects);

    Irit2ScnExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps the data for SCN to stdout.         	                             *
*                                                                            *
* PARAMETERS:                                                                *
*   PObjects:   To dump into file.                                           *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpDataForSCN(IPObjectStruct *PObjects)
{
    static char *Header1[] = {
	"%",
	"% This file was automatically created from IRIT solid modeller data",
	"% using Irit2Scn - IRIT to SCN filter.",
	"%",
	"% (c) Copyright 1992 Antonio Costa, Non commercial use only.",
	"%",
	"",
	NULL
    };
    static char *Header2[] = {
	"",
	"eye  0 0 10",
	"look 0 0 0",
	"up   0 1 0",
	"fov 6",
	"",
	"light point 10 30 10",
	"",
	NULL
    };
    int i;
    char Line[128];
    MatrixType UnitMat;

    sprintf(Line, "%s.scn", OutFileName);
    if (!GlblDumpOnlyGeometry) {
	if ((OutputFileScn = fopen(Line, "w")) == NULL) {
	    fprintf(stderr, "Failed to open \"%s\".\n", Line);
	    Irit2ScnExit(2);
	}
    }
    else
	OutputFileScn = NULL;

    sprintf(Line, "%s.geom", OutFileName);
    if ((OutputFileGeom = fopen(Line, "w")) == NULL) {
#	if defined(OS2GCC) || defined(__WINNT__)
	    sprintf(Line, "%s.geo", OutFileName);
	    if ((OutputFileGeom = fopen(Line, "w")) == NULL)
#	endif /* OS2GCC || __WINNT__ */
	    {
		fprintf(stderr, "Failed to open \"%s\".\n", Line);
		Irit2ScnExit(2);
	    }
    }

    if (OutputFileScn != NULL)
	for (i = 0; Header1[i] != NULL; i++)
	    fprintf(OutputFileScn, "%s\n", Header1[i]);
    for (i = 0; Header1[i] != NULL; i++)
	fprintf(OutputFileGeom, "%s\n", Header1[i]);

    MatGenUnitMat(UnitMat);
    IPTraverseObjListHierarchy(PObjects, UnitMat, DumpOneTraversedObject);

    if (OutputFileScn != NULL) {
	fprintf(OutputFileScn, "#include \"%s\"\n", Line);
	for (i = 0; Header2[i] != NULL; i++)
	    fprintf(OutputFileScn, "%s\n", Header2[i]);
	fclose(OutputFileScn);
    }

    fclose(OutputFileGeom);

    fprintf(stderr, "\nTotal number of polygons - %d\n", GlblTotalPolys);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Call back function of IPTraverseObjListHierarchy. Called on every non    *
* list object found in hierarchy.                                            *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:       Non list object to handle.                                   *
*   Mat:        Transformation matrix to apply to this object.               *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpOneTraversedObject(IPObjectStruct *PObj, MatrixType Mat)
{
    MatrixType SaveCrntMat;
    IPObjectStruct *PObjs;

    MAT_COPY(SaveCrntMat, CrntViewMat);
    MatMultTwo4by4(CrntViewMat, Mat, CrntViewMat);

    if (IP_IS_FFGEOM_OBJ(PObj))
        PObjs = ProcessFreeForm(PObj, &FFCState);
    else
	PObjs = PObj;

    for (PObj = PObjs; PObj != NULL; PObj = PObj -> Pnext)
	GlblTotalPolys += DumpOneObject(OutputFileScn, OutputFileGeom, PObj);

    MAT_COPY(CrntViewMat, SaveCrntMat);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one object PObject to files.                                         *
*                                                                            *
* PARAMETERS:                                                                *
*   FScn:         SCN file to dump object to.                                *
*   FGeom:        Geometry file to dump object to.                           *
*   PObject:      Object to dump to file f.                                  *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static int DumpOneObject(FILE *FScn, FILE *FGeom, IPObjectStruct *PObject)
{
    static int
	ObjectSeqNum = 1;
    int i, j, Color, RGBIColor[3],
        PolyCount = 0,
	HasColor = FALSE,
	HasSrfProp = FALSE;
    char *p, Name[LINE_LEN], SrfPropString[LINE_LEN_LONG];
    RealType RGBColor[3];
    IPPolygonStruct *PList;

    if (!IP_IS_POLY_OBJ(PObject) ||
	!IP_IS_POLYGON_OBJ(PObject) ||
	PObject -> U.Pl == NULL)
	return 0;

    PList = PObject -> U.Pl;

    if (strlen(PObject -> Name) == 0)
	sprintf(Name, "Object_%d", ObjectSeqNum);
    else
	strcpy(Name, PObject -> Name);

    fprintf(FGeom, "begin %% %s\n", Name);

    if (AttrGetObjectRGBColor(PObject,
			      &RGBIColor[0], &RGBIColor[1], &RGBIColor[2])) {
	HasColor = TRUE;
	for (i = 0; i < 3; i++)
	    RGBColor[i] = RGBIColor[i];
    }
    else if ((Color = AttrGetObjectColor(PObject)) != IP_ATTR_NO_COLOR) {
	for (i = 0; TransColorTable[i][0] >= 0; i++) {
	    if (TransColorTable[i][0] == Color) {
		HasColor = TRUE;
		for (j = 0; j < 3; j++)
		    RGBColor[j] = TransColorTable[i][j+1];
		break;
	    }
	}
    }

    if ((p = AttrGetObjectStrAttrib(PObject, "SCNsurface")) != NULL) {
	HasSrfProp = TRUE;
	strcpy(SrfPropString, p);
    }

    if (HasColor || HasSrfProp) {
	if (FScn != NULL) {
	    fprintf(FScn, "#define Surface_%s \\\n", Name);
	    if (HasColor && !HasSrfProp) {
		for (i = 0; i < 3; i++)
		    RGBColor[i] /= 255.0;

		fprintf(FScn, "\tsurface matte %g %g %g\n",
			RGBColor[0],
			RGBColor[1],
			RGBColor[2]);
	    }
	    else if (!HasColor && HasSrfProp) {
		fprintf(FScn, "\t%s\n", SrfPropString);
	    }
	    else {
		for (i = 0; i < 3; i++)
		    RGBColor[i] /= 255.0;

		fprintf(FScn, "\t%s\n", SrfPropString);
		fprintf(FScn, "\t%% color %g %g %g\n",
			RGBColor[0],
			RGBColor[1],
			RGBColor[2]);
	    }
	}

	fprintf(FGeom, "Surface_%s\n", Name);
    }

    if ((p = AttrGetObjectStrAttrib(PObject, "SCNrefraction")) != NULL) {
	if (FScn != NULL)
	    fprintf(FScn, "#define Refraction_%s %s\n", Name, p);
    }

    if ((p = AttrGetObjectStrAttrib(PObject, "SCNtexture")) != NULL) {
	if (FScn != NULL) {
	    fprintf(FScn, "#define Texture_%s %s\n", Name, p);
	    fprintf(FGeom, "Texture_%s\n", Name);
	}
    }

    while (PList) {
	PolyCount += DumpOnePolygon(FGeom, PList, IP_IS_POLYGON_OBJ(PObject));
	PList =	PList -> Pnext;
    }
    fprintf(FGeom, "end\n");

    if (FFCState.Talkative)
	fprintf(stderr, "Converting \"%s\" - %d triangles.\n",
		Name, PolyCount);

    fprintf(FGeom, "\n");
    if (FScn != NULL)
	fprintf(FScn, "\n");

    ObjectSeqNum++;

    return PolyCount;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one polygon, using global Matrix transform CrntViewMat.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   FGeom:        Geometry file to dump object to.                           *
*   PPolygon:     Polygon to dump to file f.                                 *
*   IsPolygon:    Is it a polygon or a polyline?                             *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:          Number of triangles.                                       *
*****************************************************************************/
static int DumpOnePolygon(FILE *FGeom,
			  IPPolygonStruct *PPolygon,
			  int IsPolygon)
{
    int i,
	TriCount = 0;
    RealType *MappedNormal[3], *MappedPoint[3], Normal[3], Vec1[3], Vec2[3];
    IPVertexStruct *VFirst, *V1, *V2,
	*VList = PPolygon -> PVertex;

    if (VList == NULL)
	return 0;

    if (!IritPrsrIsConvexPolygon(PPolygon)) {
	static int
	    Printed = FALSE;

	if (!Printed) {
	    fprintf(stderr,
		    "\nWARNING: Non convex polygon(s) might be in data (see CONVEX in IRIT),\n\t\t\t\toutput can be wrong as the result!\n");
	    Printed = TRUE;
	}
    }

    if (IsPolygon) {
	VFirst = VList;
	V1 = VFirst -> Pnext;
	V2 = V1 -> Pnext;

	while (V2 != NULL) {
	    MappedPoint[0] = MapPoint(VFirst -> Coord);
	    MappedPoint[1] = MapPoint(V1 -> Coord);
	    MappedPoint[2] = MapPoint(V2 -> Coord);

	    /* Test for two type of degeneracies. Make sure that no two  */
	    /* points in the triangle are the same and that they are     */
	    /* not colinear.					     */
	    if (!PT_APX_EQ(MappedPoint[0], MappedPoint[1]) &&
		!PT_APX_EQ(MappedPoint[0], MappedPoint[2]) &&
		!PT_APX_EQ(MappedPoint[1], MappedPoint[2])) {

		PT_SUB(Vec1, MappedPoint[0], MappedPoint[1]);
		PT_SUB(Vec2, MappedPoint[1], MappedPoint[2]);
		PT_NORMALIZE(Vec1);
		PT_NORMALIZE(Vec2);
		CROSS_PROD(Normal, Vec1, Vec2);

		if (PT_LENGTH(Normal) > SIZE_IRIT_EPS) {
		    PT_NORMALIZE(Normal);

		    MappedNormal[0] =
		        MapVector(VFirst -> Coord, VFirst -> Normal);
		    MappedNormal[1] =
		        MapVector(V1 -> Coord, V1 -> Normal);
		    MappedNormal[2] =
		        MapVector(V2 -> Coord, V2 -> Normal);

		    if (DOT_PROD(Normal, MappedNormal[0]) < -SIZE_IRIT_EPS ||
			DOT_PROD(Normal, MappedNormal[1]) < -SIZE_IRIT_EPS ||
			DOT_PROD(Normal, MappedNormal[2]) < -SIZE_IRIT_EPS) {
			SWAP(RealType *, MappedPoint[1], MappedPoint[2]);
			SWAP(RealType *, MappedNormal[1], MappedNormal[2]);
			PT_SCALE(Normal, -1.0);
		    }

		    /* Make sure all normals are set properly: */
		    if (DOT_PROD(MappedNormal[0],
				 MappedNormal[0]) < SIZE_IRIT_EPS)
		        PT_COPY(MappedNormal[0], Normal);
		    if (DOT_PROD(MappedNormal[1],
				 MappedNormal[1]) < SIZE_IRIT_EPS)
		        PT_COPY(MappedNormal[1], Normal);
		    if (DOT_PROD(MappedNormal[2],
				 MappedNormal[2]) < SIZE_IRIT_EPS)
		        PT_COPY(MappedNormal[2], Normal);

		    TriCount++;

		    for (i = 0; i < 3; i++)
		      fprintf(FGeom,
			      "%s %10.7f %10.7f %10.7f  %9.6f %9.6f %9.6f\n",
			      i == 0 ? "triangle normal\n\t" : "\t",
				  MappedPoint[i][0],
				  MappedPoint[i][1],
				  MappedPoint[i][2],
				  MappedNormal[i][0],
				  MappedNormal[i][1],
			          MappedNormal[i][2]);
		}
	    }

	    V1 = V2;
	    V2 = V2 -> Pnext;
	}
    }

    return TriCount;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Maps the given E3 point using the CrntViewMat.			     *
*                                                                            *
* PARAMETERS:                                                                *
*   Pt:        Point to map.                                                 *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType *:   Mapped point, in local static place (for up to 3 calls).   *
*****************************************************************************/
static RealType *MapPoint(RealType *Pt)
{
    static int Count = 0;
    static RealType MappedPts[3][3];
    RealType *MappedPt = MappedPts[Count++];

    if (Count >= 3)
	Count = 0;

    MatMultVecby4by4(MappedPt, Pt, CrntViewMat);

    return MappedPt;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Maps the given E3 unit vector using the CrntViewMat.			     *
*   This routine will return a zero vector if unit normal is not computable. *
*                                                                            *
* PARAMETERS:                                                                *
*   Pt:         Position of normal.                                          *
*   Vec:        Normal at Pt.                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType *:    Mapped normal.                                            *
*****************************************************************************/
static RealType *MapVector(RealType *Pt, RealType *Vec)
{
    static int
	Count = 0,
	WasWarning = 0;
    static RealType MappedVecs[3][3];
    RealType MappedPt[3], Pt2[3], MappedPt2[3],
	*MappedVec = MappedVecs[Count++];

    if (Count >= 3)
	Count = 0;

    if (DOT_PROD(Vec, Vec) < SIZE_IRIT_EPS) {
	MappedVec[0] = MappedVec[1] = MappedVec[2] = 0.0;
	if (!WasWarning) {
	    WasWarning = 1;
	    fprintf(stderr,
	    "Non computable normals detected. Approximated from geometry.\n");
	}
    }
    else {
    	MatMultVecby4by4(MappedPt, Pt, CrntViewMat);

    	PT_ADD(Pt2, Pt, Vec);
    	MatMultVecby4by4(MappedPt2, Pt2, CrntViewMat);

    	PT_SUB(MappedVec, MappedPt2, MappedPt);
    	PT_NORMALIZE(MappedVec);
    }

    return MappedVec;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Irit2Scn exit routine.						     *
*                                                                            *
* PARAMETERS:                                                                *
*   ExitCode:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void Irit2ScnExit(int ExitCode)
{
    exit(ExitCode);
}

#ifdef DEBUG

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Dummy function to link at debugging time.                               *
*                                                                            *
* PARAMETERS:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*                                                                            *
* KEYWORDS:                                                                  *
*****************************************************************************/
void DummyLinkCagdDebug(void)
{
    IritPrsrDbg();
}

#endif /* DEBUG */
