/*****************************************************************************
* Filter to convert IRIT data files to ray shade format.		     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Gershon Elber				Ver 1.0, Sep 1991    *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "iritgrap.h"
#include "getarg.h"
#include "genmat.h"
#include "ffcnvrt.h"
#include "ip_cnvrt.h"

#define DIST_IRIT_EPS	2e-4
#define SIZE_IRIT_EPS	1e-5

#define DEFAULT_POLYLINE_WIDTH  0.05

#define CONE_SIZE	5e-4

#define MIN_RATIO	0.01

#ifdef NO_CONCAT_STR
static char *VersionStr =
	"Irit2Ray		Version 7.0,	Gershon Elber,\n\
	 (C) Copyright 1989/90-97 Gershon Elber, Non commercial use only.";
#else
static char *VersionStr = "Irit2Ray	" IRIT_VERSION ",	Gershon Elber,	"
	__DATE__ ",   " __TIME__ "\n" IRIT_COPYRIGHT ", Non commercial use only.";
#endif /* NO_CONCAT_STR */

static char
#ifdef IRIT_DOUBLE
    *CtrlStr = "irit2ray l%- 4%- G%-GridSize%d F%-PolyOpti|FineNess!d!F f%-PolyOpti|SampTol!d!F o%-OutName!s g%- p%-Zmin|Zmax!F!F P%- M%- T%- t%-AnimTime!F I%-#UIso[:#VIso[:#WIso]]!s s%-ObjSeq#!d z%- DFiles!*s";
#else
    *CtrlStr = "irit2ray l%- 4%- G%-GridSize%d F%-PolyOpti|FineNess!d!f f%-PolyOpti|SampTol!d!f o%-OutName!s g%- p%-Zmin|Zmax!f!f P%- M%- T%- t%-AnimTime!f I%-#UIso[:#VIso[:#WIso]]!s s%-ObjSeq#!d z%- DFiles!*s";
#endif /* IRIT_DOUBLE */

static char
    *GlblOutFileName = "irit2ray";

static FILE
    *OutputFileRay = NULL,
    *OutputFileGeom = NULL;

static int
    GlblTotalPolys = 0,
    GlblDumpPolylines = FALSE,
    GlblDumpOnlyGeomFile = FALSE,
    GlblObjectSeqNum = 1;

static RealType
    GlblZBBox[2] = { IRIT_INFNTY, -IRIT_INFNTY },
    GlblPolylineDepthCue[3] = { -IRIT_INFNTY, IRIT_INFNTY };

static MatrixType CrntViewMat;			/* This is the current view! */

static int TransColorTable[][4] = {
    { /* BLACK		*/ 0,    0,   0,   0 },
    { /* BLUE		*/ 1,    0,   0, 255 },
    { /* GREEN		*/ 2,    0, 255,   0 },
    { /* CYAN		*/ 3,    0, 255, 255 },
    { /* RED		*/ 4,  255,   0,   0 },
    { /* MAGENTA 	*/ 5,  255,   0, 255 },
    { /* BROWN		*/ 6,   50,   0,   0 },
    { /* LIGHTGRAY	*/ 7,  127, 127, 127 },
    { /* DARKGRAY	*/ 8,   63,  63,  63 },
    { /* LIGHTBLUE	*/ 9,    0,   0, 255 },
    { /* LIGHTGREEN	*/ 10,   0, 255,   0 },
    { /* LIGHTCYAN	*/ 11,   0, 255, 255 },
    { /* LIGHTRED	*/ 12, 255,   0,   0 },
    { /* LIGHTMAGENTA	*/ 13, 255,   0, 255 },
    { /* YELLOW		*/ 14, 255, 255,   0 },
    { /* WHITE		*/ 15, 255, 255, 255 },
    { /* BROWN		*/ 20,  50,   0,   0 },
    { /* DARKGRAY	*/ 56,  63,  63,  63 },
    { /* LIGHTBLUE	*/ 57,   0,   0, 255 },
    { /* LIGHTGREEN	*/ 58,   0, 255,   0 },
    { /* LIGHTCYAN	*/ 59,   0, 255, 255 },
    { /* LIGHTRED	*/ 60, 255,   0,   0 },
    { /* LIGHTMAGENTA	*/ 61, 255,   0, 255 },
    { /* YELLOW		*/ 62, 255, 255,   0 },
    { /* WHITE		*/ 63, 255, 255, 255 },
    {			   -1,   0,   0,   0 }
};

static void DumpDataForRayShade(IPObjectStruct *PObjects);
static void DumpOneTraversedObject(IPObjectStruct *PObj, MatrixType Mat);
static int DumpOneObject(FILE *FRay, FILE *FGeom, IPObjectStruct *PObject);
static int DumpOnePoly(FILE *FGeom,
		       IPPolygonStruct *PPoly,
		       IPObjectStruct *PObj,
		       int ObjNum,
		       char *Name,
		       RealType Width);
static int DumpOnePolygon(FILE *f, IPPolygonStruct *PPolygon, char *Name);
static int DumpOnePolyline(FILE *f,
			   IPPolygonStruct *PPolyline,
			   int ObjNum,
			   char *Name,
			   RealType Width,
			   int IsPolyline);
static void DumpCone(RealType R1,
		     RealType X1,
		     RealType Y1,
		     RealType Z1,
		     RealType R2,
		     RealType X2,
		     RealType Y2,
		     RealType Z2,
		     FILE *FGeom,
		     char *Name,
		     int ObjNum);
static RealType *MapPoint(RealType *Pt);
static RealType *MapVector(RealType *Pt, RealType *Vec);
static void Irit2RayExit(int ExitCode);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of irit2scn - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    int Error,
	HasTime = FALSE,
	SrfFineNessFlag = FALSE,
	CrvFineNessFlag = FALSE,
	VerFlag = FALSE,
	ObjSeqFlag = FALSE,
	OutFileFlag = FALSE,
        BBoxGrid = TRUE,
        GridSize = 5,
	NumOfIsolinesFlag = FALSE,
	NumFiles = 0;
    char Line[LINE_LEN_LONG], *p,
	*StrNumOfIsolines = NULL,
	**FileNames = NULL;
    RealType CurrentTime;
    IPObjectStruct *PObjects;

    if ((Error = GAGetArgs(argc, argv, CtrlStr,
			   &FFCState.LinearOnePolyFlag,
			   &FFCState.FourPerFlat, &BBoxGrid,
			   &GridSize, &SrfFineNessFlag,
			   &FFCState.OptimalPolygons,
			   &FFCState.FineNess, &CrvFineNessFlag,
			   &FFCState.CrvApproxMethod,
			   &FFCState.CrvApproxTolSamples, &OutFileFlag,
			   &GlblOutFileName, &GlblDumpOnlyGeomFile,
			   &GlblDumpPolylines,
			   &GlblPolylineDepthCue[0],
			   &GlblPolylineDepthCue[1],
			   &FFCState.DumpObjsAsPolylines,
			   &FFCState.DrawFFMesh, &FFCState.Talkative,
			   &HasTime, &CurrentTime,
			   &NumOfIsolinesFlag, &StrNumOfIsolines,
			   &ObjSeqFlag, &GlblObjectSeqNum, &VerFlag,
			   &NumFiles, &FileNames)) != 0) {

	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	Irit2RayExit(1);
    }

    if (VerFlag) {
	fprintf(stderr, "\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	Irit2RayExit(0);
    }

    FFCState.BBoxGrid = BBoxGrid ? GridSize : 0;

    if (GlblDumpPolylines ||
	FFCState.DrawFFMesh ||
	&FFCState.DumpObjsAsPolylines) {
	if (APX_EQ(GlblPolylineDepthCue[0], GlblPolylineDepthCue[1])) {
	    fprintf(stderr, "Cannot compute depth cue if Zmin == Zmax\n");
	    Irit2RayExit(1);
	}
	else {
	    /* Precompute DZ. */
	    GlblPolylineDepthCue[2] =
		GlblPolylineDepthCue[1] - GlblPolylineDepthCue[0]; 
	}
    }

    if (NumOfIsolinesFlag && StrNumOfIsolines != NULL) {
	if (sscanf(StrNumOfIsolines, "%d:%d:%d",
		   &FFCState.NumOfIsolines[0],
		   &FFCState.NumOfIsolines[1],
		   &FFCState.NumOfIsolines[2]) != 3) {
	    if (sscanf(StrNumOfIsolines, "%d:%d",
		       &FFCState.NumOfIsolines[0],
		       &FFCState.NumOfIsolines[1]) != 2) {
		if (sscanf(StrNumOfIsolines, "%d",
			   &FFCState.NumOfIsolines[1]) != 1) {
		    fprintf(stderr,
			    "Number(s) of isolines (-I) cannot be parsed.\n");
		    GAPrintHowTo(CtrlStr);
		    Irit2RayExit(1);
		}
		else {
		    FFCState.NumOfIsolines[2] =
			FFCState.NumOfIsolines[1] =
			    FFCState.NumOfIsolines[0];
		}
	    }
	    else {
		FFCState.NumOfIsolines[2] = FFCState.NumOfIsolines[0];
	    }

	}
    }

    fprintf(stderr, "%s triangles per flat will be created.\n",
	    FFCState.FourPerFlat ? "Four" : "Two");

    if (!NumFiles) {
	fprintf(stderr, "No data file names were given, exit.\n");
	GAPrintHowTo(CtrlStr);
	Irit2RayExit(1);
    }

    if (!OutFileFlag) {		/* Pick the first input name as output name. */
	strcpy(Line, FileNames[0]);
	if ((p = strstr(Line, ".dat")) != NULL)	    /* Remove old file type. */
	    *p = 0;
	GlblOutFileName = IritStrdup(Line);
    }

    /* Get the data files: */
    IritPrsrSetFlattenObjects(FALSE);
    if ((PObjects = IritPrsrGetDataFiles(FileNames, NumFiles, TRUE, FALSE)) ==
									NULL)
	Irit2RayExit(1);
    PObjects = IritPrsrResolveInstances(PObjects);
    if (HasTime)
	AnimEvalAnimation(CurrentTime, PObjects);

    if (IritPrsrWasPrspMat)
	MatMultTwo4by4(CrntViewMat, IritPrsrViewMat, IritPrsrPrspMat);
    else
	GEN_COPY(CrntViewMat, IritPrsrViewMat, sizeof(MatrixType));

    DumpDataForRayShade(PObjects);

    if (FFCState.DumpObjsAsPolylines && GlblZBBox[0] < GlblZBBox[1])
	fprintf(stderr, "Z depth cueing of polylines spans [%f : %f]\n",
		GlblZBBox[0], GlblZBBox[1]);

    Irit2RayExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps the data for Ray Shade to stdout.                                    *
*                                                                            *
* PARAMETERS:                                                                *
*   PObjects:   To dump into file.                                           *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpDataForRayShade(IPObjectStruct *PObjects)
{
    static char *Header1[] = {
	"/*",
	" * This file was automatically created from IRIT solid modeller data",
	" * using Irit2ray - IRIT to RayShade filter.",
	" *",
	" *            (c) Copyright 1991/92 Gershon Elber, Non commercial use only.",
	" */",
	"",
	NULL
    };
    static char *Header2[] = {
	"",
	"eyep   0  0 10",
	"lookp  0  0  0",
	"up     0  1  0",
	"fov 12",
	"",
	"light 1 1 1 point 10 30 10",
	"",
	NULL
    };
    int i;
    char Line[128];
    MatrixType UnitMat;

    sprintf(Line, "%s.ray", GlblOutFileName);
    if (!GlblDumpOnlyGeomFile) {
	if ((OutputFileRay = fopen(Line, "w")) == NULL) {
	    fprintf(stderr, "Failed to open \"%s\".\n", Line);
	    Irit2RayExit(2);
	}
    }
    else
	OutputFileRay = NULL;

    sprintf(Line, "%s.geom", GlblOutFileName);
    if ((OutputFileGeom = fopen(Line, "w")) == NULL) {
#	if defined(OS2GCC) || defined(__WINNT__)
	    sprintf(Line, "%s.geo", GlblOutFileName);
	    if ((OutputFileGeom = fopen(Line, "w")) == NULL)
#	endif /* OS2GCC || __WINNT__ */
	    {
		fprintf(stderr, "Failed to open \"%s\".\n", Line);
		Irit2RayExit(2);
	    }
    }

    if (OutputFileRay != NULL)
	for (i = 0; Header1[i] != NULL; i++)
	    fprintf(OutputFileRay, "%s\n", Header1[i]);
    for (i = 0; Header1[i] != NULL; i++)
	fprintf(OutputFileGeom, "%s\n", Header1[i]);

    MatGenUnitMat(UnitMat);
    IPTraverseObjListHierarchy(PObjects, UnitMat, DumpOneTraversedObject);

    if (OutputFileRay != NULL) {
	fprintf(OutputFileRay, "#include \"%s\"\n", Line);
	for (i = 0; Header2[i] != NULL; i++)
	    fprintf(OutputFileRay, "%s\n", Header2[i]);
	fclose(OutputFileRay);
    }

    fclose(OutputFileGeom);

    fprintf(stderr, "\nTotal number of polygons - %d\n", GlblTotalPolys);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Call back function of IPTraverseObjListHierarchy. Called on every non    *
* list object found in hierarchy.                                            *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:       Non list object to handle.                                   *
*   Mat:        Transformation matrix to apply to this object.               *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpOneTraversedObject(IPObjectStruct *PObj, MatrixType Mat)
{
    MatrixType SaveCrntMat;
    IPObjectStruct *PObjs;

    MAT_COPY(SaveCrntMat, CrntViewMat);
    MatMultTwo4by4(CrntViewMat, Mat, CrntViewMat);

    if (IP_IS_FFGEOM_OBJ(PObj))
        PObjs = ProcessFreeForm(PObj, &FFCState);
    else
	PObjs = PObj;

    for (PObj = PObjs; PObj != NULL; PObj = PObj -> Pnext)
	GlblTotalPolys += DumpOneObject(OutputFileRay, OutputFileGeom, PObj);

    MAT_COPY(CrntViewMat, SaveCrntMat);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one object PObject to files.                                         *
*                                                                            *
* PARAMETERS:                                                                *
*   FRay:         Ray file to dump object to.                                *
*   FGeom:        Geometry file to dump object to.                           *
*   PObject:      Object to dump to file f.                                  *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static int DumpOneObject(FILE *FRay, FILE *FGeom, IPObjectStruct *PObject)
{
    static char
        *StrAttribs[] = {
	"specpow",
	"reflect",
	"transp",
	"body",
	"index",
	NULL
    };
    static int ObjNum = 0;
    int i, j, Color, RGBIColor[3],
        PolyCount = 0,
	HasColor = FALSE,
	HasSrfProp = FALSE;
    char *p, Name[LINE_LEN], SrfPropString[LINE_LEN_LONG];
    RealType RGBColor[3],
	Width = DEFAULT_POLYLINE_WIDTH;
    IPPolygonStruct *PList;

    if (!IP_IS_POLY_OBJ(PObject) ||
	IP_IS_POINTLIST_OBJ(PObject) ||
	PObject -> U.Pl == NULL)
	return 0;

    ObjNum++;
    PList = PObject -> U.Pl;

    if (strlen(PObject -> Name) == 0 || strcmp(PObject -> Name, "_") == 0)
	sprintf(Name, "ObjSeq%d", GlblObjectSeqNum);
    else
	strcpy(Name,
	       PObject -> Name[0] == '_' ? &PObject -> Name[1]
				         : PObject -> Name);

    SrfPropString[0] = 0;
    for (i = 0; StrAttribs[i] != NULL; i++) {
	if ((p = AttrGetObjectStrAttrib(PObject, StrAttribs[i])) != NULL) {
	    strcat(SrfPropString, StrAttribs[i]);
	    strcat(SrfPropString, " ");
	    strcat(SrfPropString, p);
	    strcat(SrfPropString, " ");
	    HasSrfProp = TRUE;
	}
    }

    if ((Width = AttrGetObjectRealAttrib(PObject, "Width")) >
							    IP_ATTR_BAD_REAL)
        Width = DEFAULT_POLYLINE_WIDTH;

    if (FFCState.BBoxGrid) {
	char
	    *GridStr = AttrGetObjectStrAttrib(PObject, "GridSize");

	if (GridStr != NULL)
	    fprintf(FGeom, "name RS%s%d grid %s\n", Name, ObjNum, GridStr);
	else
	    fprintf(FGeom, "name RS%s%d grid %d %d %d\n", Name, ObjNum,
		    FFCState.BBoxGrid, FFCState.BBoxGrid, FFCState.BBoxGrid);
    }
    else
	fprintf(FGeom, "name RS%s%d list\n", Name, ObjNum);

    while (PList) {
	PolyCount += DumpOnePoly(FGeom, PList, PObject, ObjNum, Name, Width);
	PList =	PList -> Pnext;
    }
    fprintf(FGeom, "end\n");

    if (FFCState.Talkative)
	fprintf(stderr, "Converting \"%s\" - %d triangles.\n",
		Name, PolyCount);

    if (AttrGetObjectRGBColor(PObject,
			      &RGBIColor[0], &RGBIColor[1], &RGBIColor[2])) {
	HasColor = TRUE;
	for (i = 0; i < 3; i++)
	    RGBColor[i] = RGBIColor[i];
    }
    else if ((Color = AttrGetObjectColor(PObject)) != IP_ATTR_NO_COLOR) {
	for (i = 0; TransColorTable[i][0] >= 0; i++) {
	    if (TransColorTable[i][0] == Color) {
		HasColor = TRUE;
		for (j = 0; j < 3; j++)
		    RGBColor[j] = TransColorTable[i][j+1];
		break;
	    }
	}
    }

    if (HasColor || HasSrfProp) {
	if (FRay != NULL) {
	    fprintf(FRay, "surface RS%sSrfProp%d\n", Name, ObjNum);
	    if (HasColor) {
		for (i = 0; i < 3; i++)
		    RGBColor[i] /= 255.0;

		fprintf(FRay, "\tambient  %7.4f %7.4f %7.4f\n",
			0.1 * RGBColor[0],
			0.1 * RGBColor[1],
			0.1 * RGBColor[2]);
		fprintf(FRay, "\tdiffuse  %7.4f %7.4f %7.4f\n",
			0.7 * RGBColor[0],
			0.7 * RGBColor[1],
			0.7 * RGBColor[2]);
		fprintf(FRay, "\tspecular %7.4f %7.4f %7.4f\n",
			0.8, 0.8, 0.8);
	    }
	    if (HasSrfProp)
		fprintf(FRay, "\t%s\n", SrfPropString);
	}

	fprintf(FGeom, "object RS%sSrfProp%d RS%s%d", Name, ObjNum, Name, ObjNum);
    }
    else
	fprintf(FGeom, "object RS%s%d", Name, ObjNum);

    if ((p = AttrGetObjectStrAttrib(PObject, "texture")) != NULL) {
	char Scale[LINE_LEN],
	    *q = strchr(p, ',');

	/* Remove the scaling factor, if any. */
	Scale[0] = 0;
	if (q != NULL) {
	    RealType
		R;

	    *q = 0;
	    q++;

#ifdef IRIT_DOUBLE
	    if (sscanf(q, "%lf", &R))
#else
	    if (sscanf(q, "%f", &R))
#endif /* IRIT_DOUBLE */
		sprintf(Scale, " scale %f %f %f", R, R, R);
	}

	if (FRay != NULL)
	    fprintf(FRay, "#define RS%s%dTEXTURE %s%s\n",
		    Name, ObjNum, p, Scale);
	fprintf(FGeom, " texture RS%s%dTEXTURE", Name, ObjNum);
    }
    fprintf(FGeom, "\n\n");
    if (FRay != NULL)
	fprintf(FRay, "\n\n");

    GlblObjectSeqNum++;

    return PolyCount;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one polygon/line, using global transform CrntViewMat.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   FGeom:        Geometry file to dump poly to.                             *
*   PPoly:        Poly to dump to file f.                                    *
*   PObj:         Object PPoly came from to idetify a polygon from polyline. *
*   ObjNum:	  Number of object. 	                                     *
*   Name:         Name of object.	                                     *
*   Width:	  Of polylinesdumped as skinny cones/spheres.		     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:          Number of Polys.                                           *
*****************************************************************************/
static int DumpOnePoly(FILE *FGeom,
		       IPPolygonStruct *PPoly,
		       IPObjectStruct *PObj,
		       int ObjNum,
		       char *Name,
		       RealType Width)
{
    if (IP_IS_POLYGON_OBJ(PObj)) {
	if (FFCState.DumpObjsAsPolylines)
	    return DumpOnePolyline(FGeom, PPoly, ObjNum, Name, Width, FALSE);
	else
	    return DumpOnePolygon(FGeom, PPoly, Name);
    }
    else if (IP_IS_POLYLINE_OBJ(PObj)) {
	return DumpOnePolyline(FGeom, PPoly, ObjNum, Name, Width, TRUE);
    }
    else
	return 0;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one polygon, using global Matrix transform CrntViewMat.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   FGeom:        Geometry file to dump object to.                           *
*   PPolygon:     Polygon to dump to file f.                                 *
*   Name:         Name of object.                                            *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:          Number of triangles.                                       *
*****************************************************************************/
static int DumpOnePolygon(FILE *FGeom, IPPolygonStruct *PPolygon, char *Name)
{
    int i,
	TriCount = 0;
    RealType *MappedNormal[3], *MappedPoint[3], Normal[3], Vec1[3], Vec2[3];
    IPVertexStruct *VFirst, *V1, *V2,
	*VList = PPolygon -> PVertex;

    if (VList == NULL)
	return 0;

    if (!IritPrsrIsConvexPolygon(PPolygon)) {
	static int
	    Printed = FALSE;

	if (!Printed) {
	    fprintf(stderr,
		    "\nWARNING: Non convex polygon(s) might be in data (see CONVEX in IRIT),\n\t\t\t\toutput can be wrong as the result!\n");
	    Printed = TRUE;
	}
    }

    VFirst = VList;
    V1 = VFirst -> Pnext;
    V2 = V1 -> Pnext;
    
    while (V2 != NULL) {
	MappedPoint[0] = MapPoint(VFirst -> Coord);
	MappedPoint[1] = MapPoint(V1 -> Coord);
	MappedPoint[2] = MapPoint(V2 -> Coord);
	
	/* Test for two type of degeneracies. Make sure that no two  */
	/* points in the triangle are the same and that they are     */
	/* not colinear.					     */
	if (!PT_APX_EQ(MappedPoint[0], MappedPoint[1]) &&
	    !PT_APX_EQ(MappedPoint[0], MappedPoint[2]) &&
	    !PT_APX_EQ(MappedPoint[1], MappedPoint[2])) {
	    
	    PT_SUB(Vec1, MappedPoint[0], MappedPoint[1]);
	    PT_SUB(Vec2, MappedPoint[1], MappedPoint[2]);
	    PT_NORMALIZE(Vec1);
	    PT_NORMALIZE(Vec2);
	    CROSS_PROD(Normal, Vec1, Vec2);
	    
	    if (PT_LENGTH(Normal) > SIZE_IRIT_EPS) {
		PT_NORMALIZE(Normal);
		
		MappedNormal[0] =
		    MapVector(VFirst -> Coord, VFirst -> Normal);
		MappedNormal[1] =
		    MapVector(V1 -> Coord, V1 -> Normal);
		MappedNormal[2] =
		    MapVector(V2 -> Coord, V2 -> Normal);
		
		if (DOT_PROD(Normal, MappedNormal[0]) < -SIZE_IRIT_EPS ||
		    DOT_PROD(Normal, MappedNormal[1]) < -SIZE_IRIT_EPS ||
		    DOT_PROD(Normal, MappedNormal[2]) < -SIZE_IRIT_EPS) {
		    SWAP(RealType *, MappedPoint[1], MappedPoint[2]);
		    SWAP(RealType *, MappedNormal[1], MappedNormal[2]);
		    PT_SCALE(Normal, -1.0);
		}
		
		/* Make sure all normals are set properly: */
		if (DOT_PROD(MappedNormal[0], MappedNormal[0]) < SIZE_IRIT_EPS)
		    PT_COPY(MappedNormal[0], Normal);
		if (DOT_PROD(MappedNormal[1], MappedNormal[1]) < SIZE_IRIT_EPS)
		    PT_COPY(MappedNormal[1], Normal);
		if (DOT_PROD(MappedNormal[2], MappedNormal[2]) < SIZE_IRIT_EPS)
		    PT_COPY(MappedNormal[2], Normal);
		
		TriCount++;
		
		for (i = 0; i < 3; i++)
		    fprintf(FGeom,
			    "%s %10.7f %10.7f %10.7f  %9.6f %9.6f %9.6f\n",
			    i == 0 ? "    triangle" : "\t    ",
			    MappedPoint[i][0],
			    MappedPoint[i][1],
			    MappedPoint[i][2],
			    MappedNormal[i][0],
			    MappedNormal[i][1],
			    MappedNormal[i][2]);
	    }
	}
	
	V1 = V2;
	V2 = V2 -> Pnext;
    }

    return TriCount;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one polyline, using global Matrix transform CrntViewMat.	     *
*                                                                            *
* PARAMETERS:                                                                *
*   FGeomaaa:     Geometry file to dump polyline to.  		             *
*   PPolyline:    Polyline to dump to file f.                                *
*   ObjNum:	  Number of object. 	                                     *
*   Name:         Name of object.                                            *
*   Width:	  Of polyline.						     *
*   IsPolyline:   Is it a polyline?					     *
*                                                                            *
* RETURN VALUE:                                                              *
*   int:          Number of polylines.                                       *
*****************************************************************************/
static int DumpOnePolyline(FILE *FGeom,
			   IPPolygonStruct *PPolyline,
			   int ObjNum,
			   char *Name,
			   RealType Width,
			   int IsPolyline)
{
    int DumpedSphere = FALSE;
    RealType *MappedPoint[2];
    IPVertexStruct *V1, *V2,
	*VList = PPolyline -> PVertex;

    if (VList == NULL)
	return 0;

    V1 = VList;
    V2 = V1 -> Pnext;
    
    while (V2 != NULL) {
	MappedPoint[0] = MapPoint(V1 -> Coord);
	MappedPoint[1] = MapPoint(V2 -> Coord);

	if (GlblZBBox[1] < MappedPoint[0][2])
	    GlblZBBox[1] = MappedPoint[0][2];
	if (GlblZBBox[0] > MappedPoint[0][2])
	    GlblZBBox[0] = MappedPoint[0][2];

	if (!PT_APX_EQ(MappedPoint[0], MappedPoint[1]) &&
	    (IsPolyline || !IP_IS_INTERNAL_VRTX(V1))) {
	    DumpCone(Width * MAX(MappedPoint[0][2] - GlblPolylineDepthCue[0],
				 MIN_RATIO) / GlblPolylineDepthCue[2],
		     MappedPoint[0][0],
		     MappedPoint[0][1],
		     MappedPoint[0][2],
		     Width * MAX(MappedPoint[1][2] - GlblPolylineDepthCue[0],
				 MIN_RATIO) / GlblPolylineDepthCue[2],
		     MappedPoint[1][0],
		     MappedPoint[1][1],
		     MappedPoint[1][2],
		     FGeom, Name, ObjNum);

	    if (!DumpedSphere)
		fprintf(FGeom,
			"sphere RS%sSrfProp%d %7.5f %7.5f %7.5f %7.5f\n",
			Name, ObjNum, Width * MAX(MappedPoint[0][2] -
					  GlblPolylineDepthCue[0],
					  MIN_RATIO) / GlblPolylineDepthCue[2],
			MappedPoint[0][0],
			MappedPoint[0][1],
			MappedPoint[0][2]);
	    fprintf(FGeom,
		    "sphere RS%sSrfProp%d %7.5f %7.5f %7.5f %7.5f\n",
		    Name, ObjNum, Width * MAX(MappedPoint[1][2] -
				      GlblPolylineDepthCue[0],
				      MIN_RATIO) / GlblPolylineDepthCue[2],
		    MappedPoint[1][0],
		    MappedPoint[1][1],
		    MappedPoint[1][2]);
	    DumpedSphere = TRUE;
	}
	else {
	    DumpedSphere = FALSE;
	}
	
	V1 = V2;
	V2 = V2 -> Pnext;
    }

    if (!IsPolyline && !IP_IS_INTERNAL_VRTX(V1)) {
	MappedPoint[0] = MapPoint(V1 -> Coord);
	MappedPoint[1] = MapPoint(VList -> Coord);

	if (!PT_APX_EQ(MappedPoint[0], MappedPoint[1])) {
	    DumpCone(Width * MAX(MappedPoint[0][2] - GlblPolylineDepthCue[0],
				 MIN_RATIO) / GlblPolylineDepthCue[2],
		     MappedPoint[0][0],
		     MappedPoint[0][1],
		     MappedPoint[0][2],
		     Width * MAX(MappedPoint[1][2] - GlblPolylineDepthCue[0],
				 MIN_RATIO) / GlblPolylineDepthCue[2],
		     MappedPoint[1][0],
		     MappedPoint[1][1],
		     MappedPoint[1][2],
		     FGeom, Name, ObjNum);

	    if (!DumpedSphere)
		fprintf(FGeom,
			"sphere RS%sSrfProp%d %7.5f %7.5f %7.5f %7.5f\n",
			Name, ObjNum, Width * MAX(MappedPoint[0][2] -
					  GlblPolylineDepthCue[0],
					  MIN_RATIO) / GlblPolylineDepthCue[2],
			MappedPoint[0][0],
			MappedPoint[0][1],
			MappedPoint[0][2]);
	    fprintf(FGeom,
		    "sphere RS%sSrfProp%d %7.5f %7.5f %7.5f %7.5f\n",
		    Name, ObjNum, Width * MAX(MappedPoint[1][2] -
				      GlblPolylineDepthCue[0],
				      MIN_RATIO) / GlblPolylineDepthCue[2],
		    MappedPoint[1][0],
		    MappedPoint[1][1],
		    MappedPoint[1][2]);
	}
    }

    return 1;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps a truncated cone (can degenerate to a cylinder or nothing).          *
*                                                                            *
* PARAMETERS:                                                                *
*   R1:          Radius of first base of cone.                               *
*   X1, Y1, Z1:  Center of first base of cone.                               *
*   R2:          Radius of second base of cone.                              *
*   X2, Y2, Z2:  Center of second base of cone.                              *
*   FGeom:       Geometry file when this cone goes to.                       *
*   Name:        Name of object                                              *
*   ObjNum:	 Number of object. 	                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpCone(RealType R1,
		     RealType X1,
		     RealType Y1,
		     RealType Z1,
		     RealType R2,
		     RealType X2,
		     RealType Y2,
		     RealType Z2,
		     FILE *FGeom,
		     char *Name,
		     int ObjNum)
{
    RealType
	Len = sqrt(SQR(X1 - X2) + SQR(Y1 - Y2) + SQR(Z1 - Z2));

    if (Len < CONE_SIZE)
	return;

    if (ABS(R1 - R2) < CONE_SIZE) {
	fprintf(FGeom,
		"cylinder RS%sSrfProp%d %7.5f  %7.5f %7.5f %7.5f  %7.5f %7.5f %7.5f\n",
		Name, ObjNum, R1, X1, Y1, Z1, X2, Y2, Z2);
    }
    else {
	fprintf(FGeom,
		"cone RS%sSrfProp%d %7.5f %7.5f %7.5f %7.5f  %7.5f %7.5f %7.5f %7.5f\n",
		Name, ObjNum, R1, X1, Y1, Z1, R2, X2, Y2, Z2);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Maps the given E3 point using the CrntViewMat.			     *
*                                                                            *
* PARAMETERS:                                                                *
*   Pt:        Point to map.                                                 *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType *:   Mapped point, in local static place (for up to 3 calls).   *
*****************************************************************************/
static RealType *MapPoint(RealType *Pt)
{
    static int Count = 0;
    static RealType MappedPts[3][3];
    RealType
	*MappedPt = MappedPts[Count++];

    if (Count >= 3)
	Count = 0;

    MatMultVecby4by4(MappedPt, Pt, CrntViewMat);

    return MappedPt;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Maps the given E3 unit vector using the CrntViewMat.			     *
*   This routine will return a zero vector if unit normal is not computable. *
*                                                                            *
* PARAMETERS:                                                                *
*   Pt:         Position of normal.                                          *
*   Vec:        Normal at Pt.                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType *:    Mapped normal.                                            *
*****************************************************************************/
static RealType *MapVector(RealType *Pt, RealType *Vec)
{
    static int
	Count = 0,
	WasWarning = 0;
    static RealType MappedVecs[3][3];
    RealType MappedPt[3], Pt2[3], MappedPt2[3],
	*MappedVec = MappedVecs[Count++];

    if (Count >= 3)
	Count = 0;

    if (DOT_PROD(Vec, Vec) < SIZE_IRIT_EPS) {
	MappedVec[0] = MappedVec[1] = MappedVec[2] = 0.0;
	if (!WasWarning) {
	    WasWarning = 1;
	    fprintf(stderr, "Non computable normals detected. Approximated from geometry.\n");
	}
    }
    else {
    	MatMultVecby4by4(MappedPt, Pt, CrntViewMat);

    	PT_ADD(Pt2, Pt, Vec);
    	MatMultVecby4by4(MappedPt2, Pt2, CrntViewMat);

    	PT_SUB(MappedVec, MappedPt2, MappedPt);
    	PT_NORMALIZE(MappedVec);
    }

    return MappedVec;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Irit2Ray exit routine.						     *
*                                                                            *
* PARAMETERS:                                                                *
*   ExitCode:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void Irit2RayExit(int ExitCode)
{
    exit(ExitCode);
}

#ifdef DEBUG

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Dummy function to link at debugging time.                               *
*                                                                            *
* PARAMETERS:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*                                                                            *
* KEYWORDS:                                                                  *
*****************************************************************************/
void DummyLinkCagdDebug(void)
{
    IritPrsrDbg();
}

#endif /* DEBUG */
