/*****************************************************************************
* Conversion routines from curvesand surfaces to polygons and polylines.     *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Written by:  Gershon Elber				Ver 1.0, Apr 1992    *
*****************************************************************************/

#include "irit_sm.h"
#include "iritgrap.h"
#include "iritprsr.h"
#include "allocate.h"
#include "ffcnvrt.h"
#include "ip_cnvrt.h"

FreeformConvState FFCState = {
    FALSE,   /* Talkative */
    FALSE,   /* DumpObjsAsPolylines */
    TRUE,    /* DrawFFGeom */
    FALSE,   /* DrawFFMesh */
    { IG_DEFAULT_NUM_OF_ISOLINES,
	  IG_DEFAULT_NUM_OF_ISOLINES,
	      IG_DEFAULT_NUM_OF_ISOLINES },
    IG_DEFAULT_SAMPLES_PER_CURVE,
    SYMB_CRV_APPROX_UNIFORM,  /* CrvApproxMethod */
    FALSE,   /* ShowIntrnal */
    FALSE,   /* CubicCrvsAprox */
    IG_DEFAULT_POLYGON_FINENESS, /* FineNess */
    FALSE,   /* ComputeUV */
    TRUE,    /* ComputeNrml */
    FALSE,   /* FourPerFlat */
    0,       /* OptimalPolygons */
    FALSE,   /* BBoxGrid */
    TRUE,    /* LinearOnePolyFlag */
    FALSE
};

/*****************************************************************************
* DESCRIPTION:                                                               M
* Convert a curve into a polyline.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   Crv:           A curve to piecewise linear approximate.                  M
*   DrawCurve:     Do we want to draw the curve?                             M
*   DrawCtlPoly:   Do we want to draw its control polygon?                   M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPPolygonStruct *:   Polyline(s) representing the curve and its          M
*                        control polygon.                                    M
* KEYWORDS:                                                                  M
*   Curve2Polylines                                                          M
*****************************************************************************/
IPPolygonStruct *Curve2Polylines(CagdCrvStruct *Crv,
				 int DrawCurve,
				 int DrawCtlPoly,
				 CagdRType TolSamples,
				 SymbCrvApproxMethodType Method)
{
    IPPolygonStruct *Poly,
	*PolyHead =
	    DrawCurve ? IritCurve2Polylines(Crv, TolSamples, Method)
		      : NULL;

    if (DrawCtlPoly) {
	Poly = IritCurve2CtlPoly(Crv);
	Poly -> Pnext = PolyHead;
	PolyHead = Poly;
    }

    return PolyHead;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a single surface into polylines, NumOfIsolines isolines in each   M
* axis into a polyline object list.					     M
*                                                                            *
* PARAMETERS:                                                                M
*   Srf:           A surface to piecewise linear approximate.                M
*   DrawSurface:   Do we want to draw the surface?                           M
*   DrawMesh:      Do we want to draw its control mesh?                      M
*   NumOfIsolines: Number of isocurves in each parametric direction.         M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPPolygonStruct *:  Polyline(s) representing the surface and its         M
*                       control mesh.	                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   Surface2Polylines                                                        M
*****************************************************************************/
IPPolygonStruct *Surface2Polylines(CagdSrfStruct *Srf,
				   int DrawSurface,
				   int DrawMesh,
				   int NumOfIsolines[2],
				   CagdRType TolSamples,
				   SymbCrvApproxMethodType Method)
{
    IPPolygonStruct
	*PolyHead = DrawSurface ?
	    IritSurface2Polylines(Srf, NumOfIsolines, TolSamples, Method) :
	    NULL;

    if (DrawMesh)
	PolyHead = IritPrsrAppendPolyLists(IritSurface2CtlMesh(Srf),
					   PolyHead);

    return PolyHead;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a single trimmed surface into polylines, NumOfIsolines isolines   M
* in each axis into a polyline object list.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   TrimSrf:       A trimmed surface to piecewise linear approximate.        M
*   DrawSurface:   Do we want to draw the surface?                           M
*   DrawMesh:      Do we want to draw its control mesh?                      M
*   NumOfIsolines: Number of isocurves in each parametric direction.         M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPPolygonStruct *:  Polyline(s) representing the trimmed surface and its M
*                       control mesh.	                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   TSrf2Polylines                                                           M
*****************************************************************************/
IPPolygonStruct *TSrf2Polylines(TrimSrfStruct *TrimSrf,
				int DrawSurface,
				int DrawMesh,
				int NumOfIsolines[2],
				CagdRType TolSamples,
				SymbCrvApproxMethodType Method)
{
    IPPolygonStruct
	*PolyHead = DrawSurface ?
	    IritTrimSrf2Polylines(TrimSrf, NumOfIsolines, TolSamples,
				  Method, TRUE, TRUE) :
	    NULL;

    if (DrawMesh)
	PolyHead = IritPrsrAppendPolyLists(IritSurface2CtlMesh(TrimSrf -> Srf),
					   PolyHead);

    return PolyHead;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a single trivariate function into polylines, NumOfIsolines        M
* isolines in each axis into a polyline object list.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   Trivar:        A trivariate function to piecewise linear approximate.    M
*   DrawTrivar:    Do we want to draw the trivariate?                        M
*   DrawMesh:      Do we want to draw its control mesh?                      M
*   NumOfIsolines: Number of isocurves in each parametric direction.         M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPPolygonStruct *:  Polyline(s) representing the trivariate and its      M
*                       control mesh.	                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   TV2Polylines                                                             M
*****************************************************************************/
IPPolygonStruct *TV2Polylines(TrivTVStruct *Trivar,
			      int DrawTrivar,
			      int DrawMesh,
			      int NumOfIsolines[3],
			      CagdRType TolSamples,
			      SymbCrvApproxMethodType Method)
{
    IPPolygonStruct
	*PolyHead = DrawTrivar ?
	    IritTrivar2Polylines(Trivar, NumOfIsolines, TolSamples, Method) :
	    NULL;

    if (DrawMesh)
	PolyHead = IritPrsrAppendPolyLists(IritTrivar2CtlMesh(Trivar),
					   PolyHead);

    return PolyHead;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a single triangular surface into polylines with SamplesPerCurve   M
* samples, NumOfIsolines isolines into a polyline object list.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   TriSrf:           A triangular surface to piecewise linear approximate.  M
*   DrawSurface:      Do we want to draw the surface?                        M
*   DrawMesh:         Do we want to draw its control mesh?                   M
*   NumOfIsolines:    Number of isocurves in each parametric direction.      M
*   SamplesPerCurve:  Number of samples on curve.                            M
*   OptimalPolyline:  Do we want optimal yet expensive sampling?             M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPPolygonStruct *:  Polyline(s) representing the triangular surface and  M
*                       its control mesh.	                             M
*                                                                            *
* KEYWORDS:                                                                  M
*   Surface2Polylines                                                        M
*****************************************************************************/
IPPolygonStruct *TriangSrf2Polylines(TrngTriangSrfStruct *TriSrf,
				     int DrawSurface,
				     int DrawMesh,
				     int NumOfIsolines[3],
				     int SamplesPerCurve,
				     int OptimalPolylines)
{
    IPPolygonStruct
	*PolyHead = DrawSurface ?
	    IritTriSrf2Polylines(TriSrf, NumOfIsolines, SamplesPerCurve,
				 OptimalPolylines) :
	    NULL;

    if (DrawMesh)
	PolyHead = IritPrsrAppendPolyLists(IritTriSrf2CtlMesh(TriSrf),
					   PolyHead);

    return PolyHead;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a single triangular surface into polylines with SamplesPerCurve   M
* samples, NumOfIsolines isolines into a polyline object list.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   Model:         A model to piecewise linear approximate.		     M
*   DrawSurface:   Do we want to draw the surface?                           M
*   DrawMesh:      Do we want to draw its control mesh?                      M
*   NumOfIsolines: Number of isocurves in each parametric direction.         M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPPolygonStruct *:  Polyline(s) representing the triangular surface and  M
*                       its control mesh.	                             M
*                                                                            *
* KEYWORDS:                                                                  M
*   Surface2Polylines                                                        M
*****************************************************************************/
IPPolygonStruct *Model2Polylines(MdlModelStruct *Model,
				 int DrawSurface,
				 int DrawMesh,
				 int NumOfIsolines[2],
				 CagdRType TolSamples,
				 SymbCrvApproxMethodType Method)

{
    TrimSrfStruct *TSrf,
	*TrimSrfs = MdlTrimConvert(Model);
    IPPolygonStruct
        *RetPolys = NULL;

    for (TSrf = TrimSrfs; TSrf != NULL; TSrf = TSrf -> Pnext) {
	IPPolygonStruct
	    *Polys = TSrf2Polylines(TSrf, DrawSurface, DrawMesh,
				    NumOfIsolines, TolSamples, Method);

	RetPolys = IritPrsrAppendPolyLists(RetPolys, Polys);
    }

    TrimSrfFreeList(TrimSrfs);

    return RetPolys;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a whole set of geometry to polylines, in place.                   M
*                                                                            *
* PARAMETERS:                                                                M
*   FreeForms:     Crvs/Srfs/Trimmed Srfs/Trivariates read from a file by    M
*                  the irit parser.					     M
*   Talkative:     Do we want some more information printed.		     M
*   DrawGeom:      Do we want to draw the geometry?                          M
*   DrawMesh:      Do we want to draw its control mesh?                      M
*   NumOfIsolines: Number of isocurves in each parametric direction.         M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Polyline(s) representing the trivariate and its      M
*                       control mesh.	                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   FreeForm2Polylines                                                       M
*****************************************************************************/
IPObjectStruct *FreeForm2Polylines(IritPrsrFreeFormStruct *FreeForms,
				   int Talkative,
				   int DrawGeom,
				   int DrawMesh,
				   int NumOfIsolines[3],
				   CagdRType TolSamples,
				   SymbCrvApproxMethodType Method)
{
    IPObjectStruct *PObj,
	*CrvObjs = FreeForms -> CrvObjs,
	*SrfObjs = FreeForms -> SrfObjs,
	*TrimSrfObjs = FreeForms -> TrimSrfObjs,
	*TrivarObjs = FreeForms -> TrivarObjs,
	*TriSrfObjs = FreeForms -> TriSrfObjs,
	*ModelObjs = FreeForms -> ModelObjs;
    CagdCrvStruct *Crv, *Crvs;
    CagdSrfStruct *Srf, *Srfs;
    TrimSrfStruct *TrimSrf, *TrimSrfs;
    TrivTVStruct *TV, *TVs;
    TrngTriangSrfStruct *TriSrf, *TriSrfs;
    IPPolygonStruct *PPolygon;

    if (CrvObjs) {
	for (PObj = CrvObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing curve object \"%s\"\n",
			PObj -> Name);

	    Crvs = PObj -> U.Crvs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYLINE_OBJ(PObj);
	    for (Crv = Crvs; Crv != NULL; Crv = Crv -> Pnext) {
		PPolygon = Curve2Polylines(Crv, DrawGeom, DrawMesh,
					   TolSamples, Method);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    CagdCrvFreeList(Crvs);
	}
    }

    if (SrfObjs) {
	for (PObj = SrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing surface object \"%s\"\n",
			PObj -> Name);
	    
	    Srfs = PObj -> U.Srfs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYLINE_OBJ(PObj);
	    for (Srf = Srfs; Srf != NULL; Srf = Srf -> Pnext) {
		PPolygon = Surface2Polylines(Srf, DrawGeom, DrawMesh,
					     NumOfIsolines, TolSamples,
					     Method);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    CagdSrfFreeList(Srfs);
	}
    }

    if (TrimSrfObjs) {
	for (PObj = TrimSrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing trimmed surface object \"%s\"\n",
			PObj -> Name);
		
	    TrimSrfs = PObj -> U.TrimSrfs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYLINE_OBJ(PObj);
	    for (TrimSrf = TrimSrfs;
		 TrimSrf != NULL;
		 TrimSrf = TrimSrf -> Pnext) {
		PPolygon = TSrf2Polylines(TrimSrf, DrawGeom, DrawMesh,
					  NumOfIsolines, TolSamples,
					  Method);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    TrimSrfFreeList(TrimSrfs);
	}
    }

    if (TrivarObjs) {
	for (PObj = TrivarObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing trivariate object \"%s\"\n",
			PObj -> Name);
		
	    TVs = PObj -> U.Trivars;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYLINE_OBJ(PObj);
	    for (TV = TVs; TV != NULL; TV = TV -> Pnext) {
		PPolygon = TV2Polylines(TV, DrawGeom, DrawMesh, NumOfIsolines,
					TolSamples, Method);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    TrivTVFreeList(TVs);
	}
    }

    if (TriSrfObjs) {
        int SamplesPerCurve = Method == SYMB_CRV_APPROX_UNIFORM ?
			      (int) TolSamples : IG_DEFAULT_SAMPLES_PER_CURVE;

	for (PObj = TriSrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing triangular surface object \"%s\"\n",
			PObj -> Name);
	    
	    TriSrfs = PObj -> U.TriSrfs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYLINE_OBJ(PObj);
	    for (TriSrf = TriSrfs; TriSrf != NULL; TriSrf = TriSrf -> Pnext) {
		PPolygon = TriangSrf2Polylines(TriSrf, DrawGeom, DrawMesh,
					       NumOfIsolines, SamplesPerCurve,
					       FALSE);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    TrngTriSrfFreeList(TriSrfs);
	}
    }

    if (ModelObjs) {
	IPObjectStruct *PObjTmp,
	    *PTrimSrfObjs = NULL;

	/* Convert to a set of trimmed surfaces and call recursively. */
	for (PObj = ModelObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    TrimSrfStruct *TSrf,
		*TrimSrfs = MdlTrimConvert(PObj -> U.Mdls);

	    for (TSrf = TrimSrfs; TSrf != NULL; ) {
		TrimSrfStruct
		    *TSrfTmp = TSrf -> Pnext;

		TSrf -> Pnext = NULL;
		PObjTmp = GenTRIMSRFObject(TSrf);
		LIST_PUSH(PObjTmp, PTrimSrfObjs);

		TSrf = TSrfTmp;
	    }
	}

	IPFreeObjectList(ModelObjs);
	FreeForms -> TrimSrfObjs = PTrimSrfObjs;
	FreeForms -> ModelObjs = NULL;
	return FreeForm2Polylines(FreeForms, Talkative, DrawGeom, DrawMesh,
				  NumOfIsolines, TolSamples, Method);
    }

    return IritPrsrConcatFreeForm(FreeForms);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a whole set of geometry to cubic bezier curves, in place.         M
*                                                                            *
* PARAMETERS:                                                                M
*   FreeForms:     Crvs/Srfs/Trimmed Srfs/Trivariates read from a file by    M
*                  the irit parser.					     M
*   Talkative:     Do we want some more information printed.		     M
*   DrawGeom:      Do we want to draw the geometry?                          M
*   DrawMesh:      Do we want to draw its control mesh?                      M
*   NumOfIsolines: Number of isocurves in each parametric direction.         M
*   TolSamples:    Tolerance of approximation error (Method = 2) or          M
*                  Number of samples to compute on polyline (Method = 0, 1). M
*   Method:        0 - TolSamples are set uniformly in parametric space,     M
*                  1 - TolSamples are set optimally, considering the curve's M
*		       curvature.					     M
*		   2 - TolSamples sets the maximum error allowed between the M
*		       piecewise linear approximation and original curve.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Polyline(s) representing the trivariate and its      M
*                       control mesh.	                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   FreeForm2CubicBzr                                                        M
*****************************************************************************/
IPObjectStruct *FreeForm2CubicBzr(IritPrsrFreeFormStruct *FreeForms,
				  int Talkative,
				  int DrawGeom,
				  int DrawMesh,
				  int NumOfIsolines[3],
				  CagdRType TolSamples,
				  SymbCrvApproxMethodType Method)
{
    IPObjectStruct *PObj,
	*CrvObjs = FreeForms -> CrvObjs,
	*SrfObjs = FreeForms -> SrfObjs,
	*TrimSrfObjs = FreeForms -> TrimSrfObjs,
	*TrivarObjs = FreeForms -> TrivarObjs,
	*TriSrfObjs = FreeForms -> TriSrfObjs,
	*ModelObjs = FreeForms -> ModelObjs;
    IPPolygonStruct *PPolygon;
    CagdCrvStruct *CubicBzrCrvs;

    /* Convert curves and surfaces into cubic polynomials (bezier). */
    if (CrvObjs) {
	for (PObj = CrvObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing curve object \"%s\"\n",
			PObj -> Name);

	    PPolygon = NULL;
	    CubicBzrCrvs = IritCurvesToCubicBzrCrvs(PObj -> U.Crvs, &PPolygon,
						    DrawGeom, DrawMesh,
						    IRIT_INFNTY);

	    if (CubicBzrCrvs) {
		CagdCrvFreeList(PObj -> U.Crvs);
		PObj -> U.Crvs = CubicBzrCrvs;
		PObj -> ObjType = IP_OBJ_CURVE;
	    }
	    else
		PObj -> ObjType = IP_OBJ_UNDEF;

	    if (PPolygon) {
		PObj -> Pnext = IPAllocObject("Mesh", IP_OBJ_POLY,
					      PObj -> Pnext);
		strcat(PObj -> Pnext -> Name, PObj -> Name);
		IP_SET_POLYLINE_OBJ(PObj -> Pnext);
		PObj = PObj -> Pnext;
		PObj -> U.Pl = PPolygon;
	    }
	}
    }

    if (SrfObjs) {
	for (PObj = SrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing surface object \"%s\"\n",
			PObj -> Name);

	    PPolygon = NULL;
	    CubicBzrCrvs =
		IritSurfacesToCubicBzrCrvs(PObj -> U.Srfs, &PPolygon,
					   DrawGeom, DrawMesh, NumOfIsolines,
					   IRIT_INFNTY);
	    if (CubicBzrCrvs) {
		CagdSrfFreeList(PObj -> U.Srfs);
		PObj -> U.Crvs = CubicBzrCrvs;
		PObj -> ObjType = IP_OBJ_CURVE;
	    }
	    else
		PObj -> ObjType = IP_OBJ_UNDEF;

	    if (PPolygon) {
		PObj -> Pnext = IPAllocObject("", IP_OBJ_POLY, PObj -> Pnext);
		IP_SET_POLYLINE_OBJ(PObj -> Pnext);
		PObj = PObj -> Pnext;
		PObj -> U.Pl = PPolygon;
	    }
	}
    }

    if (TrimSrfObjs) {
	for (PObj = TrimSrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing trimmed surface object \"%s\"\n",
			PObj -> Name);

	    PPolygon = NULL;
	    CubicBzrCrvs =
		IritTrimSrfsToCubicBzrCrvs(PObj -> U.TrimSrfs, &PPolygon,
					   DrawGeom, DrawMesh, NumOfIsolines,
					   IRIT_INFNTY);

	    /* Get the trimming curves as well. */
	    PObj -> Pnext = IPAllocObject("", IP_OBJ_POLY, PObj -> Pnext);
	    IP_SET_POLYLINE_OBJ(PObj -> Pnext);
	    PObj -> Pnext -> U.Pl = IritTrimSrf2Polylines(PObj -> U.TrimSrfs,
							  NumOfIsolines,
							  TolSamples,
							  Method,
							  TRUE, FALSE);

	    if (CubicBzrCrvs) {
		TrimSrfFreeList(PObj -> U.TrimSrfs);
		PObj -> U.Crvs = CubicBzrCrvs;
		PObj -> ObjType = IP_OBJ_CURVE;
	    }
	    else
		PObj -> ObjType = IP_OBJ_UNDEF;

	    /* And finally the control mesh, if any. */
	    PObj = PObj -> Pnext;       /* Skip the trimming curves object. */
	    if (PPolygon) {
		PObj -> Pnext = IPAllocObject("Mesh", IP_OBJ_POLY,
					      PObj -> Pnext);
		strcat(PObj -> Pnext -> Name, PObj -> Name);
		IP_SET_POLYLINE_OBJ(PObj -> Pnext);
		PObj = PObj -> Pnext;
		PObj -> U.Pl = PPolygon;
	    }
	}
    }

    if (TrivarObjs) {
	for (PObj = TrivarObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing trivariate object \"%s\"\n",
			PObj -> Name);

	    PPolygon = NULL;
	    CubicBzrCrvs =
		IritTrivarToCubicBzrCrvs(PObj -> U.Trivars, &PPolygon,
					 DrawGeom, DrawMesh, NumOfIsolines,
					 IRIT_INFNTY);
	    if (CubicBzrCrvs) {
		TrivTVFreeList(PObj -> U.Trivars);
		PObj -> U.Crvs = CubicBzrCrvs;
		PObj -> ObjType = IP_OBJ_CURVE;
	    }
	    else
		PObj -> ObjType = IP_OBJ_UNDEF;

	    if (PPolygon) {
		PObj -> Pnext = IPAllocObject("Mesh", IP_OBJ_POLY,
					      PObj -> Pnext);
		strcat(PObj -> Pnext -> Name, PObj -> Name);
		IP_SET_POLYLINE_OBJ(PObj -> Pnext);
		PObj = PObj -> Pnext;
		PObj -> U.Pl = PPolygon;
	    }
	}
    }

    if (TriSrfObjs) {
	for (PObj = TriSrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    if (Talkative)
		fprintf(stderr, "Processing surface object \"%s\"\n",
			PObj -> Name);

	    PPolygon = NULL;
	    CubicBzrCrvs =
		IritTriSrfsToCubicBzrCrvs(PObj -> U.TriSrfs, &PPolygon,
					  DrawGeom, DrawMesh, NumOfIsolines,
					  IRIT_INFNTY);
	    if (CubicBzrCrvs) {
		TrngTriSrfFreeList(PObj -> U.TriSrfs);
		PObj -> U.Crvs = CubicBzrCrvs;
		PObj -> ObjType = IP_OBJ_CURVE;
	    }
	    else
		PObj -> ObjType = IP_OBJ_UNDEF;

	    if (PPolygon) {
		PObj -> Pnext = IPAllocObject("", IP_OBJ_POLY, PObj -> Pnext);
		IP_SET_POLYLINE_OBJ(PObj -> Pnext);
		PObj = PObj -> Pnext;
		PObj -> U.Pl = PPolygon;
	    }
	}
    }

    if (ModelObjs) {
	IPObjectStruct *PObjTmp,
	    *PTrimSrfObjs = NULL;

	/* Convert to a set of trimmed surfaces and call recursively. */
	for (PObj = ModelObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    TrimSrfStruct *TSrf,
		*TrimSrfs = MdlTrimConvert(PObj -> U.Mdls);

	    for (TSrf = TrimSrfs; TSrf != NULL; ) {
		TrimSrfStruct
		    *TSrfTmp = TSrf -> Pnext;

		TSrf -> Pnext = NULL;
		PObjTmp = GenTRIMSRFObject(TSrf);
		LIST_PUSH(PObjTmp, PTrimSrfObjs);

		TSrf = TSrfTmp;
	    }
	}

	IPFreeObjectList(ModelObjs);
	FreeForms -> TrimSrfObjs = PTrimSrfObjs;
	FreeForms -> ModelObjs = NULL;
	return FreeForm2CubicBzr(FreeForms, Talkative, DrawGeom, DrawMesh,
				 NumOfIsolines, TolSamples, Method);
    }

    return IritPrsrConcatFreeForm(FreeForms);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Converts a whole set of geometry to polygons, in place.                    M
*                                                                            *
* PARAMETERS:                                                                M
*   FreeForms:        Crvs/Srfs/Trimmed Srfs/Trivariates read from a file by M
*                     the irit parser.				             M
*   Talkative:        Do we want some more information printed?		     M
*   FourPerFlat:      See IritSurface2Polygons.	                             M
*   FineNess:         See IritSurface2Polygons.	                             M
*   ComputeUV:        See IritSurface2Polygons.	                             M
*   ComputeNrml:      See IritSurface2Polygons.	                             M
*   Optimal:          See IritSurface2Polygons.	                             M
*   BBoxGridFlag:     Do we want bounding box values and grid estimation.    M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Polygon/line(s) representing the geometry and its    M
*                       control mesh.	                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   FreeForm2Polygons                                                        M
*****************************************************************************/
IPObjectStruct *FreeForm2Polygons(IritPrsrFreeFormStruct *FreeForms,
				  int Talkative,
				  int FourPerFlat,
				  int FineNess,
				  int ComputeUV,
				  int ComputeNrml,
				  int Optimal,
				  int BBoxGrid)
{
    int LocalFourPerFlat;
    CagdSrfStruct *Srf, *Srfs;
    TrimSrfStruct *TrimSrf, *TrimSrfs;
    TrivTVStruct *TV, *TVs;
    TrngTriangSrfStruct *TriSrf, *TriSrfs;
    IPPolygonStruct *PPolygon;
    IPObjectStruct *PObj,
	*SrfObjs = FreeForms -> SrfObjs,
	*TrimSrfObjs = FreeForms -> TrimSrfObjs,
	*TrivarObjs = FreeForms -> TrivarObjs,
	*TriSrfObjs = FreeForms -> TriSrfObjs,
	*ModelObjs = FreeForms -> ModelObjs;

    if (SrfObjs) {
	for (PObj = SrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    CagdBBoxStruct BBox, TempBBox;
	    char GridStr[LINE_LEN];
	    RealType RelativeFineNess;

	    if (Talkative)
		fprintf(stderr, "Processing surface object \"%s\"\n",
			PObj -> Name);

	    Srfs = PObj -> U.Srfs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYGON_OBJ(PObj);

	    LocalFourPerFlat = FourPerFlat;
		
	    if (AttrGetObjectStrAttrib(PObj, "twoperflat"))
		LocalFourPerFlat = FALSE;
	    if (AttrGetObjectStrAttrib(PObj, "fourperflat"))
		LocalFourPerFlat = TRUE;
		
	    if ((RelativeFineNess = AttrGetObjectRealAttrib(PObj,
					   "resolution")) > IP_ATTR_BAD_REAL)
		RelativeFineNess = 1.0;
		
	    for (Srf = Srfs; Srf != NULL; Srf = Srf -> Pnext) {
		RealType t;

		if (BBoxGrid) {
		    /* Generate bounding box to surfaces and estimate */
		    /* the grid size for it using GlblGridSize.	  */
		    if (Srf == Srfs)
			CagdSrfBBox(Srf, &BBox);
		    else {
			CagdSrfBBox(Srf, &TempBBox);
			CagdMergeBBox(&BBox, &TempBBox);
		    }
		}

		if ((t = AttrGetObjectRealAttrib(PObj, "u_resolution"))
							    < IP_ATTR_BAD_REAL)
		    AttrSetRealAttrib(&Srf -> Attr, "u_resolution", t);
		if ((t = AttrGetObjectRealAttrib(PObj, "v_resolution"))
							    < IP_ATTR_BAD_REAL)
		    AttrSetRealAttrib(&Srf -> Attr, "v_resolution", t);

		PPolygon = IritSurface2Polygons(Srf, LocalFourPerFlat,
						RelativeFineNess * FineNess,
						ComputeUV, ComputeNrml,
						Optimal);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    CagdSrfFreeList(Srfs);

	    if (BBoxGrid) {
		RealType
		    Dx = BBox.Max[0] - BBox.Min[0],
		    Dy = BBox.Max[1] - BBox.Min[1],
		    Dz = BBox.Max[2] - BBox.Min[2],
		    M = MAX(MAX(Dx, Dy), Dz);
		int IDx = (int) (BBoxGrid * (Dx / M)),
		    IDy = (int) (BBoxGrid * (Dy / M)),
		    IDz = (int) (BBoxGrid * (Dz / M));
		    
		/* Save grid information derived from the surface bbox. */
		sprintf(GridStr, "%d %d %d",
			IDx > 0 ? IDx : 1,
			IDy > 0 ? IDy : 1,
			IDz > 0 ? IDz : 1);
		AttrSetObjectStrAttrib(PObj, "GridSize", GridStr);
		sprintf(GridStr, "%f %f %f %f %f %f",
			BBox.Min[0], BBox.Max[0],
			BBox.Min[1], BBox.Max[1],
			BBox.Min[2], BBox.Max[2]);
		AttrSetObjectStrAttrib(PObj, "BBox", GridStr);
	    }
	}
    }

    if (TrimSrfObjs) {
	for (PObj = TrimSrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    CagdBBoxStruct BBox, TempBBox;
	    char GridStr[LINE_LEN];
	    RealType RelativeFineNess;

	    if (Talkative)
		fprintf(stderr, "Processing trimmed surface object \"%s\"\n",
			PObj -> Name);

	    TrimSrfs = PObj -> U.TrimSrfs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYGON_OBJ(PObj);

	    LocalFourPerFlat = FourPerFlat;
		
	    if (AttrGetObjectStrAttrib(PObj, "twoperflat"))
		LocalFourPerFlat = FALSE;
	    if (AttrGetObjectStrAttrib(PObj, "fourperflat"))
		LocalFourPerFlat = TRUE;
		
	    if ((RelativeFineNess = AttrGetObjectRealAttrib(PObj,
					   "resolution")) > IP_ATTR_BAD_REAL)
		RelativeFineNess = 1.0;
		
	    for (TrimSrf = TrimSrfs;
		 TrimSrf != NULL;
		 TrimSrf = TrimSrf -> Pnext) {
		RealType t;

		if (BBoxGrid) {
		    /* Generate bounding box to surfaces and estimate */
		    /* the grid size for it using GlblGridSize.	  */
		    if (TrimSrf == TrimSrfs)
			CagdSrfBBox(TrimSrf -> Srf, &BBox);
		    else {
			CagdSrfBBox(TrimSrf -> Srf, &TempBBox);
			CagdMergeBBox(&BBox, &TempBBox);
		    }
		}

		if ((t = AttrGetObjectRealAttrib(PObj, "u_resolution"))
							    < IP_ATTR_BAD_REAL)
		    AttrSetRealAttrib(&TrimSrf -> Attr, "u_resolution", t);
		if ((t = AttrGetObjectRealAttrib(PObj, "v_resolution"))
							    < IP_ATTR_BAD_REAL)
		    AttrSetRealAttrib(&TrimSrf -> Attr, "v_resolution", t);

		PPolygon = IritTrimSrf2Polygons(TrimSrf, LocalFourPerFlat,
						RelativeFineNess * FineNess,
						ComputeUV, ComputeNrml,
						Optimal);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    TrimSrfFreeList(TrimSrfs);

	    if (BBoxGrid) {
		RealType
		    Dx = BBox.Max[0] - BBox.Min[0],
		    Dy = BBox.Max[1] - BBox.Min[1],
		    Dz = BBox.Max[2] - BBox.Min[2],
		    M = MAX(MAX(Dx, Dy), Dz);
		int IDx = (int) (BBoxGrid * (Dx / M)),
		    IDy = (int) (BBoxGrid * (Dy / M)),
		    IDz = (int) (BBoxGrid * (Dz / M));
		    
		/* Save grid information derived from the surface bbox. */
		sprintf(GridStr, "%d %d %d",
			IDx > 0 ? IDx : 1,
			IDy > 0 ? IDy : 1,
			IDz > 0 ? IDz : 1);
		AttrSetObjectStrAttrib(PObj, "GridSize", GridStr);
		sprintf(GridStr, "%f %f %f %f %f %f",
			BBox.Min[0], BBox.Max[0],
			BBox.Min[1], BBox.Max[1],
			BBox.Min[2], BBox.Max[2]);
		AttrSetObjectStrAttrib(PObj, "BBox", GridStr);
	    }
	}
    }

    if (TrivarObjs) {
	for (PObj = TrivarObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    CagdBBoxStruct BBox, TempBBox;
	    char GridStr[LINE_LEN];
	    RealType RelativeFineNess;

	    if (Talkative)
		fprintf(stderr, "Processing trivariate object \"%s\"\n",
			PObj -> Name);

	    TVs = PObj -> U.Trivars;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYGON_OBJ(PObj);

	    LocalFourPerFlat = FourPerFlat;
		
	    if (AttrGetObjectStrAttrib(PObj, "twoperflat"))
		LocalFourPerFlat = FALSE;
	    if (AttrGetObjectStrAttrib(PObj, "fourperflat"))
		LocalFourPerFlat = TRUE;
		
	    if ((RelativeFineNess = AttrGetObjectRealAttrib(PObj,
					   "resolution")) > IP_ATTR_BAD_REAL)
		RelativeFineNess = 1.0;

	    for (TV = TVs; TV != NULL; TV = TV -> Pnext) {
		if (BBoxGrid) {
		    /* Generate bounding box to surfaces and estimate */
		    /* the grid size for it using BBoxGrid.	      */
		    if (TV == TVs)
			TrivTVBBox(TV, &BBox);
		    else {
			TrivTVBBox(TV, &TempBBox);
			CagdMergeBBox(&BBox, &TempBBox);
		    }
		}
		PPolygon = IritTrivar2Polygons(TV, LocalFourPerFlat,
					       RelativeFineNess * FineNess,
					       ComputeUV, ComputeNrml,
					       Optimal);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    TrivTVFreeList(TVs);

	    if (BBoxGrid) {
		RealType
		    Dx = BBox.Max[0] - BBox.Min[0],
		    Dy = BBox.Max[1] - BBox.Min[1],
		    Dz = BBox.Max[2] - BBox.Min[2],
		    M = MAX(MAX(Dx, Dy), Dz);
		int IDx = (int) (BBoxGrid * (Dx / M)),
		    IDy = (int) (BBoxGrid * (Dy / M)),
		    IDz = (int) (BBoxGrid * (Dz / M));
		    
		/* Save grid information derived from the surface bbox. */
		sprintf(GridStr, "%d %d %d",
			IDx > 0 ? IDx : 1,
			IDy > 0 ? IDy : 1,
			IDz > 0 ? IDz : 1);
		AttrSetObjectStrAttrib(PObj, "GridSize", GridStr);
		sprintf(GridStr, "%f %f %f %f %f %f",
			BBox.Min[0], BBox.Max[0],
			BBox.Min[1], BBox.Max[1],
			BBox.Min[2], BBox.Max[2]);
		AttrSetObjectStrAttrib(PObj, "BBox", GridStr);
	    }
	}
    }

    if (TriSrfObjs) {
	for (PObj = TriSrfObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    CagdBBoxStruct BBox, TempBBox;
	    char GridStr[LINE_LEN];
	    RealType RelativeFineNess;

	    if (Talkative)
		fprintf(stderr, "Processing surface object \"%s\"\n",
			PObj -> Name);

	    TriSrfs = PObj -> U.TriSrfs;
	    PObj -> U.Pl = NULL;
	    PObj -> ObjType = IP_OBJ_POLY;
	    IP_SET_POLYGON_OBJ(PObj);

	    if ((RelativeFineNess = AttrGetObjectRealAttrib(PObj,
					   "resolution")) > IP_ATTR_BAD_REAL)
		RelativeFineNess = 1.0;

	    for (TriSrf = TriSrfs; TriSrf != NULL; TriSrf = TriSrf -> Pnext) {
		if (BBoxGrid) {
		    /* Generate bounding box to surfaces and estimate */
		    /* the grid size for it using GlblGridSize.	  */
		    if (TriSrf == TriSrfs)
			TrngTriSrfBBox(TriSrf, &BBox);
		    else {
			TrngTriSrfBBox(TriSrf, &TempBBox);
			CagdMergeBBox(&BBox, &TempBBox);
		    }
		}

		PPolygon = IritTriSrf2Polygons(TriSrf,
					       RelativeFineNess * FineNess,
					       ComputeUV, ComputeNrml,
					       Optimal);

		PObj -> U.Pl = IritPrsrAppendPolyLists(PPolygon,
						       PObj -> U.Pl);
	    }
	    TrngTriSrfFreeList(TriSrfs);

	    if (BBoxGrid) {
		RealType
		    Dx = BBox.Max[0] - BBox.Min[0],
		    Dy = BBox.Max[1] - BBox.Min[1],
		    Dz = BBox.Max[2] - BBox.Min[2],
		    M = MAX(MAX(Dx, Dy), Dz);
		int IDx = (int) (BBoxGrid * (Dx / M)),
		    IDy = (int) (BBoxGrid * (Dy / M)),
		    IDz = (int) (BBoxGrid * (Dz / M));
		    
		/* Save grid information derived from the surface bbox. */
		sprintf(GridStr, "%d %d %d",
			IDx > 0 ? IDx : 1,
			IDy > 0 ? IDy : 1,
			IDz > 0 ? IDz : 1);
		AttrSetObjectStrAttrib(PObj, "GridSize", GridStr);
		sprintf(GridStr, "%f %f %f %f %f %f",
			BBox.Min[0], BBox.Max[0],
			BBox.Min[1], BBox.Max[1],
			BBox.Min[2], BBox.Max[2]);
		AttrSetObjectStrAttrib(PObj, "BBox", GridStr);
	    }
	}
    }

    if (ModelObjs) {
	IPObjectStruct *PObjTmp,
	    *PTrimSrfObjs = NULL;

	/* Convert to a set of trimmed surfaces and call recursively. */
	for (PObj = ModelObjs; PObj != NULL; PObj = PObj -> Pnext) {
	    TrimSrfStruct *TSrf,
		*TrimSrfs = MdlTrimConvert(PObj -> U.Mdls);

	    for (TSrf = TrimSrfs; TSrf != NULL; ) {
		TrimSrfStruct
		    *TSrfTmp = TSrf -> Pnext;

		TSrf -> Pnext = NULL;
		PObjTmp = GenTRIMSRFObject(TSrf);
		LIST_PUSH(PObjTmp, PTrimSrfObjs);

		TSrf = TSrfTmp;
	    }
	}

	IPFreeObjectList(ModelObjs);
	FreeForms -> TrimSrfObjs = PTrimSrfObjs;
	FreeForms -> ModelObjs = NULL;
	return FreeForm2Polygons(FreeForms, Talkative, FourPerFlat, FineNess,
				 ComputeUV, ComputeNrml, Optimal, BBoxGrid);
    }

    return IritPrsrConcatFreeForm(FreeForms);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to convert a single freeform geometry to polylines/polygons, in    M
* place.								     M
*                                                                            *
* PARAMETERS:                                                                M
*   PObj:  A Crv/Srf/Trimmed Srf/Trivariate freeform geoemtry.               M
*   State: The way the freeform geometry should be converted.                M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:   Processed freeform geometry.                         M
*                                                                            *
* SEE ALSO:                                                                  M
*   FFCnvrtSrfsToBicubicBezier                                               M
*                                                                            *
* KEYWORDS:                                                                  M
*   ProcessFreeForm, conversion                                              M
*****************************************************************************/
IPObjectStruct *ProcessFreeForm(IPObjectStruct *PObj,
				FreeformConvState *State)
{
    static int
	FirstTime = TRUE;
    IritPrsrFreeFormStruct FreeForms;
    int DumpObjsAsPolylines = State -> DumpObjsAsPolylines,
	OldBspMultUsingInter = BspMultInterpFlag(FALSE);
    IPObjectStruct
	*PObjNext = PObj -> Pnext;

    PObj -> Pnext = NULL;

    if (FirstTime) {
	FirstTime = FALSE;

	if (FFCState.LinearOnePolyFlag)
	    CagdSetLinear2Poly(CAGD_ONE_POLY_PER_COLIN);
	else
	    CagdSetLinear2Poly(CAGD_REG_POLY_PER_LIN);
    }

    FreeForms.CrvObjs = NULL;
    FreeForms.SrfObjs = NULL;
    FreeForms.TrimSrfObjs = NULL;
    FreeForms.TrivarObjs = NULL;
    FreeForms.TriSrfObjs = NULL;
    FreeForms.ModelObjs = NULL;

    switch (PObj -> ObjType) {
	case IP_OBJ_CURVE:
	    FreeForms.CrvObjs = PObj;
	    DumpObjsAsPolylines = TRUE;
	    break;
	case IP_OBJ_SURFACE:
	    FreeForms.SrfObjs = PObj;
	    break;
	case IP_OBJ_TRIMSRF:
	    FreeForms.TrimSrfObjs = PObj;
	    break;
	case IP_OBJ_TRIVAR:
	    FreeForms.TrivarObjs = PObj;
	    break;
	case IP_OBJ_TRISRF:
	    FreeForms.TriSrfObjs = PObj;
	    break;
	case IP_OBJ_MODEL:
	    FreeForms.ModelObjs = PObj;
	    break;
	default:
	    fprintf(stderr, "Non freeform geometry detected\n");
	    BspMultInterpFlag(OldBspMultUsingInter);
	    return NULL;
    }

    if (FFCState.CubicCrvsAprox) {
	PObj = FreeForm2CubicBzr(&FreeForms,
				 State -> Talkative,
				 State -> DrawFFGeom,
				 State -> DrawFFMesh,
				 State -> NumOfIsolines,
				 State -> CrvApproxTolSamples,
				 State -> CrvApproxMethod);
    }
    else if (DumpObjsAsPolylines) {
	PObj = FreeForm2Polylines(&FreeForms,
				  State -> Talkative,
				  State -> DrawFFGeom,
				  State -> DrawFFMesh,
				  State -> NumOfIsolines,
				  State -> CrvApproxTolSamples,
				  State -> CrvApproxMethod);
    }
    else {
	IPObjectStruct
	    *MeshObj = NULL;
	int CnvrtToPolys = TRUE;

	if (State -> DrawFFMesh) {
	    FreeformConvState
		TState = *State;

	    TState.DrawFFGeom = FALSE;
	    TState.DumpObjsAsPolylines = TRUE;
	    MeshObj = ProcessFreeForm(CopyObject(NULL, PObj, TRUE), &TState);
	}

	if (FFCState.SrfsToBicubicBzr && PObj -> ObjType == IP_OBJ_SURFACE) {
	    CagdSrfStruct *Srf,
		*NoConvertionSrfs = NULL,
		*ConvertedSrfs = NULL;

	    /* Convert all surfaces that can be converted to bicubic Bezier. */
	
	    for (Srf = PObj -> U.Srfs; Srf != NULL; Srf = Srf -> Pnext) {
		CagdSrfStruct *TSrfs, *TNoSrfs;

		TSrfs = IritSurfacesToCubicBzrSrfs(Srf, &TNoSrfs);
		if (TSrfs != NULL && TNoSrfs != NULL) {
		    fprintf(stderr, "Both convertable and non convertable to bicubic Bezier surfaces in object\n");
		    fprintf(stderr, "named \"%s\", convertion is ignored.\n",
			    PObj -> Name);
		    CagdSrfFreeList(TSrfs);
		    CagdSrfFreeList(TNoSrfs);
		    CagdSrfFreeList(ConvertedSrfs);
		    CagdSrfFreeList(NoConvertionSrfs);
		    ConvertedSrfs = NoConvertionSrfs = NULL;
		    break;
		}

		ConvertedSrfs = CagdListAppend(TSrfs, ConvertedSrfs);
		NoConvertionSrfs = CagdListAppend(TNoSrfs, NoConvertionSrfs);
	    }

	    if (ConvertedSrfs != NULL || NoConvertionSrfs != NULL) {
		if (ConvertedSrfs != NULL) {
		    CagdSrfFreeList(PObj -> U.Srfs);
		    PObj -> U.Srfs = ConvertedSrfs;
		    CnvrtToPolys = FALSE;
		}
		else {
		    CagdSrfFreeList(NoConvertionSrfs);
		}
	    }
	}

	if (CnvrtToPolys)
	    PObj = FreeForm2Polygons(&FreeForms,
				     State -> Talkative,
				     State -> FourPerFlat,
				     (int) State -> FineNess,
				     State -> ComputeUV,
				     State -> ComputeNrml,
				     State -> OptimalPolygons,
				     State -> BBoxGrid);
	if (MeshObj) {
	    MeshObj -> Pnext = PObj -> Pnext;
	    PObj -> Pnext = MeshObj;
	}
    }

    BspMultInterpFlag(OldBspMultUsingInter);

    return IritPrsrAppendObjLists(PObj, PObjNext);
}
