/******************************************************************************
* SBsp-Int.c - Bspline surface interpolation/approximation schemes.	      *
*******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                 *
*******************************************************************************
* Written by Gershon Elber, Feb. 94.					      *
******************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "cagd_loc.h"
#include "extra_fn.h"

/*****************************************************************************
* DESCRIPTION:                                                               M
* Given a set of points, PtList, computes a Bspline surface of order UOrder  M
* by VOrder that interpolates or least square approximates the given set of  M
* points.	                                                             M
*   PtList is a NULL terminated array of linked lists of CagdPtStruct        M
* structs.                                                 	             M
*   All linked lists in PtList must have the same length.		     M
*   U direction of surface is associated with array, V with the linked       M
* lists.		                                   	             M
*   The size of the control mesh of the resulting Bspline surface defaults   M
* to the number of points in PtList (if SrfUSize = SrfVSize = 0).	     M
*   However, either numbers can smaller to yield a least square              M
* approximation of the gievn data set.                     	             M
*   The created surface can be parametrized as specified by ParamType.       M
*                                                                            *
* PARAMETERS:                                                                M
*   PtList:      A NULL terminating array of linked list of points.          M
*   UOrder:      Of the to be created surface.                               M
*   VOrder:      Of the to be created surface.                               M
*   SrfUSize:    U size of the to be created surface. Must be at least as    M
*                large as the array PtList.			             M
*   SrfVSize:    V size of the to be created surface. Must be at least as    M
*                large as the length of each list in PtList.                 M
*   ParamType:   Type of parametrization.                                    M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:   Constructed interpolating/approximating surface.      M
*                                                                            *
* KEYWORDS:                                                                  M
*   BspSrfInterpPts, interpolation, least square approximation               M
*****************************************************************************/
CagdSrfStruct *BspSrfInterpPts(CagdPtStruct **PtList,
			       int UOrder,
			       int VOrder,
			       int SrfUSize,
			       int SrfVSize,
			       CagdParametrizationType ParamType)
{
    int i, NumLinkedLists, NumPtsInList;
    CagdPtStruct *Pt, **PtPtr;
    CagdRType *UKV, *VKV, *PtUKnots, *PtVKnots, *AveKV, *RU, *RV, *R2;
    CagdSrfStruct *Srf;
    CagdCtlPtStruct
	*CtlPt = NULL,
	*CtlPtList = NULL;

    for (NumLinkedLists = 0, PtPtr = PtList;
	 *PtPtr != NULL;
	 NumLinkedLists++, PtPtr++);
    for (NumPtsInList = 0, Pt = *PtList;
	 Pt != NULL;
	 NumPtsInList++, Pt = Pt -> Pnext);

    if (SrfUSize == 0)
	SrfUSize = NumLinkedLists;
    if (SrfVSize == 0)
	SrfVSize = NumPtsInList;
    if (NumLinkedLists < 3 ||
	NumLinkedLists < UOrder ||
	NumLinkedLists < SrfUSize ||
	SrfUSize < UOrder ||
	NumPtsInList < 3 ||
	NumPtsInList < VOrder ||
	NumPtsInList < SrfVSize ||
	SrfVSize < VOrder)
	return NULL;

    RU = PtUKnots =
	(CagdRType *) IritMalloc(sizeof(CagdRType) * NumLinkedLists);
    RV = PtVKnots =
	(CagdRType *) IritMalloc(sizeof(CagdRType) * NumPtsInList);

    /* Compute parameter values at which surface will interpolate PtList. */
    switch (ParamType) {
	case CAGD_CHORD_LEN_PARAM:
        case CAGD_CENTRIPETAL_PARAM:
	    /* No real support for chord length although we might be able to */
	    /* do something useful better that uniform parametrization.      */
	case CAGD_UNIFORM_PARAM:
	default:
	    for (i = 0; i < NumLinkedLists; i++)
		*RU++ = ((RealType) i) / (NumLinkedLists - 1);
	    for (i = 0; i < NumPtsInList; i++)
		*RV++ = ((RealType) i) / (NumPtsInList - 1);
	    break;
    }

    /* Construct the two knot vectors of the Bspline surface. */
    UKV = (CagdRType *) IritMalloc(sizeof(CagdRType) * (SrfUSize + UOrder));
    VKV = (CagdRType *) IritMalloc(sizeof(CagdRType) * (SrfVSize + VOrder));

    AveKV = BspKnotAverage(PtUKnots, NumLinkedLists,
			   NumLinkedLists + UOrder - SrfUSize - 1);
    BspKnotAffineTrans(AveKV, SrfUSize - UOrder + 2, PtUKnots[0] - AveKV[0],
		       (PtUKnots[NumLinkedLists - 1] - PtUKnots[0]) /
			   (AveKV[SrfUSize - UOrder + 1] - AveKV[0]));
    for (i = 0, RU = UKV, R2 = AveKV; i < UOrder; i++)
	*RU++ = *R2;
    for (i = 0; i < SrfUSize - UOrder; i++)
	*RU++ = *++R2;
    for (i = 0, R2++; i < UOrder; i++)
	*RU++ = *R2;
    IritFree((VoidPtr) AveKV);

    AveKV = BspKnotAverage(PtVKnots, NumPtsInList,
			   NumPtsInList + VOrder - SrfVSize - 1);
    BspKnotAffineTrans(AveKV, SrfVSize - VOrder + 2, PtVKnots[0] - AveKV[0],
		       (PtVKnots[NumPtsInList - 1] - PtVKnots[0]) /
			   (AveKV[SrfVSize - VOrder + 1] - AveKV[0]));
    for (i = 0, RV = VKV, R2 = AveKV; i < VOrder; i++)
	*RV++ = *R2;
    for (i = 0; i < SrfVSize - VOrder; i++)
	*RV++ = *++R2;
    for (i = 0, R2++; i < VOrder; i++)
	*RV++ = *R2;
    IritFree((VoidPtr) AveKV);

    /* Convert to control points in a linear list */
    for (PtPtr = PtList; *PtPtr != NULL; PtPtr++) {
	for (Pt = *PtPtr, i = 0; Pt != NULL; Pt = Pt -> Pnext, i++) {
	    int j;

	    if (CtlPtList == NULL)
		CtlPtList = CtlPt = CagdCtlPtNew(CAGD_PT_E3_TYPE);
	    else {
		CtlPt ->Pnext = CagdCtlPtNew(CAGD_PT_E3_TYPE);
		CtlPt = CtlPt -> Pnext;
	    }
	    for (j = 0; j < 3; j++)
		CtlPt -> Coords[j + 1] = Pt -> Pt[j];
	}
	if (i != NumPtsInList) {
	    CagdCtlPtFreeList(CtlPtList);

	    IritFree((VoidPtr *) PtUKnots);
	    IritFree((VoidPtr *) PtVKnots);
	    IritFree((VoidPtr *) UKV);
	    IritFree((VoidPtr *) VKV);
    
	    CAGD_FATAL_ERROR(CAGD_ERR_PT_OR_LEN_MISMATCH);
	    return NULL;
	}
    }

    Srf = BspSrfInterpolate(CtlPtList, NumLinkedLists, NumPtsInList,
			    PtUKnots, PtVKnots, UKV, VKV,
			    SrfUSize, SrfVSize, UOrder, VOrder);
    CagdCtlPtFreeList(CtlPtList);

    IritFree((VoidPtr *) PtUKnots);
    IritFree((VoidPtr *) PtVKnots);
    IritFree((VoidPtr *) UKV);
    IritFree((VoidPtr *) VKV);

    return Srf;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Given a set of points on a rectangular grid, PtList, parameter values the  M
* surface should interpolate or approximate these grid points, U/VParams,    M
* the expected two knot vectors of the surface, U/VKV, the expected lengths  M
* U/VLength and orders U/VOrder of the Bspline surface, computes the Bspline M
* surface's coefficients.						     M
*   All points in PtList are assumed of the same type.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   PtList:     A long linked list (NumUPts * NumVPts) of points to          M
*               interpolated or least square approximate.                    M
*   NumUPts:    Number of points in PtList in the U direction.               M
*   NumVPts:    Number of points in PtList in the V direction.               M
*   UParams:    Parameter at which surface should interpolate or             M
*               approximate PtList in the U direction.                       M
*   VParams:    Parameter at which surface should interpolate or             M
*               approximate PtList in the V direction.                       M
*   UKV:        Requested knot vector form the surface in the U direction.   M
*   VKV:        Requested knot vector form the surface in the V direction.   M
*   ULength:    Requested length of control mesh of surface in U direction.  M
*   VLength:    Requested length of control mesh of surface in V direction.  M
*   UOrder:     Requested order of surface in U direction.                   M
*   VOrder:     Requested order of surface in U direction.                   M
*									     *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:   Constructed interpolating/approximating surface.      M
*                                                                            *
* KEYWORDS:                                                                  M
*   BspSrfInterpolate, interpolation, least square approximation             M
*****************************************************************************/
CagdSrfStruct *BspSrfInterpolate(CagdCtlPtStruct *PtList,
				 int NumUPts,
				 int NumVPts,
				 CagdRType *UParams,
				 CagdRType *VParams,
				 CagdRType *UKV,
				 CagdRType *VKV,
				 int ULength,
				 int VLength,
				 int UOrder,
				 int VOrder)
{
    CagdPointType
	PtType = PtList -> PtType;
    CagdBType
	IsRational = CAGD_IS_RATIONAL_PT(PtType);
    int i, j,
	NumCoords = CAGD_NUM_OF_PT_COORD(PtType);
    CagdCtlPtStruct *Pt;
    CagdRType **SPoints;
    CagdSrfStruct *Srf;
    CagdCrvStruct **Crvs;

    /* Construct the Bspline surface and its two knot vectors. */
    Srf = BspSrfNew(ULength, VLength, UOrder, VOrder, PtType);
    SPoints = Srf -> Points;
    CAGD_GEN_COPY(Srf -> UKnotVector, UKV,
		  (ULength + UOrder) * sizeof(CagdRType));
    CAGD_GEN_COPY(Srf -> VKnotVector, VKV,
		  (VLength + VOrder) * sizeof(CagdRType));

    /* Interpolate the rows as set of curves. */
    Crvs = (CagdCrvStruct **) IritMalloc(sizeof(CagdCrvStruct *) * NumVPts);
    for (i = 0, Pt = PtList; i < NumVPts; i++) {
	Crvs[i] = BspCrvInterpolate(Pt, NumUPts, UParams, UKV, ULength,
				    UOrder, FALSE);

	for (j = 0; j < NumUPts; j++)		     /* Advance to next row. */
	    Pt = Pt -> Pnext;
    }

    /* Interpolate the columns from the curves of the rows. */
    for (i = 0; i < ULength; i++) {
	int k;
	CagdCrvStruct *Crv;
	CagdCtlPtStruct
	    *CtlPtList = NULL,
	    *CtlPt = NULL;
	CagdRType **CPoints;

	for (j = 0; j < NumVPts; j++) {
	    CagdRType
		**Points = Crvs[j] -> Points;

	    if (CtlPtList == NULL)
		CtlPtList = CtlPt = CagdCtlPtNew(CAGD_PT_E3_TYPE);
	    else {
		CtlPt ->Pnext = CagdCtlPtNew(CAGD_PT_E3_TYPE);
		CtlPt = CtlPt -> Pnext;
	    }

	    for (k = !IsRational; k <= NumCoords; k++)
		CtlPt -> Coords[k] = Points[k][i];
	}

	/* Interpolate the column, copy to mesh, and free the curve. */
	Crv = BspCrvInterpolate(CtlPtList, NumVPts, VParams, VKV, VLength,
				VOrder, FALSE);
	CPoints = Crv -> Points;
	CagdCtlPtFreeList(CtlPtList);

	for (j = 0; j < VLength; j++)
	    for (k = !IsRational; k <= NumCoords; k++)
		SPoints[k][i + j * ULength] = CPoints[k][j];

	CagdCrvFree(Crv);
    }

    for (i = 0; i < NumVPts; i++)
	CagdCrvFree(Crvs[i]);
    IritFree((VoidPtr) Crvs);

    return Srf;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Given a set of scattered points, PtList, computes a Bspline surface of     M
* order UOrder by VOrder that interpolates or least square approximates the  M
* given set of scattered points.                                             M
*   PtList is a NULL terminated lists of CagdPtStruct structs, with each     M
* point holding (u, v, x [, y[, z]]).  This is, E3 points create an E1       M
* scalar surface and E5 points create an E3 surface,           	             M
*                                                                            *
* PARAMETERS:                                                                M
*   PtList:      A NULL terminating array of linked list of points.          M
*   UOrder:      Of the to be created surface.                               M
*   VOrder:      Of the to be created surface.                               M
*   USize:       U size of the to be created surface.                        M
*   VSize:       V size of the to be created surface.                        M
*   UKV:	 Expected knot vector in U direction, NULL for uniform open. M
*   VKV:	 Expected knot vector in V direction, NULL for uniform open. M
*                                                                            *
* RETURN VALUE:                                                              M
*   CagdSrfStruct *:   Constructed interpolating/approximating surface.      M
*                                                                            *
* KEYWORDS:                                                                  M
*   BspSrfInterpScatPts, interpolation, least square approximation           M
*****************************************************************************/
CagdSrfStruct *BspSrfInterpScatPts(CagdCtlPtStruct *PtList,
				   int UOrder,
				   int VOrder,
				   int USize,
				   int VSize,
				   CagdRType *UKV,
				   CagdRType *VKV)
{
    int i, j,
	NumCoords = CAGD_NUM_OF_PT_COORD(PtList -> PtType),
	PtListLen = CagdListLength(PtList),
	Size = USize * VSize;
    CagdBType
	NewUKV = FALSE,
	NewVKV = FALSE;
    CagdRType *M, *R, *InterpPts,
	*ULine = (CagdRType *) IritMalloc(sizeof(CagdRType) * UOrder),
	*Mat = (CagdRType *) IritMalloc(sizeof(CagdRType) * Size *
					MAX(Size, PtListLen));
    CagdCtlPtStruct *Pt;
    CagdSrfStruct *Srf;

    if (NumCoords < 3) {
	CAGD_FATAL_ERROR(CAGD_ERR_PT_OR_LEN_MISMATCH);
	return NULL;
    }

    ZAP_MEM(Mat, sizeof(CagdRType) * Size * MAX(Size, PtListLen));

    if (UKV == NULL) {
	UKV = BspKnotUniformOpen(USize, UOrder, NULL);
	BspKnotAffineTrans2(UKV, USize + UOrder, 0.0, 1.0);
	NewUKV = TRUE;
    }
    if (VKV == NULL) {
	VKV = BspKnotUniformOpen(VSize, VOrder, NULL);
	BspKnotAffineTrans2(VKV, VSize + VOrder, 0.0, 1.0);
	NewVKV = TRUE;
    }

    for (Pt = PtList, M = Mat; Pt != NULL; Pt = Pt -> Pnext, M += Size) {
	int UIndex, VIndex;
	CagdRType *VLine;

	if (NumCoords != CAGD_NUM_OF_PT_COORD(Pt -> PtType)) {
	    CAGD_FATAL_ERROR(CAGD_ERR_PT_OR_LEN_MISMATCH);
	    IritFree((VoidPtr *) ULine);
	    IritFree((VoidPtr *) Mat);
	    return NULL;
	}

	VLine = BspCrvCoxDeBoorBasis(UKV, UOrder, USize,
				     Pt -> Coords[1], &UIndex);
	GEN_COPY(ULine, VLine, sizeof(CagdRType) * UOrder);
	VLine = BspCrvCoxDeBoorBasis(VKV, VOrder, VSize,
				     Pt -> Coords[2], &VIndex);

	for (j = VIndex; j < VIndex + VOrder; j++)
	    for (i = UIndex; i < UIndex + UOrder; i++)
		M[j * USize + i] = ULine[i - UIndex] * VLine[j - VIndex];
    }
    IritFree((VoidPtr *) ULine);

#   ifdef DEBUG_PRINT_INPUT_SVD
	for (i = 0; i < PtListLen; i++) {
	    fprintf(stderr, "[");
	    for (j = 0; j < Size; j++) {
		fprintf(stderr, "%7.4f ", Mat[i * Size + j]);
	    }
	    fprintf(stderr, "]\n");
	}
#   endif /* DEBUG_PRINT_INPUT_SVD */

    /* Compute SVD decomposition for Mat. */
    if (fabs(SvdLeastSqr(Mat, NULL, NULL,
			 MAX(Size, PtListLen), Size)) < IRIT_UEPS &&
	Size <= PtListLen) {
	CAGD_FATAL_ERROR(CAGD_ERR_NO_SOLUTION);

	IritFree((VoidPtr *) Mat);
	return NULL;
    }
    IritFree((VoidPtr *) Mat);

    /* Construct the Bspline surface and copy its knot vectors. */
    Srf = BspSrfNew(USize, VSize, UOrder, VOrder,
		    CAGD_MAKE_PT_TYPE(FALSE, NumCoords - 2));
    CAGD_GEN_COPY(Srf -> UKnotVector, UKV,
		  (CAGD_SRF_UPT_LST_LEN(Srf) + UOrder) * sizeof(CagdRType));
    CAGD_GEN_COPY(Srf -> VKnotVector, VKV,
		  (CAGD_SRF_VPT_LST_LEN(Srf) + VOrder) * sizeof(CagdRType));

    /* Solve for the coefficients of all the coordinates of the curve. */
    InterpPts = (CagdRType *) IritMalloc(sizeof(CagdRType) *
					 MAX(Size, PtListLen));
    for (i = 3; i <= NumCoords; i++) {
	for (Pt = PtList, R = InterpPts; Pt != NULL; Pt = Pt -> Pnext)
	    *R++ = Pt -> Coords[i];

	SvdLeastSqr(NULL, Srf -> Points[i - 2], InterpPts, PtListLen, Size);
    }
    IritFree((VoidPtr *) InterpPts);

    if (NewUKV)
	IritFree(UKV);
    if (NewVKV)
	IritFree(VKV);

    return Srf;
}
