/*****************************************************************************
* Filter to convert IRIT data files to SENAC .sdat files.                    *
* Based on dat2irit 		                                             *
*   Written by:  Gershon Elber				Ver 1.0, Sep 1991    *
*                                                                            *
* Written by: M Glanvill 1995, contact senac@waikato.ac.nz                   *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "irit_sm.h"
#include "iritprsr.h"
#include "allocate.h"
#include "attribut.h"
#include "cagd_lib.h"
#include "symb_lib.h"
#include "triv_lib.h"
#include "trim_lib.h"
#include "getarg.h"
#include "genmat.h"

#ifdef NO_CONCAT_STR
static char *VersionStr =
	"Dat2Sen		Version 1.2,	Martin Glanvill	,\n\
	 (C) Copyright 1995/6 M Glanvill, U.O.W..";
#else
static char *VersionStr = "Dat2sen	" VERSION ",	Gershon Elber/M. Glanvill,     "
	__DATE__ ",   " __TIME__ "\n" COPYRIGHT ", Non commercial use only.";
#endif /* NO_CONCAT_STR */

static char
    *CtrlStr = "dat2sen z%- DFiles!*s";

static void DumpDataForIrit(IPObjectStruct *PObjects);
static void DumpOneObject(FILE *f, IPObjectStruct *PObject, int Level);
static void DumpOnePolygon(FILE *f,
			   IPPolygonStruct *PPolygon,
			   int PolyNum,
			   int IsPolyline);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of dat2sen - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    int Error,
	VerFlag = FALSE,
	NumFiles = 0;
    char
	**FileNames = NULL;
    IPObjectStruct *PObjects;

    if ((Error = GAGetArgs(argc, argv, CtrlStr,
			   &VerFlag, &NumFiles, &FileNames)) != 0) {
	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	exit(1);
    }

    if (VerFlag) {
	fprintf(stderr, "\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	exit(0);
    }

    if (!NumFiles) {
	fprintf(stderr, "No data file names were given, exit.\n");
	GAPrintHowTo(CtrlStr);
	exit(1);
    }

    /* Get the data files: */
    IritPrsrSetFlattenObjects(FALSE);
    if ((PObjects = IritPrsrGetDataFiles(FileNames, NumFiles, TRUE, FALSE)) ==
									NULL)
	exit(0);

    DumpDataForIrit(PObjects);

    exit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps the given Objects to stdout.                                         *
*                                                                            *
* PARAMETERS:                                                                *
*   PObject:    Objects todump                                               *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpDataForIrit(IPObjectStruct *PObjects)
{
    int NameCount = 1;
    IPObjectStruct
	*PObjHead = PObjects;

    for (PObjects = PObjHead; PObjects != NULL; PObjects = PObjects -> Pnext)
	DumpOneObject(stdout, PObjects,1);

}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one object PObject to files.                                         *
*                                                                            *
* PARAMETERS:                                                                *
*   f:            File to dump object to.                                    *
*   PObject:      Object to dump to file f.                                  *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void DumpOneObject(FILE *f, IPObjectStruct *PObject, int Level)
{
    static int
	NameCount = 1;
    int i, j;
    char Name[LINE_LEN_LONG];
    IPPolygonStruct *Pl;
    IPObjectStruct *PTmp;

    if (PObject -> Name == NULL || strlen(PObject -> Name) == 0)
	sprintf(Name, "NoName%d", NameCount++);
    else
	strcpy(Name, PObject -> Name);

    switch (PObject -> ObjType) {
	case IP_OBJ_LIST_OBJ:
	    for (i = 0; (PTmp = ListObjectGet(PObject, i)) != NULL; i++)
		DumpOneObject(f, PTmp, Level + 1);

	    break;
	case IP_OBJ_POLY:
	    for (Pl = PObject -> U.Pl, i = 1;
		 Pl != NULL; 
		 Pl = Pl -> Pnext, i++)
	      DumpOnePolygon(f, Pl, i, IP_IS_POLYLINE_OBJ(PObject));
	    
            fprintf(f, "EnterPoly %s\n", Name);
	    break;
	default:
	    fprintf(stderr,"Error: non-polygonal object - use irit2nff\n");
	    exit(1);
	    break;

    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Dumps one polygon.                                                         *
*                                                                            *
* PARAMETERS:                                                                *
*   f:		  File to dump polygon to. 		                     *
*   PPolygon:     Polygon to dump to file f.                                 *
*   PolyNum:      Number of polygon.                                         *
*   IsPolyline:   Is it a polyline or a polygon?                             *
*                                                                            *
* RETURN VALUE:                                                              *
*   void		                                                     *
*****************************************************************************/
static void DumpOnePolygon(FILE *f,
			   IPPolygonStruct *PPolygon,
			   int PolyNum,
			   int IsPolyline)
{
    int i,
	VCount = 1;
    IPVertexStruct
	*V = PPolygon -> PVertex,
	*VFirst = V;

    fprintf(f,"{\n");
    do {
	fprintf(f, "%14.7lg %14.7lg %14.7lg\n", V -> Coord[0], V -> Coord[1], 
		V -> Coord[2], VCount++);
	V = V -> Pnext;
    }
    while (V != VFirst && V != NULL);
    fprintf(f, "} #polyline %s\n", IsPolyline ? "true" : "false" );
}

