/*****************************************************************************
* Points (distribution) Shader program to render freeform models.	     *
*									     *
* Written by:  Gershon Elber			     Ver 1.0, September 1996 *
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "irit_sm.h"
#include "config.h"
#include "iritgrap.h"
#include "attribut.h"
#include "allocate.h"
#include "iritgrap.h"
#include "poly_cln.h"
#include "geomat3d.h"
#include "geomvals.h"
#include "getarg.h"
#include "ip_cnvrt.h"

#define NO_DECLARE_MALLOC /* For rle.h */
#define USE_PROTOTYPES
#define USE_STDLIB_H
#include <rle.h>
#include <rle_raw.h>

typedef struct PixelStruct {
    signed char Gray;
} PixelStruct;

typedef struct ImageStruct {
    int xSize, ySize;                  
    PixelStruct *data;
    struct ImageStruct *DxImage;                     
    struct ImageStruct *DyImage;                     
} ImageStruct;

typedef enum {		           /* Type of distance function computation. */
    NO_SHADER = 0,
    PTS_DIST_REGULAR,
    PTS_DIST_PHONG,
    PTS_DIST_PHONG_2L,
    PTS_DIST_PHONG_SPEC,
    PTS_DIST_PHONG_2L_SPEC,
    PTS_DIST_ZNORMAL,
    PTS_DIST_POINT_E3
} PtsDistCompType;

#define SRF_AREA_FINENESS	20
#define PTS_DENSITY		1000
#define SRF_MARCH_STEP		0.05
#define WOOD_LENGTH_DECAY_POWER 4
#define WOOD_MIN_LENGTH		0.01
#define MAX_X_DIFF		0.02

#ifdef NO_CONCAT_STR
static char *VersionStr =
    "PtsShad		Version 6.0,	Gershon Elber,\n\
	 (C) Copyright 1989/90/91/92/93 Gershon Elber, Non commercial use only.";
#else
static char *VersionStr =
    "PtsShad		" VERSION ",	Gershon Elber,	"
	__DATE__ ",   " __TIME__ "\n" COPYRIGHT ", Non commercial use only.";
#endif /* NO_CONCAT_STR */

static char *CtrlStr =
#ifdef DOUBLE
    "PtsShad o%-OutName!s m%- r%-RndrMdl!d c%-CosPwr!F s%-SdrPwr!F i%-Intensity!F l%-Lx|Ly|Lz!F!F!F v%-Vx|Vy|Vz!F!F!F w%-Width!F d%-Density!F Z%-SrfZTrans!F t%-Texture!s z%- DFiles!*s";
#else
    "PtsShad o%-OutName!s m%- r%-RndrMdl!d c%-CosPwr!f s%-SdrPwr!f i%-Intensity!f l%-Lx|Ly|Lz!f!f!f v%-Vx|Vy|Vz!f!f!f w%-Width!f d%-Density!f Z%-SrfZTrans!f t%-Texture!s z%- DFiles!*s";
#endif

static char
    *GlblTexture = "",
    *GlblLightSourceStr = "",
    *GlblViewDirStr = "",
    *GlblOutFileName = "PtsShad.dat";

static PtsDistCompType
    GlblDistRndrModel = NO_SHADER;

static int
    GlblTalkative = FALSE;

static RealType
    GlblMaxWidth = 0.01,
    GlblCosinePower = 1.0,
    GlblShaderPower = 2.0,
    GlblIntensity = 1.0,
    GlblLightSource[3] = { 0.5, 0.1, 2.0 },
    GlblViewDir[3] = { 0.0, 0.0, 1.0 },
    GlblSrfTranslate[3] = { 0.0, 0.0, 0.01 },
    GlblPtsDensity = 1.0;
static FILE
    *GlblOutFile = NULL;

static ConfigStruct SetUp[] =
{
  { "RenderModel",    "-r", (VoidPtr) &GlblDistRndrModel,   SU_INTEGER_TYPE },
  { "MoreVerbose",    "-m", (VoidPtr) &GlblTalkative,	    SU_BOOLEAN_TYPE },
  { "Texture",        "-t", (VoidPtr) &GlblTexture,	    SU_STRING_TYPE },
  { "LightSrcDir",    "-l", (VoidPtr) &GlblLightSourceStr,  SU_STRING_TYPE },
  { "Density",        "-d", (VoidPtr) &GlblPtsDensity,      SU_REAL_TYPE },
  { "Intensity",      "-i", (VoidPtr) &GlblIntensity,       SU_REAL_TYPE },
  { "MaxWidth",       "-w", (VoidPtr) &GlblMaxWidth,        SU_REAL_TYPE },
  { "CosinePower",    "-c", (VoidPtr) &GlblCosinePower,     SU_REAL_TYPE },
  { "ShaderPower",    "-s", (VoidPtr) &GlblShaderPower,     SU_REAL_TYPE },
  { "GlblViewDir",    "-v", (VoidPtr) &GlblViewDirStr,      SU_STRING_TYPE },
  { "SrfZTrans",      "-Z", (VoidPtr) &GlblSrfTranslate[2], SU_REAL_TYPE },
};
#define NUM_SET_UP	(sizeof(SetUp) / sizeof(ConfigStruct))

static void ProcessUVPts(CagdSrfStruct *Srf,
			 IPObjectStruct *PObj,
			 CagdUVType *UVPts,
			 int n,
			 char *SrfTexture,
			 CagdSrfStruct *SrfTextureSrf,
			 ImageStruct *ImageTexture);
static CagdCrvStruct *MarchOnSurface(CagdVType Dir,
				     CagdSrfStruct *Srf,
				     CagdSrfStruct *NSrf,
				     CagdSrfStruct *DuSrf,
				     CagdSrfStruct *DvSrf,
				     CagdUVType UVOrig,
				     CagdRType Length,
				     CagdRType *BreakNormal,
				     CagdSrfStruct *SrfTextureSrf,
				     ImageStruct *ImageTexture);
static RealType LineShader(CagdSrfStruct *Srf,
			   CagdSrfStruct *NSrf,
			   CagdUVType UV);
static RealType GetSrfArea(CagdSrfStruct *Srf);
static ImageStruct *RLELoadImage(char *File);
static ImageStruct *DiffImage(ImageStruct *Image, int XDir);
static void PtsShaderExit(int ExitCode);

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of PtsShad - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    int Error,
	DistCompFlag = FALSE,
	CosinePowerFlag = FALSE,
	ShaderPowerFlag = FALSE,
	SrfZTransFlag = FALSE,
	TextureFlag = FALSE,
	MaxWidthFlag = FALSE,
	IntensityFlag = FALSE,
        LightSrcFlag = FALSE,
        ViewDirFlag = FALSE,
	PtsDensityFlag = FALSE,
        MinSubdivFlag = FALSE,
	OutFileFlag = FALSE,
	VerFlag = FALSE,
	NumFiles = 0;
    char
	**FileNames = NULL;
    IPObjectStruct *PObjects, *NoProcessObjs, *PObj,
	*PObjWaves = NULL;
    MatrixType CrntViewMat;

    Config("ptsshad", SetUp, NUM_SET_UP);   /* Read config. file if exists. */
    if (GlblLightSourceStr != NULL && strlen(GlblLightSourceStr) > 0) {
#ifdef DOUBLE
	if (sscanf(GlblLightSourceStr, "%lf,%lf,%lf",
#else
	if (sscanf(GlblLightSourceStr, "%f,%f,%f",
#endif /* DOUBLE */
		   &GlblLightSource[0],
		   &GlblLightSource[1],
		   &GlblLightSource[2]) != 3) {
	    fprintf(stderr,
		    "Fail to parse LightSrcDir in configuration file.\n");
	    PtsShaderExit(-1);
	}
    }
    if (GlblViewDirStr != NULL && strlen(GlblViewDirStr) > 0) {
#ifdef DOUBLE
	if (sscanf(GlblViewDirStr, "%lf,%lf,%lf",
#else
	if (sscanf(GlblViewDirStr, "%f,%f,%f",
#endif /* DOUBLE */
		   &GlblViewDirStr[0],
		   &GlblViewDirStr[1],
		   &GlblViewDirStr[2]) != 3) {
	    fprintf(stderr,
		    "Fail to parse ViewDir in configuration file.\n");
	    PtsShaderExit(-1);
	}
    }

    if ((Error = GAGetArgs(argc, argv, CtrlStr,
			   &OutFileFlag, &GlblOutFileName, &GlblTalkative,
			   &DistCompFlag, &GlblDistRndrModel,
			   &CosinePowerFlag, &GlblCosinePower,
			   &ShaderPowerFlag, &GlblShaderPower,
			   &IntensityFlag, &GlblIntensity,
			   &LightSrcFlag, &GlblLightSource[0],
			   &GlblLightSource[1], &GlblLightSource[2],
			   &ViewDirFlag, &GlblViewDir[0],
			   &GlblViewDir[1], &GlblViewDir[2],
			   &MaxWidthFlag, &GlblMaxWidth,
			   &PtsDensityFlag, &GlblPtsDensity,
			   &SrfZTransFlag, &GlblSrfTranslate[2],
			   &TextureFlag, &GlblTexture,
			   &VerFlag, &NumFiles, &FileNames)) != 0) {
	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	PtsShaderExit(1);
    }

    if (VerFlag) {
	fprintf(stderr, "\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	ConfigPrint(SetUp, NUM_SET_UP);
	PtsShaderExit(0);
    }

    if (!NumFiles) {
	fprintf(stderr, "No data file names where given, exit.\n");
	GAPrintHowTo(CtrlStr);
	PtsShaderExit(1);
    }

    if (GlblDistRndrModel != PTS_DIST_POINT_E3)
	PT_NORMALIZE(GlblLightSource);

    /* Get the data files: */
    if ((PObjects = IritPrsrGetDataFiles(FileNames, NumFiles, TRUE, FALSE)) ==
									NULL)
	PtsShaderExit(0);

    if (IritPrsrWasPrspMat)
	MatMultTwo4by4(CrntViewMat, IritPrsrViewMat, IritPrsrPrspMat);
    else
	GEN_COPY(CrntViewMat, IritPrsrViewMat, sizeof(MatrixType));

    PObj = GMTransformObjectList(PObjects, CrntViewMat);
    IPFreeObjectList(PObjects);
    PObjects = PObj;

    /* Open output file, if necessary. */
    if (OutFileFlag) {
	if ((GlblOutFile = fopen(GlblOutFileName, "w")) == NULL) {
	    fprintf(stderr, "Failed to open \"%s\".\n", GlblOutFileName);
	    PtsShaderExit(2);
	}
    }
    else
	GlblOutFile = stdout;

    /* Dumps out all original surfaces. */
    for (PObj = PObjects; PObj != NULL; PObj = PObj -> Pnext) {
	if (GlblTexture != NULL && strlen(GlblTexture) > 0)
	    AttrSetObjectStrAttrib(PObj, "ltexture", GlblTexture);

	if (IP_IS_GEOM_OBJ(PObj) &&
	    AttrGetObjectRealAttrib(PObj, "transp") > IP_ATTR_BAD_REAL / 10)
	    IritPrsrPutObjectToFile(GlblOutFile, PObj);
    }	    

    /* Traverse all the objects and generate the coverage for them. */
    for (PObj = PObjects; PObj != NULL; PObj = PObj -> Pnext) {
	if (GlblTalkative)
	    fprintf(stderr, "Processing object \"%s\"\n", PObj -> Name);

	if (IP_IS_SRF_OBJ(PObj)) {
	    RealType
		RelPtsDensity = AttrGetObjectRealAttrib(PObj, "PtsDensity");
	    char
	        *SrfTexture = AttrGetObjectStrAttrib(PObj, "ltexture"),
	        *ImgTexture = AttrGetObjectStrAttrib(PObj, "itexture");
	    IPObjectStruct
		*SrfTextureObj = AttrGetObjectObjAttrib(PObj, "lTextSrf");
	    ImageStruct
		*ImageText = ImgTexture != NULL ? RLELoadImage(ImgTexture)
						: NULL;
	    CagdSrfStruct *Srf,
		*SrfTextureSrf = SrfTextureObj != NULL ?
						SrfTextureObj -> U.Srfs : NULL;

	    if (RelPtsDensity > IP_ATTR_BAD_REAL / 10)
		RelPtsDensity = 1.0;

	    for (Srf = PObj -> U.Srfs; Srf != NULL; Srf = Srf -> Pnext) {
		int n = (int) (GlblPtsDensity * RelPtsDensity *
			       GetSrfArea(Srf) * PTS_DENSITY);
		CagdUVType
		    *UVPts = SymbUniformAprxPtOnSrfDistrib(Srf, FALSE,
							   MAX(n, 2));

		ProcessUVPts(Srf, PObj, UVPts, n, SrfTexture,
			     SrfTextureSrf, ImageText);

		IritFree(UVPts);
	    }
	}
    }

    fclose(GlblOutFile);

    PtsShaderExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Process the even distribution of points on the surface according to the  *
* current shader and texture mapping and dump the result out.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   Srf:           To draw using line art.                                   *
*   UVPts:         Even distribution of points on Srf, in Euclidean space.   *
*   n:             Number of UV points in UVPts.                             *
*   SrfTexture:    Texture mapping function, if any, NULL otherwise.         *
*   SrfTextureSrf: Texture surface to verify positive against, if not NULL.  *
*   ImageTexture:  Image holding the gray level texture map of this surface. *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void ProcessUVPts(CagdSrfStruct *Srf,
			 IPObjectStruct *PObj,
			 CagdUVType *UVPts,
			 int n,
			 char *SrfTexture,
			 CagdSrfStruct *SrfTextureSrf,
			 ImageStruct *ImageTexture)
{
    int i;
    CagdCrvStruct *TCrv, *TCrv1, *TCrv2, *TCrv3, *TCrv4,
	*ShadingCrvs = NULL;
    CagdSrfStruct
	*DuSrf = CagdSrfDerive(Srf, CAGD_CONST_U_DIR),
	*DvSrf = CagdSrfDerive(Srf, CAGD_CONST_V_DIR),
	*NSrf = SymbSrfNormalSrf(Srf);
    CagdRType UMin, VMin, UMax, VMax;

    CagdSrfDomain(Srf, &UMin, &UMax, &VMin, &VMax);

    if (SrfTexture == NULL) {
	fprintf(GlblOutFile, "[OBJECT PTLIST\n    [POINTLIST %d\n", n);

	for (i = 0; i < n ; i++) {
	    CagdPType Pt;
	    CagdRType
		*R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

	    CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

	    fprintf(GlblOutFile, "\t[%f %f %f]\n", Pt[0], Pt[1], Pt[2]);
	}

	fprintf(GlblOutFile, "    ]\n]\n");
    }
    else {
	if (strnicmp(SrfTexture, "isoparam", 8) == 0) {
	    char
	        *AuxTexture = AttrGetObjectStrAttrib(PObj, "isoDir");
	    int VaryWidth, UIso;

	    if (AuxTexture == NULL)
		AuxTexture = "v_";

	    UIso = AuxTexture != NULL &&
	               (AuxTexture[0] == 'u' || AuxTexture[0] == 'U');
	    VaryWidth = AuxTexture != NULL &&
	               (AuxTexture[1] == 'w' || AuxTexture[1] == 'W');

	    /* Compute no texture - just shading. */
	    for (i = 0; i < n ; i++) {
		CagdPType Pt;
		CagdRType t1, t2,
		    Shade = LineShader(Srf, NSrf, UVPts[i]),
		    *R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

		/* Evaluate position. */
		CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		/* Extract the isocurve out of the surface. */
		if (UIso) {
		    CagdRType USpeed, ULength;
		    CagdVType V;
		    R = CagdSrfEval(DuSrf, UVPts[i][0], UVPts[i][1]);

		    CagdCoerceToE3(V, &R, -1, Srf -> PType);

		    USpeed = PT_LENGTH(V);
		    ULength = GlblIntensity / USpeed;
		    if (!VaryWidth)
		        ULength *= Shade;
		    TCrv = CagdCrvFromSrf(Srf, UVPts[i][1],
					  CAGD_CONST_V_DIR);

		    if (ULength > UMax - UMin) {
			LIST_PUSH(TCrv, ShadingCrvs);
		    }
		    else {
			t1 = MAX(UMin, UVPts[i][0] - ULength / 2);
			t2 = MIN(UMax, UVPts[i][0] + ULength / 2);
			TCrv1 = CagdCrvRegionFromCrv(TCrv, t1, t2);
			CagdCrvFree(TCrv);
			LIST_PUSH(TCrv1, ShadingCrvs);
		    }
		}
		else {
		    CagdRType VSpeed, VLength;
		    CagdVType V;
		    R = CagdSrfEval(DvSrf, UVPts[i][0], UVPts[i][1]);

		    CagdCoerceToE3(V, &R, -1, Srf -> PType);

		    VSpeed = PT_LENGTH(V);
		    VLength = GlblIntensity / VSpeed;
		    if (!VaryWidth)
		        VLength *= Shade;
		    TCrv = CagdCrvFromSrf(Srf, UVPts[i][0],
					  CAGD_CONST_U_DIR);

		    if (VLength > VMax - VMin) {
			LIST_PUSH(TCrv, ShadingCrvs);
		    }
		    else {
			t1 = MAX(VMin, UVPts[i][1] - VLength / 2);
			t2 = MIN(VMax, UVPts[i][1] + VLength / 2);
			TCrv1 = CagdCrvRegionFromCrv(TCrv, t1, t2);
			CagdCrvFree(TCrv);
			LIST_PUSH(TCrv1, ShadingCrvs);
		    }
		}

		if (VaryWidth)
		    AttrSetRealAttrib(&ShadingCrvs -> Attr, "width",
				      Shade * GlblMaxWidth);
	    }
	}
	else if (strnicmp(SrfTexture, "wood", 4) == 0) {
	    VectorType WoodDir, WoodDirInv;

#ifdef DOUBLE
	    if (sscanf(&SrfTexture[5], "%lf,%lf,%lf",
#else
	    if (sscanf(&SrfTexture[5], "%f,%f,%f",
#endif /* DOUBLE */
		       &WoodDir[0], &WoodDir[1], &WoodDir[2]) != 3) {
		fprintf(stderr,
			"Failed to extract direction of wood texture \"%s\"\n",
			SrfTexture);
		return;
	    }
	    PT_NORMALIZE(WoodDir);
	    PT_COPY(WoodDirInv, WoodDir);
	    PT_SCALE(WoodDirInv, -1.0);

	    /* Compute the wood texture. */
	    for (i = 0; i < n ; i++) {
		CagdPType Pt;
		CagdVType Nrml;
		CagdRType Length, Val,
		    *R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

		/* Evaluate position. */
		CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		/* Evaluate normal. */
		R = CagdSrfEval(NSrf, UVPts[i][0], UVPts[i][1]);
		CagdCoerceToE3(Nrml, &R, -1, NSrf -> PType);
		PT_NORMALIZE(Nrml);

		Length = GlblIntensity *
		    (WOOD_MIN_LENGTH +
		     (1 - fabs(DOT_PROD(Nrml, WoodDir))));

		/* We should favour short edges as they "cover" less area. */
		Val = IritRandom(0.0, 1.1);
		if (Length < SQR(SQR(Val))) {
		    TCrv1 = MarchOnSurface(WoodDir, Srf, NSrf, DuSrf, DvSrf,
					   UVPts[i], Length / 2, Nrml,
					   SrfTextureSrf, ImageTexture);
		    TCrv2 = MarchOnSurface(WoodDirInv, Srf, NSrf, DuSrf, DvSrf,
					   UVPts[i], Length / 2, Nrml,
					   SrfTextureSrf, ImageTexture);

		    /* Add width attribute. */
		    if (GlblDistRndrModel == NO_SHADER)
		        Val = SQR(SQR(IritRandom(0.0, 1.0)));
		    else
		        Val = LineShader(Srf, NSrf, UVPts[i]);

		    if (TCrv1 != NULL) {
			AttrSetRealAttrib(&TCrv1 -> Attr, "width",
					  GlblMaxWidth * Val);
		        LIST_PUSH(TCrv1, ShadingCrvs);
		    }
		    if (TCrv2 != NULL) {
			AttrSetRealAttrib(&TCrv2 -> Attr, "width",
					  GlblMaxWidth * Val);
		        LIST_PUSH(TCrv2, ShadingCrvs);
		    }
		}
	    }
	}
	else if (strnicmp(SrfTexture, "vood", 4) == 0) {
	    /* Compute the Vood texture. */
	    for (i = 0; i < n ; i++) {
		CagdPType Pt;
		CagdVType Nrml;
		CagdRType Length, Val,
		    *R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

		/* Evaluate position. */
		CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		/* Evaluate normal. */
		R = CagdSrfEval(NSrf, UVPts[i][0], UVPts[i][1]);
		CagdCoerceToE3(Nrml, &R, -1, NSrf -> PType);
		PT_NORMALIZE(Nrml);

		TCrv1 = MarchOnSurface(NULL, Srf, NSrf,
				       DuSrf, DvSrf,
				       UVPts[i], GlblIntensity / 2, Nrml,
				       SrfTextureSrf, ImageTexture);

		TCrv2 = MarchOnSurface(NULL, Srf, NSrf,
				       DuSrf, DvSrf,
				       UVPts[i], GlblIntensity / 2, Nrml,
				       SrfTextureSrf, ImageTexture);

		/* Add width attribute. */
		if (GlblDistRndrModel == NO_SHADER)
		    Val = SQR(SQR(IritRandom(0.0, 1.0)));
		else
		    Val = LineShader(Srf, NSrf, UVPts[i]);

		if (TCrv1 != NULL) {
		    AttrSetRealAttrib(&TCrv1 -> Attr, "width",
				      GlblMaxWidth * Val);
		    LIST_PUSH(TCrv1, ShadingCrvs);
		}

		if (TCrv2 != NULL) {
		    AttrSetRealAttrib(&TCrv2 -> Attr, "width",
				      GlblMaxWidth * Val);
		    LIST_PUSH(TCrv2, ShadingCrvs);
		}
	    }
	}
	else if (strnicmp(SrfTexture, "isomarch", 8) == 0) {
	    char
	        *AuxTexture = AttrGetObjectStrAttrib(PObj, "isoDir");
	    int UIso;
	    VectorType Dir, DirInv;

	    if (AuxTexture == NULL)
		AuxTexture = "v_";

	    PT_RESET(Dir);
	    PT_RESET(DirInv);
	    if (AuxTexture != NULL &&
		(AuxTexture[0] == 'u' || AuxTexture[0] == 'U')) {
		Dir[0] = 1.0; 
		DirInv[0] = -1.0; 
	    }
	    else {
		Dir[1] = 1.0; 
		DirInv[1] = -1.0; 
	    }
	    
	    /* Compute the isomatch texture. */
	    for (i = 0; i < n ; i++) {
		CagdPType Pt;
		CagdVType Nrml;
		CagdRType Length, RandVal,
		    *R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

		/* Evaluate position. */
		CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		/* Evaluate normal. */
		R = CagdSrfEval(NSrf, UVPts[i][0], UVPts[i][1]);
		CagdCoerceToE3(Nrml, &R, -1, NSrf -> PType);
		PT_NORMALIZE(Nrml);

		Length = GlblIntensity;

		/* We should favour short edges as they "cover" less area. */
		RandVal = IritRandom(0.0, 1.1);
		if (Length < SQR(SQR(RandVal))) {
		    TCrv1 = MarchOnSurface(Dir, Srf, NSrf, NULL, NULL,
					   UVPts[i], Length / 2, Nrml,
					   SrfTextureSrf, ImageTexture);
		    TCrv2 = MarchOnSurface(DirInv, Srf, NSrf, NULL, NULL,
					   UVPts[i], Length / 2, Nrml,
					   SrfTextureSrf, ImageTexture);

		    /* Add width attribute. */
		    RandVal = SQR(SQR(IritRandom(0.0, 1.0)));

		    if (TCrv1 != NULL) {
			AttrSetRealAttrib(&TCrv1 -> Attr, "width",
					  GlblMaxWidth * RandVal);
		        LIST_PUSH(TCrv1, ShadingCrvs);
		    }
		    if (TCrv2 != NULL) {
			AttrSetRealAttrib(&TCrv2 -> Attr, "width",
					  GlblMaxWidth * RandVal);
		        LIST_PUSH(TCrv2, ShadingCrvs);
		    }
		}
	    }
	}
	else if (strnicmp(SrfTexture, "silhouette", 10) == 0) {
	    static VectorType MarchDir, MarchDirInv;
	    VectorType ViewDirInv;
	    int TangCrvs, NrmlCrvs;
	    char
	        *AuxTexture = AttrGetObjectStrAttrib(PObj, "SilDir");

	    PT_NORMALIZE(GlblViewDir);
	    PT_COPY(ViewDirInv, GlblViewDir);
	    PT_SCALE(ViewDirInv, -1);

	    if (AuxTexture == NULL)
		AuxTexture = "tn";
	    TangCrvs = AuxTexture[0] == 't' || AuxTexture[0] == 'T';
	    NrmlCrvs = AuxTexture[1] == 'n' || AuxTexture[1] == 'N';

	    /* Compute the silheoutte texture. */
	    for (i = 0; i < n ; i++) {
		CagdPType Pt;
		CagdVType Nrml;
		CagdRType Length, RandVal,
		    *R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

		/* Evaluate position. */
		CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		/* Evaluate normal. */
		R = CagdSrfEval(NSrf, UVPts[i][0], UVPts[i][1]);
		CagdCoerceToE3(Nrml, &R, -1, NSrf -> PType);
		PT_NORMALIZE(Nrml);

		Length = pow(1 - fabs(DOT_PROD(Nrml, GlblViewDir)),
			     GlblCosinePower);

		/* We should favour long edges near the silhouette. */
		RandVal = IritRandom(0.0, 1.0);
		if (Length > RandVal) {
		    Length *= GlblIntensity;

		    CROSS_PROD(MarchDir, Nrml, GlblViewDir);
		    PT_NORMALIZE(MarchDir);
		    PT_COPY(MarchDirInv, MarchDir);
		    PT_SCALE(MarchDirInv, -1.0);

		    if (TangCrvs) {
			TCrv1 = MarchOnSurface(MarchDir, Srf, NSrf,
					       DuSrf, DvSrf,
					       UVPts[i], Length / 2, Nrml,
					       SrfTextureSrf, ImageTexture);
			TCrv2 = MarchOnSurface(MarchDirInv, Srf, NSrf,
					       DuSrf, DvSrf,
					       UVPts[i], Length / 2, Nrml,
					       SrfTextureSrf, ImageTexture);

			if (TCrv1 != NULL) {
			    LIST_PUSH(TCrv1, ShadingCrvs);
			}
			if (TCrv2 != NULL) {
			    LIST_PUSH(TCrv2, ShadingCrvs);
			}
		    }

		    if (NrmlCrvs) {
			TCrv1 = MarchOnSurface(GlblViewDir, Srf, NSrf,
					       DuSrf, DvSrf,
					       UVPts[i], Length / 2, Nrml,
					       SrfTextureSrf, ImageTexture);
			TCrv2 = MarchOnSurface(ViewDirInv, Srf, NSrf,
					       DuSrf, DvSrf,
					       UVPts[i], Length / 2, Nrml,
					       SrfTextureSrf, ImageTexture);
			
			if (TCrv1 != NULL) {
			    LIST_PUSH(TCrv1, ShadingCrvs);
			}
			if (TCrv2 != NULL) {
			    LIST_PUSH(TCrv2, ShadingCrvs);
			}
		    }
		}
	    }
	}
	else if (strnicmp(SrfTexture, "itexture", 8) == 0) {
	    VectorType ViewDirInv;

	    PT_NORMALIZE(GlblViewDir);
	    PT_COPY(ViewDirInv, GlblViewDir);
	    PT_SCALE(ViewDirInv, -1);

	    for (i = 0; i < n ; i++) {
		CagdPType Pt;
		CagdVType Nrml;
		CagdRType Length,
		    *R = CagdSrfEval(Srf, UVPts[i][0], UVPts[i][1]);

		/* Evaluate position. */
		CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		/* Evaluate normal. */
		R = CagdSrfEval(NSrf, UVPts[i][0], UVPts[i][1]);
		CagdCoerceToE3(Nrml, &R, -1, NSrf -> PType);
		PT_NORMALIZE(Nrml);

		Length = GlblIntensity;

		/* We should favour short edges as they "cover" less area. */
		TCrv1 = MarchOnSurface(GlblViewDir, Srf, NSrf, DuSrf, DvSrf,
				       UVPts[i], Length / 2, Nrml,
				       SrfTextureSrf, ImageTexture);
		TCrv2 = MarchOnSurface(ViewDirInv, Srf, NSrf, DuSrf, DvSrf,
				       UVPts[i], Length / 2, Nrml,
				       SrfTextureSrf, ImageTexture);

		if (TCrv1 != NULL) {
		    LIST_PUSH(TCrv1, ShadingCrvs);
		}
		if (TCrv2 != NULL) {
		    LIST_PUSH(TCrv2, ShadingCrvs);
		}
	    }
	}
	else if (strnicmp(SrfTexture, "curvature", 9) == 0) {
	    SymbEvalSrfCurvPrep(Srf, TRUE);

	    for (i = 0; i < n ; i++) {
		CagdRType K1, K2,
		    Length = GlblIntensity;
		CagdVType D1, D2, D;

		if (GlblDistRndrModel == NO_SHADER)
		    Length *= SQR(SQR(IritRandom(0.0, 1.0)));
		else
		    Length *= LineShader(Srf, NSrf, UVPts[i]);

		SymbEvalSrfCurvature(Srf, UVPts[i][0], UVPts[i][1], TRUE,
				     &K1, &K2, D1, D2);

		/* We should favour short edges as they "cover" less area. */
		PT_COPY(D, D1);
		TCrv1 = MarchOnSurface(D, Srf, NULL, NULL, NULL,
				       UVPts[i], Length / 2, NULL,
				       NULL, NULL);
		PT_COPY(D, D1);
		PT_SCALE(D, -1);
		TCrv2 = MarchOnSurface(D, Srf, NULL, NULL, NULL,
				       UVPts[i], Length / 2, NULL,
				       NULL, NULL);
		PT_COPY(D, D2);
		TCrv3 = MarchOnSurface(D, Srf, NULL, NULL, NULL,
				       UVPts[i], Length / 2, NULL,
				       NULL, NULL);
		PT_COPY(D, D2);
		PT_SCALE(D, -1);
		TCrv4 = MarchOnSurface(D, Srf, NULL, NULL, NULL,
				       UVPts[i], Length / 2, NULL,
				       NULL, NULL);

		if (TCrv1 != NULL) {
		    LIST_PUSH(TCrv1, ShadingCrvs);
		}
		if (TCrv2 != NULL) {
		    LIST_PUSH(TCrv2, ShadingCrvs);
		}
		if (TCrv3 != NULL) {
		    LIST_PUSH(TCrv3, ShadingCrvs);
		}
		if (TCrv4 != NULL) {
		    LIST_PUSH(TCrv4, ShadingCrvs);
		}
	    }

	    SymbEvalSrfCurvPrep(Srf, FALSE);
	}
	else if (strnicmp(SrfTexture, "pattern,", 8) == 0) {
	    IPObjectStruct
	        *PatternObj = AttrGetObjectObjAttrib(PObj, "objPattern");
	    CagdCrvStruct
		*PatternCrv = IP_IS_CRV_OBJ(PatternObj) ?
						PatternObj -> U.Crvs : NULL;
	    int i, j,
		FineNess = atoi(&SrfTexture[8]),
		PCLength = PatternCrv -> Length,
		PCOrder = PatternCrv -> Order,
		UVLen = 0;
	    CagdRType *R, *PCKV, T1, T2;
	    CagdUVType *UVPattern;

	    if (PatternCrv == NULL) {
		fprintf(stderr, "Expected a curve as a pattern texture\n");
		exit(1);
	    }

	    if (CAGD_IS_BEZIER_CRV(PatternCrv)) {
		PatternCrv = CnvrtBezier2BsplineCrv(PatternCrv);
	    }
	    else {
		PatternCrv = CagdCrvCopy(PatternCrv);
	    }
	    PCKV = PatternCrv -> KnotVector;

	    if (FineNess < 1)
		FineNess = 1;
	    if (FineNess > 10)
		FineNess = 10;

	    /* Evaluate the pattern only once. */
	    UVPattern = (CagdUVType *) IritMalloc(FineNess
						        * (PCLength + PCOrder)
				                        * sizeof(CagdUVType));
	    for (i = PCOrder - 1; i < PCLength; ) {
		T1 = PCKV[i];
		T2 = PCKV[++i];

		while (APX_EQ(T1, T2) && i < PCLength)
		    T2 = PCKV[++i];

		for (j = 0; j < FineNess; j++) {
		    CagdRType
			t = ((CagdRType) j) / FineNess;

		    t = t * T2 + (1 - t) * T1;
		    R = CagdCrvEval(PatternCrv, t);
		    CagdCoerceToE2((CagdRType *) &UVPattern[UVLen++], &R, -1,
				   PatternCrv -> PType);
		}
	    }
	    R = CagdCrvEval(PatternCrv, T2);
	    CagdCoerceToE2((CagdRType *) &UVPattern[UVLen++], &R, -1,
			   PatternCrv -> PType);
	    
	    /* Compute the pattern based texture. */
	    for (i = 0; i < n ; i++) {
		CagdRType **Points, Val;

		TCrv1 = BspCrvNew(UVLen, MIN(UVLen, 2), CAGD_PT_E3_TYPE);
		BspKnotUniformOpen(UVLen, MIN(UVLen, 2), TCrv1 -> KnotVector);
		Points = TCrv1 -> Points;

		for (j = 0; j < UVLen; j++) {
		    CagdPType Pt;
		    CagdRType Val, *R,
			U = UVPts[i][0] + UVPattern[j][0],
			V = UVPts[i][1] + UVPattern[j][1];

		    U = BOUND(U, UMin, UMax);
		    V = BOUND(V, VMin, VMax);
		    R = CagdSrfEval(Srf, U, V);
		    CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

		    Points[1][j] = Pt[0];
		    Points[2][j] = Pt[1];
		    Points[3][j] = Pt[2];
		}
 
		/* Add width attribute. */
		if (GlblDistRndrModel == NO_SHADER)
		    Val = SQR(SQR(IritRandom(0.0, 1.0)));
		else
		    Val = LineShader(Srf, NSrf, UVPts[i]);

		if (TCrv1 != NULL) {
		    AttrSetRealAttrib(&TCrv1 -> Attr, "width",
				      GlblMaxWidth * Val);
		    LIST_PUSH(TCrv1, ShadingCrvs);
		}
	    }

	    IritFree(PatternCrv);
	    CagdCrvFree(PatternCrv);
	}
	else {
	    fprintf(stderr, "Undefined texture \"%s\" ignored\n", SrfTexture);
	}
    }

    if (ShadingCrvs != NULL) {
	while (ShadingCrvs) {
	    RealType Width;
	    IPObjectStruct
	        *PObj = GenCrvObject("LineArt", ShadingCrvs, NULL);

	    ShadingCrvs = ShadingCrvs -> Pnext;
	    PObj -> U.Crvs -> Pnext = NULL;

	    if ((Width = AttrGetRealAttrib(PObj -> U.Crvs -> Attr, "width"))
						<= IP_ATTR_BAD_REAL / 10.0)
	        AttrSetObjectRealAttrib(PObj, "width", Width);

	    IritPrsrPutObjectToFile(GlblOutFile, PObj);
	    IPFreeObject(PObj);
	}
    }

    CagdSrfFree(NSrf);
    CagdSrfFree(DuSrf);
    CagdSrfFree(DvSrf);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Marches and create a curve on the surface of given Length in principal   *
* direction Dir.                                                             *
*   If DuSrf = DvSrf = NULL, the direction is taken directly from Dir.	     *
*   If Dir = NULL, the direction is coerced to by the vertical direction.    *
*   If NSrf == NULL, direction is derived from surface principal directions. *
*                                                                            *
* PARAMETERS:                                                                *
*   Dir:         Direction to march on surface (projected to tangent plane). *
*   Srf:         Surface to march on.                                        *
*   NSrf:        Normal field of surface to march on.                        *
*   DuSrf:       Partial with respect to u.                                  *
*   DvSrf:       Partial with respect to v.                                  *
*   UVOrig:      Origin on surface.                                          *
*   Length:      Length of March.                                            *
*   BreakNormal: If not NULL, compare current normal with BreatNormal and    *
*		 break of too much of a difference.			     *
*   SrfTextureSrf:  A texture surface to verify positivity against, if not   *
*		 NULL.							     *
*   ImageTexture:  Image holding the gray level texture map of this surface. *
*                                                                            *
* RETURN VALUE:                                                              *
*   CagdCrvStruct *:   A curve that approximates the march.                  *
*****************************************************************************/
static CagdCrvStruct *MarchOnSurface(CagdVType Dir,
				     CagdSrfStruct *Srf,
				     CagdSrfStruct *NSrf,
				     CagdSrfStruct *DuSrf,
				     CagdSrfStruct *DvSrf,
				     CagdUVType UVOrig,
				     CagdRType Length,
				     CagdRType *BreakNormal,
				     CagdSrfStruct *SrfTextureSrf,
				     ImageStruct *ImageTexture)
{
    static int
	ToggleDir = FALSE;
    int i, j, CrvLen;
    CagdPType Pt;
    CagdVType Nrml, Du, Dv, V, TangentDir;
    CagdRType Det, Wu, Wv, UMin, VMin, UMax, VMax, **Points, d, *R,
	SrfMarchStep = Dir == NULL ? SRF_MARCH_STEP / 10 : SRF_MARCH_STEP;
    CagdUVType UV;
    CagdPtStruct *CrvPt,
	*CrvPtList = NULL;
    CagdCrvStruct *Crv;

    CagdSrfDomain(Srf, &UMin, &UMax, &VMin, &VMax);

    GEN_COPY(UV, UVOrig, 2 * sizeof(CagdRType));

    if (SrfMarchStep * 5 > Length)
        SrfMarchStep = Length / 5;

    ToggleDir = !ToggleDir;

    do {
	if (SrfTextureSrf != NULL) {
	    R = CagdSrfEval(SrfTextureSrf, UV[0], UV[1]);
	    if (R[1] < 0.0)
		break;
	}

	R = CagdSrfEval(Srf, UV[0], UV[1]);
	CagdCoerceToE3(Pt, &R, -1, Srf -> PType);

	if (Dir == NULL &&
	    CrvPtList != NULL &&
	    fabs(Pt[0] - CrvPtList -> Pt[0]) > MAX_X_DIFF)
	    break;

	CrvPt = CagdPtNew();
	PT_COPY(CrvPt -> Pt, Pt);
	LIST_PUSH(CrvPt, CrvPtList);

	if (NSrf == NULL) {
	    CagdRType K1, K2;
	    CagdVType D1, D2;

	    SymbEvalSrfCurvature(Srf, UV[0], UV[1], TRUE, &K1, &K2, D1, D2);
	    PT_NORMALIZE(D1);
	    PT_NORMALIZE(D2);
	    if (fabs(K1 = DOT_PROD(D1, Dir)) > fabs(K2 = DOT_PROD(D2, Dir))) {
		if (K1 < 0)
		    PT_SCALE(D1, -1);
		PT_COPY(Dir, D1);
		PT_NORMALIZE(D1);
		Wu = D1[0] * SrfMarchStep;
		Wv = D1[1] * SrfMarchStep;
	    }
	    else {
		if (K2 < 0)
		    PT_SCALE(D2, -1);
		PT_COPY(Dir, D2);
		PT_NORMALIZE(D2);
		Wu = D2[0] * SrfMarchStep;
		Wv = D2[1] * SrfMarchStep;
	    }
	}
	else if (ImageTexture != NULL) {
	    i = (ImageTexture -> xSize - 1) * (UV[0] - UMin) / (UMax - UMin);
	    j = (ImageTexture -> ySize - 1) * (UV[1] - VMin) / (VMax - VMin);

	    i = i + j * (ImageTexture -> xSize - 1);
	    Wu = -ImageTexture -> DyImage -> data[i].Gray;
	    Wv =  ImageTexture -> DxImage -> data[i].Gray;
	    if ((d = sqrt(SQR(Wu) + SQR(Wv))) > 0.0) {
		Wu *= SrfMarchStep / d;
		Wv *= SrfMarchStep / d;
	    }
	    else {
	    }
	}
	else if (DuSrf != NULL && DvSrf != NULL) {
	    R = CagdSrfEval(DuSrf, UV[0], UV[1]);
	    CagdCoerceToE3(Du, &R, -1, DuSrf -> PType);

	    R = CagdSrfEval(DvSrf, UV[0], UV[1]);
	    CagdCoerceToE3(Dv, &R, -1, DvSrf -> PType);

	    R = CagdSrfEval(NSrf, UV[0], UV[1]);
	    CagdCoerceToE3(Nrml, &R, -1, NSrf -> PType);
	    PT_NORMALIZE(Nrml);

	    if (BreakNormal != NULL &&
		DOT_PROD(Nrml, BreakNormal) < IritRandom(0.9, 1.0))
	        break;

	    if (Dir == NULL) {
		static VectorType
		    XAxis = { 1, 0, 0 };

		XAxis[0] = ToggleDir ? 1 : -1;
		CROSS_PROD(TangentDir, XAxis, Nrml);
	    }
	    else {
		d = DOT_PROD(Nrml, Dir);

		for (i = 0; i < 3; i++)
		    TangentDir[i] = Dir[i] - Nrml[i] * d;
	    }

	    /* Now figure out weights of Du and Dv to produce a vector in   */
	    /* direction TangentDir, solving "Wu Du + Wv Dv = TangentDir".  */
	    if (fabs(Nrml[0]) > fabs(Nrml[1]) &&
		fabs(Nrml[0]) > fabs(Nrml[2])) {
		/* Solve in the YZ plane. */
		Det = Du[1] * Dv[2] - Du[2] * Dv[1];
		Wu = (TangentDir[1] * Dv[2] - TangentDir[2] * Dv[1]) / Det;
		Wv = (TangentDir[2] * Du[1] - TangentDir[1] * Du[2]) / Det;
	    }
	    else if (fabs(Nrml[1]) > fabs(Nrml[0]) &&
		     fabs(Nrml[1]) > fabs(Nrml[2])) {
		/* Solve in the XZ plane. */
		Det = Du[0] * Dv[2] - Du[2] * Dv[0];
		Wu = (TangentDir[0] * Dv[2] - TangentDir[2] * Dv[0]) / Det;
		Wv = (TangentDir[2] * Du[0] - TangentDir[0] * Du[2]) / Det;
	    }
	    else {
		/* Solve in the XY plane. */
		Det = Du[0] * Dv[1] - Du[1] * Dv[0];
		Wu = (TangentDir[0] * Dv[1] - TangentDir[1] * Dv[0]) / Det;
		Wv = (TangentDir[1] * Du[0] - TangentDir[0] * Du[1]) / Det;
	    }

	    for (i = 0; i < 3; i++)
	        V[i] = Du[i] * Wu + Dv[i] * Wv;

	    if (PT_LENGTH(V) == 0.0)
	        break;

	    Wu *= SrfMarchStep / PT_LENGTH(V);
	    Wv *= SrfMarchStep / PT_LENGTH(V);
	}
	else {
	    Wu = Dir[0] * SrfMarchStep;
	    Wv = Dir[1] * SrfMarchStep;
	}

	UV[0] += Wu;
	UV[1] += Wv;

	if (UV[0] > UMax || UV[0] < UMin || UV[1] > VMax || UV[1] < VMin)
	    break;

	Length -= SrfMarchStep;
    }
    while (Length >= 0.0);

    CrvLen = CagdListLength(CrvPtList);
    if (CrvLen >= 2) {
#define SHRINK_VERT_POLYLINES
#ifdef SHRINK_VERT_POLYLINES
	if (Dir == NULL) {
	    /* It is a vertical line - drop the intermediate points. */
	    CrvPt = CagdPtCopy(CagdListLast(CrvPtList));
	    CagdPtFreeList(CrvPtList -> Pnext);
	    CrvPtList -> Pnext = CrvPt;
	    CrvLen = 2;
	}
#endif /* SHRINK_VERT_POLYLINES */

	Crv = BspCrvNew(CrvLen, MIN(CrvLen, 3), CAGD_PT_E3_TYPE);
	BspKnotUniformOpen(CrvLen, MIN(CrvLen, 3), Crv -> KnotVector);
	Points = Crv -> Points;
	for (CrvPt = CrvPtList, i = 0;
	     i < CrvLen;
	     CrvPt = CrvPt -> Pnext, i++) {
	    Points[1][i] = CrvPt -> Pt[0];
	    Points[2][i] = CrvPt -> Pt[1];
	    Points[3][i] = CrvPt -> Pt[2];
	}
    }
    else
	Crv = NULL;

    CagdPtFreeList(CrvPtList);

    return Crv;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Computes a shading value for a given UV point on Srf.                    *
*                                                                            *
* PARAMETERS:                                                                *
*   Srf:     The surface with work with.                                     *
*   NSrf:    The normal field of the surface with work with.                 *
*   UV:      The current location to treat.                                  *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType:    Value between zero (black) and one (maximum light).         *
*****************************************************************************/
static RealType LineShader(CagdSrfStruct *Srf,
			   CagdSrfStruct *NSrf,
			   CagdUVType UV)
{
    CagdVType N;
    CagdPType P;
    CagdRType *R, Z;

    switch (GlblDistRndrModel) {
	case NO_SHADER:
	    Z = 1.0;
	    break;
	case PTS_DIST_PHONG:
	case PTS_DIST_PHONG_2L:
	case PTS_DIST_PHONG_SPEC:
	case PTS_DIST_PHONG_2L_SPEC:
	    R = CagdSrfEval(NSrf, UV[0], UV[1]);

	    CagdCoerceToE3(N, &R, -1, NSrf -> PType);
	    PT_NORMALIZE(N);

	    if (GlblDistRndrModel == PTS_DIST_PHONG_2L ||
		GlblDistRndrModel == PTS_DIST_PHONG_2L_SPEC) {
		Z = fabs(DOT_PROD(N, GlblLightSource));

		/* Add specular term, if have one. */
		if (GlblDistRndrModel == PTS_DIST_PHONG_2L_SPEC)
		    Z = pow(Z, GlblCosinePower);
	    }
	    else { /* PTS_DIST_PHONG || PTS_DIST_PHONG_SPEC */
		Z = (DOT_PROD(N, GlblLightSource) + 1.0) / 2.0;

		/* Add specular term, if have one. */
		if (GlblDistRndrModel == PTS_DIST_PHONG_SPEC)
		    Z = pow(Z, GlblCosinePower);
	    }
	    break;
	case PTS_DIST_ZNORMAL:
	    R = CagdSrfEval(NSrf, UV[0], UV[1]);

	    CagdCoerceToE3(N, &R, -1, NSrf -> PType);
	    PT_NORMALIZE(N);

	    Z = pow(0.95 * (1.0 - N[2]) + 0.05, GlblCosinePower);
	    break;
	case PTS_DIST_POINT_E3:
	    R = CagdSrfEval(Srf,UV[0], UV[1]);
	    CagdCoerceToE3(P, &R, -1, Srf -> PType);

	    Z = CGDistPointPoint(P, GlblLightSource);
	    Z = pow(Z, GlblShaderPower);
	    break;
	default:
	    fprintf(stderr, "Undefined shader %d ignored\n",
		    GlblDistRndrModel);
	    Z = 1.0;
	    break;
    }

    return Z;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   kReturns a crude approximation of the surface's area.                    *
*                                                                            *
* PARAMETERS:                                                                *
*   Srf:      To estimate its area.                                          *
*                                                                            *
* RETURN VALUE:                                                              *
*   RealType:    Estimated area                                              *
*****************************************************************************/
static RealType GetSrfArea(CagdSrfStruct *Srf)
{
    IPPolygonStruct
	*Polys = IritSurface2Polygons(Srf, FALSE, SRF_AREA_FINENESS,
				      FALSE, FALSE, 0);
    IPObjectStruct
	*PObj = GenPOLYObject(Polys);
    RealType
	Area = PolyObjectArea(PObj);

    if (GlblTalkative)
        fprintf(stderr, "\tSurface area ~= %f\n", Area);

    IPFreeObject(PObj);

    return Area;    
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Loads image file in RLE format.                                          *
*                                                                            *
* PARAMETERS:                                                                *
*   File:  Name of the image file.                                           *
*                                                                            *
* RETURN VALUE:                                                              *
*   ImageStruct *:  Pointer to dynamicaly created image.                     *
*****************************************************************************/
static ImageStruct *RLELoadImage(char *File)
{
    rle_hdr Header;
    rle_pixel **Rows;
    ImageStruct *PImage;
    PixelStruct *p;
    int Error, x, y;

    Header.rle_file = rle_open_f_noexit("RleLoadImage", File, "r");
    if (!Header.rle_file)
        return NULL;
    if (Error = rle_get_setup(&Header)) {
        rle_get_error(Error, "RleLoadImage", File);
        return NULL;
    }
    rle_row_alloc(&Header, &Rows);
    PImage = (ImageStruct *) IritMalloc(sizeof(ImageStruct));
    PImage -> xSize = Header.xmax - Header.xmin;
    PImage -> ySize = Header.ymax - Header.ymin;
    PImage -> data = p = (PixelStruct *) IritMalloc(sizeof(PixelStruct) *
				(PImage -> ySize + 1) * (PImage -> xSize + 1));
    for (y = 0; y <= PImage -> ySize; y++) {
        rle_getrow(&Header, Rows);
        for (x = 0; x <= PImage -> xSize; x++, p++) {
	    if (Header.ncolors == 1)
	        p -> Gray = Rows[RLE_RED][x] / 2;   /* Between zero and 127. */
	    else /* 3 colors */
	        p -> Gray = (Rows[RLE_RED][x] +
                             Rows[RLE_GREEN][x] +
                             Rows[RLE_BLUE][x]) / 6;/* Between zero and 127. */
        }
    }

    PImage -> DxImage = DiffImage(PImage, TRUE);
    PImage -> DyImage = DiffImage(PImage, FALSE);

    return PImage;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Differentiate an image.	                                             *
*                                                                            *
* PARAMETERS:                                                                *
*   Image:  Image to differentiate.                                          *
*   XDir:   Differentiate in X if TRUE, in Y otherwise.                      *
*                                                                            *
* RETURN VALUE:                                                              *
*   ImageStruct *:  Differentiated image.		                     *
*****************************************************************************/
static ImageStruct *DiffImage(ImageStruct *Image, int XDir)
{
    ImageStruct
	*DImage = (ImageStruct *) IritMalloc(sizeof(ImageStruct));
    int x, y;

    DImage -> data = (PixelStruct *) IritMalloc(sizeof(PixelStruct) *
				     Image -> ySize * Image -> xSize);
    DImage -> xSize = Image -> xSize - 1;
    DImage -> ySize = Image -> ySize - 1;

    for (y = 0; y <= DImage -> ySize; y++) {
	PixelStruct
	    *p  = DImage -> data + (DImage -> xSize + 1) * y,
	    *p1 = Image -> data + (Image -> xSize + 1) * y,
	    *p2 = Image -> data + (Image -> xSize + 1) * (y + 1);

        for (x = 0; x <= DImage -> xSize; x++) {
	    if (XDir) {
		p -> Gray = p1[1].Gray - p1 -> Gray;
		p++;
		p1++;
	    }
	    else {
		p -> Gray = p2 -> Gray - p1 -> Gray;
		p++;
		p1++;
		p2++;
	    }
        }
    }

    DImage -> DxImage = DImage -> DyImage = NULL;

    return DImage;
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* PtsShad Exit routine.							     M
*                                                                            *
* PARAMETERS:                                                                M
*   ExitCode:    To notify O.S. with result of program.                      M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   PtsShaderExit                                                            M
*****************************************************************************/
static void PtsShaderExit(int ExitCode)
{
    exit(ExitCode);
}

#ifdef DEBUG

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Dummy function to link at debugging time.                               *
*                                                                            *
* PARAMETERS:                                                                *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*                                                                            *
* KEYWORDS:                                                                  *
*****************************************************************************/
void DummyLinkCagdDebug(void)
{
    IritPrsrDbg();
}

#endif /* DEBUG */
