/* wptGPG.cpp - GnuPG configuration
 *    	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <string.h>
#include <stdio.h>
#include <windows.h>
#include <shlobj.h>
#include <ctype.h>
#include <io.h>

#include "wptGPG.h"
#include "wptGPGCmds.h"
#include "wptGPGOptSkel.h"
#include "wptTypes.h"
#include "wptNLS.h"
#include "wptRegistry.h"
#include "wptErrors.h"
#include "wptW32API.h"

#define GPG_CONF "gpg.conf"

struct gpg_watcher_s {
    FILETIME last_access;
    FILETIME access;
    char * object;
    int modified;
};

/* XXX need to watch for gpg.conf due to the fact keyring entries could be changed */
static const char * gpg_objs[] = {"pubring.gpg", "secring.gpg", "trustdb.gpg"};
static gpg_watcher_s gpg_table[3];
static int gpg_table_count = DIM (gpg_table);

int idea_available = 0;

static int check_keyring (char ** r_path);


static char*
multi_gnupg_path (void)
{
    static char buf[256+64];
    BOOL ec;

    /* MSDN: buf must be at least MAX_PATH=256 bytes */
    memset (buf, 0, sizeof (buf));
    ec = SHGetSpecialFolderPath (HWND_DESKTOP, buf, CSIDL_APPDATA, TRUE);
    if (ec != 1)
	return NULL;
    strcat (buf, "\\gnupg");
    if (access (buf, 00))
	return NULL;
    return buf;
}

/*
 * Return the full path of the GnuPG application. First the registry is scanned
 * for the entry 'HomeDir'. If it wasn't set, the default dir C:\GNUPG is used.
 */
char*
get_gnupg_path (void)
{
    char *p = NULL, *path = NULL;    
    
    p = get_reg_entry_gpg ("HomeDir");
    if (p) {
	path = m_strdup (p);
	free_if_alloc (p);
	return path;
    }
    else {
	p = multi_gnupg_path ();
	if (p)
	    return m_strdup (p);
    }
    return m_strdup ("c:\\gnupg");
} /* get_gnupg_path */


char*
get_gnupg_cfgfile (void)
{    
    char *p = NULL, *optfile = NULL, *path = NULL;
    size_t nlen = 0;

    path = get_gnupg_path ();
    if (path == NULL)
	return NULL;
    p = get_reg_entry_gpg ("OptFile");
    if (p && !strcmp (p, "")) 
    {
	nlen = strlen (path) + 64;
	optfile = new char[nlen + 1];
	if (optfile == NULL)
	    BUG (0);
	_snprintf (optfile, nlen, "%s\\"GPG_CONF, path);
    }
    else if( p ) {
	nlen = strlen( p ) + 4;
	optfile = new char[nlen + 1];
	if( optfile == NULL )
	    BUG( NULL );
	_snprintf( optfile, nlen, "%s", p );
    }
    else {
	nlen = strlen( path ) + 64;
	optfile = new char[nlen + 1];
	if( optfile == NULL )
	    BUG( NULL );
	_snprintf( optfile, nlen, "%s\\"GPG_CONF, path );
    }
    free_if_alloc (path);
    free_if_alloc (p);

    return optfile;
} /* get_gnupg_cfgfile */


char *
get_gnupg_keyring (int pub, int strict)
{    
    char * optfile = NULL, * path = NULL;
    char * keyring = NULL;

    path = get_gnupg_path ();
    if (!path)
	return NULL;
    keyring = make_filename (path, pub? "pubring" : "secring", "gpg");
    if (!strict && !file_exist_check (keyring)) {
	free_if_alloc (path);
	return keyring;
    }
    if (file_exist_check (keyring) || get_file_size (keyring) == 0) {
	free_if_alloc (keyring);
	optfile = make_filename (path, GPG_CONF, NULL);
	keyring = get_gnupg_keyring_from_options (optfile, pub);
    }
    free_if_alloc (path);
    free_if_alloc (optfile);
    return keyring;
} /* get_gnupg_keyring */


/*
 * Return the full path (with the gpg exe name). First the registry is scanned
 * for the entry 'gpgProgram'. If it wasn't set, the default path is the
 * appended string 'gpg.exe' is used.
 */
char*
get_gnupg_prog( void )
{    
    char *p, *path, *pgm = NULL;
    size_t nlen = 0;
	
    path = get_gnupg_path ();
    if (path == NULL)
	return NULL;

    p = get_reg_entry_gpg ("gpgProgram");
    if (p == NULL)
	pgm = make_filename (path, "gpg", "exe");
    else {
	pgm = m_strdup (p);
	free_if_alloc (p);
    }
    free_if_alloc (path);
    return pgm;
} /* get_gnupg_prog */


static char *
default_key_from_cache (int * ret_no_useable)
{
    const char * s;
    char * keyid = NULL;
    gpgme_key_t key;
    gpgme_keycache_t sec = keycache_get_ctx (0);

    if (!sec)
	BUG (0);
    gpgme_keycache_rewind (sec);
    while (!gpgme_keycache_next_key (sec, 1, &key)) 
    {
	if (gpgme_key_get_ulong_attr (key, GPGME_ATTR_KEY_USABLE, NULL, 0))
	{
	    s = gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0);
	    if (s)    
		keyid = m_strdup (s+8);	
	    break;
	}
    }
    if (!keyid) 
    {
	*ret_no_useable = 1;
	msg_box( NULL, _("No useable secret key found."), _("GPG Error"), MB_ERR);
    }
    return keyid;
} /* default_key_from_cache */


char *
get_gnupg_default_key (void)
{    
    gpg_optfile_t opt = NULL;
    gpg_option_t e;
    char * keyid = NULL, * optfile = NULL;
    int no_usable=0, rc = 0;

    optfile = get_gnupg_cfgfile ();
    if (!optfile)
	return default_key_from_cache( &no_usable );
    rc = parse_gpg_options( optfile, &opt );
    if( rc ) {
	free_if_alloc( optfile );
	return default_key_from_cache( &no_usable );
    }
    e = find_option( opt, "default-key" );
    if ( e )
	keyid = m_strdup( e->val );
    if( !e ) {
	e = find_option( opt, "local-user" );
	if( e )
	    keyid = m_strdup( e->val );
    }
    if( !e ) {
	e = find_option( opt, "encrypt-to" );
	if( e )
	    keyid = m_strdup( e->val );
    }
    free_if_alloc( optfile );
    release_gpg_options( opt );    

    if( !keyid )
	keyid = default_key_from_cache( &no_usable );
    return keyid;
} /* get_gnupg_default_key */

/*
 * Check if the gpg application (exe file) is available.
 */
int
check_gnupg_prog( void )
{
    char *pgm = NULL;
    int rc = 0;

    pgm = get_gnupg_prog( );
    if( pgm == NULL )
	rc = WPTERR_GPG_EXEFILE;
    if( file_exist_check( pgm ) )
	rc = WPTERR_GPG_EXEFILE;
    free_if_alloc( pgm );
    return rc;
} /* check_gpg_prog */


static int
parse_version_nr( const char * buf, int *major, int *minor, int *patch )
{
    char tmp[8];
    int i;
    
    if( strncmp( buf, "gpg ", 4 ) )
	return -1;    
    buf += 4;
    if( strncmp( buf, "(GnuPG) ", 8 ) )
	return -1;    
    buf += 8;
    i=0;
    while( buf && *buf != '.' && i < 8 )
	tmp[i++] = *buf++;
    tmp[i] = 0; buf++;
    *major = atol( tmp );
    i=0;
    while( buf && *buf != '.' && i < 8 )
	tmp[i++] = *buf++;
    tmp[i] = 0; buf++;
    *minor = atol( tmp );
    i=0;
    while( buf && isdigit( *buf ) && i < 8 )
	tmp[i++] = *buf++;
    tmp[i] = 0;
    *patch = atol( tmp );
    return 0;
}


int
check_gnupg_engine (int * r_major, int * r_minor, int * r_patch)
{
    gpgme_error_t err;
    char * eng = NULL;
    int major=0, minor=0, patch=0;
    int rc;
	
    err = gpgme_op_version( &eng );
    if( err )
	return -1;
    if( strstr( eng, "IDEA" ) )
	idea_available = 1;
    rc = parse_version_nr( eng, &major, &minor, &patch );
    free( eng ); eng = NULL;
    if( rc )
	return rc;
    if( major < *r_major 
     || minor < *r_minor)
	rc = 1;
    else {
	if (patch < *r_patch )
	    rc = 1;
	rc = 0;
    }
    *r_major = major;
    *r_minor = minor;
    *r_patch = patch;
    return rc;
} /* check_gnupg_engine */


int
check_gnupg_cfgfile (const char *fname, int *r_secrings, int *r_pubrings)
{
    gpg_optfile_t opt;    
    gpg_option_t e;
    int rc = 0;

    if( r_secrings )
	*r_secrings = 0;
    if( r_pubrings )
	*r_pubrings = 0;
    rc = parse_gpg_options( fname, &opt );
    if( rc )
	return WPTERR_FILE_OPEN;

    for( e = opt->list; e; e = e->next ) {
	if( !strcmp( e->name, "secret-keyring" ) ) {
	    if( !file_exist_check( e->val ) )
		r_secrings[0]++;
	}
	else if( !strcmp( e->name, "keyring" ) ) {
	    if( !file_exist_check( e->val ) )
		r_pubrings[0]++;
	}
    }
    release_gpg_options( opt );
    return 0;
} /* check_gnupg_cfgfile */

/*
 * Check if both keyrings are located in the gnupg home directory.
 */
int
gnupg_access_files (void)
{
    int rc = 0;
    int pubring_ok = 0, secring_ok = 0;
    int secrings = 0, pubrings = 0;
    char *optfile;

    if (gnupg_access_keyring (1))
	rc = WPTERR_GPG_KEYRINGS;
    else
	pubring_ok = 1;

    if (gnupg_access_keyring (0))
	rc = WPTERR_GPG_KEYRINGS;
    else
	secring_ok = 1;
    if (!pubring_ok || !secring_ok) {
	optfile = get_gnupg_cfgfile ();
	if (!optfile)
	    return WPTERR_GPG_KEYRINGS;
	rc = file_exist_check (optfile);
	if (!rc && get_file_size(optfile) > 0) {
	    rc = check_gnupg_cfgfile (optfile, &secrings, &pubrings);
	    if (!rc && secrings && pubrings) {
		free_if_alloc (optfile);
		return 0; /* found two keyrings in the option file */
	    }
	    else if ((!rc && pubrings && secring_ok) 
		  || (!rc && secrings && pubring_ok)) {
		free_if_alloc (optfile);
		return 0; /* found one keyring and one entry in the options file */
	    }
	    else
		return WPTERR_GPG_OPT_KEYRINGS;
	}
	free_if_alloc (optfile);
	rc = WPTERR_GPG_KEYRINGS;
    }
    return rc;
} /* gnupg_access_files */


static int
create_gpg_options( void )
{
    FILE *fp;
    char *s, *optfile;

    s = get_gnupg_path( );
    if( s == NULL )
	return WPTERR_FILE_CREAT;
    optfile = make_filename( s, GPG_CONF, NULL );
    fp = fopen( optfile, "wb" );
    if( fp == NULL ) {	
	return WPTERR_FILE_CREAT;
	goto fail;
    }
    fwrite( options_skel, 1, strlen( options_skel ), fp );
    fclose( fp );

fail:
    free_if_alloc( s );
    free_if_alloc( optfile );
    return 0;
} /* create_gpg_options */


/*
 * Return the contents of the options file as a char buf.
 */
char *
get_gnupg_config( void )
{
    FILE * fp;
    char * p = NULL, * optfile = NULL;
    int fsize, rc = 0;
	
    optfile = get_gnupg_cfgfile( );
    if( optfile == NULL )
	return NULL;
    fsize = get_file_size( optfile );
    if( !fsize ) {
	rc = create_gpg_options( );
	if ( rc )
	    return NULL;
	fsize = get_file_size( optfile );
    }
    if( fsize > 100000 )
	goto leave; /* too large */
    p = new char[fsize+1];
    if( p == NULL )
	BUG( NULL );
    fp = fopen( optfile, "rb" );
    if( fp == NULL ) {
	free_if_alloc( p );
	return NULL;
    }
    fread( p, 1, fsize, fp );
    fclose( fp );
    p[fsize] = '\0';
    free_if_alloc( optfile );

leave:
    return p;
} /* get_gnupg_config */


int
set_gnupg_default_key (const char * key)
{
    gpg_optfile_t opt;
    gpg_option_t e;
    char *optfile = NULL;
    int rc = 0;

    optfile = get_gnupg_cfgfile ();
    if (!optfile)
	return -1;
    rc = parse_gpg_options (optfile, &opt);
    if( rc ) {
	free_if_alloc (optfile);
	return -1;
    }
    e = find_option (opt, "default-key");
    if (e) {
	free_if_alloc (e->val);
	e->val = m_strdup (key);
	e->used = 1;
    }
    else
	add_entry (opt, ENTRY_MULTI, "default-key", key);
    rc = commit_gpg_options (optfile, opt);

    free_if_alloc (optfile);
    release_gpg_options (opt);

    return rc;
} /* set_gnupg_default_key */


/*
 * Set the contents of the options file.
 */
int
set_gnupg_options( const char *buf, size_t buflen )
{
    FILE *fp;	
    char *optfile = NULL;

    optfile = get_gnupg_cfgfile( );
    if( optfile == NULL )
	return WPTERR_FILE_CREAT;

    fp = fopen( optfile, "wb" );
    if( fp == NULL ) {
	free_if_alloc( optfile );
	return WPTERR_FILE_CREAT;
    }
    fwrite( buf, 1, buflen, fp );
    fclose( fp );
    free_if_alloc( optfile );
    return 0;
} /* set_gnupg_options */

/*
 * Check if the line contains a valid GPG argument.
 */
static int
check_line( const char *buf )
{
    int j, len;
    int rc = 0;

    if( *buf == '#' || *buf == '\r' || *buf == '\n' )
	return 1;
    rc = 0;
    for ( j = 0; valid_gpg_args[j]; j++ ) {
	len = strlen( valid_gpg_args[j] );
	if( !strncmp( valid_gpg_args[j], buf, len ) )
	    rc = 1;	
    }

    return rc;
} /* check_line */


int
check_gnupg_options( const char *buf )
{
    char line[1024];
    int nbytes = 0;
    unsigned j;
	
    for ( j = 0; j<strlen( buf ) && j < sizeof(line); j++ ) {
	line[nbytes++] = buf[j];
	if ( buf[j] == '\n' || j == ( strlen( buf ) - 1 ) ) {
	    line[nbytes] = '\0';
	    if( !check_line( line ) ) {
		msg_box( NULL, line, "options", MB_OK );
		return 1;	
	    }
	    nbytes = 0;
	}	
    }

    return 0;
} /* check_gnupg_options */


static int
get_last_gnupg_access (gpg_watcher_s * ctx)
{
    HANDLE fd;
    char *path = NULL, *file = NULL;

    path = get_gnupg_path ();
    file =  make_filename (path, ctx->object, NULL);
    fd = CreateFile (file, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_ALWAYS, 0, NULL);
    if( fd == INVALID_HANDLE_VALUE ) {
	free_if_alloc (path);
	free_if_alloc (file);
	return WPTERR_FILE_OPEN;
    }
    GetFileTime (fd, NULL, NULL, &ctx->access);
    CloseHandle (fd);
    free_if_alloc (path);	
    free_if_alloc (file);
    return 0;
} /* get_last_gnupg_access */


static void
check_last_gnupg_access( gpg_watcher_s *ctx )
{		
    ctx->modified = 0;

    if( ctx->last_access.dwHighDateTime != ctx->access.dwHighDateTime &&
	ctx->last_access.dwLowDateTime != ctx->access.dwLowDateTime )	
	ctx->modified = 1;

    ctx->last_access.dwLowDateTime = ctx->access.dwLowDateTime;
    ctx->last_access.dwHighDateTime = ctx->access.dwHighDateTime;
} /* check_last_gnupg_access */


void
init_gnupg_table (void)
{	
    int j;

    for (j = 0; j < gpg_table_count; j++) {
	gpg_table[j].object = m_strdup (gpg_objs[j]);
	memset (&gpg_table[j].access, 0, sizeof (FILETIME));
	memset (&gpg_table[j].last_access, 0, sizeof (FILETIME));
	gpg_table[j].modified = 0;
    }
} /* init_gnupg_table */


void
free_gnupg_table (void)
{
    int j;

    for (j=0; j < gpg_table_count; j++)
	free_if_alloc (gpg_table[j].object);
} /* free_gnupg_table */


int
keyring_check_last_access (void)
{	
    int rc, j;

    rc = 0;
    for (j = 0; j < gpg_table_count; j++) {
	get_last_gnupg_access( &gpg_table[j] );
	check_last_gnupg_access( &gpg_table[j] );
	if ( gpg_table[j].modified )
	    rc++;	
    }

    return rc;
} /* keyring_check_last_access */


const char *
gnupg_check_file_ext (const char * fname)
{		
    char file_ext[5];

    if (!strchr( fname, '.' ))
	return "UNKNOWN";

    strncpy (file_ext, fname + strlen (fname) - 4, 4);
    file_ext[4] = '\0';
    if (!stricmp (file_ext, ".asc"))
	return "ARMORED";
    else if (!stricmp (file_ext, ".sig"))
	return "SIGNED";
    else if  (!stricmp (file_ext, ".gpg")
	|| !stricmp (file_ext, ".pgp"))
	return "ENCRYPTED";
    else
	return "UNKNOWN";
	
    return "UNKNOWN";
} /* gnupg_check_file_ext */


char *
get_gnupg_keyring_from_options (const char * fname, int pub)
{
    gpg_optfile_t opt;
    gpg_option_t e;
    char * kring = NULL;
    int rc = 0;

    rc = parse_gpg_options (fname, &opt);
    if (rc)
	return NULL;
    if (pub)
	e = find_option (opt, "keyring");
    else
	e = find_option (opt, "secret-keyring");
    if (e)
	kring = m_strdup (e->val);
    release_gpg_options (opt);

    return kring;
} /* get_gnupg_keyring_from_options */



/* XXX: does not work with write-protected floppies */
static int
my_access (const char * fname)
{
    HANDLE hd;
    hd = CreateFile (fname, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
    if (hd == INVALID_HANDLE_VALUE)
	return -1;
    CloseHandle (hd);
    return 0;
} /* my_access */


int
gpg_check_permissions (int showmsg)
{
    char * p, * name = NULL;
    int failed = 0, ans=0, attrs=0;

    p = get_gnupg_path ();
    check_keyring (&p);
    if (p) {
	name = make_filename (p, "pubring", "gpg");
	free_if_alloc (p);
	if ((attrs=GetFileAttributes (name)) & FILE_ATTRIBUTE_READONLY) {
	    ans = msg_box (NULL, 
			   _("The selected keyring has the read-only file\n"
			     "attribute. In this state you do not have write\n"
			     "access. Do you want to remove the attribute?"),
		     _("GPG Information"), MB_YESNO);
	    if (ans == IDYES) {
		attrs &= ~FILE_ATTRIBUTE_READONLY;
		if (!SetFileAttributes (name, attrs)) {
		    msg_box (NULL, _("Could not reset read-only state."), 
			     _("GPG Error"), MB_ERR);
		    failed = 1;
		}
	    }
	    else if (ans == IDNO) {
		/*
		msg_box (NULL, _("All commands with write access to the keyring\n"
				 "will be disabled."), _("GPG Information"), MB_INFO);
		*/
		failed = 1;
	    }
	}
	if (my_access (name)) {
	    if (showmsg)
		msg_box (NULL, 
		_("You do not have file access to modify the contents of\n"
		  "one or both of the selected keyrings.\n"
		  "\n"
		  "The keyrings are in a read-only state which is propably\n"
		  "caused by another program which already opened the files.\n"),
		   _("GPG Warning"), MB_WARN);
	    failed = 2;
	}
    }
    free_if_alloc (name);
    return failed;
} /* gpg_check_permissions */


static int
check_homedir (void)
{	
    char * homedir = NULL;
    int yes = 0;

    homedir = get_reg_entry_gpg ("HomeDir");
    if (!homedir)
	homedir = multi_gnupg_path ();
    if (!homedir)
	homedir = m_strdup ("c:\\gnupg");
    if (homedir) {
	if (GetFileAttributes (homedir) == 0xFFFFFFFF) {
	    yes = log_box (_("Preferences"), MB_YESNO,
			   _("%s does not exit.\n"
			     "Do you want to create this directory?"), homedir);
	    if (yes == IDYES) {
		if (CreateDirectory (homedir, NULL) == FALSE) {
		    free_if_alloc (homedir);
		    return WPTERR_DIR_CREAT;
		}
	    }
	    return WPTERR_DIR_OPEN;
	}
	free_if_alloc (homedir);
    }
    return 0;
} /* check_homedir */


int
gnupg_check_homedir (void)
{	
    char *homedir = NULL, *prog = NULL;
    int rc = 0, ec = 0;
    
    rc = check_homedir ();
    if (rc)
	return rc;
    if ((homedir = get_reg_entry_gpg ("HomeDir")) &&
	!(prog = get_reg_entry_gpg ("gpgProgram" ))) {
	prog = make_filename (homedir, "gpg", "exe");
	if (file_exist_check (prog) == 0) {
	    rc = set_reg_entry_gpg ("gpgProgram", prog);
	    if (rc)
		goto fail;
	}
	free_if_alloc (homedir);
	free_if_alloc (prog);
	return rc;
    }
    if ((prog = get_reg_entry_gpg ("gpgProgram"))
	&& file_exist_check (prog)) {
	free_if_alloc (prog);
	homedir = get_reg_entry_gpg ("HomeDir");
	if (!homedir) {
	    rc = WPTERR_GENERAL;
	    goto fail;
	}
	prog = make_filename (homedir, "gpg", "exe");
	if (file_exist_check (prog) == 0) {
	    rc = set_reg_entry_gpg ("gpgProgram", prog);
	    if (rc)
		goto fail;
	    free_if_alloc (prog);
	    return rc;
	}
    }
    
    /* Change the return code if homedir doesn't exist or if the program 
       doesn't exist. Note that exist_checks return 0 to suggest existance. */
    if ((!homedir || dir_exist_check (homedir)))
	rc = WPTERR_GENERAL;
    
fail:
    free_if_alloc (homedir);
    free_if_alloc (prog);
    return rc;
} /* gnupg_check_homedir */


int
gnupg_copy_keyrings (void)
{
    const char * pring, * sring;
    char * file = NULL, * path = NULL;
    int id = 0, rc = 0;
    HWND hwnd;
    
    path = get_gnupg_path ();
    if (!path)
	return WPTERR_GENERAL;
    hwnd = GetDesktopWindow ();

    pring = get_filename_dlg (hwnd, FILE_OPEN, _("Please choose your public keyring"),
			      _("GPG Keyrings (*.gpg)\0*.gpg\0\0"),NULL);
    if (!pring) {
	msg_box (hwnd, _("No keyring was chosen. Exit."), _("WinPT Error"), MB_ERR);
	free_if_alloc (path);
	return WPTERR_GENERAL;
    }
    file = make_filename (path, "pubring", "gpg");
    if (file_exist_check (file) == 0) {
	id = msg_box (hwnd, _("Overwrite old public keyring?"), "WinPT", MB_INFO|MB_YESNO);
	if (id == IDNO)
	    goto fail;
    }
    if (!CopyFile (pring, file, FALSE)) {
	msg_box (hwnd, _("Could not copy file."), _("WinPT Error"), MB_ERR);
	rc = WPTERR_FILE_READ;
	goto fail;
    }
    free_if_alloc (file);

    sring = get_filename_dlg (hwnd, FILE_OPEN, _("Please choose your secret keyring"),
			      _("GPG Keyrings (*.gpg)\0*.gpg\0\0"), NULL);
    if (!sring) {
	msg_box( NULL, _("No keyring was chosen. Exit."), _("WinPT Error"), MB_ERR );
	return WPTERR_GENERAL;
    }
    file = make_filename (path, "secring", "gpg");
    if (file_exist_check (file) == 0) {
	id = msg_box (hwnd, _("Overwrite old secret keyring?"), "WinPT", MB_INFO|MB_YESNO);
	if( id == IDNO )
	    goto fail;
    }
    if (!CopyFile (sring, file, FALSE)) {
	msg_box( NULL, _("Could not copy file."), _("WinPT Error"), MB_ERR);
	rc = WPTERR_FILE_READ;
    }

fail:
    free_if_alloc (file);
    free_if_alloc (path);
    return rc;
} /* gnupg_import_keyrings */


void
gnupg_backup_options (int keep)
{
    char *optfile = NULL;
    char bak[1024];

    optfile = get_gnupg_cfgfile ();
    if (optfile == NULL)
	return;
    if (keep)
	_snprintf (bak, DIM (bak)-1, "%s.old", optfile);
    else
	_snprintf (bak, DIM (bak)-1, "%s.O", optfile);
    CopyFile (optfile, bak, keep);
    free_if_alloc (optfile);
} /* gnupg_backup_options */



static int
backup_one_file (const char * srcpath, const char * srcn,
		 const char * dstpath, const char * dstn)
{
    char * src, * dst;
    BOOL rc;

    src = make_filename (srcpath, srcn, "gpg");
    if (!src)
        BUG (NULL);
    dst = make_filename (dstpath, dstn, "gpg");
    if (!dst)
        BUG (NULL);
    rc = CopyFile (src, dst, FALSE);
    free_if_alloc (src);
    free_if_alloc (dst);
    if (!rc)
    {
	log_box (_("Backup"), MB_ERR, _("Backup keyring \"%s\" failed"), srcn);
        return WPTERR_GENERAL;
    }
    return 0;
} /* backup_one_file */


static int
check_keyring (char ** r_path)
{
    char * p;
    char * opt, * name;

    if (!*r_path)
	return 0;
    p = make_filename (*r_path, "pubring", "gpg");
    if (!p || get_file_size (p) > 0)
	return 0;

    opt = get_gnupg_cfgfile ();
    if (!opt)
	BUG (0);
    name = get_gnupg_keyring_from_options (opt, 1);
    free_if_alloc (opt);
    free_if_alloc (p);
    if (!name)
	return 0;
    p = strrchr (name, '\\');
    if (!p) 
    {
	free_if_alloc (name);
	return 0;	
    }
    free_if_alloc (*r_path);
    *r_path = new char [strlen (name)+1];
    memset (*r_path, 0, strlen (name));
    strncpy (*r_path, name, (p-name));
    free_if_alloc (name);
    return 1;
}


void
gnupg_backup_keyrings (void)
{
    char * srcpath = NULL, * dstpath = NULL;
    int rc, bakmode=0;

    if (!reg_prefs.auto_backup)
	return;
    bakmode = reg_prefs.backup.mode;
    srcpath =  get_gnupg_path ();
    check_keyring (&srcpath);
    if (bakmode == 1) 
    {
	dstpath = get_gnupg_path ();
	check_keyring (&dstpath);
    }
    else if (bakmode == 2) 
    {
	char * tmpfile;
	FILE * fp;

	dstpath = m_strdup (reg_prefs.backup.path);
	if (!dstpath)
	    BUG (0);
	tmpfile = make_filename (reg_prefs.backup.path, "test", "tmp");
	fp = fopen (tmpfile, "wb");
	if (!fp)
	    rc = log_box (_("Backup"), MB_WARN|MB_RETRYCANCEL, _("The backup drive '%s' does not seems to accessable.\n"
							         "Please insert/check the drive to continue."), dstpath);
	else
	{
	    rc = 0;
	    fclose (fp);
	    unlink (tmpfile);
	}
	free_if_alloc (tmpfile);
	if (!fp || rc == IDCANCEL)
	    return;
    }
    else 
    {
	log_box (_("Backup"), MB_ERR, _("Invalid backup mode %d"), bakmode);
	return;
    }
    rc = backup_one_file (srcpath, "pubring", dstpath, "pubring-bak");
    if (!rc)
	rc = backup_one_file (srcpath, "secring", dstpath, "secring-bak");
    free_if_alloc (srcpath);
    free_if_alloc (dstpath);
} /* gnupg_backup_keyrings */


void
gnupg_display_error (void)
{	
    char tmpath[512], * errstr;
    size_t size = 0;
    FILE * fp;

    GetTempPath (sizeof tmpath - 32, (tmpath));
    strcat (tmpath, "gpg_stderr");
    size = get_file_size (tmpath);
    if (file_exist_check (tmpath) || size <= 0)
	return;
    fp = fopen( tmpath, "rb" );
    if (!fp) {
	msg_box( NULL, _("No GPG error description available."), _("GPG Error"), MB_INFO );
	return;
    }
    errstr = new char[size+1];
    if (!errstr)
	BUG (0);
    fread (errstr, 1, size, fp);
    errstr[size] = '\0';
    fclose (fp);
    msg_box (NULL, errstr, _("GPG Error"), MB_INFO);
    free_if_alloc (errstr);
} /* gnupg_display_error */


int
gnupg_access_keyring (int _pub)
{
    int rc = 0;
    char * name = get_gnupg_keyring (_pub, 1);
    if (file_exist_check (name))
	rc = -1;
    free_if_alloc (name);
    return rc;
}

