/* wptClipDecryptDlg.cpp - Clipboard decrypt dialog
 *	Copyright (C) 2000-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "wptTypes.h"
#include "wptW32API.h"
#include "wptAgent.h"
#include "wptNLS.h"
#include "wptGPG.h"
#include "wptVersion.h"
#include "wptErrors.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptDlgs.h"
#include "wptKeylist.h"
#include "wptFileManager.h"
#include "../resource.h"

char *
get_key_userid (const char * keyid)
{
    gpgme_key_t key;
    const char * s;
    char * p;

    if( get_pubkey( keyid, &key ) )
	return m_strdup( "" );
    s = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
    if( !s )
	s = _("user ID not found");
    p = new char[strlen( s ) + 4 + 8];
    if( !p )
	BUG( NULL );
    sprintf( p, "\n    \"%s\"", s );
    return p;
} /* get_key_userid */


static gpgme_error_t
list_recipients (gpgme_ctx_t ctx, gpgme_recipients_t *r_rset)
{
    gpgme_error_t rc;
    gpgme_data_t clipdat;

    rc = gpgme_data_new_from_clipboard (&clipdat);
    if (!rc)
	rc = gpgme_op_list_keys (clipdat, NULL, r_rset);
    gpgme_data_release (clipdat);
    return rc;
} /* list_recipients */


int
algo_from_list (gpgme_recipients_t rset, const char * keyid)
{
    void * ctx = NULL;
    const char * s;

    gpgme_recipients_enum_open (rset, &ctx);
    while ((s = gpgme_recipients_enum_read (rset, &ctx))) {
	if (!strcmp( s+1, keyid))
	    return *s;
    }
    return 0;
} /* algo_from_list */


int
clip_decrypt_dlg (HWND hwnd)
{
    gpgme_error_t rc;
    gpgme_ctx_t ctx = NULL;
    gpgme_recipients_t keys = NULL;
    gpgme_sig_t sig = NULL;
    gpgme_op_flags_t flags;
    passphrase_cb_s pwd;
    const char *created, *s;
    char keyid[17] = {0};
    int novalid = 0;
    unsigned int pkalgo = 0;
    u32 t, sigstat;
    
    rc = gpgme_new (&ctx);
    if (rc)
	BUG (0);
    gpgme_enable_logging (ctx);

    /* allow to verify data generated by 'gpg -a --sign foo' */
    if (fm_assume_onepass_sig (NULL) == 0) {
	rc = list_recipients (ctx, &keys);
	if (rc) {
	    gpgme_show_error (hwnd, rc, ctx, _("Decryption"), MB_ERR);
	    gpgme_release (ctx);
	    return rc;
	}
    }

    set_gpg_passphrase_cb (ctx, &pwd, GPG_CMD_DECRYPT, hwnd, _("Decryption"));
    pwd.enc_to = keys;
    rc = gpgme_op_clip_decrypt (ctx);
    memset (pwd.pwd, 0, sizeof pwd.pwd);
    if (pwd.cancel)
	goto leave;

    if (rc == GPGME_Bad_Passphrase)
	agent_del_cache (pwd.keyid);
    gpgme_decrypt_get_status (ctx, keyid, &flags);
    if (rc == GPGME_No_Seckey && (flags & GPGME_OPFLAG_NOSECKEY)) {
	char * p = get_key_userid (keyid+8);
	int pkalgo = algo_from_list (keys, keyid);
	log_box (_("Decryption"), MB_ERR, 
		 _("Encrypted with %s key, ID %s.%s\n"
		   "Decryption failed: secret key not available."), 
		   gpgme_key_expand_attr( GPGME_ATTR_ALGO, pkalgo),
		   keyid+8, p);
	free_if_alloc (p);
	goto leave;
    }
    else if (rc) {
	gpgme_cliptype_t pgp_type;
	gpgme_clip_get_pgptype (&pgp_type);
	if (rc == GPGME_No_Data && (pgp_type & GPGME_CLIP_MESSAGE))
	    msg_box (hwnd, _("Broken OpenPGP message (maybe: quoted printable character in armor)."), _("Decryption"), MB_INFO);
	else
	    gpgme_show_error (hwnd, rc, ctx, _("Decryption"), MB_ERR);
	goto leave;
    }

    if (flags & GPGME_OPFLAG_BADMDC) {
	const char *s;
	s = _("WARNING: encrypted message has been manipulated!\n"
	    "\n"
	    "Do *NOT* trust any text or data output from this file!\n"
	    "It is likely, the data was corrupted during the transport\n"
	    "but it might be also possible that this is part of an attack.");
	msg_box (hwnd, s, _("*** IMPORTANT ***"), MB_INFO);
    }
    
    show_msg (hwnd, 1500, _("GnuPG Status: Finished"));
    gpgme_decrypt_get_sig_ctx (ctx, &sig);
    sigstat = gpgme_sig_get_ulong_attr (sig, 0, GPGME_ATTR_VALIDITY);
    if (sig && sigstat != GPGME_SIG_STAT_ERROR) {
	gpgme_key_t key;
	const char * val;
	char keyid[16+1];
	
	val = gpgme_sig_get_string_attr( sig, GPGME_ATTR_KEYID );
	if( !val )
	    val = "DEADBEEFDEADBEEF";
	_snprintf( keyid, sizeof(keyid)-1, "%s", val+8 );
	sigstat = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_VALIDITY );
	if( get_pubkey( keyid, &key ) )
	    goto leave;

	t = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_OTRUST );
	if( t == GPGME_VALIDITY_FULL || t == GPGME_VALIDITY_ULTIMATE )
	    s = _("Signature Status: Created with a fully trusted key");
	else if ( t == GPGME_VALIDITY_MARGINAL )
	    s = _("Signature Status: Created with a marginal trusted key");
	else if ( t == GPGME_VALIDITY_UNKNOWN
                  || t == GPGME_VALIDITY_UNDEFINED
                  || t == GPGME_VALIDITY_NEVER ) {
	    novalid = 1;
	    s =  _("Signature Status: Created with an UNTRUSTED key");
	}
	else
	    s = _("Signature Status: Created with an invalid key");
	t = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_CREATED );
	created = strtimestamp( t );
        t = gpgme_key_get_ulong_attr( key, GPGME_ATTR_VALIDITY, NULL, 0 );
	val = gpgme_sig_get_string_attr( sig, GPGME_ATTR_USERID );
	if( !val )
	    val = _("Invalid User ID");
	log_box( _("WinPT Verify"), MB_OK, 
		 _("%s\n"
		   "%s\n"	
                   "Signature made %s\n"
                   "From \"%s\" using key ID 0x%s"
		   "%s %s"),
		   s, gpg_sigstat[sigstat % SIGSTAT_MASK], created,
		   val, keyid, 
		   novalid? "\nPrimary key fingerprint: " : "",
		   novalid? get_key_fpr( key ) : ""
		   );
    }
    
leave:
    gpgme_release( ctx );
    gpgme_recipients_release( keys );
    gpgme_sig_release( sig );
    return rc;
} /* clip_decrypt_dlg */
