#ifndef GPGLIB_OPENPGP_H
#define GPGLIB_OPENPGP_H

#ifdef __cplusplus
extern "C" {
#endif

#ifndef HAVE_BYTE_TYPEDEF
typedef unsigned char byte;
#define HAVE_BYTE_TYPEDEF
#endif

#ifndef HAVE_U32_TYPEDEF
typedef unsigned long u32;
#define HAVE_U32_TYPEDEF
#endif

#ifndef HAVE_ULONG_TYPEDEF
typedef unsigned long ulong;
#define SIZEOF_UNSIGNED_LONG 4
#define HAVE_ULONG_TYPEDEF
#endif

#ifndef HAVE_U16_TYPEDEF
typedef unsigned short u16;
#define HAVE_U16_TYPEDEF
#endif

#ifndef DIM
#define DIM(v) (sizeof(v)/sizeof((v)[0]))
#endif

#ifndef safe_free
#define safe_free( p ) do { \
    if( p ) { free( p ); p = NULL; } \
} while( 0 )
#endif

#define G10ERR_GENERAL	       1
#define G10ERR_UNKNOWN_PACKET  2
#define G10ERR_UNKNOWN_VERSION 3 /* Unknown version (in packet) */
#define G10ERR_PUBKEY_ALGO     4 /* Unknown pubkey algorithm */
#define G10ERR_DIGEST_ALGO     5 /* Unknown digest algorithm */
#define G10ERR_BAD_PUBKEY      6 /* Bad public key */
#define G10ERR_BAD_SECKEY      7 /* Bad secret key */
#define G10ERR_BAD_SIGN        8 /* Bad signature */
#define G10ERR_NO_PUBKEY       9 /* public key not found */
#define G10ERR_CHECKSUM       10 /* checksum error */
#define G10ERR_BAD_PASS       11 /* Bad passphrase */
#define G10ERR_CIPHER_ALGO    12 /* Unknown cipher algorithm */
#define G10ERR_KEYRING_OPEN   13
#define G10ERR_INVALID_PACKET 14
#define G10ERR_INVALID_ARMOR  15
#define G10ERR_NO_USER_ID     16
#define G10ERR_NO_SECKEY      17 /* secret key not available */
#define G10ERR_WRONG_SECKEY   18 /* wrong seckey used */
#define G10ERR_UNSUPPORTED    19
#define G10ERR_BAD_KEY	      20 /* bad (session) key */
#define G10ERR_READ_FILE      21
#define G10ERR_WRITE_FILE     22
#define G10ERR_COMPR_ALGO     23 /* Unknown compress algorithm */
#define G10ERR_OPEN_FILE      24
#define G10ERR_CREATE_FILE    25
#define G10ERR_PASSPHRASE     26 /* invalid passphrase */
#define G10ERR_NI_PUBKEY      27
#define G10ERR_NI_CIPHER      28
#define G10ERR_SIG_CLASS      29
#define G10ERR_BAD_MPI	      30
#define G10ERR_RESOURCE_LIMIT 31
#define G10ERR_INV_KEYRING    32
#define G10ERR_TRUSTDB	      33 /* a problem with the trustdb */
#define G10ERR_BAD_CERT       34 /* bad certicate */
#define G10ERR_INV_USER_ID    35
#define G10ERR_CLOSE_FILE     36
#define G10ERR_RENAME_FILE    37
#define G10ERR_DELETE_FILE    38
#define G10ERR_UNEXPECTED     39
#define G10ERR_TIME_CONFLICT  40
#define G10ERR_WR_PUBKEY_ALGO 41 /* unusabe pubkey algo */
#define G10ERR_FILE_EXISTS    42
#define G10ERR_WEAK_KEY       43 /* NOTE: hardcoded into the cipher modules */
#define G10ERR_WRONG_KEYLEN   44 /* NOTE: hardcoded into the cipher modules */
#define G10ERR_INV_ARG	      45
#define G10ERR_BAD_URI	      46  /* syntax error in URI */
#define G10ERR_INVALID_URI    47  /* e.g. unsupported scheme */
#define G10ERR_NETWORK	      48  /* general network error */
#define G10ERR_UNKNOWN_HOST   49
#define G10ERR_SELFTEST_FAILED 50
#define G10ERR_NOT_ENCRYPTED  51
#define G10ERR_NOT_PROCESSED  52
#define G10ERR_UNU_PUBKEY     53
#define G10ERR_UNU_SECKEY     54
#define G10ERR_KEYSERVER      55

#define CIPHER_ALGO_NONE	 0
#define CIPHER_ALGO_IDEA	 1
#define CIPHER_ALGO_3DES	 2
#define CIPHER_ALGO_CAST5	 3
#define CIPHER_ALGO_BLOWFISH	 4  /* blowfish 128 bit key */
#define CIPHER_ALGO_SAFER_SK128  5
#define CIPHER_ALGO_DES_SK	 6
#define CIPHER_ALGO_AES          7
#define CIPHER_ALGO_AES192       8
#define CIPHER_ALGO_AES256       9
#define CIPHER_ALGO_RIJNDAEL     CIPHER_ALGO_AES
#define CIPHER_ALGO_RIJNDAEL192  CIPHER_ALGO_AES192
#define CIPHER_ALGO_RIJNDAEL256  CIPHER_ALGO_AES256
#define CIPHER_ALGO_TWOFISH	10  /* twofish 256 bit */
#define CIPHER_ALGO_SKIPJACK   101  /* experimental: skipjack */
#define CIPHER_ALGO_TWOFISH_OLD 102 /* experimental: twofish 128 bit */
#define CIPHER_ALGO_DUMMY      110  /* no encryption at all */

#define PUBKEY_ALGO_RSA        1
#define PUBKEY_ALGO_RSA_E      2     /* RSA encrypt only */
#define PUBKEY_ALGO_RSA_S      3     /* RSA sign only */
#define PUBKEY_ALGO_ELGAMAL_E 16     /* encrypt only ElGamal (but not for v3)*/
#define PUBKEY_ALGO_DSA       17
#define PUBKEY_ALGO_ELGAMAL   20     /* sign and encrypt elgamal */

#define PUBKEY_USAGE_SIG     1	    /* key is good for signatures */
#define PUBKEY_USAGE_ENC     2	    /* key is good for encryption */
#define PUBKEY_USAGE_CERT    4      /* key is also good to certify other keys*/

#define DIGEST_ALGO_MD5       1
#define DIGEST_ALGO_SHA1      2
#define DIGEST_ALGO_RMD160    3
#define DIGEST_ALGO_SHA256    8
#define DIGEST_ALGO_SHA384    9
#define DIGEST_ALGO_SHA512    10

#define MD_MD5    DIGEST_ALGO_MD5
#define MD_SHA1   DIGEST_ALGO_SHA1
#define MD_RMD160 DIGEST_ALGO_RMD160
#define MD_SHA256 DIGEST_ALGO_SHA256
#define MD_SHA384 DIGEST_ALGO_SHA384
#define MD_SHA512 DIGEST_ALGO_SHA512

#define is_RSA(a)     ((a)==PUBKEY_ALGO_RSA || (a)==PUBKEY_ALGO_RSA_E \
		       || (a)==PUBKEY_ALGO_RSA_S )
#define is_ELGAMAL(a) ((a)==PUBKEY_ALGO_ELGAMAL || (a)==PUBKEY_ALGO_ELGAMAL_E)


#define PUBKEY_MAX_NPKEY  4
#define PUBKEY_MAX_NSKEY  6
#define PUBKEY_MAX_NSIG   2
#define PUBKEY_MAX_NENC   2

#define MAX_FINGERPRINT_LEN 20

#define IOBUFCTRL_INIT	    1
#define IOBUFCTRL_FREE	    2
#define IOBUFCTRL_UNDERFLOW 3
#define IOBUFCTRL_FLUSH     4
#define IOBUFCTRL_DESC	    5
#define IOBUFCTRL_CANCEL    6
#define IOBUFCTRL_USER	    16


/*-- md.c --*/
struct         gpgm_md_s;
typedef struct gpgm_md_s *gpg_md_t;

gpg_md_t     gpg_md_open( int algo );
void         gpg_md_close( gpg_md_t md );
void         gpg_md_final( gpg_md_t md );
void         gpg_md_write( gpg_md_t md, byte *inbuf, size_t len );
void         gpg_md_putc( gpg_md_t md, int c );
const byte * gpg_md_read( gpg_md_t md );
int          gpg_md_selftest (void);


/*-- iobuf.c --*/
struct gpg_iobuf_s {
    int use;	       /* 1 input , 2 output, 3 temp */
    _off_t nlimit;
    _off_t nbytes;      /* used together with nlimit */
    _off_t ntotal;      /* total bytes read (position of stream) */
    int nofast; 	/* used by the iobuf_get() */
    void *directfp;
    struct {
	size_t size;   /* allocated size */
	size_t start;  /* number of invalid bytes at the begin of the buffer */
	size_t len;    /* currently filled to this size */
	byte *buf;
    } d;
    int filter_eof;
    int error;
    int (*filter)( void *opaque, int control,
		   struct gpg_iobuf_s *chain, byte *buf, size_t *len);
    void *filter_ov;	/* value for opaque */
    int filter_ov_owner;
    char *real_fname;
    struct gpg_iobuf_s *chain;	/* next iobuf used for i/o if any (passed to filter) */
    int no, subno;
    const char *desc;
    void *opaque;      /* can be used to hold any information	 */
		       /* this value is copied to all instances */
    struct {
	size_t size;   /* allocated size */
	size_t start;  /* number of invalid bytes at the begin of the buffer */
	size_t len;    /* currently filled to this size */
	byte *buf;
    } unget;
};
typedef struct gpg_iobuf_s *gpg_iobuf_t;

typedef struct {
    /* these fields may be initialized */
    int what;		    /* what kind of armor headers to write */
    int only_keyblocks;     /* skip all headers but ".... key block" */
    const char *hdrlines;   /* write these headerlines */

    /* these fileds must be initialized to zero */
    int no_openpgp_data;    /* output flag: "No valid OpenPGP data found" */

    /* the following fields must be initialized to zero */
    int inp_checked;	    /* set if the input has been checked */
    int inp_bypass;	    /* set if the input is not armored */
    int in_cleartext;	    /* clear text message */
    int not_dash_escaped;   /* clear text is not dash escaped */
    int hashes; 	    /* detected hash algorithms */
    int faked;		    /* we are faking a literal data packet */
    int truncated;	    /* number of truncated lines */
    int qp_detected;
    int pgp2mode;

    byte *buffer;	    /* malloced buffer */
    unsigned buffer_size;   /* and size of this buffer */
    unsigned buffer_len;    /* used length of the buffer */
    unsigned buffer_pos;    /* read position */

    byte radbuf[4];
    int idx, idx2;
    u32 crc;

    int status; 	    /* an internal state flag */
    int cancel;
    int any_data;	    /* any valid armored data seen */
    int pending_lf;	    /* used together with faked */
} armor_filter_context_t;

struct unarmor_pump_s;
typedef struct unarmor_pump_s *UnarmorPump;

struct gpg_mpi_s {
    int alloced;    /* array size (# of allocated limbs) */
    int nlimbs;     /* number of valid limbs */
    int nbits;	    /* the real number of valid bits (info only) */
    int sign;	    /* indicates a negative number */
    unsigned flags; /* bit 0: array must be allocated in secure memory space */
		    /* bit 1: the mpi is encrypted */
		    /* bit 2: the limb is a pointer to some m_alloced data */
    unsigned long *d;  /* array with the limbs */
};
typedef struct gpg_mpi_s *gpg_mpi_t;

typedef enum {
	PKT_NONE	   =0,
	PKT_PUBKEY_ENC	   =1, /* public key encrypted packet */
	PKT_SIGNATURE	   =2, /* secret key encrypted packet */
	PKT_SYMKEY_ENC	   =3, /* session key packet (OpenPGP)*/
	PKT_ONEPASS_SIG    =4, /* one pass sig packet (OpenPGP)*/
	PKT_SECRET_KEY	   =5, /* secret key */
	PKT_PUBLIC_KEY	   =6, /* public key */
	PKT_SECRET_SUBKEY  =7, /* secret subkey (OpenPGP) */
	PKT_COMPRESSED	   =8, /* compressed data packet */
	PKT_ENCRYPTED	   =9, /* conventional encrypted data */
	PKT_MARKER	  =10, /* marker packet (OpenPGP) */
	PKT_PLAINTEXT	  =11, /* plaintext data with filename and mode */
	PKT_RING_TRUST	  =12, /* keyring trust packet */
	PKT_USER_ID	  =13, /* user id packet */
	PKT_PUBLIC_SUBKEY =14, /* public subkey (OpenPGP) */
	PKT_OLD_COMMENT   =16, /* comment packet from an OpenPGP draft */
	PKT_ATTRIBUTE     =17, /* PGP's attribute packet */
	PKT_ENCRYPTED_MDC =18, /* integrity protected encrypted data */
	PKT_MDC 	  =19, /* manipulaion detection code packet */
	PKT_COMMENT	  =61, /* new comment packet (private) */
        PKT_GPG_CONTROL   =63  /* internal control packet */
} pkttype_t;

typedef struct packet_struct PACKET;

/* PKT_GPG_CONTROL types */
typedef enum {
    CTRLPKT_CLEARSIGN_START = 1,
    CTRLPKT_PIPEMODE        = 2,
    CTRLPKT_PLAINTEXT_MARK  =3
} ctrlpkttype_t;

typedef enum {
    PREFTYPE_NONE = 0,
    PREFTYPE_SYM  = 1,
    PREFTYPE_HASH = 2,
    PREFTYPE_ZIP  = 3
} preftype_t;

typedef struct {
    byte type; 
    byte value;
} prefitem_t;

typedef struct {
    int  mode;
    byte hash_algo;
    byte salt[8];
    u32  count;
} STRING2KEY;

typedef struct {
    byte version;
    byte cipher_algo;	 /* cipher algorithm used */
    STRING2KEY s2k;
    byte seskeylen;   /* keylength in byte or 0 for no seskey */
    byte seskey[1];
} PKT_symkey_enc;

typedef struct {
    u32     keyid[2];	    /* 64 bit keyid */
    byte    version;
    byte    pubkey_algo;    /* algorithm used for public key scheme */
    byte    throw_keyid;
    gpg_mpi_t     data[PUBKEY_MAX_NENC];
} PKT_pubkey_enc;


typedef struct {
    u32     keyid[2];	    /* 64 bit keyid */
    byte    sig_class;	    /* sig classification */
    byte    digest_algo;    /* algorithm used for digest */
    byte    pubkey_algo;    /* algorithm used for public key scheme */
    byte    last;	    /* a stupid flag */
} PKT_onepass_sig;


typedef struct {
    size_t size;  /* allocated */
    size_t len;   /* used */
    byte data[1];
} subpktarea_t;

struct revocation_key {
    byte rclass;
    byte algid;
    byte fpr[MAX_FINGERPRINT_LEN];
};

typedef struct {
    ulong   local_id;	    /* internal use, valid if > 0 */
    struct {
	unsigned checked:1; /* signature has been checked */
	unsigned valid:1;   /* signature is good (if checked is set) */
	unsigned unknown_critical:1;
        unsigned exportable:1;
        unsigned revocable:1;
        unsigned policy_url:1; /* Policy URL is present */
        unsigned notation:1; /* At least one notation is present */
        unsigned expired:1;
    } flags;
    u32     keyid[2];	    /* 64 bit keyid */
    u32     timestamp;	    /* signature made */
    u32     expiredate;     /* expires at this date or 0 if not at all */
    byte    version;
    byte    sig_class;	    /* sig classification, append for MD calculation*/
    byte    pubkey_algo;    /* algorithm used for public key scheme */
    /* (PUBKEY_ALGO_xxx) */
    byte    digest_algo;    /* algorithm used for digest (DIGEST_ALGO_xxxx) */
    struct revocation_key **revkey;
    int numrevkeys;
    subpktarea_t *hashed;    /* all subpackets with hashed  data (v4 only) */
    subpktarea_t *unhashed;  /* ditto for unhashed data */
    byte digest_start[2];   /* first 2 bytes of the digest */
    gpg_mpi_t  data[PUBKEY_MAX_NSIG];
} PKT_signature;

#define ATTRIB_IMAGE 1

/* This is the cooked form of attributes */
struct user_attribute {
    byte type;
    const byte *data;
    u32 len;
};

typedef struct {
    int ref;              /* reference counter */
    int len;		  /* length of the name */
    struct user_attribute *attribs;
    int numattribs;
    byte *attrib_data;    /* if this is not NULL, the packet is an attribute */
    unsigned long attrib_len;
    int help_key_usage;
    u32 help_key_expire;
    int is_primary;
    int is_revoked;
    int is_expired;
    u32 expiredate;       /* expires at this date or 0 if not at all */
    prefitem_t *prefs;    /* list of preferences (may be NULL)*/
    int mdc_feature;
    u32 created;          /* according to the self-signature */
    byte selfsigversion;
    char name[1];
} PKT_user_id;


/****************
 * Note about the pkey/skey elements:  We assume that the secret keys
 * has the same elemts as the public key at the begin of the array, so
 * that npkey < nskey and it is possible to compare the secret and
 * public keys by comparing the first npkey elements of pkey againts skey.
 */
typedef struct {
    u32     timestamp;	    /* key made */
    u32     expiredate;     /* expires at this date or 0 if not at all */
    u32     max_expiredate; /* must not expire past this date */
    byte    hdrbytes;	    /* number of header bytes */
    byte    version;
    byte    selfsigversion; /* highest version of all of the self-sigs */
    byte    pubkey_algo;    /* algorithm used for public key scheme */
    byte    pubkey_usage;   /* for now only used to pass it to getkey() */
    byte    req_usage;      /* hack to pass a request to getkey() */
    byte    req_algo;       /* Ditto */
    u32     has_expired;    /* set to the expiration date if expired */ 
    int     is_revoked;     /* key has been revoked */
    int     is_valid;       /* key (especially subkey) is valid */
    int     dont_cache;     /* do not cache this */
    ulong   local_id;	    /* internal use, valid if > 0 */
    u32     main_keyid[2];  /* keyid of the primary key */
    u32     keyid[2];	    /* calculated by keyid_from_pk() */
    prefitem_t *prefs;      /* list of preferences (may be NULL) */
    int     mdc_feature;    /* mdc feature set */
    byte    *namehash;	    /* if != NULL: found by this name */
    PKT_user_id *user_id;   /* if != NULL: found by that uid */
    struct revocation_key *revkey;
    int     numrevkeys;
    int is_primary;
    gpg_mpi_t     pkey[PUBKEY_MAX_NPKEY];
} PKT_public_key;

typedef struct {
    u32     timestamp;	    /* key made */
    u32     expiredate;     /* expires at this date or 0 if not at all */
    u32     max_expiredate; /* must not expire past this date */
    byte    hdrbytes;	    /* number of header bytes */
    byte    version;
    byte    pubkey_algo;    /* algorithm used for public key scheme */
    byte    pubkey_usage;
    byte    req_usage;
    byte    req_algo;
    u32     has_expired;    /* set to the expiration date if expired */ 
    int     is_revoked;     /* key has been revoked */
    int     is_valid;       /* key (especially subkey) is valid */
    u32     main_keyid[2];  /* keyid of the primary key */
    u32     keyid[2];   
    byte is_primary;
    byte is_protected;	/* The secret info is protected and must */
    /* be decrypted before use, the protected */
    /* GPG_MPIs are simply (void*) pointers to memory */
    /* and should never be passed to a mpi_xxx() */
    struct {
	byte algo;  /* cipher used to protect the secret information*/
        byte sha1chk;  /* SHA1 is used instead of a 16 bit checksum */ 
	STRING2KEY s2k;
	byte ivlen;  /* used length of the iv */
	byte iv[16]; /* initialization vector for CFB mode */
    } protect;
    gpg_mpi_t skey[PUBKEY_MAX_NSKEY];
    u16 csum;		/* checksum */
} PKT_secret_key;


typedef struct {
    int  len;		  /* length of data */
    char data[1];
} PKT_comment;

typedef struct {
    u32  len;		  /* reserved */
    byte  new_ctb;
    byte  algorithm;
    gpg_iobuf_t buf;		  /* gpg_iobuf_t reference */
} PKT_compressed;

typedef struct {
    u32  len;		  /* length of encrypted data */
    int  extralen;        /* this is (blocksize+2) */
    byte new_ctb;	  /* uses a new CTB */
    byte mdc_method;	  /* > 0: integrity protected encrypted data packet */
    gpg_iobuf_t buf;		  /* gpg_iobuf_t reference */
} PKT_encrypted;

typedef struct {
    byte hash[20];
} PKT_mdc;

typedef struct {
    unsigned int trustval;
    unsigned int sigcache;
} PKT_ring_trust;

typedef struct {
    u32  len;		  /* length of encrypted data */
    gpg_iobuf_t buf;		  /* gpg_iobuf_t reference */
    byte new_ctb;
    byte is_partial;      /* partial length encoded */
    int mode;
    u32 timestamp;
    int  namelen;
    char name[1];
} PKT_plaintext;

typedef struct {
    int  control;
    size_t datalen;
    char data[1];
} PKT_gpg_control;

/* combine all packets into a union */
struct packet_struct {
    pkttype_t pkttype;
    union {
	void *generic;
	PKT_symkey_enc	*symkey_enc;	/* PKT_SYMKEY_ENC */
	PKT_pubkey_enc	*pubkey_enc;	/* PKT_PUBKEY_ENC */
	PKT_onepass_sig *onepass_sig;	/* PKT_ONEPASS_SIG */
	PKT_signature	*signature;	/* PKT_SIGNATURE */
	PKT_public_key	*public_key;	/* PKT_PUBLIC_[SUB)KEY */
	PKT_secret_key	*secret_key;	/* PKT_SECRET_[SUB]KEY */
	PKT_comment	*comment;	/* PKT_COMMENT */
	PKT_user_id	*user_id;	/* PKT_USER_ID */
	PKT_compressed	*compressed;	/* PKT_COMPRESSED */
	PKT_encrypted	*encrypted;	/* PKT_ENCRYPTED[_MDC] */
	PKT_mdc 	*mdc;		/* PKT_MDC */
	PKT_ring_trust	*ring_trust;	/* PKT_RING_TRUST */
	PKT_plaintext	*plaintext;	/* PKT_PLAINTEXT */
        PKT_gpg_control *gpg_control;   /* PKT_GPG_CONTROL */
    } pkt;
};

struct gpg_kbnode_s {
    struct gpg_kbnode_s *next;
    PACKET *pkt;
    int flag;
    int private_flag;
    ulong recno;  /* used while updating the trustdb */
};
typedef struct gpg_kbnode_s *gpg_kbnode_t;

#define gpg_init_packet(a) do { (a)->pkttype = (pkttype_t)0;		\
			    (a)->pkt.generic = NULL;	\
			    } while(0)

typedef enum {
    SIGSUBPKT_TEST_CRITICAL=-3,
    SIGSUBPKT_LIST_UNHASHED=-2,
    SIGSUBPKT_LIST_HASHED  =-1,
    SIGSUBPKT_NONE	   = 0,
    SIGSUBPKT_SIG_CREATED  = 2, /* signature creation time */
    SIGSUBPKT_SIG_EXPIRE   = 3, /* signature expiration time */
    SIGSUBPKT_EXPORTABLE   = 4, /* exportable */
    SIGSUBPKT_TRUST	   = 5, /* trust signature */
    SIGSUBPKT_REGEXP	   = 6, /* regular expression */
    SIGSUBPKT_REVOCABLE    = 7, /* revocable */
    SIGSUBPKT_KEY_EXPIRE   = 9, /* key expiration time */
    SIGSUBPKT_ARR	   =10, /* additional recipient request */
    SIGSUBPKT_PREF_SYM	   =11, /* preferred symmetric algorithms */
    SIGSUBPKT_REV_KEY	   =12, /* revocation key */
    SIGSUBPKT_ISSUER	   =16, /* issuer key ID */
    SIGSUBPKT_NOTATION	   =20, /* notation data */
    SIGSUBPKT_PREF_HASH    =21, /* preferred hash algorithms */
    SIGSUBPKT_PREF_COMPR   =22, /* preferred compression algorithms */
    SIGSUBPKT_KS_FLAGS	   =23, /* key server preferences */
    SIGSUBPKT_PREF_KS	   =24, /* preferred key server */
    SIGSUBPKT_PRIMARY_UID  =25, /* primary user id */
    SIGSUBPKT_POLICY	   =26, /* policy URL */
    SIGSUBPKT_KEY_FLAGS    =27, /* key flags */
    SIGSUBPKT_SIGNERS_UID  =28, /* signer's user id */
    SIGSUBPKT_REVOC_REASON =29, /* reason for revocation */
    SIGSUBPKT_FEATURES     =30, /* feature flags */
    SIGSUBPKT_PRIV_VERIFY_CACHE =101, /* cache verification result (obsolete)*/

    SIGSUBPKT_FLAG_CRITICAL=128
} sigsubpkttype_t;

/*-- iobuf.c --*/
void        gpg_iobuf_enable_special_filenames ( int yes );
gpg_iobuf_t gpg_iobuf_alloc(int use, size_t bufsize);
gpg_iobuf_t gpg_iobuf_temp(void);
gpg_iobuf_t gpg_iobuf_temp_with_content( const char *buffer, size_t length );
gpg_iobuf_t gpg_iobuf_open( const char *fname );
gpg_iobuf_t gpg_iobuf_fdopen( int fd, const char *mode );
gpg_iobuf_t gpg_iobuf_sockopen( int fd, const char *mode );
gpg_iobuf_t gpg_iobuf_create( const char *fname );
gpg_iobuf_t gpg_iobuf_append( const char *fname );
gpg_iobuf_t gpg_iobuf_openrw( const char *fname );
int         gpg_iobuf_ioctl ( gpg_iobuf_t a, int cmd, int intval, void *ptrval );
int         gpg_iobuf_close( gpg_iobuf_t iobuf );
int         gpg_iobuf_cancel( gpg_iobuf_t iobuf );

int gpg_iobuf_push_filter( gpg_iobuf_t a, int (*f)(void *opaque, int control,
		       gpg_iobuf_t chain, byte *buf, size_t *len), void *ov );
int gpg_iobuf_push_filter2( gpg_iobuf_t a,
		    int (*f)(void *opaque, int control,
		    gpg_iobuf_t chain, byte *buf, size_t *len),
		    void *ov, int rel_ov );
int gpg_iobuf_flush(gpg_iobuf_t a);
void gpg_iobuf_clear_eof(gpg_iobuf_t a);
#define gpg_iobuf_set_error(a)    do { (a)->error = 1; } while(0)
#define gpg_iobuf_error(a)	      ((a)->error)

void gpg_iobuf_set_limit( gpg_iobuf_t a, _off_t nlimit );

_off_t gpg_iobuf_tell( gpg_iobuf_t a );
int    gpg_iobuf_seek( gpg_iobuf_t a, _off_t newpos );

int  gpg_iobuf_readbyte(gpg_iobuf_t a);
int  gpg_iobuf_read(gpg_iobuf_t a, byte *buf, unsigned buflen );
unsigned gpg_iobuf_read_line( gpg_iobuf_t a, byte **addr_of_buffer,
			  unsigned *length_of_buffer, unsigned *max_length );
int  gpg_iobuf_peek(gpg_iobuf_t a, byte *buf, unsigned buflen );
int  gpg_iobuf_writebyte(gpg_iobuf_t a, unsigned c);
int  gpg_iobuf_write(gpg_iobuf_t a, byte *buf, unsigned buflen );
int  gpg_iobuf_writestr(gpg_iobuf_t a, const char *buf );

void gpg_iobuf_flush_temp( gpg_iobuf_t temp );
int  gpg_iobuf_write_temp( gpg_iobuf_t a, gpg_iobuf_t temp );
size_t gpg_iobuf_temp_to_buffer( gpg_iobuf_t a, byte *buffer, size_t buflen );
void gpg_iobuf_unget_and_close_temp( gpg_iobuf_t a, gpg_iobuf_t temp );

off_t gpg_iobuf_get_filelength( gpg_iobuf_t a );
#define IOBUF_FILELENGTH_LIMIT 0xffffffff
const char *gpg_iobuf_get_real_fname( gpg_iobuf_t a );
const char *gpg_iobuf_get_fname( gpg_iobuf_t a );

void gpg_iobuf_set_block_mode( gpg_iobuf_t a, size_t n );
void gpg_iobuf_set_partial_block_mode( gpg_iobuf_t a, size_t len );
int  gpg_iobuf_in_block_mode( gpg_iobuf_t a );

int gpg_iobuf_translate_file_handle ( int fd, int for_write );

/* get a byte form the iobuf; must check for eof prior to this function
 * this function returns values in the range 0 .. 255 or -1 to indicate EOF
 * iobuf_get_noeof() does not return -1 to indicate EOF, but masks the
 * returned value to be in the range 0 ..255.
 */
#define gpg_iobuf_get(a)  \
     (	((a)->nofast || (a)->d.start >= (a)->d.len )?  \
	gpg_iobuf_readbyte((a)) : ( (a)->nbytes++, (a)->d.buf[(a)->d.start++] ) )
#define gpg_iobuf_get_noeof(a)    (gpg_iobuf_get((a))&0xff)

/* write a byte to the iobuf and return true on write error
 * This macro does only write the low order byte
 */
#define gpg_iobuf_put(a,c)	gpg_iobuf_writebyte(a,c)

#define gpg_iobuf_where(a)	"[don't know]"
#define gpg_iobuf_id(a)	((a)->no)

#define gpg_iobuf_get_temp_buffer(a) ( (a)->d.buf )
#define gpg_iobuf_get_temp_length(a) ( (a)->d.len )
#define gpg_iobuf_is_temp(a)	 ( (a)->use == 3 )

/*-- parse-packet.c --*/
int gpg_parse_packet( gpg_iobuf_t inp, PACKET *ret_pkt );
int gpg_read_keyblock( gpg_iobuf_t a, PACKET **pending_pkt, 
		       gpg_kbnode_t *ret_root );

const byte * gpg_parse_sig_subpkt( const subpktarea_t * buffer,
                                   sigsubpkttype_t reqtype,
                                   size_t *ret_n );
const byte * gpg_parse_sig_subpkt2( PKT_signature * sig,
                                    sigsubpkttype_t reqtype,
                                    size_t *ret_n );
void gpg_free_packet( PACKET *pkt );

int pubkey_get_npkey( int algo );
int pubkey_get_nskey( int algo );
int pubkey_get_nsig( int algo );
int pubkey_get_nenc( int algo );


u32 gpg_keyid_from_sk( PKT_secret_key * sk, byte *fprint );
u32 gpg_keyid_from_pk( PKT_public_key * pk, byte *fprint );

/*-- armor.c --*/
int gpg_use_armor_filter( gpg_iobuf_t a );
int gpg_armor_filter( void *opaque, int control, gpg_iobuf_t a, byte *buf,
		      size_t *ret_len);

/*-- rand.c --*/
void gpg_randomize( unsigned char *buffer, size_t length, int level );
char gpg_random_char( int level );
int  gpg_quick_random_gen( int onoff );
char gpg_random_char( int level );

/*-- kbnode.c --*/
gpg_kbnode_t gpg_new_kbnode( PACKET *pkt );
void         gpg_release_kbnode( gpg_kbnode_t n );
void         gpg_add_kbnode( gpg_kbnode_t root, gpg_kbnode_t node );
gpg_kbnode_t gpg_walk_kbnode( gpg_kbnode_t root, gpg_kbnode_t *context, int all );

#ifdef __cplusplus
}
#endif


#endif /*GPGLIB_OPENPGP_H*/