//------------------------------ $Keywords ----------------------------------
// GPGee - GNU Privacy Guard Explorer Extension
// GPGeeUtility.cpp - Miscelaneous utility functions
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of GPGee.
//
// GPGee is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// GPGee is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 1 $ $Revision: 12 $
/*
$History: **** V 1.0 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-05-13 11:14:16 PM - 10303 Bytes
$History: * gpgeeutility.h - 2005-05-12 9:05:36 AM - 5764 Bytes
$History: * Initial check-in
$History: **** V 1.1 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-05-17 2:16:33 AM - 15129 Bytes
$History: * gpgeeutility.h - 2005-05-17 1:16:26 AM - 5979 Bytes
$History: * Add persistent key cache
$History: **** V 1.2 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-08-08 6:48:08 AM - 15307 Bytes
$History: * gpgeeutility.h - 2005-08-08 6:37:04 AM - 5915 Bytes
$History: * License change - remove option for later versions of GPL
$History: **** V 1.3 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-08-31 6:58:42 PM - 18161 Bytes
$History: * gpgeeutility.h - 2005-08-31 5:03:10 PM - 5989 Bytes
$History: * Add version check function CheckVersion()
$History: **** V 1.4 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-05 3:19:23 AM - 20097 Bytes
$History: * gpgeeutility.h - 2005-09-05 2:41:26 AM - 6060 Bytes
$History: * Add PassphraseCache cleanup function now that the 
$History: * passphrase cache is global.
$History: **** V 1.5 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 3:05:29 AM - 22076 Bytes
$History: * gpgeeutility.h - 2005-09-05 2:41:26 AM - 6060 Bytes
$History: * Add logging to recently added functions.  Also fixed 
$History: * "Could not convert variant of type (string) into type (Double)" error.
$History: **** V 1.6 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 6:11:21 AM - 24119 Bytes
$History: * gpgeeutility.h - 2005-09-06 5:51:28 AM - 6099 Bytes
$History: * Make workaround for unknown access violation bug in 
$History: * version checking on some platforms.  Shooting blind 
$History: * here but this should do it.  The function is now as 
$History: * robust as it can be made.
$History: **** V 1.7 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 2:30:22 PM - 24443 Bytes
$History: * gpgeeutility.h - 2005-09-06 5:51:28 AM - 6099 Bytes
$History: * Fix bug where new version checks aren't done only once a day.
$History: **** V 1.8 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 4:49:04 PM - 24665 Bytes
$History: * gpgeeutility.h - 2005-09-06 4:48:58 PM - 6195 Bytes
$History: * Add messages for signature appending
$History: **** V 1.9 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 5:46:57 PM - 27076 Bytes
$History: * gpgeeutility.h - 2005-09-06 5:10:16 PM - 6315 Bytes
$History: * Add GetTempFileName()
$History: **** V 1.10 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 5:49:11 PM - 27295 Bytes
$History: * gpgeeutility.h - 2005-09-06 5:48:56 PM - 6363 Bytes
$History: * Add append error message string
$History: **** V 1.11 by kfitzner ****
$History: * gpgeeutility.cpp - 2005-09-06 6:56:40 PM - 27948 Bytes
$History: * gpgeeutility.h - 2005-09-06 5:48:56 PM - 6363 Bytes
$History: * Have the version check system detect based on build 
$History: * numbers rather than a simple string check.  No notifications 
$History: * for people to upgrade to an earlier version any more.
$History: **** Latest ** V 1.12 by kfitzner ** 2005-09-06 7:32:58 PM ****
$History: * Update upgrade function to change version checking url
*/
//----------------------------  $NoKeywords ---------------------------------


//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 21 March 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// This unit is simply a respository for utility functions that are used by
// more than one form or unit.
//---------------------------------------------------------------------------
// 6 May 2006 - Kurt Fitzner (kfitzner@excelcia.org>
//
// I've used this unit as the location for the language file message loading
// and formatting functions GetMessage() and FormMessage().  The former
// loads a string from the current language DLL, the latter loads and formats
// the string through vprintf.
//---------------------------------------------------------------------------
#include <vcl.h>
#include <FileCtrl.hpp>
#include <DateUtils.hpp>
#include <IdBaseComponent.hpp>
#include <IdComponent.hpp>
#include <IdHTTP.hpp>
#include <IdTCPClient.hpp>
#include <IdTCPConnection.hpp>
#pragma hdrstop

#include "gpgme/gpgme.h"
#include "GPGeeExceptions.h"
#include "TProgramLog.h"
#include "TConfiguration.h"
#include "TProgressForm.h"
#include "GPGeeNewVersionNotify.h"
#include "GPGeeUtility.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
//---------------------------------------------------------------------------
extern TConfiguration *GPGeeConfig;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Take a filename and reduce it so that it fits inside the space given on
// an edit box.
//
AnsiString __fastcall MinimizeName(AnsiString FileName, TEdit *Edit)
{
  TCanvas *FileNameCanvas;

  __ENTERFUNCTION__;
  LOG1(LOG_DEBUG, "Minimizing filename \"%s\"", FileName.c_str());
  FileNameCanvas = new TCanvas;
  FileNameCanvas->Handle = GetDC(GetActiveWindow());
  FileNameCanvas->Font->Assign(Edit->Font);
  Edit->Text = Filectrl::MinimizeName(FileName, FileNameCanvas, Edit->Width-4);
  ReleaseDC(GetActiveWindow(),FileNameCanvas->Handle);
  __RETURNFUNCTION(Edit->Text);
}  // AnsiString __fastcall MinimizeName(AnsiString FileName, TControl *Control)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Take a filename and reduce it so that it fits inside the space given on
// an edit box.
//
AnsiString __fastcall MinimizeName(AnsiString FileName, TFont *Font, int Width)
{
  AnsiString retval;
  TCanvas *FileNameCanvas;

  __ENTERFUNCTION__;
  LOG1(LOG_DEBUG, "Minimizing filename \"%s\"", FileName.c_str());
  FileNameCanvas = new TCanvas;
  FileNameCanvas->Handle = GetDC(GetActiveWindow());
  FileNameCanvas->Font->Assign(Font);
  retval = Filectrl::MinimizeName(FileName, FileNameCanvas, Width);
  ReleaseDC(GetActiveWindow(),FileNameCanvas->Handle);
  __RETURNFUNCTION(retval);
}  // AnsiString __fastcall MinimizeName(AnsiString FileName, TControl *Control)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Identify a GnuPG file.  This is done in one of two ways.  If the file is
// ascii armored, it will have a .asc extension.  These files have text
// headers which identify their type.  If the file has a .sig extension,
// then it is a detached signature.  If it has a .gpg extension, then it
// could be almost anything.  These are treated simply as "messages" which
// a decrypt operation will be attempted on.
//
gpg_file_type __fastcall IdentifyFile(AnsiString FileName)
{
  AnsiString Extension;
  gpg_file_type retval = GPG_UNKNOWN;

  __ENTERFUNCTION__;
  LOG1(LOG_DEBUG, "Identifying file \"%s\"", FileName.c_str());
  Extension = ExtractFileExt(FileName);
  if (Extension.AnsiCompareIC(".gpg") == 0)
    retval = GPG_MESSAGE;
  else if (Extension.AnsiCompareIC(".sig") == 0)
    retval = GPG_SIGNATURE;
  else if (Extension.AnsiCompareIC(".asc") == 0) {
    gpgme_pgptype_t pgptype;
    if (gpgme_file_get_pgptype(FileName.c_str(), &pgptype) == GPGME_No_Error)
      pgptype &= (GPGME_TYPE_MESSAGE | GPGME_TYPE_SIG | GPGME_TYPE_CLEARSIG);
      switch (pgptype) {
        case GPGME_TYPE_MESSAGE:
          retval = GPG_MESSAGE;
          break;
        case GPGME_TYPE_SIG:
          retval = GPG_SIGNATURE;
          break;
        case GPGME_TYPE_CLEARSIG:
        case GPGME_TYPE_CLEARSIG | GPGME_TYPE_SIG:
          retval = GPG_CLEARSIGNED;
          break;
        default:
          LOG2(LOG_WARNING, "Could not identify .asc file \"%s\": pgptype=0x%X.", FileName.c_str(), pgptype);
          retval = GPG_UNKNOWN;
      }  // switch (pgptype)
  }  // else if (Extension.AnsiCompareIC(".asc") == 0)
  __RETURNFUNCTION(retval);
}  // int __fastcall IdentifyFile(AnsiString FileName)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Convert GnuPG dates to TDateTime
//
TDateTime __fastcall GPGToDateTime(unsigned long GPGDate, bool LocalTime) {
  TIME_ZONE_INFORMATION tzInfo;
  DWORD tzResult;
  static double tzBias = -1.0;
  TDateTime retval;

  __ENTERFUNCTION__;

  // GnuPG returns timestamps in GMT - we convert that to local time here, so we need to get the time zone info
  if (tzBias == -1.0) {
    tzResult = GetTimeZoneInformation(&tzInfo);
    if (tzResult == TIME_ZONE_ID_STANDARD)
      tzBias = ((double)tzInfo.Bias + (double)tzInfo.StandardBias) / 1440.0;
    else if (tzResult == TIME_ZONE_ID_DAYLIGHT)
      tzBias = ((double)tzInfo.Bias + (double)tzInfo.DaylightBias) / 1440.0;
    else
      tzBias = 0.0;
  }  // if (tzBias = -1.0)
  retval = (double)GPGDate / (24.0 * 60.0 * 60.0) + UnixDateDelta - (LocalTime?tzBias:0.0);
  LOG2(LOG_DEBUG, "GPG date %d converted to %s.", GPGDate, retval.DateTimeString().c_str());

  __RETURNFUNCTION(retval);
}  // TDateTime __fastcall GPGToDateTime(unsigned long GPGDate)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Return a short string description for a GnuPG public key id number
//
AnsiString __fastcall KeyTypeToString(gpgme_pk_cipher_t KeyType)
{
  AnsiString retval;

  __ENTERFUNCTION__;

  switch (KeyType) {
    case GPGME_PK_RSA:
      retval = "RSA"; break;
    case GPGME_PK_RSA_E:
      retval = "RSAe"; break;
    case GPGME_PK_RSA_S:
      retval = "RSAs"; break;
    case GPGME_PK_ELG_E:
      retval = "ELG"; break;
    case GPGME_PK_DSA:
      retval = "DSA"; break;
    case GPGME_PK_ELG_ES:
      retval = "ELGse"; break;
  }  // switch (KeyType)

  __RETURNFUNCTION(retval);
}  // AnsiString __fastcall KeyTypeToString(gpgme_pk_cipher_t KeyType)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
AnsiString __fastcall TrustToString(gpgme_validity_t Trust)
{
  AnsiString retval;
  __ENTERFUNCTION__;
  retval = GetMessage(MSG_GPGME_TRUST_UNKNOWN + Trust);
  __RETURNFUNCTION(retval);
}  // AnsiString __fastcall TrustToString(GpgmeOwnertrust Trust)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
AnsiString __fastcall DigestToString(gpgme_md_t Digest)
{
  AnsiString retval;
  __ENTERFUNCTION__;
  retval = GetMessage(MSG_GPGME_MD_UNKNOWN + Digest);
  __RETURNFUNCTION(retval);
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Return a string from the language resource module
//
AnsiString __fastcall GetMessage(int ident)
{
  static HINSTANCE ResourceHInstance = 0;
  AnsiString Message;

  __ENTERFUNCTION__;

  LOG1(LOG_DEBUG, "Retrieving message #%d from the language string table.", ident);
  if (!ResourceHInstance) {
    ResourceHInstance = (HINSTANCE)FindResourceHInstance((unsigned int)HInstance);
    if (!ResourceHInstance)
      ResourceHInstance = HInstance;
  }  // if (!ResourceHInstance)
  Message.LoadString(ResourceHInstance, ident);

  __RETURNFUNCTION(Message);
}  // AnsiString GetMessage(int ident)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Return a formatted message composed of a language resource string that
// has been run through vprintf()
//
AnsiString FormMessage(int ident, ...)
{
  va_list valist;
  AnsiString Message;

  __ENTERFUNCTION__;
  va_start(valist, ident);
  Message.vprintf(GetMessage(ident).c_str(),valist);
  va_end(valist);
  __RETURNFUNCTION(Message);
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void ProgressCallback(void *opaque, const char *what, int type, unsigned current, unsigned total)
{
  __ENTERFUNCTION__;
  if (!total)
    return;
  if (!ProgressForm)
    ProgressForm = new TProgressForm(NULL, "GPGee is caching your " + String((char *)opaque) + " keyring...");
  ProgressForm->SetPosition(current, total);
  ProgressForm->Show();
  Application->ProcessMessages();
  __RETURNFUNCTION__;
}  // void ProgressCallback(void *opaque, const char *what, int type, unsigned current, unsigned total)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Return a GPGME key cache.  This function will keep a persistent cache
// if GPGee is configured to do so, otherwise it will return a newly
// initialized cache each time it is called.
//
gpgme_keycache_t __fastcall GetKeyCache(bool CacheSecretKeys, bool Cleanup) {
  static gpgme_keycache_t cachePublic = NULL;
  static gpgme_keycache_t cacheSecret = NULL;
  static int PublicCacheStamp = -1;
  static int SecretCacheStamp = -1;
  gpgme_keycache_t *pCache;
  int *pStamp;
  int CurrentStamp;
  AnsiString CacheFile;
  gpgme_error_t err;
  bool PersistentCache;
  AnsiString CountConfig;

  __ENTERFUNCTION__;

  // Cleanup the old caches if they exist and we are called to do so.
  if (Cleanup) {
    if (cachePublic) {
      gpgme_keycache_release(cachePublic);
      cachePublic = NULL;
    }  // if (cachePublic);
    if (cacheSecret) {
      gpgme_keycache_release(cacheSecret);
      cacheSecret = NULL;
    }  // if (cacheSecret);
    __RETURNFUNCTION(NULL);
  }  // if (Cleanup)

  try {
    PersistentCache = GPGeeConfig->Values["Cache Keys"];
    CountConfig = CacheSecretKeys?"Secring Count":"Pubring Count";

    // Configure our cache and timestamp pointers and get the keyring filename.
    if (CacheSecretKeys) {
      pCache = &cacheSecret;
      pStamp = &SecretCacheStamp;
      CacheFile = GPGeeConfig->Values["Secret Keyring"];
    } else {
      pCache = &cachePublic;
      pStamp = &PublicCacheStamp;
      CacheFile = GPGeeConfig->Values["Keyring"];
    }  // if (CacheSecretKeys);

    if (PersistentCache && !CacheFile.IsEmpty()) {
      CurrentStamp = FileAge(CacheFile);
      if (CurrentStamp == *pStamp && *pCache) {
        LOG1(LOG_DEBUG, "Keyring timestamp matches and existing cache exists - reusing %s key cache.", CacheSecretKeys?"secret":"public");
        __RETURNFUNCTION(*pCache);
      } else {
        // If we get here it means we are configured to use a persistent cache but either the cache
        // hasn't been initialized yet or the timestamps don't match so we store the current timestamp.
        if (CurrentStamp)
          *pStamp = CurrentStamp;
      }  // if (CurrentStamp == *pStamp && *pCache)
    }  // if (GPGeeConfig->Values["Cache Keys"])

    if (*pCache) {
      gpgme_keycache_release(*pCache);
      *pCache = NULL;
    }  // if (*pCache)
    err = gpgme_keycache_new(pCache);
    if (err)
      throw EGPGMEError(FormMessage(CacheSecretKeys?MSG_ERROR_GPGME_KEYCACHE_INIT_PUB:MSG_ERROR_GPGME_KEYCACHE_INIT_PUB,gpgme_strerror(err)));
    gpgme_keycache_set_cb(*pCache, ProgressCallback, CacheSecretKeys?"secret":"public", GPGeeConfig->Values[CountConfig]);
    err = gpgme_keycache_init(*pCache, NULL, CacheSecretKeys);
    GPGeeConfig->Values[CountConfig] = gpgme_keycache_count(*pCache);
    gpgme_keycache_set_cb(*pCache, NULL, NULL, 0);
    if (ProgressForm) {
      ProgressForm->Close();
      ProgressForm->Release();
      ProgressForm = NULL;
    }  // if (ProgressForm)
    if (err)
      throw EGPGMEError(FormMessage(CacheSecretKeys?MSG_ERROR_GPGME_KEYCACHE_LOAD_SEC:MSG_ERROR_GPGME_KEYCACHE_LOAD_PUB,gpgme_strerror(err)));    
    __RETURNFUNCTION(*pCache);
  }  // try

  catch (...) {
    if (cachePublic) {
      gpgme_keycache_release(cachePublic);
      cachePublic = NULL;
    }  // if (cachePublic);
    if (cacheSecret) {
      gpgme_keycache_release(cacheSecret);
      cacheSecret = NULL;
    }  // if (cacheSecret);
    __RETURNFUNCTION(NULL);
  }  // catch (...)

}  // gpgme_keycache_t __fastcall GetKeyCache(bool GetSecretKeys)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Format the last Windows error message as an AnsiString
//
AnsiString GetLastErrorMessage(void) {
  char *message;
  AnsiString retval;

  __ENTERFUNCTION__;
  FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, NULL, GetLastError(),
                MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (char *) &message, 0, NULL);
  retval = String(message);
  LocalFree(message);
  __RETURNFUNCTION(retval);
}  // AnsiString GetLastErrorMessage(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Check to see if GPGee is up-to-date - notify the user if it isn't
//
void CheckVersion(void) {

  TDateTime *dtLastCheck;
  char *filename = NULL;
  char *buffer = NULL;
  char *verbuff = NULL;
  unsigned nVersionInfoSize;
  int nBuildPos;
  static AnsiString sMyVersion = "";
  static int nMyBuild = 0;
  AnsiString sCurrentVersion, sURL;
  int nCurrentBuild = 0;
  TIdHTTP *HTTP;
  DWORD nDummy;

  __ENTERFUNCTION__;

  try {
    try {
      formGPGeeNewVersionNotify = NULL;
      if (!GPGeeConfig->Values["Update Notify"]) {
        LOG(LOG_DEBUG, "Update Notify is turned off - not checking for new version.");
        __RETURNFUNCTION__;
      }  // if (!GPGeeConfig->Values["Update Notify"])

      if (String(GPGeeConfig->Values["Last Version Check"]).IsEmpty())
        GPGeeConfig->Values["Last Version Check"] = (Now()-1).DateTimeString();
      dtLastCheck = new TDateTime(String(GPGeeConfig->Values["Last Version Check"]));
      if (!DaysBetween(Now(), *dtLastCheck)) {
        LOG1(LOG_DEBUG, "Last new version check was performed %s - not checking again.", dtLastCheck->DateTimeString().c_str());
        // __RETURNFUNCTION__;
      }  // if (!DaysBetween(Now(), *dtLastCheck))

      GPGeeConfig->Values["Last Version Check"] = Now().DateTimeString();

      // Get this DLL's version - this is a pain, so let's just do it once for each time GPGee is instantiated
      if (!nMyBuild) {
        LOG(LOG_DEBUG, "Retrieving internal version number.");
        filename = (char *)malloc(MAXPATH);
        GetModuleFileName(HInstance, filename, MAXPATH);
        nVersionInfoSize = GetFileVersionInfoSize(filename, &nDummy);
        // These following three exceptions should never make it to the user except through the logs in a debug build.
        // Since they are only for logging, they are allowed to have text that doesn't come from the string table.
        if (!nVersionInfoSize)
          throw EGPGeeSystemError("GetFileVersionInfoSize() failed: " + GetLastErrorMessage());
        buffer = (char *)malloc(nVersionInfoSize);
        if (!GetFileVersionInfo(filename, NULL, nVersionInfoSize, buffer))
          throw EGPGeeSystemError("GetFileVersionInfo() failed:" + GetLastErrorMessage());
        if (!VerQueryValue(buffer, "\\StringFileInfo\\100904E4\\FileVersion", (void **)&verbuff, NULL))
          throw EGPGeeSystemError("VerQueryValue() failed:" + GetLastErrorMessage());
        sMyVersion = verbuff;
        nBuildPos = sMyVersion.LastDelimiter(".");
        nMyBuild = sMyVersion.SubString(nBuildPos + 1, sMyVersion.Length() - nBuildPos).ToInt();
        sMyVersion = sMyVersion.SubString(1,nBuildPos-1) + " (Build " + String(nMyBuild) + ")";
        LOG1(LOG_MESSAGE, "Internal version string retrieved from resources: \"%s\".", sMyVersion.c_str());
      }  // if (sMyVersion.IsEmpty())

      // Get GPGee's current version from the interweb - woo, this is cool
      HTTP = new TIdHTTP(NULL);
      sURL = GPGeeConfig->Values["Version URL"];
      LOG1(LOG_DEBUG, "Attempting to retrieve version string from URL \"%s\".", sURL.c_str());
      try {
        sCurrentVersion = HTTP->Get(GPGeeConfig->Values["Version URL"]);
        nBuildPos = sCurrentVersion.Pos("(Build ");
        nCurrentBuild = sCurrentVersion.SubString(nBuildPos + 7, sCurrentVersion.Length() - nBuildPos - 7).ToInt();
      }
      catch (...) { sCurrentVersion = ""; }
      delete HTTP;
      #ifdef __LOGGINGENABLED__
      if (!sCurrentVersion.IsEmpty())
        LOG2(LOG_MESSAGE, "Version string \"%s\" retrieved from %s.", sCurrentVersion.c_str(), sURL.c_str())
      else
        LOG2(LOG_WARNING, "Unable to retrieve version string from URL \"%s\": %s", sURL.c_str(), HTTP->ResponseText.c_str());
      #endif

      // Compare the two...
      if (nCurrentBuild && nMyBuild < nCurrentBuild && sCurrentVersion != String(GPGeeConfig->Values["Notify Squelch"])) {
        LOG(LOG_DEBUG, "Notifying user of new version.");
        formGPGeeNewVersionNotify = new TformGPGeeNewVersionNotify(NULL, sMyVersion, sCurrentVersion);
        if (formGPGeeNewVersionNotify->ShowModal() == mrCancel)
          GPGeeConfig->Values["Notify Squelch"] = sCurrentVersion;
      }  // if (different versions)
    }  // inner try
    catch (Exception &e) {
      LOG2(LOG_ERROR, "Exception of type \"%s\" raised during new version check with message \"%s\"", String(e.ClassName()).c_str(), e.Message.c_str());
    }  // catch
  }  // outer try
  __finally {
    if (filename)
      free(filename);
    if (buffer)
      free(buffer);
    if (formGPGeeNewVersionNotify) {
      delete formGPGeeNewVersionNotify;
      formGPGeeNewVersionNotify = NULL;
    }  // if (formGPGeeNewVersionNotify)
  }
  __RETURNFUNCTION__;
}  // void CheckVersion(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Cleanup the passphrase cache
//
void CleanupPassphraseCache(TStringList *Cache) {
  __ENTERFUNCTION__;

  LOG(LOG_DEBUG, "Clearing and unlocking all passphrases in the passphrase cache.");
  // Overwrite the passphrases in the cache, unlock and free the memory.
  if (Cache) {
    for (int n=0; n < Cache->Count; n++) {
      void *data = (void *)(Cache->Strings[n].data());
      int Length = Cache->Strings[n].Length();
      memset(data, '*', Length);
      Cache->Strings[n].StringOfChar('*', 128);
      VirtualUnlock(data, Length);
    }  // for (int n=0; n < PassphraseCache->Count; n++)
    Cache->Clear();
  }  // if (PassphraseCache)

  __RETURNFUNCTION__;
}  // void CleanupPassphraseCache(TStringList *Cache)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// From time to time changes to the registry need to be done for when we
// are upgrading from one version to the next.
//
void Upgrade(void) {
  __ENTERFUNCTION__;

  TRegistry *Registry = new TRegistry;
  Registry->RootKey = HKEY_CURRENT_USER;
  Registry->OpenKey("Software\\GPGee", false);

  // In versions 1.1.3 and earlier, GPGee could only sign with one key at a time.  The last signing key used was
  // stored in "HKCU\Software\GPGee\Signing Key" as a DWORD.  In version 1.2.0 and later, this becomes the string
  // value "Signing Keys" which is a space-separated list of all keys to sign.
  if (Registry->ValueExists("Signing Key")) {
    LOG(LOG_DEBUG, "Old \"Signing Key\" registry value found - converting to \"Signing Keys\".");
    int nSigningKey = Registry->ReadInteger("Signing Key");
    if (nSigningKey) {
      AnsiString sSigningKey = "0x" + IntToHex(nSigningKey,8);
      GPGeeConfig->Values["Signing Keys"] = sSigningKey;
    }  // if (nSigningKey)
    Registry->DeleteValue("Signing Key");
  }  // if (Registry->ValueExists("Signing Key"))

  // In version 1.2.0, the URL for the version checking system pointed to a web server that cached the files.  This
  // resulted in lots of people receiving upgrade notices to upgrade to an old version long after the GPGeeCurrentVersion.txt
  // file had been updated.  This URL should now point to a different server.
  if (GPGeeConfig->Values["Version URL"] == String("http://members.shaw.ca/GPGee/GPGeeCurrentVersion.txt"))
    GPGeeConfig->Values["Version URL"] = "http://gpgee.excelcia.org/GPGeeCurrentVersion.txt";

  Registry->CloseKey();
  delete Registry;

  __RETURNFUNCTION__;
}  // void Upgrade(void);
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Wrapper for system GetTempFileName() call.  Why are windows functions
// always so klunky to use?
//
AnsiString GetTempFileName(AnsiString Prefix) {
  char *temppath;
  char *tempname;
  AnsiString retval;

  __ENTERFUNCTION__;
  temppath = (char *)malloc(MAXPATH);
  tempname = (char *)malloc(MAXPATH);
  GetTempPath(MAXPATH, temppath);
  if (GetTempFileName(temppath, Prefix.c_str(), 0, tempname))
    retval = String(tempname);
  else
    retval = "";
  free(tempname);
  free(temppath);
  __RETURNFUNCTION(retval);
}  // AnsiString GetTempFileName(AnsiString Prefix)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Append one file to the end of another.
//
bool AppendFile(AnsiString AppendFilename, AnsiString TargetFilename) {
  TFileStream *AppendStream = NULL, *TargetStream = NULL;
  __int64 nnCopyCount;
  bool retval;

  __ENTERFUNCTION__;
  LOG(LOG_DEBUG, "Appending file \"" + AppendFilename + "\" to \"" + TargetFilename + "\".");
  try {
    try {
      AppendStream = new TFileStream(AppendFilename, fmOpenRead);
      TargetStream = new TFileStream(TargetFilename, fmOpenReadWrite);
      TargetStream->Seek(0, soFromEnd);
      nnCopyCount = TargetStream->CopyFrom(AppendStream, 0);
      if (nnCopyCount == AppendStream->Size)
        retval = true;
      else
        retval = false;
    }  // inner try
    catch (...) {
      retval = false;
    }  // catch
  }  // outer try
  __finally {
    if (AppendStream)
      delete AppendStream;
    if (TargetStream)
      delete TargetStream;
    __RETURNFUNCTION(retval);
  }  // __finally
}  // bool AppendFile(AnsiString AppendFilename, AnsiString TargetFilename)
//---------------------------------------------------------------------------

