/* Copyright (C) 1995-2003 Ghostgum Software Pty Ltd.  All rights reserved.

  This software is provided AS-IS with no warranty, either express or
  implied.

  This software is distributed under licence and may not be copied,
  modified or distributed except as expressly authorised under the terms
  of the licence contained in the file LICENCE in this distribution.

  For more information about licensing, please refer to
  http://www.ghostgum.com.au/ or contact Ghostsgum Software Pty Ltd, 
  218 Gallaghers Rd, Glen Waverley VIC 3150, AUSTRALIA, 
  Fax +61 3 9886 6616.
*/

/* $Id: epstool.c,v 1.33 2003/05/06 13:17:40 ghostgum Exp $ */

#include "common.h"
#include "dscparse.h"
#include "errors.h"
#include "iapi.h"
#include "gdevdsp.h"
#define DEFINE_COPT
#include "copt.h"
#define DEFINE_CAPP
#include "capp.h"
#include "cbmp.h"
#define DEFINE_CDOC
#include "cdoc.h"
#include "cdll.h"
#include "ceps.h"
#include "cimg.h"
#include "cres.h"
#ifdef __WIN32__
#include "wgsver.h"
#endif
#ifdef UNIX
#include <fcntl.h>
#include <sys/wait.h>
#include <errno.h>
#endif

const char *epstool_name = "epstool";
const char *epstool_version = "3.02";	 /* should be EPSTOOL_VERSION */
const char *epstool_date = "2003-05-06"; /* should be EPSTOOL_DATE */
const char *copyright = "Copyright 1995-2003 Ghostgum Software Pty Ltd";

const char *cmd_help = "\
Commands (one only):\n\
  --add-tiff4-preview        or  -t4\n\
  --add-tiff6u-preview       or  -t6u\n\
  --add-tiff6p-preview       or  -t6p\n\
  --add-tiff-preview         or  -tg\n\
  --add-interchange-preview  or  -i\n\
  --add-metafile-preview     or  -w\n\
  --add-user-preview filename\n\
  --dcs2-multi\n\
  --dcs2-single\n\
  --dcs2-report\n\
  --extract-postscript       or  -p\n\
  --extract-preview          or  -v\n\
  --bitmap\n\
  --copy\n\
  --help                     or  -h\n\
";

const char *opt_help = "\
Options:\n\
  --bbox                     or  -b\n\
  --custom-colours filename\n\
  --debug                    or  -d\n\
  --device name\n\
  --doseps-reverse\n\
  --dpi resolution\n\
  --dpi-render resolution\n\
  --ignore-warnings\n\
  --ignore-errors\n\
  --gs command\n\
  --gs-args arguments\n\
  --missing-separations\n\
  --output filename\n\
  --quiet\n\
  --rename-separation old_name new_name\n\
  --replace-composite\n\
";


typedef enum {
    CMD_UNKNOWN,
    CMD_TIFF4,
    CMD_TIFF6U,
    CMD_TIFF6P,
    CMD_TIFF,
    CMD_INTERCHANGE,
    CMD_WMF,
    CMD_USER,
    CMD_DCS2_MULTI,
    CMD_DCS2_SINGLE,
    CMD_DCS2_REPORT,
    CMD_PREVIEW,
    CMD_POSTSCRIPT,
    CMD_BITMAP,
    CMD_COPY,
    CMD_HELP
} CMD;

typedef enum{
    CUSTOM_CMYK,
    CUSTOM_RGB
} CUSTOM_COLOUR_TYPE;
typedef struct CUSTOM_COLOUR_s CUSTOM_COLOUR;
struct CUSTOM_COLOUR_s {
    char name[MAXSTR];
    CUSTOM_COLOUR_TYPE type;
    /* if type=CUSTOM_CMYK */
    float cyan;
    float magenta;
    float yellow;
    float black;
    /* if type=CUSTOM_RGB */
    float red;
    float green;
    float blue;
    /* Next colour */
    CUSTOM_COLOUR *next;
};
typedef struct OPT_s {
    CMD cmd;
    TCHAR device[64];	/* --device name for --bitmap or --add-tiff-preview */
    BOOL composite;		/* --replace-composite */
    BOOL bbox;			/* --bbox */
    int dscwarn;		/* --ignore-warnings or --ignore-errors */
    TCHAR gs[MAXSTR];		/* --gs command */
    TCHAR gsargs[MAXSTR];	/* --gs-args arguments */
    TCHAR input[MAXSTR];	/* filename */
    TCHAR output[MAXSTR];	/* --output filename or (second) filename */
    TCHAR user_preview[MAXSTR];	/* --add-user-preview filename */
    BOOL quiet;			/* --quiet */
    BOOL debug;			/* --debug */
    BOOL doseps_reverse;	/* --doseps-reverse */
    float dpi;			/* --dpi resolution */
    float dpi_render;		/* --dpi-render resolution */
    BOOL help;			/* --help */
    TCHAR custom_colours[MAXSTR]; /* --custom-colours filename */
    CUSTOM_COLOUR *colours;
    BOOL missing_separations;	/* --missing-separations */
    RENAME_SEPARATION *rename_sep; /* --rename-separation */
} OPT;


#define MSGOUT stdout
#ifdef UNIX
const char gsexe[] = "gs";
#define COLOUR_DEVICE "ppmraw"
#define GREY_DEVICE "pgmraw"
#define MONO_DEVICE "pbmraw"
#else
TCHAR gsexe[MAXSTR] = TEXT("gswin32c.exe");
#define COLOUR_DEVICE TEXT("bmp16m")
#define GREY_DEVICE TEXT("bmpgray")
#define MONO_DEVICE TEXT("bmpmono")
#endif


static void print_help(void);
static int parse_args(OPT *opt, int argc, TCHAR *argv[]);
static int epstool_add_preview(Doc *doc, OPT *opt);
static int epstool_dcs2_copy(Doc *doc, OPT *opt);
static int epstool_dcs2_report(Doc *doc);
static int epstool_dcs2_composite(Doc *doc, OPT *opt, GFile *compfile);
static int epstool_dcs2_check_files(Doc *doc, OPT *opt);
static int epstool_extract_doseps(Doc *doc, OPT *opt);
static int epstool_bitmap(Doc *doc, OPT *opt);
static int epstool_copy(Doc *doc, OPT *opt);

static IMAGE *make_preview_image(Doc *doc, OPT *opt, int page, LPCTSTR device,
    CDSCBBOX *bbox, CDSCFBBOX *hires_bbox, int calc_bbox);
static int make_preview_file(Doc *doc, OPT *opt, int page, 
    LPCTSTR preview, LPCTSTR device,
    float dpi, CDSCBBOX *bbox, CDSCFBBOX *hires_bbox, int calc_bbox);
static int calculate_bbox(Doc *doc, OPT *opt, LPCTSTR psname, 
    CDSCBBOX *bbox, CDSCFBBOX *hires_bbox);
static int calc_device_size(float dpi, CDSCBBOX *bbox, CDSCFBBOX *hires_bbox,
    int *width, int *height, float *xoffset, float *yoffset);
static int exec_program(LPTSTR command,
    int hstdin, int hstdout, int hstderr,
    LPCTSTR stdin_name, LPCTSTR stdout_name, LPCTSTR stderr_name);
static int custom_colours_read(OPT *opt);
static CUSTOM_COLOUR *custom_colours_find(OPT *opt, const char *name);
static void custom_colours_free(OPT *opt);
static int colour_to_cmyk(CDSC *dsc, const char *name, 
   float *cyan, float *magenta, float *yellow, float *black);


static void 
print_help(void)
{
    fprintf(MSGOUT, "%s %s %s\n", epstool_name, epstool_version, epstool_date);
    fprintf(MSGOUT, "%s\n", copyright);
    fprintf(MSGOUT, "Usage: epstool command [options] inputfile outputfile\n");
    fprintf(MSGOUT, "%s", cmd_help);
    fprintf(MSGOUT, "%s", opt_help);
}

/* return 0 on success, or argument index if there is a duplicated
 * command, a missing parameter, or an unknown argument.
 */
static int
parse_args(OPT *opt, int argc, TCHAR *argv[])
{
    int arg;
    const TCHAR *p;
    memset(opt, 0, sizeof(OPT));
    opt->cmd = CMD_UNKNOWN;
    opt->dpi = 72.0;
    opt->dpi_render = 0.0;
    csncpy(opt->gs, gsexe, sizeof(opt->gs)/sizeof(TCHAR)-1);
    for (arg=1; arg<argc; arg++) {
	p = argv[arg];
	if ((cscmp(p, TEXT("--add-tiff4-preview")) == 0) ||
	    (cscmp(p, TEXT("-t4")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_TIFF4;
	}
	else if ((cscmp(p, TEXT("--add-tiff6u-preview")) == 0) ||
	    (cscmp(p, TEXT("-t6u")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_TIFF6U;
	}
	else if ((cscmp(p, TEXT("--add-tiff6p-preview")) == 0) ||
	    (cscmp(p, TEXT("-t6p")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_TIFF6P;
	}
	else if ((cscmp(p, TEXT("--add-tiff-preview")) == 0) ||
	    (cscmp(p, TEXT("-tg")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_TIFF;
	}
	else if ((cscmp(p, TEXT("--add-interchange-preview")) == 0) ||
	    (cscmp(p, TEXT("-i")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_INTERCHANGE;
	}
	else if ((cscmp(p, TEXT("--add-metafile-preview")) == 0) ||
	    (cscmp(p, TEXT("-w")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_WMF;
	}
	else if (cscmp(p, TEXT("--add-user-preview")) == 0) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_USER;
	    arg++;
	    if (arg == argc)
		return arg;
	    csncpy(opt->user_preview, argv[arg], 
		sizeof(opt->user_preview)/sizeof(TCHAR)-1);
	}
	else if (cscmp(p, TEXT("--dcs2-multi")) == 0) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_DCS2_MULTI;
	}
	else if (cscmp(p, TEXT("--dcs2-single")) == 0) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_DCS2_SINGLE;
	}
	else if (cscmp(p, TEXT("--dcs2-report")) == 0) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_DCS2_REPORT;
	}
	else if (cscmp(p, TEXT("--replace-composite")) == 0) {
	    opt->composite = TRUE;
	}
	else if (cscmp(p, TEXT("--missing-separations")) == 0) {
	    opt->missing_separations = TRUE;
	}
	else if (cscmp(p, TEXT("--rename-separation")) == 0) {
	    RENAME_SEPARATION *rs = opt->rename_sep;
	    arg++;
	    if (arg+1 == argc)
		return arg;
	    if (rs) {
		while (rs && (rs->next != NULL))
		    rs = rs->next;
		rs->next = 
		    (RENAME_SEPARATION *)malloc(sizeof(RENAME_SEPARATION));
		rs = rs->next;
	    }
	    else {
		opt->rename_sep = rs = 
		    (RENAME_SEPARATION *)malloc(sizeof(RENAME_SEPARATION));
	    }
	    if (rs) {
		char oldname[MAXSTR];
		char newname[MAXSTR];
		char *p;
		char *q;
		int len;
		memset(rs, 0, sizeof(RENAME_SEPARATION));
		rs->next = NULL;
		memset(oldname, 0, sizeof(oldname));
	        cs_to_narrow(oldname, sizeof(oldname)-1, 
		    argv[arg], cslen(argv[arg])+1);
		len = strlen(oldname)+1;
		p = (char *)malloc(len);
		if (p) {
		    memcpy(p, oldname, len);
		    rs->oldname = p;
		}
		arg++;
		memset(newname, 0, sizeof(newname));
	        cs_to_narrow(newname, sizeof(newname)-1, 
		    argv[arg], cslen(argv[arg])+1);
		len = strlen(newname)+1;
		q = (char *)malloc(len);
		if (q) {
		    memcpy(q, newname, len);
		    rs->newname = q;
		}
		if ((p == NULL) || (q == NULL)) {
		    fprintf(stderr, "Out of memory\n");
		    return arg;
		}
	    }
	    else {
		fprintf(stderr, "Out of memory\n");
		return arg;
	    }
	}
	else if (cscmp(p, TEXT("--custom-colours")) == 0) {
	    arg++;
	    if (arg == argc)
		return arg;
	    csncpy(opt->custom_colours, argv[arg], 
		sizeof(opt->custom_colours)/sizeof(TCHAR)-1);
	}
	else if ((cscmp(p, TEXT("--extract-preview")) == 0) ||
	    (cscmp(p, TEXT("-v")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_PREVIEW;
	}
	else if ((cscmp(p, TEXT("--extract-postscript")) == 0) ||
	    (cscmp(p, TEXT("-p")) == 0)) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_POSTSCRIPT;
	}
	else if (cscmp(p, TEXT("--bitmap")) == 0) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_BITMAP;
	}
	else if (cscmp(p, TEXT("--copy")) == 0) {
	    if (opt->cmd != CMD_UNKNOWN)
		return arg;
	    opt->cmd = CMD_COPY;
	}
	else if (cscmp(p, TEXT("--device")) == 0) {
	    arg++;
	    if (arg == argc)
		return arg;
	    csncpy(opt->device, argv[arg], sizeof(opt->device)/sizeof(TCHAR)-1);
	}
	else if ((cscmp(p, TEXT("--bbox")) == 0) ||
	    (cscmp(p, TEXT("-b")) == 0)) {
	    opt->bbox = TRUE;
	}
	else if (cscmp(p, TEXT("--ignore-warnings")) == 0) {
	    opt->dscwarn = CDSC_ERROR_WARN;
	}
	else if (cscmp(p, TEXT("--ignore-errors")) == 0) {
	    opt->dscwarn = CDSC_ERROR_ERROR;
	}
	else if (cscmp(p, TEXT("--gs")) == 0) {
	    arg++;
	    if (arg == argc)
		return arg;
	    csncpy(opt->gs, argv[arg], sizeof(opt->gs)/sizeof(TCHAR)-1);
	}
	else if (cscmp(p, TEXT("--gs-args")) == 0) {
	    arg++;
	    if (arg == argc)
		return arg;
	    csncpy(opt->gsargs, argv[arg], sizeof(opt->gsargs)/sizeof(TCHAR)-1);
	}
	else if (cscmp(p, TEXT("--output")) == 0) {
	    arg++;
	    if (arg == argc)
		return arg;
	    csncpy(opt->output, argv[arg], sizeof(opt->output)/sizeof(TCHAR)-1);
	}
	else if (cscmp(p, TEXT("--quiet")) == 0) {
	    opt->quiet = TRUE;
	}
	else if ((cscmp(p, TEXT("--debug")) == 0) ||
	    (cscmp(p, TEXT("-d")) == 0)) {
	    opt->debug = TRUE;
	}
	else if ((cscmp(p, TEXT("--doseps-reverse")) == 0) ||
	    (cscmp(p, TEXT("-d")) == 0)) {
	    opt->doseps_reverse = TRUE;
	}
	else if (cscmp(p, TEXT("--dpi")) == 0) {
	    char buf[MAXSTR];
	    arg++;
	    if (arg == argc)
		return arg;
	    cs_to_narrow(buf, sizeof(buf)-1, argv[arg], cslen(argv[arg])+1);
	    opt->dpi = (float)atof(buf);
	}	
	else if (cscmp(p, TEXT("--dpi-render")) == 0) {
	    char buf[MAXSTR];
	    arg++;
	    if (arg == argc)
		return arg;
	    cs_to_narrow(buf, sizeof(buf)-1, argv[arg], cslen(argv[arg])+1);
	    opt->dpi_render = (float)atof(buf);
	}	
	else if ((cscmp(p, TEXT("--help")) == 0) || (cscmp(p, TEXT("-h"))==0)) {
	    opt->help = TRUE;
	}
	else if (*p != '-') {
	    if (opt->input[0] == 0)
		csncpy(opt->input, argv[arg], 
		    sizeof(opt->input)/sizeof(TCHAR)-1);
	    else if (opt->output[0] == 0)
		csncpy(opt->output, argv[arg], 
		    sizeof(opt->output)/sizeof(TCHAR)-1);
	    else
		return arg;
	}
	else {
	    return arg;
	}
    }
    return 0;
}


#ifdef UNICODE
int wmain(int argc, TCHAR *argv[])
#else
int main(int argc, char *argv[])
#endif
{
    GSview *app;
    Doc *doc;
    int code = 0;
    int i, arg;
    OPT opt;

#ifdef __WIN32__
    { 	/* Find latest version of Ghostscript */
	char buf[MAXSTR];
	if (find_gs(buf, sizeof(buf)-1, 550, FALSE)) {
	    narrow_to_cs(gsexe, sizeof(gsexe)/sizeof(TCHAR)-1,
		buf, strlen(buf)+1);
	}
    }
#endif

    memset(&opt, 0, sizeof(opt));
    arg = parse_args(&opt, argc, argv);
    debug = 0;
    if (opt.debug) {
	debug = DEBUG_LOG | DEBUG_MEM | DEBUG_GENERAL;
	opt.quiet = 0;
    }
    if (!opt.quiet)
	debug |= DEBUG_LOG;	/* enable output */
    if (opt.help) {
	print_help();
	return 1;
    }
    if (opt.dpi_render < opt.dpi)
	opt.dpi_render = opt.dpi;

    app = app_new(NULL, FALSE);
    if (app == NULL) {
        fprintf(MSGOUT, "Can't create epstool app\n");
	return 1;
    }

    if (arg != 0) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("epstool: Error in command line arguments:\n  "));
	for (i=0; i<arg; i++)
	    app_csmsgf(app, TEXT("%s "), argv[i]);
	app_csmsgf(app, TEXT("\n    The next argument is unrecognised, missing, or conflicts:\n  "));
	for (; i<argc; i++)
	    app_csmsgf(app, TEXT("%s "), argv[i]);
	app_csmsgf(app, TEXT("\n"));
	app_unref(app);
	return 1;
    }
    
    if ((opt.cmd == CMD_TIFF) && (opt.device[0] == '\0')) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("--add-tiff-preview requires a device to be specified with --device\n"));
	code = -1;
    }
    if (opt.cmd == CMD_UNKNOWN) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("No command specified.\n"));
	code = -1;
    }
    if (opt.input[0] == '\0') {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("Input file not specified.\n"));
	code = -1;
    }
    if (opt.custom_colours[0] != '\0') {
	FILE *f = csfopen(opt.custom_colours, TEXT("rb"));
	if (f == (FILE*)NULL) {
	    debug |= DEBUG_LOG;
	    app_csmsgf(app, TEXT("Failed to open \042%s\042.\n"), 
		opt.custom_colours);
	    code = -1;
	}
	else
	    fclose(f);
    }
    if (code == 0) {
	FILE *f = csfopen(opt.input, TEXT("rb"));
	if (f == (FILE*)NULL) {
	    debug |= DEBUG_LOG;
	    app_csmsgf(app, TEXT("Failed to open \042%s\042.\n"), opt.input);
	    code = -1;
	}
	else
	    fclose(f);
    }
    if ((opt.output[0] == '\0') && (opt.cmd != CMD_DCS2_REPORT)) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("Output file not specified.\n"));
	code = -1;
    }
    if (code != 0) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("Run \042epstool --help\042 for more details.\n"));
	app_unref(app);
	return 1;
    }

    /* Create a document */
    doc = doc_new(app);
    if (doc == NULL) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, TEXT("Failed to create new Doc\n"));
	app_unref(app);
	return 1;
    }
    /* Attach it to application */
    doc_add(doc, app);
    doc_dsc_warn(doc, opt.dscwarn);

    if (code == 0) {
	code = doc_open(doc, opt.input);
	if (code < 0) {
	    debug |= DEBUG_LOG;
	    app_csmsgf(app, TEXT("Error opening file \042%s\042.\n"), opt.input);
	    code = -1;
 	}
	else if (code > 0) {
	    debug |= DEBUG_LOG;
	    app_csmsgf(app, TEXT(
	    "Input file \042%s\042 didn't have DSC comments.\n"), opt.input);
	    code = -1;
	}
    }
    if ((code == 0) && (!doc->dsc->epsf) && (opt.cmd != CMD_DCS2_REPORT)) {
	debug |= DEBUG_LOG;
	app_csmsgf(app, 
	    TEXT("Input file \042%s\042 is not EPSF.\n"), opt.input);
	code = -1;
    }

    if ((code == 0) && opt.bbox) {
	switch (opt.cmd) {
	    case CMD_TIFF4:
	    case CMD_TIFF6U:
	    case CMD_TIFF6P:
	    case CMD_INTERCHANGE:
	    case CMD_WMF:
	    case CMD_COPY:
		if (doc->dsc->dcs2) {
		    debug |= DEBUG_LOG;
		    app_csmsgf(app, TEXT("Ignoring --bbox for DCS 2.0.\n"));
		    opt.bbox = 0;
		}
		/* all others are these OK */
		break;
	    default:
		debug |= DEBUG_LOG;
		app_csmsgf(app, TEXT(
		  "Can't use --bbox with this command.  Ignoring --bbox.\n"));
		opt.bbox = 0;
	}
    }

    if (code == 0) {
      switch (opt.cmd) {
	case CMD_TIFF4:
	case CMD_TIFF6U:
	case CMD_TIFF6P:
	case CMD_TIFF:
	case CMD_INTERCHANGE:
	case CMD_WMF:
    	case CMD_USER:
	    code = epstool_add_preview(doc, &opt);
	    break;
	case CMD_DCS2_SINGLE:
	case CMD_DCS2_MULTI:
	    if (doc->dsc->dcs2)
		code = epstool_dcs2_check_files(doc, &opt);
	    if (code == 0)
		code = epstool_dcs2_copy(doc, &opt); 
	    break;
	case CMD_DCS2_REPORT:
	    code = epstool_dcs2_report(doc); 
	    break;
	case CMD_PREVIEW:
	case CMD_POSTSCRIPT:
	    code = epstool_extract_doseps(doc, &opt);
	    break;
	case CMD_BITMAP:
	    code = epstool_bitmap(doc, &opt);
	    break;
	case CMD_COPY:
	    code = epstool_copy(doc, &opt);
	    break;
 	default:
	case CMD_UNKNOWN:
	case CMD_HELP:
	    /* should reach here */
	    code = -1;
	    break;
      }
    }


    doc_close(doc);

    /* detach doc from app */
    doc_remove(doc);

    doc_unref(doc);
    app_unref(app);

    debug_memory_report();

    debug &= ~DEBUG_MEM;
    while (opt.rename_sep) {
	RENAME_SEPARATION *rs = opt.rename_sep;
	if (rs->oldname)
	    free((void *)rs->oldname);
	if (rs->newname)
	    free((void *)rs->newname);
	opt.rename_sep = rs->next;
	free(rs);
    }

    if (!opt.quiet)
        fprintf(MSGOUT, "%s\n", code == 0 ? "OK" : "Failed");
    return code;
}

/****************************************************************/

static int
epstool_add_preview(Doc *doc, OPT *opt)
{
    int code = 0;
    CDSCBBOX devbbox = {0,0,595,842};
    CDSCBBOX bbox = {0, 0, 0, 0};
    CDSCFBBOX hires_bbox = {0.0, 0.0, 0.0, 0.0};
    CDSCFBBOX *phires_bbox = NULL;
    const TCHAR *device = COLOUR_DEVICE;
    FILE *f;
    TCHAR preview[MAXSTR];
    IMAGE *img = NULL;
    if (opt->device[0] != '\0')
	device = opt->device;
    if (opt->cmd == CMD_TIFF4)
	device = MONO_DEVICE;

    if (doc->dsc->bbox) {
	bbox = *doc->dsc->bbox;
    }
    else
	opt->bbox = 1;

    if (doc->dsc->hires_bbox) {
	hires_bbox = *doc->dsc->hires_bbox;
	phires_bbox = &hires_bbox;
    }

    if (opt->cmd == CMD_USER) {
	/* Attempt to load BMP or PPM */
	/* If these fail, assume it is TIFF or WMF */
	img = bmpfile_to_image(opt->user_preview);
	if (img == NULL)
	    img = pnmfile_to_image(opt->user_preview);
    }
    else if (opt->cmd == CMD_TIFF) {
	/* We'll use ghostscript to make the image. */
    }
    else {
	img = make_preview_image(doc, opt, 0, device, 
	    &bbox, &hires_bbox, opt->bbox);
	if (img == NULL) {
	    app_csmsgf(doc->app, TEXT("Couldn't make preview image\n"));
	    return -1;
	}
	if ((hires_bbox.fllx < hires_bbox.furx) &&
	    (hires_bbox.flly < hires_bbox.fury))
	    phires_bbox = &hires_bbox;
	
    }
    if (img) {
	devbbox.llx = devbbox.lly = 0;
	devbbox.urx = img->width;
	devbbox.ury = img->height;
    }

    /* add preview to EPS file */

    switch (opt->cmd) {
	case CMD_TIFF4:
	    code = make_eps_tiff(doc, img, devbbox, &bbox, phires_bbox, 
		opt->dpi, opt->dpi, TRUE, FALSE, opt->doseps_reverse,
		opt->output);
	    break;
	case CMD_TIFF6U:
	    code = make_eps_tiff(doc, img, devbbox, &bbox, phires_bbox, 
		opt->dpi, opt->dpi, FALSE, FALSE, opt->doseps_reverse,
		opt->output);
	    break;
	case CMD_TIFF6P:
	    code = make_eps_tiff(doc, img, devbbox, &bbox, phires_bbox, 
		opt->dpi, opt->dpi, FALSE, TRUE, opt->doseps_reverse,
		opt->output);
	    break;
	case CMD_TIFF:
	    /* create preview file */
	    if ((f = app_temp_file(doc->app, preview, 
		sizeof(preview)/sizeof(TCHAR))) == (FILE *)NULL) {
		app_csmsgf(doc->app, 
		    TEXT("Can't create temporary tiff file \042%s\042\n"),
		    preview);
		code = -1;
	    }
	    else
		fclose(f);
	    if (code == 0)
		code = make_preview_file(doc, opt, 0, preview, 
		device, opt->dpi, &bbox, &hires_bbox, opt->bbox);
	    if (code == 0) {
		code = make_eps_user(doc, preview, opt->doseps_reverse,
		    opt->output);
		csunlink(preview);
	    }
	    break;
	case CMD_INTERCHANGE:
	    code = make_eps_interchange(doc, img, devbbox, 
		&bbox, phires_bbox, opt->output);
	    break;
	case CMD_WMF:
	    code = make_eps_metafile(doc, img, devbbox, &bbox, phires_bbox, 
		opt->dpi, opt->dpi, opt->doseps_reverse, opt->output);
	    break;
    	case CMD_USER:
	    if (img)
		code = make_eps_tiff(doc, img, devbbox, &bbox, phires_bbox,
		    opt->dpi, opt->dpi, FALSE, TRUE, opt->doseps_reverse, 
		    opt->output);
	    else
	        code = make_eps_user(doc, opt->user_preview, 
		    opt->doseps_reverse, opt->output);
	    break;
	default:
	    return 0;
    }

    if (img)
	bitmap_image_free(img);
    return code;
}

/****************************************************************/

static int
epstool_extract_doseps(Doc *doc, OPT *opt)
{
    int code;
    code = extract_doseps(doc, opt->output, opt->cmd == CMD_PREVIEW);
    return code;
}

/****************************************************************/

static int 
epstool_bitmap(Doc *doc, OPT *opt)
{
    int code = 0;
    CDSCBBOX bbox;
    CDSCFBBOX hires_bbox;
    LPCTSTR device = COLOUR_DEVICE;
    if (opt->device[0] != '\0')
	device = opt->device;

    if (doc->dsc->bbox)
	bbox = *doc->dsc->bbox;
    else {
	bbox.llx = bbox.lly = bbox.urx = bbox.ury = 0;
	opt->bbox = 1;
    }

    if (doc->dsc->hires_bbox)
	hires_bbox = *doc->dsc->hires_bbox;
    else {
	hires_bbox.fllx = (float)bbox.llx;
	hires_bbox.flly = (float)bbox.lly;
	hires_bbox.furx = (float)bbox.urx;
	hires_bbox.fury = (float)bbox.ury;
    }

    code = make_preview_file(doc, opt, 0, opt->output,
	device, opt->dpi, &bbox, &hires_bbox, opt->bbox);

    return code;
}

/****************************************************************/

static int 
epstool_copy(Doc *doc, OPT *opt)
{
    int code = 0;
    if (opt->bbox) {
	CDSCBBOX bbox;
	CDSCFBBOX hires_bbox;
	GFile *f;
	TCHAR tpsname[MAXSTR];

	memset(tpsname, 0, sizeof(tpsname));
	memset(&bbox, 0, sizeof(bbox));
	memset(&hires_bbox, 0, sizeof(hires_bbox));
	if (doc->dsc->bbox)
	    memcpy(&bbox, &doc->dsc->bbox, sizeof(bbox)); 
	if (doc->dsc->hires_bbox)
	    memcpy(&hires_bbox, &doc->dsc->hires_bbox, sizeof(hires_bbox)); 

	/* Copy page to temporary file */
	if ((f = app_temp_gfile(doc->app, tpsname, 
	    sizeof(tpsname)/sizeof(TCHAR))) == (GFile *)NULL) {
	    app_csmsgf(doc->app, 
		TEXT("Can't create temporary ps file \042%s\042\n"),
		tpsname);
	    return -1;
	}
	code = copy_page_temp(doc, f, 0);
	gfile_close(f);
	if (code != 0) {
	    if (!(debug & DEBUG_GENERAL))
		csunlink(tpsname);
	    return -1;
	}
	code = calculate_bbox(doc, opt, tpsname, &bbox, &hires_bbox);

	if (code == 0)
	    code = copy_eps(doc, opt->output, &bbox, &hires_bbox, 0, FALSE); 

	/* delete temporary ps file */
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
    }
    else {
	code = copy_eps(doc, opt->output, doc->dsc->bbox, 
		doc->dsc->hires_bbox, 0, FALSE); 
    }

    return code;
}

/****************************************************************/

/* If multi is TRUE, write DCS 2.0 multiple file, otherwise
 * write the single file version.
 */
static int 
epstool_dcs2_copy(Doc *doc, OPT *opt)
{
    GFile *infile = NULL;
    GFile *outfile = NULL;
    DSC_OFFSET complen = 0;
    int code = 0;
    GFile *compfile = NULL;
    TCHAR compname[MAXSTR];

    if (doc->dsc->dcs2 == NULL) {
	app_csmsgf(doc->app, TEXT("Input file is not DCS 2.0\n"));
	return -1;
    }
    memset(compname, 0, sizeof(compname));
    if (opt->composite) {
	/* Replace composite (first page) with a raster generated
	 * from the separations.
	 */
	if ((compfile = app_temp_gfile(doc->app, compname, 
	    sizeof(compname)/sizeof(TCHAR))) == (GFile *)NULL) {
	    app_csmsgf(doc->app, 
		TEXT("Can't create temporary composite EPS file \042%s\042\n"),
		compname);
	    return -1;
	}
	custom_colours_read(opt);
	code = epstool_dcs2_composite(doc, opt, compfile);
	custom_colours_free(opt);
	gfile_close(compfile);
	if (code == 0) {
	    compfile = gfile_open(compname, gfile_modeRead);
	    if (compfile == (GFile *)NULL)
		code = -1;
	}
	if (code) {
	    if (!(debug & DEBUG_GENERAL) && compname[0])
		csunlink(compname);
	    return -1;
	}
    }

    /* Let's convert from single file to multi file */
    if (code == 0)
	code = (infile = gfile_open(doc_name(doc), gfile_modeRead)) 
	    == (GFile *)NULL;
    if (code == 0)
	code = (outfile = gfile_open(opt->output, 
	    gfile_modeWrite | gfile_modeCreate)) == (GFile *)NULL;
    if (opt->cmd == CMD_DCS2_SINGLE) {
	if (code == 0)
	    code = copy_dcs2(doc, infile, outfile, opt->output,
		0 /* offset */, 
		FALSE /* multi */,
		FALSE /* write all */, 
		opt->missing_separations,
		&complen, compfile, opt->rename_sep);
	gfile_seek(infile, 0, gfile_begin);
	if (compfile)
	    gfile_seek(compfile, 0, gfile_begin);
	gfile_close(outfile);
	outfile = NULL;
    }
    if (code == 0)
	code = (outfile = gfile_open(opt->output, 
	    gfile_modeWrite | gfile_modeCreate)) == (GFile *)NULL;
    if (code == 0)
	code = copy_dcs2(doc, infile, outfile, opt->output,
	    0 /* offset */, 
	    opt->cmd == CMD_DCS2_MULTI,
	    TRUE /* write all */, 
	    opt->missing_separations,
	    &complen, compfile, opt->rename_sep);
    if (infile != (GFile *)NULL)
	gfile_close(infile);
    if (outfile != (GFile *)NULL)
	gfile_close(outfile);
    if (compfile != (GFile *)NULL)
	gfile_close(compfile);
    if (!(debug & DEBUG_GENERAL) && compname[0])
	csunlink(compname);

    return code;
}

static int 
epstool_dcs2_report(Doc *doc)
{
    int i;
    int code = 0;
    CDSC *dsc = doc->dsc;
    const char *type = "Unknown";
    const char *preview = "Unknown";

    if (dsc->dcs2) {
	DSC_OFFSET length = 0;
	GFile *f;
	/* Check for missing composite */
	if ((dsc->page_count == 0) ||
	     (dsc->page[0].begin == dsc->page[0].end)) {
	    app_msgf(doc->app, TEXT("WARNING: Missing composite, so a separation offset is probably wrong.\n"));
	    code = 2;
	}
	/* Check for separations that extend beyond EOF */
	if ((f = gfile_open(doc_name(doc), gfile_modeRead)) != (GFile *)NULL) {
	    length = gfile_get_length(f);
	    gfile_close(f);
	}
	for (i=0; i<(int)dsc->page_count; i++) {
	    if ((dsc->page[i].begin > length) ||
	        (dsc->page[i].end > length)) {
		app_msgf(doc->app, 
		    TEXT("WARNING: separation %s extends beyond EOF\n"),
		    dsc->page[i].label);
		code = 2;
	    }
	}
    }

    /* Document type */
    if (dsc == NULL)
	type = "Unknown";
    else if (dsc->dcs2)
	type = "DCS2.0";
    else if (dsc->epsf)
	type = "EPSF";
    else if (dsc->pdf)
	type = "PDF";
    else
	type = "DSC";
    fprintf(stdout, "Type\t%s\n", type);

    /* Preview type */
    switch (dsc->preview) {
	default:
	case CDSC_NOPREVIEW:
	    preview = "None";
	    break;
	case CDSC_EPSI:
	    preview = "Interchange";
	    break;
	case CDSC_TIFF:
	    preview = "TIFF";
	    break;
	case CDSC_WMF:
	    preview = "WMF";
	    break;
	case CDSC_PICT:
	    preview = "PICT";
	    break;
    }
    fprintf(stdout, "Preview\t%s\n", preview);

    if (dsc->bbox)
        fprintf(stdout, "BoundingBox\t%d\t%d\t%d\t%d\n", 
	    dsc->bbox->llx, dsc->bbox->lly,
	    dsc->bbox->urx, dsc->bbox->ury);
    else
        fprintf(stdout, "BoundingBox\n");
    if (dsc->hires_bbox)
        fprintf(stdout, "HiBoundingBox\t%g\t%g\t%g\t%g\n", 
	    dsc->hires_bbox->fllx, dsc->hires_bbox->flly,
	    dsc->hires_bbox->furx, dsc->hires_bbox->fury);
    else
        fprintf(stdout, "HiResBoundingBox\n");


    if (!dsc->dcs2)
	return 1;

    /* Pages */
    for (i=0; i<(int)dsc->page_count; i++) {
	int found = -1;
	const char *name = dsc->page[i].label;
	float cyan, magenta, yellow, black;
	cyan = magenta = yellow = black = 0.0;
	if (name == NULL)
	    name = "Unknown";
	if (strcmp(name, "Composite") != 0)
	    found = colour_to_cmyk(dsc, name, &cyan, &magenta, &yellow, &black);
	if (found == 0)
	    fprintf(stdout, "%s\t%lu\t%g\t%g\t%g\t%g\n", name, 
		(unsigned long)(dsc->page[i].end - dsc->page[i].begin),
		cyan, magenta, yellow, black);
	else
	    fprintf(stdout, "%s\t%lu\n", name, 
		(unsigned long)(dsc->page[i].end - dsc->page[i].begin));
    }
    return code;
}

/* Check that all separations actually exist */
static int 
epstool_dcs2_check_files(Doc *doc, OPT *opt)
{
    int code = 0;
    int i;
    CDSC *dsc = doc->dsc;
    GFile *f;
    const char *fname;
    for (i=1; (i<(int)dsc->page_count); i++) {
	/* First find length of separation */
	fname = dsc_find_platefile(dsc, i);
	if (fname) {
	    TCHAR wfname[MAXSTR];
	    narrow_to_cs(wfname, sizeof(wfname)/sizeof(TCHAR)-1,
		fname, strlen(fname)+1);
	    if ((f = gfile_open(wfname, gfile_modeRead)) != (GFile *)NULL) {
		gfile_close(f);
	    }
	    else {
		if (!opt->missing_separations)
		    code = -1;
		if (!opt->quiet)
		    app_msgf(doc->app, 
		    "Separation \042%s\042 is missing file \042%s\042\n",
			dsc->page[i].label, fname);
	    }
	}
	else {
	    if (dsc->page[i].end <= dsc->page[i].begin) {
		if (!opt->quiet)
	            app_msgf(doc->app, 
			"Separation \042%s\042 is empty\n",
			dsc->page[i].label);
		if (!opt->missing_separations)
		    code = -1;
	    }
	}
    }
    return code;
}

/****************************************************************/
/* Read --custom-colours file etc. */

static int 
custom_colours_read(OPT *opt)
{
    FILE *f;
    char line[MAXSTR];
    CUSTOM_COLOUR colour;
    CUSTOM_COLOUR *pcolour;
    CUSTOM_COLOUR *tail = opt->colours;
    int i;
    char *s;
    int code = 0;
    if (opt->custom_colours[0]) {
	/* read CMYK colours from file */
	f =  csfopen(opt->custom_colours, TEXT("r"));
	if (f == NULL)
	    return -1;
	while (tail && tail->next)
	    tail = tail->next;
	while (fgets(line, sizeof(line), f) != NULL) {
	    memset(&colour, 0, sizeof(colour));
	    i = 0;
	    s = line;
	    if (strncmp(line, "%%CMYKCustomColor:", 18) == 0) {
		s += 18;
		colour.type = CUSTOM_CMYK;
		while (*s && (*s == ' '))
		    s++;
		/* Get the colour values */
		s = strtok(s, " \t\r\n");
		if (s != NULL) {
		    colour.cyan = (float)atof(s);
		    s = strtok(NULL, " \t\r\n");
		}
		if (s != NULL) {
		    colour.magenta = (float)atof(s);
		    s = strtok(NULL, " \t\r\n");
		}
		if (s != NULL) {
		    colour.yellow = (float)atof(s);
		    s = strtok(NULL, " \t\r\n");
		}
		if (s != NULL) {
		    colour.black = (float)atof(s);
		    s = strtok(NULL, "\t\r\n");
		}
	    }
	    else if (strncmp(line, "%%RGBCustomColor:", 17) == 0) {
		s += 17;
		colour.type = CUSTOM_RGB;
		while (*s && (*s == ' '))
		    s++;
		/* Get the colour values */
		s = strtok(s, " \t\r\n");
		if (s != NULL) {
		    colour.red = (float)atof(s);
		    s = strtok(NULL, " \t\r\n");
		}
		if (s != NULL) {
		    colour.green = (float)atof(s);
		    s = strtok(NULL, " \t\r\n");
		}
		if (s != NULL) {
		    colour.blue = (float)atof(s);
		    s = strtok(NULL, "\t\r\n");
		}
	    }
	    else {
		s = NULL;
	    }

	    if (s != NULL) {
		/* Get the colour name */
		while (*s && (*s == ' '))
		    s++;
		if (*s == '(') {
		    s++;
		    while (*s && (*s != ')')) {
			if (i < sizeof(colour.name)-1)
			    colour.name[i++] = *s;
			s++;
		    }
		    if (*s == ')')
			s++;
		}
		else {
		    while (*s && (*s != ' ') && (*s != '\t') &&
			(*s != '\r') && (*s != '\n')) {
			if (i < sizeof(colour.name)-1)
			    colour.name[i++] = *s;
			s++;
		    }
		}
		colour.name[i] = '\0';
	    }
	    if (debug & DEBUG_GENERAL) {
		if (s == NULL)
		    fprintf(stdout, "Unrecognised line: %s\n", line);
		else if (colour.type == CUSTOM_CMYK)
		    fprintf(stdout, "CMYK Colour: %g %g %g %g (%s)\n",
			colour.cyan, colour.magenta, colour.yellow, 
		        colour.black, colour.name);
		else if (colour.type == CUSTOM_RGB)
		    fprintf(stdout, "RGB Colour: %g %g %g (%s)\n",
			colour.red, colour.green, colour.blue, 
		        colour.name);
		else 
		    fprintf(stdout, "Unrecognised colour\n");
	    }
	    if (s == NULL) {
		if (code == 0)
		    code = 1;
	    }
	    else {
		pcolour = (CUSTOM_COLOUR *)malloc(sizeof(CUSTOM_COLOUR));
		if (pcolour == NULL)
		    code = -1;
		else {
		    /* append to list */
		    memcpy(pcolour, &colour, sizeof(colour));
		    if (tail) {
			tail->next = pcolour;
			tail = pcolour;
		    }
		    else
			opt->colours = tail = pcolour;
		}
	    }
	}
	fclose(f);
    }
    return code;
}

static void 
custom_colours_free(OPT *opt)
{
    CUSTOM_COLOUR *pcolour;
    while (opt->colours) {
	pcolour = opt->colours;
	opt->colours = opt->colours->next;
	free(pcolour);
    }
}

static CUSTOM_COLOUR *
custom_colours_find(OPT *opt, const char *name)
{
    CUSTOM_COLOUR *pcolour = opt->colours;
    while (pcolour) {
	if (strcmp(name, pcolour->name) == 0)
	    break;
	pcolour = pcolour->next;
    }
    return pcolour;
}

/****************************************************************/

/* Calculate the bounding box using the ghostscript bbox device */
static int
calculate_bbox(Doc *doc, OPT *opt, LPCTSTR psname, CDSCBBOX *bbox, CDSCFBBOX *hires_bbox)
{
    FILE *bboxfile;
    TCHAR bboxname[MAXSTR];
    TCHAR command[MAXSTR*4];
    char line[MAXSTR];
    int got_bbox = 0;
    int got_hires_bbox = 0;
    int code = 0;
    int llx, lly, urx, ury;
    float fllx, flly, furx, fury;
    const int offset = 3000;
    if ((bboxfile = app_temp_file(doc->app, bboxname, 
	sizeof(bboxname)/sizeof(TCHAR))) == (FILE *)NULL) {
	app_csmsgf(doc->app, TEXT("Can't create temporary bbox file \042%s\042\n"),
	    bboxname);
	return -1;
    }
    fclose(bboxfile);

    csnprintf(command, sizeof(command)/sizeof(TCHAR), TEXT(
	"\042%s\042 %s %s -dNOPAUSE -dBATCH -sDEVICE=bbox \
 -c \042<</PageOffset [%d %d]>> setpagedevice\042 -f \042%s\042"), 
	opt->gs, opt->gsargs, opt->quiet ? TEXT("-dQUIET") : TEXT(""), 
	offset, offset, psname);
    if (!opt->quiet)
	app_csmsgf(doc->app, TEXT("%s\n"), command);
    code = exec_program(command, -1, fileno(stdout), -1, NULL, NULL, bboxname);
    if (code != 0)
	app_csmsgf(doc->app, 
	    TEXT("Ghostscript failed to obtain bounding box\n"));

    /* Now scan for bounding box info */
    if (code == 0) {
	if ((bboxfile = csfopen(bboxname, TEXT("rb"))) == (FILE *)NULL) {
	    app_csmsgf(doc->app, 
		TEXT("Can't open temporary bbox file \042%s\042\n"),
		bboxname);
	    code = -1;
	}
    }

    if (code == 0) {
	while (fgets(line, sizeof(line), bboxfile) != NULL) {
	    if (strncmp(line, "%%BoundingBox: ", 15) == 0) {
		if (!opt->quiet)
		    app_msgf(doc->app, "%s", line);
		if (sscanf(line+15, "%d %d %d %d", &llx, &lly, &urx, &ury) 
		    == 4) {
		    bbox->llx = llx-offset;
		    bbox->lly = lly-offset;
		    bbox->urx = urx-offset;
		    bbox->ury = ury-offset;
		    got_bbox = 1;
		}
	    }
	    else if (strncmp(line, "%%HiResBoundingBox: ", 20) == 0) {
		if (!opt->quiet)
		    app_msgf(doc->app, "%s", line);
		if (sscanf(line+20, "%f %f %f %f", &fllx, &flly, &furx, &fury) 
		    == 4) {
		    hires_bbox->fllx = fllx-offset;
		    hires_bbox->flly = flly-offset;
		    hires_bbox->furx = furx-offset;
		    hires_bbox->fury = fury-offset;
		    got_hires_bbox = 1;
		}
	    }
	    if (got_bbox && got_hires_bbox)
		break;
	}
    }

    fclose(bboxfile);
    if (!(debug & DEBUG_GENERAL))
	csunlink(bboxname);

    if ((code == 0) && (!got_bbox)) {
	app_csmsgf(doc->app, TEXT("Didn't get bounding box\n"));
	code = -1;
    }
    if ((code == 0) && (!got_hires_bbox)) {
	app_csmsgf(doc->app, TEXT("Didn't get hires bounding box\n"));
	code = -1;
    }
    return code;
}

static int
calc_device_size(float dpi, CDSCBBOX *bbox, CDSCFBBOX *hires_bbox,
    int *width, int *height, float *xoffset, float *yoffset)
{
    int code = 0;
    int hires_bbox_valid = 1;
    if ((bbox->llx >= bbox->urx) ||
	(bbox->lly >= bbox->ury)) {
	code = -1;
    }

    if ((hires_bbox->fllx > hires_bbox->furx) ||
	(hires_bbox->flly > hires_bbox->fury)) {
	hires_bbox_valid = 0;
	code = -1;
    }
    if ((hires_bbox->fllx == hires_bbox->furx) ||
	(hires_bbox->flly == hires_bbox->fury)) {
	hires_bbox_valid = 0;
	/* ignore hires_bbox */
    }

    /* Make the preview image */
    if (hires_bbox_valid) {
	*width = (int)((hires_bbox->furx - hires_bbox->fllx)*dpi/72.0+0.5);
	*height = (int)((hires_bbox->fury - hires_bbox->flly)*dpi/72.0+0.5);
	*xoffset = -hires_bbox->fllx;
	*yoffset = -hires_bbox->flly;
    }
    else {
	*width = (int)((bbox->urx - bbox->llx)*dpi/72.0+0.5);
	*height = (int)((bbox->ury - bbox->lly)*dpi/72.0+0.5);
	*xoffset = (float)(-bbox->llx);
	*yoffset = (float)(-bbox->lly);
    }
    return 0;
}


static int
make_preview_file(Doc *doc, OPT *opt, int page, 
    LPCTSTR preview, LPCTSTR device,
    float dpi, CDSCBBOX *bbox, CDSCFBBOX *hires_bbox, int calc_bbox)
{
    GFile *f;
    int code = 0;
    TCHAR tpsname[MAXSTR];
    TCHAR command[MAXSTR*4];
    int width, height;
    float xoffset, yoffset;

    /* Copy page to temporary file */
    if ((f = app_temp_gfile(doc->app, tpsname, 
	sizeof(tpsname)/sizeof(TCHAR))) == (GFile *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't create temporary ps file \042%s\042\n"),
	    tpsname);
	return -1;
    }
    code = copy_page_temp(doc, f, page);
    gfile_close(f);
    if (code != 0) {
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    /* Get bbox */
    if ((code == 0) && calc_bbox)
	code = calculate_bbox(doc, opt, tpsname, bbox, hires_bbox);
    width = height = 0;
    xoffset = yoffset = 0.0;
    if (code == 0)
        code = calc_device_size(dpi, bbox, hires_bbox, &width, &height,
	    &xoffset, &yoffset);
    if (code) {
	app_csmsgf(doc->app, TEXT("BoundingBox is invalid\n"));
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }
	
    /* Make the preview image */
    csnprintf(command, sizeof(command)/sizeof(TCHAR),
	TEXT("\042%s\042 %s %s -dNOPAUSE -dBATCH -sDEVICE=%s -sOutputFile=\042%s\042 -r%g -g%dx%d -c %f %f translate -f \042%s\042"), 
	opt->gs, opt->gsargs, opt->quiet ? TEXT("-dQUIET") : TEXT(""), 
	device, preview, dpi, width, height, 
	xoffset, yoffset, tpsname);
    if (!opt->quiet)
	app_csmsgf(doc->app, TEXT("%s\n"), command);
    code = exec_program(command, -1, fileno(stdout), fileno(stderr),
	NULL, NULL, NULL);
    if (code != 0)
	app_csmsgf(doc->app, 
	    TEXT("Ghostscript failed to create preview image\n"));

    /* delete temporary ps file */
    if (!(debug & DEBUG_GENERAL))
	csunlink(tpsname);

    return code;
}


static IMAGE *
make_preview_image(Doc *doc, OPT *opt, int page, LPCTSTR device,
    CDSCBBOX *bbox, CDSCFBBOX *hires_bbox, int calc_bbox)
{
    IMAGE *img = NULL;
    IMAGE *newimg = NULL;
    TCHAR preview[MAXSTR];
    GFile *f;
    int code = 0;

    /* Create a temporary file for ghostscript bitmap output */
    if ((f = app_temp_gfile(doc->app, preview, 
	sizeof(preview)/sizeof(TCHAR))) == (GFile *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't create temporary bitmap file \042%s\042\n"),
	    preview);
	code = -1;
    }
    else
	gfile_close(f);

    if (code == 0)
	code = make_preview_file(doc, opt, page, 
	    preview, device, opt->dpi_render, bbox, hires_bbox, calc_bbox);

    if (code == 0) {
	/* Load image */
	img = bmpfile_to_image(preview);
	if (img == NULL)
	    img = pnmfile_to_image(preview);
    }

    if (img && (opt->dpi_render != opt->dpi)) {
	/* downscale it */
	newimg = (IMAGE *)malloc(sizeof(IMAGE));
	if (newimg != NULL) {
	    int ncomp;
	    int width, height;
	    float xoffset, yoffset;
	    memset(newimg, 0, sizeof(newimg));
	    calc_device_size(opt->dpi, bbox, hires_bbox, &width, &height,
		 &xoffset, &yoffset);
	    newimg->width = width;
	    newimg->height = height;
	    newimg->format = img->format;
	    if ((newimg->format & DISPLAY_COLORS_MASK) 
		== DISPLAY_COLORS_CMYK)
		ncomp = 4;
	    else if ((newimg->format & DISPLAY_COLORS_MASK) 
		== DISPLAY_COLORS_RGB)
		ncomp = 3;
	    else
		ncomp = 1;
	    newimg->raster = newimg->width * ncomp;	/* bytes per row */
	    newimg->image = malloc(newimg->raster * newimg->height);
	    if (newimg->image == NULL) {
		free(newimg);
		newimg = NULL;
	    }
	}
	if (newimg != NULL) {
	    memset(newimg->image, 0, newimg->raster * newimg->height);
	    if (image_down_scale(newimg, img) != 0) {
		bitmap_image_free(newimg);
		newimg = NULL;
	    }
	}
	if (newimg != NULL) {
	    bitmap_image_free(img);
	    img = newimg;
	}
    }
	
    if (!(debug & DEBUG_GENERAL))
	csunlink(preview);
    return img;
}


/****************************************************************/
int
epstool_dcs2_composite(Doc *doc, OPT *opt, GFile *compfile)
{
    CDSC *dsc = doc->dsc;
    IMAGE img;
    IMAGE *layer = NULL;
    int width, height;
    float xoffset, yoffset;
    float cyan, magenta, yellow, black;
    int code = 0;
    int i;
    CDSCBBOX bbox = {0, 0, 0, 0};
    CDSCFBBOX hires_bbox = {0.0, 0.0, 0.0, 0.0};
    int hires_bbox_valid = 0;

    /* Require original to be DCS 2.0 */ 
    if (doc->dsc == NULL)
	return -1;
    if (doc->dsc->dcs2 == NULL)
	return -1;

    if (dsc->bbox) {
	bbox = *dsc->bbox;
    }
    else
	return -1;
    if (dsc->hires_bbox) {
	hires_bbox = *dsc->hires_bbox;
	hires_bbox_valid = 1;
    }

    /* Make a CMYK image for the composite */
    code = calc_device_size(opt->dpi, &bbox, &hires_bbox, &width, &height,
	    &xoffset, &yoffset);
    if (code) {
	app_csmsgf(doc->app, TEXT("BoundingBox is invalid\n"));
	return -1;
    }
    memset(&img, 0, sizeof(img));
    img.width = width;
    img.height = height;
    img.raster = img.width * 4;
    img.format = DISPLAY_COLORS_CMYK | DISPLAY_ALPHA_NONE |
       DISPLAY_DEPTH_8 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST;
    img.image = (unsigned char *)malloc(img.raster * img.height);
    if (img.image == NULL)
	return -1;
    memset(img.image, 0, img.raster * img.height);
 
    /* For each plate, make an image, then merge it into
     * the composite
     */
    for (i=1; i<(int)dsc->page_count; i++) {
	/* find colour */
	int found;
	CUSTOM_COLOUR *ccolour = NULL;
	const char *name = dsc->page[i].label;
	if (name == NULL) {
	    app_msgf(doc->app, "Page %d doesn't have a label\n", i);
	    code = -1;
	    break;
	}
	found = colour_to_cmyk(dsc, name, &cyan, &magenta, &yellow, &black);
	if ((ccolour = custom_colours_find(opt, name)) != NULL) {
	    /* Use local colour overrides from --custom-colours */
	    found = 0;
	    if (ccolour->type == CUSTOM_RGB) {
		cyan = (float)(1.0 - ccolour->red);
		magenta = (float)(1.0 - ccolour->green);
		yellow = (float)(1.0 - ccolour->blue);
		black = min(cyan, min(magenta, yellow));
		if (black > 0.0) {
		    cyan -= black;
		    magenta -= black;
		    yellow -= black;
		}
	    }
	    else {
		cyan = ccolour->cyan;
		magenta = ccolour->magenta;
		yellow = ccolour->yellow;
		black = ccolour->black;
	    }
	}
	if (found < 0) {
	    app_msgf(doc->app, "Unrecognised colour (%s)\n", name);
	    code = -1;
	    break;
	}

	if ((cyan == 0.0) && (magenta == 0.0) && (yellow == 0.0) && 
	    (black == 0.0)) {
	    if (!opt->quiet)
		app_msgf(doc->app, "Skipping blank separation %s\n", name);
	    continue;
	}
	if (!opt->quiet)
	    app_msgf(doc->app, "Creating image from separation %s\n", name);
	/* Now get the separation image */
        layer = make_preview_image(doc, opt, i, GREY_DEVICE,
    		&bbox, &hires_bbox, FALSE);
	if (layer == NULL) {
	    app_msgf(doc->app, "Failed to make image for separation (%s)\n",
		name);
	    code = -1;
	    break;
	}
	else {
	    if (!opt->quiet)
		app_msgf(doc->app, "Merging separation %g %g %g %g  %s\n", 
		    cyan, magenta, yellow, black, name);
	    code = image_merge_cmyk(&img, layer, cyan, magenta, yellow, black);
	    bitmap_image_free(layer);
	    if (code < 0) {
	        app_msgf(doc->app, "Failed to merge separation (%s)\n", name);
		code = -1;
		break;
	    }
	}
	
    } 


    if (code == 0) {
	if (!opt->quiet)
	    app_msgf(doc->app, "Writing composite as EPS\n");
	code = image_to_eps(compfile, &img, 
		bbox.llx, bbox.lly, bbox.urx, bbox.ury, 
		hires_bbox.fllx, hires_bbox.flly, 
		hires_bbox.furx, hires_bbox.fury, 
		TRUE /* use ASCII85Decode */, 
		TRUE /* use RunLengthDecode */ );
    }

    free(img.image);
    return code;
}

/* Return 0 if CMYK set from the DSC comments, -1 if not set */
static int
colour_to_cmyk(CDSC *dsc, const char *name, 
   float *cyan, float *magenta, float *yellow, float *black)
{
    int code = 0;
    CDSCCOLOUR *colour = dsc->colours;
    if (name == NULL)
	return -1;
    while (colour) {
	if (colour->name && (dsc_stricmp(colour->name, name)==0))
	    break;
	colour = colour->next;
    }

    if (colour && (colour->custom == CDSC_CUSTOM_COLOUR_CMYK)) {
	*cyan = colour->cyan;
	*magenta = colour->magenta;
	*yellow = colour->yellow;
	*black = colour->black;
    }
    else if (colour && (colour->custom == CDSC_CUSTOM_COLOUR_RGB)) {
	*cyan = (float)(1.0 - colour->red);
	*magenta = (float)(1.0 - colour->green);
	*yellow = (float)(1.0 - colour->blue);
	*black = min(*cyan, min(*magenta, *yellow));
	if (*black > 0.0) {
	    *cyan -= *black;
	    *magenta -= *black;
	    *yellow -= *black;
	}
    }
    else {
	if (dsc_stricmp(name, "Cyan") == 0) {
	    *cyan = 1.0;
	    *magenta = *yellow = *black = 0.0;
	}
	else if (dsc_stricmp(name, "Magenta") == 0) {
	    *magenta = 1.0;
	    *cyan = *yellow = *black = 0.0;
	}
	else if (dsc_stricmp(name, "Yellow") == 0) {
	    *yellow = 1.0;
	    *cyan = *magenta = *black = 0.0;
	}
	else if (dsc_stricmp(name, "Black") == 0) {
	    *black = 1.0;
	    *cyan = *yellow = *magenta = 0.0;
	}
	else {
	    code = -1;
	}
    }
    return code;
}



/****************************************************************/
/* Functions from GSview app that we need. */
/* Some of these should be moved from capp.c to a separate file. */

#ifdef _MSC_VER
# pragma warning(disable:4100) /* ignore "Unreferenced formal parameter" */
#endif

int
app_platform_init(GSview *a)
{
    return 0;
}

int
app_platform_finish(GSview *a)
{
    return 0;
}

int
app_lock(GSview *a)
{
    return 0;
}

int
app_unlock(GSview *a)
{
    return 0;
}

void
app_log(const char *str, int len)
{
    fwrite(str, 1, len, MSGOUT);
}

int
cs_to_narrow(char *nstr, int nlen, LPCTSTR wstr, int wlen)
{
#ifdef UNICODE
    return WideCharToMultiByte(CP_ACP, 0, wstr, wlen, nstr, nlen, NULL, NULL);
#else
    return char_to_narrow(nstr, nlen, wstr, wlen);
#endif
}

int 
narrow_to_cs(TCHAR *wstr, int wlen, const char *nstr, int nlen)
{
#ifdef UNICODE
    return MultiByteToWideChar(CP_ACP, 0, nstr, nlen, wstr, wlen);
#else
    return narrow_to_char(wstr, wlen, nstr, nlen);
#endif
}

int get_dsc_response(GSview *app, LPCTSTR str)
{
    app_csmsgf(app, TEXT("get_dsc_response:\n%s\n"), str);
    return CDSC_RESPONSE_OK;
}

int 
load_string(GSview *a, int id, TCHAR *buf, int len)
{
    TCHAR msg[MAXSTR];
    int dscmsg = (id - CDSC_RESOURCE_BASE) / 2;
    /* CDSC_MESSAGE_INCORRECT_USAGE should be dsc->max_error */
    if (a /* ignore unused parameter GSview *a */
	&& (dscmsg > 0) && (dscmsg <= CDSC_MESSAGE_INCORRECT_USAGE)) {
	if (len)
	    buf[0] = '\0';
	return 0;	/* Don't show these - already shown by parser */
    }

    switch (id) {
	case IDS_DSC_INFO:
	case IDS_DSC_WARN:
	case IDS_DSC_ERROR:
	case IDS_DSC_LINEFMT:
	    if (len)
		buf[0] = '\0';
	    return 0;
	default:
	    csnprintf(msg, sizeof(msg)/sizeof(TCHAR), TEXT("String %d\n"), id);
	    if (len)
		csncpy(buf, msg, len);
    }

    return cslen(buf);
}

int app_msg_box(GSview *a, LPCTSTR str, int icon)
{
    app_csmsgf(a, TEXT("%s\n"), str);
    return 0;
}

int 
gssrv_request(GSview *a, GSREQ *reqnew)
{
    return -1;
}

int 
pagecache_unref_all(GSview *a) 
{
    return 0;
}


void
doc_savestat(Doc *doc)
{
}

int 
image_platform_init(IMAGE *img)
{
    return 0;
}

unsigned int 
image_platform_format(unsigned int format)
{
    return format;
}

#ifdef _MSC_VER
# pragma warning(default:4100)
#endif

/****************************************************************/
/* platform specific code for running another program */

/*
 * If hstdin not -1, duplicate handle and give to program,
 * else if stdin_name not NULL, open filename and give to program.
 * Same for hstdout/stdout_name and hstderr/stderr_name.
 */
#ifdef __WIN32__
int
exec_program(LPTSTR command,
    int hstdin, int hstdout, int hstderr,
    LPCTSTR stdin_name, LPCTSTR stdout_name, LPCTSTR stderr_name)
{
    int code = 0;
    HANDLE hChildStdinRd = INVALID_HANDLE_VALUE;
    HANDLE hChildStdoutWr = INVALID_HANDLE_VALUE;
    HANDLE hChildStderrWr = INVALID_HANDLE_VALUE;
    HANDLE hStdin = INVALID_HANDLE_VALUE;
    HANDLE hStderr = INVALID_HANDLE_VALUE;
    HANDLE hStdout = INVALID_HANDLE_VALUE;
    SECURITY_ATTRIBUTES saAttr;
    STARTUPINFO siStartInfo;
    PROCESS_INFORMATION piProcInfo;
    DWORD exitcode = (DWORD)-1;

    /* Set the bInheritHandle flag so pipe handles are inherited. */
    saAttr.nLength = sizeof(SECURITY_ATTRIBUTES);
    saAttr.bInheritHandle = TRUE;
    saAttr.lpSecurityDescriptor = NULL;

    /* if handles are provided, use them */
    if ((hstdin != -1)) {
	long handle;
	handle = _get_osfhandle(hstdin);
	if (handle == -1)
	    code = -1;
	if (code == 0) {
	    hStdin = (HANDLE)handle;
	    if (!DuplicateHandle(GetCurrentProcess(), hStdin,
		GetCurrentProcess(), &hChildStdinRd, 0,
		TRUE,       /* inherited */
		DUPLICATE_SAME_ACCESS))
	        code = -1;
	}
    }
    if ((code==0) && (hstdout != -1)) {
	long handle;
	handle = _get_osfhandle(hstdout);
	if (handle == -1)
	    code = -1;
	if (code == 0) {
	    hStdout = (HANDLE)handle;
	    if (!DuplicateHandle(GetCurrentProcess(), hStdout,
		GetCurrentProcess(), &hChildStdoutWr, 0,
		TRUE,       /* inherited */
		DUPLICATE_SAME_ACCESS))
	        code = -1;
	}
    }
    if ((code==0) && (hstderr != -1)) {
	long handle;
	handle = _get_osfhandle(hstderr);
	if (handle == -1)
	    code = -1;
	if (code == 0) {
	    hStderr = (HANDLE)handle;
	    if (!DuplicateHandle(GetCurrentProcess(), hStderr,
		GetCurrentProcess(), &hChildStderrWr, 0,
		TRUE,       /* inherited */
		DUPLICATE_SAME_ACCESS))
	        code = -1;
	}
    }

    /* If files are requested, create them */
    if ((code==0) && stdin_name && (hChildStdinRd == INVALID_HANDLE_VALUE)) {
	hChildStdinRd = CreateFile(stdin_name, GENERIC_READ, 
	    0  /* no file sharing */,
	    &saAttr /* allow handle to be inherited */,
	    OPEN_EXISTING, 0, NULL);
	if (hChildStdinRd == INVALID_HANDLE_VALUE)
	    code = -1;
    }
    if ((code==0) && stdout_name && (hChildStdoutWr == INVALID_HANDLE_VALUE)) {
	hChildStdoutWr = CreateFile(stdout_name, GENERIC_WRITE, 
	    0  /* no file sharing */,
	    &saAttr /* allow handle to be inherited */,
	    OPEN_ALWAYS, 0, NULL);
	if (hChildStdoutWr == INVALID_HANDLE_VALUE)
	    code = -1;
    }
    if ((code==0) && stderr_name && (hChildStderrWr == INVALID_HANDLE_VALUE)) {
	hChildStderrWr = CreateFile(stderr_name, GENERIC_WRITE, 
	    0  /* no file sharing */,
	    &saAttr /* allow handle to be inherited */,
	    OPEN_ALWAYS, 0, NULL);
	if (hChildStderrWr == INVALID_HANDLE_VALUE)
	    code = -1;
    }

    /* Set up members of STARTUPINFO structure. */
    siStartInfo.cb = sizeof(STARTUPINFO);
    siStartInfo.lpReserved = NULL;
    siStartInfo.lpDesktop = NULL;
    siStartInfo.lpTitle = NULL;  /* use executable name as title */
    /* next two lines ignored */
    siStartInfo.dwX = siStartInfo.dwY = (DWORD)CW_USEDEFAULT;
    siStartInfo.dwXSize = siStartInfo.dwYSize = (DWORD)CW_USEDEFAULT;
    siStartInfo.dwXCountChars = 80;
    siStartInfo.dwYCountChars = 25;
    siStartInfo.dwFillAttribute = 0;			/* ignored */
    siStartInfo.dwFlags = STARTF_USESTDHANDLES;
    siStartInfo.wShowWindow = SW_SHOWNORMAL;		/* ignored */
    siStartInfo.cbReserved2 = 0;
    siStartInfo.lpReserved2 = NULL;
    siStartInfo.hStdInput = hChildStdinRd;
    siStartInfo.hStdOutput = hChildStdoutWr;
    siStartInfo.hStdError = hChildStderrWr;
    memset(&piProcInfo, 0, sizeof(piProcInfo));

    if ((code == 0) && !CreateProcess(NULL,
        command,
        NULL,          /* process security attributes        */
        NULL,          /* primary thread security attributes */
        TRUE,          /* handles are inherited              */
	0,             /* creation flags                     */
	NULL,          /* environment                        */
        NULL,          /* use parent's current directory     */
        &siStartInfo,  /* STARTUPINFO pointer                */
        &piProcInfo)) { /* receives PROCESS_INFORMATION  */
	code = -1;
    }

    /* close our copy of the handles */
    if (hChildStdinRd != INVALID_HANDLE_VALUE)
	CloseHandle(hChildStdinRd);
    if (hChildStdoutWr != INVALID_HANDLE_VALUE)
	CloseHandle(hChildStdoutWr);
    if (hChildStderrWr != INVALID_HANDLE_VALUE)
	CloseHandle(hChildStderrWr);

    if (code == 0) {
	/* wait for process to finish */
	WaitForSingleObject(piProcInfo.hProcess, 300000);
	GetExitCodeProcess(piProcInfo.hProcess, &exitcode);
	CloseHandle(piProcInfo.hProcess);
	CloseHandle(piProcInfo.hThread);
    }

    if (code)
	return code;
    return (int)exitcode;
}
#endif

#ifdef UNIX
int
exec_program(LPTSTR command,
    int hstdin, int hstdout, int hstderr,
    LPCTSTR stdin_name, LPCTSTR stdout_name, LPCTSTR stderr_name)
{
    int code = 0;
    int hChildStdinRd = -1;
    int hChildStdoutWr = -1;
    int hChildStderrWr = -1;
    int handle;
    pid_t pid;
    int exitcode;
#define MAXARG 64
    char *argv[MAXARG+1];
    int argc = 0;
    char *args, *d, *e, *p;

    /* Parse command line handling quotes. */
    memset(argv, 0, sizeof(argv));
    argc = 0;
    args = (char *)malloc(strlen(command)+1);
    if (args == (char *)NULL)
	return -1;
    p = command;
    d = args;
    while (*p) {
	/* for each argument */
	if (argc >= MAXARG - 1) {
	    fprintf(stderr, "Too many arguments\n");
	    free(args);
	    return -1;
	}

        e = d;
        while ((*p) && (*p != ' ')) {
	    if (*p == '\042') {
		/* Remove quotes, skipping over embedded spaces. */
		/* Doesn't handle embedded quotes. */
		p++;
		while ((*p) && (*p != '\042'))
		    *d++ =*p++;
	    }
	    else 
		*d++ = *p;
	    if (*p)
		p++;
        }
	*d++ = '\0';
	argv[argc++] = e;

	while ((*p) && (*p == ' '))
	    p++;	/* Skip over trailing spaces */
    }
    argv[argc] = NULL;

    pid = fork();
    if (pid == (pid_t)-1) {
	/* fork failed */
	fprintf(stderr, "Failed to fork, error=%d\n", errno);
	return -1;
    }
    else if (pid == 0) {
	/* child */
	/* if handles are provided, use them */
	if ((code == 0) && (hstdin != -1)) {
	    hChildStdinRd = dup2(hstdin, 0);
	    if (hChildStdinRd == -1)
		code = -1;
	}
	if ((code==0) && (hstdout != -1)) {
	    hChildStdoutWr = dup2(hstdout, 1);
	    if (hChildStdoutWr == -1)
		code = -1;
	}
	if ((code==0) && (hstderr != -1)) {
	    hChildStderrWr = dup2(hstderr, 2);
	    if (hChildStderrWr == -1)
		code = -1;
	}
	if ((code==0) && stdin_name && (hChildStdinRd == -1)) {
	    handle = open(stdin_name, O_RDONLY);
	    hChildStdinRd = dup2(handle, 0);
	    if (handle != -1)
		close(handle);
	    if (hChildStdinRd == -1)
		code = -1;
	}
	if ((code==0) && stdout_name && (hChildStdoutWr == -1)) {
	    handle = open(stdout_name, O_WRONLY | O_CREAT);
	    hChildStdoutWr = dup2(handle, 1);
	    if (handle != -1)
		close(handle);
	    if (hChildStdoutWr == -1)
		code = -1;
	}
	if ((code==0) && stderr_name && (hChildStderrWr == -1)) {
	    handle = open(stderr_name, O_WRONLY | O_CREAT);
	    hChildStderrWr = dup2(handle, 2);
	    if (handle != -1)
		close(handle);
	    if (hChildStderrWr == -1)
		code = -1;
	}

	if (code) {
	    fprintf(stderr, "Failed to open stdin/out/err, error=%d\n", errno);
	    if (hChildStdinRd)
		close(hChildStdinRd);
	    if (hChildStdoutWr)
		close(hChildStdoutWr);
	    if (hChildStderrWr)
		close(hChildStderrWr);
	    exit(1);
	}

	/* Now execute it */
	if (execvp(argv[0], argv) == -1) {
	    fprintf(stderr, "Failed to execute ghostscript, error=%d\n", errno);
	    exit(1);
	}
    }

    /* parent - wait for child to finish */
    free(args);
    wait(&exitcode);
    return exitcode;
}
#endif
