/* Copyright (C) 1993-2002 Ghostgum Software Pty Ltd.  All rights reserved.

  This software is provided AS-IS with no warranty, either express or
  implied.

  This software is distributed under licence and may not be copied,
  modified or distributed except as expressly authorised under the terms
  of the licence contained in the file LICENCE in this distribution.

  For more information about licensing, please refer to
  http://www.ghostgum.com.au/ or contact Ghostsgum Software Pty Ltd, 
  218 Gallaghers Rd, Glen Waverley VIC 3150, AUSTRALIA, 
  Fax +61 3 9886 6616.
*/

/* $Id: ceps.c,v 1.14 2002/09/16 09:37:17 ghostgum Exp $ */

/* EPS preview manipulation */

#include "common.h"
#include <time.h>
#include "gdevdsp.h"
#include "dscparse.h"
#include "capp.h"
#include "cbmp.h"
#define DEFINE_CDOC
#include "cdoc.h"
#include "ceps.h"
#include "cimg.h"
#include "cps.h"


#define DOSEPS_HEADER_SIZE 30

/* Local prototypes */
static void write_doseps_header(CDSCDOSEPS *doseps, FILE *outfile);
static void shift_preview(unsigned char *preview, int bwidth, int offset);
static void validate_devbbox(IMAGE *img, CDSCBBOX *devbbox);
int write_interchange(FILE *f, IMAGE *img, CDSCBBOX devbbox);
static void write_bitmap_info(IMAGE *img, LPBITMAP2 pbmi, FILE *f);
static void make_bmp_info(LPBITMAP2 pbmi, IMAGE *img, float xdpi, float ydpi);
void copy_nobbox(FILE *outfile, FILE *infile, 
    unsigned long begin, unsigned long end);
void copy_bbox_header(FILE *outfile, 
    FILE *infile, unsigned long begin, unsigned long end,
    CDSCBBOX *bbox, CDSCFBBOX *hiresbbox);
static int without_eol(const char *str, int length);

/* A placeable Windows Metafile header is needed
 * when the metafile is in a WMF file, but not
 * when it is in memory.
 */
typedef struct WINRECT_s {
	WORD	left;
	WORD	top;
	WORD	right;
	WORD	bottom;
} WINRECT;

typedef struct METAFILEHEADER_s {
    DWORD	key;
    WORD 	hmf;
    WINRECT 	bbox;
    WORD	inch;
    DWORD	reserved;
    WORD	checksum;
} METAFILEHEADER;


static void
write_doseps_header(CDSCDOSEPS *doseps, FILE *outfile)
{
    unsigned char doseps_id[] = {0xc5, 0xd0, 0xd3, 0xc6};
    fwrite(doseps_id, 1, 4, outfile);
    write_dword(doseps->ps_begin, outfile);
    write_dword(doseps->ps_length, outfile);
    write_dword(doseps->wmf_begin, outfile);
    write_dword(doseps->wmf_length, outfile);
    write_dword(doseps->tiff_begin, outfile);
    write_dword(doseps->tiff_length, outfile);
    write_word((WORD)(doseps->checksum), outfile);
}

/* shift preview by offset bits to the left */
/* width is in bytes */
/* fill exposed bits with 1's */
static void
shift_preview(unsigned char *preview, int bwidth, int offset)
{
int bitoffset;
int byteoffset;
int newwidth;
int shifter;
int i;
    if (offset == 0)
	return;
    byteoffset = offset / 8;
    newwidth = bwidth - byteoffset;
    /* first remove byte offset */
    memmove(preview, preview+byteoffset, newwidth);
    memset(preview+newwidth, 0xff, bwidth-newwidth);
    /* next remove bit offset */
    bitoffset = offset - byteoffset*8;
    if (bitoffset==0)
	return;
    bitoffset = 8 - bitoffset;
    for (i=0; i<newwidth; i++) {
       shifter = preview[i] << 8;
       if (i==newwidth-1)
	   shifter += 0xff;	/* can't access preview[bwidth] */
       else
	   shifter += preview[i+1];  
       preview[i] = (unsigned char)(shifter>>bitoffset);
    }
}


static void
validate_devbbox(IMAGE *img, CDSCBBOX *devbbox)
{
    /* make sure the pixel coordinates are valid */
    if ((devbbox->llx < 0) || (devbbox->llx >= (int)img->width))
	devbbox->llx = 0;
    if ((devbbox->urx < 0) || (devbbox->urx >= (int)img->width))
	devbbox->urx = img->width;
    if ((devbbox->lly < 0) || (devbbox->lly >= (int)img->height))
	devbbox->lly = 0;
    if ((devbbox->ury < 0) || (devbbox->ury >= (int)img->height))
	devbbox->ury = img->height;

    if ((devbbox->llx >= devbbox->urx) || (devbbox->lly >= devbbox->ury)) {
	devbbox->llx = devbbox->lly = 0;
	devbbox->urx = img->width;
	devbbox->ury = img->height;
    }
}

/* Write bitmap info.
 * This works even if LPBITMAP2 is not packed.
 */
static void
write_bitmap_info(IMAGE *img, LPBITMAP2 pbmi, FILE *f)
{
    int i;
    unsigned char r, g, b;
    int palcount = 0;

    /* write bitmap info */
    write_dword(BITMAP2_LENGTH, f);
    write_dword(pbmi->biWidth, f);
    write_dword(pbmi->biHeight, f);
    write_word(pbmi->biPlanes, f);
    write_word(pbmi->biBitCount, f);
    write_dword(pbmi->biCompression, f);
    write_dword(pbmi->biSizeImage, f);
    write_dword(pbmi->biXPelsPerMeter, f);
    write_dword(pbmi->biYPelsPerMeter, f);
    write_dword(pbmi->biClrUsed, f);
    write_dword(pbmi->biClrImportant, f);
  
    if (pbmi->biBitCount <= 8)
	palcount = 1 << pbmi->biBitCount;
    for (i=0; i<palcount; i++) {
	image_colour(img->format, i, &r, &g, &b);
	fputc(b, f);
	fputc(g, f);
	fputc(r, f);
	fputc('\0', f);
    }
}

/* Make a BMP header from an IMAGE.
 * WARNING: The pbmi structure might not be packed, so it
 * should only be used by write_bitmap_info(), and not written
 * out directly.
 */
static void
make_bmp_info(LPBITMAP2 pbmi, IMAGE *img, float xdpi, float ydpi)
{
    int palcount = 0;
    int depth = image_depth(img);
    if (depth <= 8)
	palcount = 1 << depth;

    pbmi->biSize = sizeof(BITMAP2); /* WARNING - MAY NOT BE PACKED */
    pbmi->biWidth = img->width;
    pbmi->biHeight = img->width;
    pbmi->biPlanes = 1;
    pbmi->biBitCount = (WORD)image_depth(img);
    pbmi->biCompression = 0;
    pbmi->biSizeImage = 0;
    pbmi->biXPelsPerMeter = (long)(1000 * xdpi / 25.4);
    pbmi->biYPelsPerMeter = (long)(1000 * ydpi / 25.4);
    pbmi->biClrUsed = palcount;
    pbmi->biClrImportant = palcount;
}

/*********************************************************/

/* extract EPS or TIFF or WMF file from DOS EPS file */
int 
extract_doseps(Doc *doc, LPCTSTR outname, BOOL preview)
{
unsigned long pos;
unsigned long len;
unsigned int count;
char *buffer;
FILE* epsfile;
BOOL is_meta = TRUE;
FILE *outfile;
CDSC *dsc = doc->dsc;

    if ((dsc == (CDSC *)NULL) || (dsc->doseps == (CDSCDOSEPS *)NULL)) {
	app_csmsgf(doc->app, 
	    TEXT("Document \042%s\042 is not a DOS EPS file\n"),
	    doc->name);
	return -1;
    }

    epsfile = csfopen(doc_name(doc),TEXT("rb"));
    pos = dsc->doseps->ps_begin;
    len = dsc->doseps->ps_length;
    if (preview) {
	pos = dsc->doseps->wmf_begin;
	len = dsc->doseps->wmf_length;
	if (pos == 0L) {
	    pos = dsc->doseps->tiff_begin;
	    len = dsc->doseps->tiff_length;
	    is_meta = FALSE;
	}
    }
    if (pos == 0L) {
	fclose(epsfile);
	app_csmsgf(doc->app, 
	    TEXT("Document \042%s\042 does not have a %s section\n"),
	    doc->name, preview ? TEXT("preview") : TEXT("PostScript"));
	return -1;
    }
    fseek(epsfile, pos, SEEK_SET);	/* seek to section to extract */

    if (*outname!='\0')
	outfile = csfopen(outname,TEXT("wb"));
    else
	outfile = stdout;

    if (outfile == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Failed to open output file \042%s\042\n"), outname);
	fclose(epsfile);
	return -1;
    }
    
    /* create buffer for file copy */
    buffer = (char *)malloc(COPY_BUF_SIZE);
    if (buffer == (char *)NULL) {
	app_csmsgf(doc->app, TEXT("Out of memory in extract_doseps\n"));
	fclose(epsfile);
	if (*outname!='\0')
	    fclose(outfile);
	return -1;
    }

    if (preview && is_meta) {
	/* check if metafile already contains header */
	DWORD key;
	char keybuf[4];
	DWORD wmf_key = 0x9ac6cdd7UL;
	fread(keybuf, 1, 4, epsfile);
	key = keybuf[0] + (keybuf[1]<<8) + 
	      (keybuf[2]<<16) + (keybuf[3]<<24);
	fseek(epsfile, pos, SEEK_SET);	/* seek to section to extract */
	if ( key != wmf_key ) {
	    /* write placeable Windows Metafile header */
	    METAFILEHEADER mfh;
	    int i, temp;
	    unsigned short *pw;
	    mfh.key = wmf_key;
	    mfh.hmf = 0;
	    /* guess the location - this might be wrong */
	    mfh.bbox.left = 0;
	    mfh.bbox.top = 0;
	    if (dsc->bbox != (CDSCBBOX *)NULL) {
		temp = (dsc->bbox->urx - dsc->bbox->llx);
		/* double transfer to avoid GCC Solaris bug */
		mfh.bbox.right = (WORD)temp;	
		mfh.bbox.bottom = (WORD)(dsc->bbox->ury - dsc->bbox->lly);
		temp = (dsc->bbox->ury - dsc->bbox->lly);
		mfh.bbox.bottom = (WORD)temp;
	    }
	    else {
		/* bbox missing, assume A4 */
		mfh.bbox.right = 595;
		mfh.bbox.bottom = 842;
	    }
	    mfh.inch = 72;	/* PostScript points */
	    mfh.reserved = 0L;
	    mfh.checksum =  0;
	    pw = (WORD *)&mfh;
	    temp = 0;
	    for (i=0; i<10; i++) {
		temp ^= *pw++;
	    }
	    mfh.checksum = (WORD)temp;
	    write_dword(mfh.key, outfile);
	    write_word(mfh.hmf, outfile);
	    write_word(mfh.bbox.left,   outfile);
	    write_word(mfh.bbox.top,    outfile);
	    write_word(mfh.bbox.right,  outfile);
	    write_word(mfh.bbox.bottom, outfile);
	    write_word(mfh.inch, outfile);
	    write_dword(mfh.reserved, outfile);
	    write_word(mfh.checksum, outfile);
	}
    }

    while ( (count = (unsigned int)min(len,COPY_BUF_SIZE)) != 0 ) {
	count = (int)fread(buffer, 1, count, epsfile);
	fwrite(buffer, 1, count, outfile);
	if (count == 0)
	    len = 0;
	else
	    len -= count;
    }
    free(buffer);
    fclose(epsfile);
    if (*outname!='\0')
	fclose(outfile);

    return 0;
}

/*********************************************************/

#define tiff_long(val, f) write_dword(val, f)
#define tiff_short(val, f) write_word_as_dword(val, f)
#define tiff_word(val, f) write_word(val, f)

#define TIFF_BYTE 1
#define TIFF_ASCII 2
#define TIFF_SHORT 3
#define TIFF_LONG 4
#define TIFF_RATIONAL 5

struct rational_s {
	DWORD numerator;
	DWORD denominator;
};
#define TIFF_RATIONAL_SIZE 8

struct ifd_entry_s {
	WORD tag;
	WORD type;
	DWORD length;
	DWORD value;
};
#define TIFF_IFD_SIZE 12

struct tiff_head_s {
	WORD order;
	WORD version;
	DWORD ifd_offset;
};
#define TIFF_HEAD_SIZE 8

/* Write tiff file from IMAGE.
 * Since this will be used by a DOS EPS file, we write an Intel TIFF file.
 * Include the pixels specified in devbbox, which is in pixel coordinates
 * not points.  If this is empty, the whole image will be used.
 * Resolution of bitmap is xdpi,ydpi.
 * If tiff4 is true, write a monochrome file compatible with TIFF 4,
 * otherwise make it compatible with TIFF 6.
 * If use_packbits is true and tiff4 is false, use packbits to
 * compress the bitmap.
 */
int write_tiff(FILE *f, 
    IMAGE *img, CDSCBBOX devbbox, float xdpi, float ydpi, 
    BOOL tiff4, BOOL use_packbits)
{
#define IFD_MAX_ENTRY 12
    WORD ifd_length;
    DWORD ifd_next;
    DWORD tiff_end, end;
    int i, j;
    unsigned char *preview;
    BYTE *line;
    int temp_bwidth, bwidth;
    BOOL soft_extra = FALSE;
    int width, height;	/* size of preview */
    int bitoffset;
    WORD *comp_length=NULL;	/* lengths of compressed lines */
    BYTE *comp_line=NULL;	/* compressed line buffer */
    int rowsperstrip;
    int stripsperimage;
    int strip, is;
    int strip_len;
    int lastrow;

    int depth;
    int preview_depth;
    int topfirst;

    if (img == NULL)
	return -1;

    topfirst = ((img->format & DISPLAY_FIRSTROW_MASK) == DISPLAY_TOPFIRST);

    /* make sure the pixel coordinates are valid */
    validate_devbbox(img, &devbbox);

    /* calculate dimensions of preview */
    width  = devbbox.urx - devbbox.llx;	/* width  of dest bitmap */
    height = devbbox.ury - devbbox.lly;	/* height of dest bitmap */
    depth = image_depth(img);
    if (depth == 1)
	preview_depth = 1;
    else if (depth == 4)
	preview_depth = 4;
    else if (depth == 8)
	preview_depth = 8;
    else 
	preview_depth = 24;
    if (tiff4)
	preview_depth = 1;

    /* byte width of source bitmap is img->raster */
    /* byte width of intermediate line, after conversion
     * to preview_depth, but before shifting to remove
     * unwanted pixels.
     */
    temp_bwidth = (img->width * preview_depth + 7) >> 3;
    /* byte width of preview */
    bwidth = (width * preview_depth + 7) >> 3;
    /* Number of bits to shift intermediate line to get preview line */
    bitoffset = devbbox.llx * preview_depth;

    if (tiff4)
	rowsperstrip = 1; /* make TIFF 4 very simple */
    else {
	/* work out RowsPerStrip, to give < 8k compressed */
	/* or uncompressed data per strip */
	rowsperstrip = (8192 - 256) / bwidth;
	if (rowsperstrip == 0)
	    rowsperstrip = 1;	/* strips are larger than 8k */
    }
    stripsperimage = (height + rowsperstrip - 1) / rowsperstrip;
    if (stripsperimage == 1)
	rowsperstrip = height;

    preview = (unsigned char *) malloc(img->raster);
    if (preview == NULL)
	return -1;
    memset(preview,0xff,img->raster);

    /* compress bitmap, throwing away result, to find out compressed size */
    if (use_packbits) {
	comp_length = (WORD *)malloc(stripsperimage * sizeof(WORD));
	if (comp_length == NULL) {
	    free(preview);
	    return -1;
	}
	comp_line = (BYTE *)malloc(bwidth + bwidth/64 + 1);
	if (comp_line == NULL) {
	    free(preview);
	    free(comp_length);
	    return -1;
	}
	if (topfirst) 
	    line = img->image + img->raster * (img->height - devbbox.ury);
	else
	    line = img->image + img->raster * (devbbox.ury-1);

	/* process each strip */
	for (strip = 0; strip < stripsperimage; strip++) {
	    is = strip * rowsperstrip;
	    lastrow = min( rowsperstrip, height - is);
	    comp_length[strip] = 0;
	    strip_len = 0;
	    /* process each line within strip */
	    for (i = 0; i< lastrow; i++) {
		if (preview_depth == 1) {
		    memset(preview,0xff,img->raster);
		    image_to_mono(img, preview, line);
		    for (j=0; j<temp_bwidth; j++)
			preview[j] ^= 0xff;
		}
		else if (preview_depth == 24)
		    image_to_24RGB(img, preview, line);
		else if (depth == preview_depth)
		    memmove(preview,  line, img->raster);
		if (bitoffset)
		    shift_preview(preview, temp_bwidth, bitoffset);
		strip_len += packbits(comp_line, preview, bwidth);
		if (topfirst)
		    line += img->raster;
		else
		    line -= img->raster;
	    }
	    comp_length[strip] = (WORD)strip_len;
	}
    }
     

    /* write header */
    tiff_end = TIFF_HEAD_SIZE;
    tiff_word(0x4949, f);	/* Intel = little endian */
    tiff_word(42, f);
    tiff_long(tiff_end, f);

    /* write count of ifd entries */
    tiff_end += 2 /* sizeof(ifd_length) */;
    if (tiff4)
	ifd_length = 10;
    else {
	switch (preview_depth) {
	    case 24:
		/* extras are BitsPerPixel, SamplesPerPixel */
		ifd_length = 15;
		break;
	    case 8:
	    case 4:
		/* extras are BitsPerPixel, ColorMap */
		ifd_length = 15;
		break;
	    default:	/* bi-level */
		ifd_length = 13;
	}
    }
    tiff_word(ifd_length, f);

    tiff_end += ifd_length * TIFF_IFD_SIZE + 4 /* sizeof(ifd_next) */;
    ifd_next = 0;

    /* write each of the ifd entries */
    if (tiff4) {
	tiff_word(0xff, f);	    /* SubfileType */
	tiff_word(TIFF_SHORT, f);  /* value type */
	tiff_long(1, f);		    /* length */
	tiff_short(0, f);		    /* value */
    }
    else {
	tiff_word(0xfe, f);	/* NewSubfileType */
	tiff_word(TIFF_LONG, f);
	tiff_long(1, f);		    /* length */
	tiff_long(0, f);		    /* value */
    }

    tiff_word(0x100, f);	/* ImageWidth */
    if (tiff4) {
	tiff_word(TIFF_SHORT, f);
	tiff_long(1, f);
	tiff_short((short)width, f);
    }
    else {
	tiff_word(TIFF_LONG, f);
	tiff_long(1, f);
	tiff_long(width, f);
    }

    tiff_word(0x101, f);	/* ImageHeight */
    if (tiff4) {
	tiff_word(TIFF_SHORT, f);
	tiff_long(1, f);
	tiff_short((short)height, f);
    }
    else {
	tiff_word(TIFF_LONG, f);
	tiff_long(1, f);
	tiff_long(height, f);
    }

    if (!tiff4 && preview_depth>1) {
	tiff_word(0x102, f);	/* BitsPerSample */
	tiff_word(TIFF_SHORT, f);
	if (preview_depth == 24) {
	    tiff_long(3, f);
	    tiff_long(tiff_end, f);
	    tiff_end += 6;
	}
	else {
	    tiff_long(1, f);
	    tiff_short((WORD)preview_depth, f);
	}
    }

    tiff_word(0x103, f);	/* Compression */
    tiff_word(TIFF_SHORT, f);
    tiff_long(1, f);
    if (use_packbits)
	tiff_short(32773U, f);	/* packbits compression */
    else
	tiff_short(1, f);		/* no compression */

    tiff_word(0x106, f);	/* PhotometricInterpretation */
    tiff_word(TIFF_SHORT, f);
    tiff_long(1, f);
    if (tiff4 || preview_depth==1)
	tiff_short(1, f);		/* black is zero */
    else if (preview_depth==24)
	tiff_short(2, f);		/* RGB */
    else /* preview_depth == 4 or 8 */
	tiff_short(3, f);		/* Palette Color */

    tiff_word(0x111, f);	/* StripOffsets */
    tiff_word(TIFF_LONG, f);
    if (stripsperimage == 1) {
	/* This is messy and fragile */
	int len = 0;
	tiff_long(1, f);
	len += TIFF_RATIONAL_SIZE * 2;		/* resolutions */
	if (!tiff4) {
	    len += (((int)strlen(szAppName)+2)&~1) + 20;	/* software and date */
	    if (preview_depth == 4 || preview_depth == 8)
		len += 2 * 3*(1<<preview_depth);	/* palette */
	}
	tiff_long(tiff_end + len, f);
    }
    else {
	tiff_long(stripsperimage, f);
	tiff_long(tiff_end, f);
	tiff_end += (stripsperimage * 4 /* sizeof(DWORD) */);
    }

    if (!tiff4 && (preview_depth==24)) {
	tiff_word(0x115, f);	/* SamplesPerPixel */
	tiff_word(TIFF_SHORT, f);
	tiff_long(1, f);
	tiff_short(3, f);		/* 3 components */
    }

    tiff_word(0x116, f);	/* RowsPerStrip */
    tiff_word(TIFF_LONG, f);
    tiff_long(1, f);
    tiff_long(rowsperstrip, f);

    tiff_word(0x117, f);	/* StripByteCounts */
    tiff_word(TIFF_LONG, f);
    if (stripsperimage == 1) {
	tiff_long(1, f);
	if (use_packbits)
	    tiff_long(comp_length[0], f);
	else
	    tiff_long(bwidth * rowsperstrip, f);
    }
    else {
	tiff_long(stripsperimage, f);
	tiff_long(tiff_end, f);
	tiff_end += (stripsperimage * 4 /* sizeof(DWORD) */);
    }

    tiff_word(0x11a, f);	/* XResolution */
    tiff_word(TIFF_RATIONAL, f);
    tiff_long(1, f);
    tiff_long(tiff_end, f);
    tiff_end += TIFF_RATIONAL_SIZE;

    tiff_word(0x11b, f);	/* YResolution */
    tiff_word(TIFF_RATIONAL, f);
    tiff_long(1, f);
    tiff_long(tiff_end, f);
    tiff_end += TIFF_RATIONAL_SIZE;

    if (!tiff4) {
	tiff_word(0x128, f);	/* ResolutionUnit */
	tiff_word(TIFF_SHORT, f);
	tiff_long(1, f);
	tiff_short(2, f);		/* inches */

	tiff_word(0x131, f);	/* Software */
	tiff_word(TIFF_ASCII, f);
	i = (int)strlen(szAppName) + 1;
	tiff_long(i, f);
	tiff_long(tiff_end, f);
	tiff_end += i;
	if (tiff_end & 1) { /* pad to word boundary */
	    soft_extra = TRUE;
	    tiff_end++;
	}

	tiff_word(0x132, f);	/* DateTime */
	tiff_word(TIFF_ASCII, f);
	tiff_long(20, f);
	tiff_long(tiff_end, f);
	tiff_end += 20;

	if (preview_depth==4 || preview_depth==8) {
	    int palcount = 1<<preview_depth;
	    tiff_word(0x140, f);	/* ColorMap */
	    tiff_word(TIFF_SHORT, f);
	    tiff_long(3*palcount, f);  /* size of ColorMap */
	    tiff_long(tiff_end, f);
	    tiff_end += 2 * 3*palcount;
	}
    }


    /* write end of ifd tag */
    tiff_long(ifd_next, f);

    /* BitsPerSample for 24 bit colour */
    if (!tiff4 && (preview_depth==24)) {
	tiff_word(8, f);
	tiff_word(8, f);
	tiff_word(8, f);
    }

    /* strip offsets */
    end = tiff_end;
    if (stripsperimage > 1) {
	int stripwidth = bwidth * rowsperstrip;
	for (i=0; i<stripsperimage; i++) {
	    tiff_long(end, f);
	    if (use_packbits)
		end += comp_length[i];
	    else
		end += stripwidth;
	}
    }

    /* strip byte counts (after compression) */
    if (stripsperimage > 1) {
	for (i=0; i<stripsperimage; i++) {
	    if (use_packbits)
		tiff_long(comp_length[i], f);
	    else {
		is = i * rowsperstrip;
		lastrow = min( rowsperstrip, height - is);
		tiff_long(lastrow * bwidth, f);
	    }
	}
    }

    /* XResolution rational */
    tiff_long((int)xdpi, f);
    tiff_long(1, f);
    /* YResolution rational */
    tiff_long((int)ydpi, f);
    tiff_long(1, f);

    /* software and time strings */
    if (!tiff4) {
	time_t t;
	char now[20];
	struct tm* dt;
	fwrite(szAppName, 1, strlen(szAppName)+1, f);
	if (soft_extra)
	    fputc('\0',f);
	t = time(NULL);
	dt = localtime(&t);
	snprintf(now, sizeof(now), "%04d:%02d:%02d %02d:%02d:%02d",
	    dt->tm_year+1900, dt->tm_mon+1, dt->tm_mday,
	    dt->tm_hour, dt->tm_min, dt->tm_sec);
	fwrite(now, 1, 20, f);
    }

    /* Palette */
    if (!tiff4 && ((preview_depth==4) || (preview_depth==8))) {
	int palcount = 1<<preview_depth;
	unsigned char r, g, b;
#define PALVAL(x) ((WORD)((x<< 8) | x))
	for (i=0; i<palcount; i++) {
	    image_colour(img->format, i, &r, &g, &b);
	    tiff_word(PALVAL(r), f);
	}
	for (i=0; i<palcount; i++) {
	    image_colour(img->format, i, &r, &g, &b);
	    tiff_word(PALVAL(g), f);
	}
	for (i=0; i<palcount; i++) {
	    image_colour(img->format, i, &r, &g, &b);
	    tiff_word(PALVAL(b), f);
	}
#undef PALVAL
    }


    if (topfirst) 
	line = img->image + img->raster * (img->height - devbbox.ury);
    else
	line = img->image + img->raster * (devbbox.ury-1);

    /* process each strip of bitmap */
    for (strip = 0; strip < stripsperimage; strip++) {
	int len;
	is = strip * rowsperstrip;
	lastrow = min( rowsperstrip, height - is);
	/* process each row of strip */
	for (i = 0; i < lastrow; i++) {
		if (preview_depth == 1) {
		    memset(preview,0,img->raster);
		    image_to_mono(img, preview, line);
		    for (j=0; j<temp_bwidth; j++)
			preview[j] ^= 0xff;
		}
		else if (preview_depth == 24)
		    image_to_24RGB(img, preview, line);
		else if (depth == preview_depth)
		    memmove(preview,  line, img->raster);
		if (bitoffset)
		    shift_preview(preview, temp_bwidth, bitoffset);
		if (use_packbits) {
		    len = (WORD)packbits(comp_line, preview, bwidth);
		    fwrite(comp_line, 1, len, f);
		}
		else
		    fwrite(preview, 1, bwidth, f);
		if (topfirst)
		    line += img->raster;
		else
		    line -= img->raster;
	}
    }

    if (use_packbits) {
	free(comp_length);
	free(comp_line);
    }
    free(preview);
    return 0;
}

/* FIX: instead of devbbox, perhaps we should use CDSCFBBOX in points,
  or simply grab this from doc. */

/* make a PC EPS file with a TIFF Preview */
/* from a PS file and a bitmap */
int
make_eps_tiff(Doc *doc, IMAGE *img, CDSCBBOX devbbox, 
    CDSCBBOX *bbox, CDSCFBBOX *hires_bbox,
    float xdpi, float ydpi, BOOL tiff4, BOOL use_packbits, LPCTSTR epsname)
{
FILE *epsfile;
FILE *tiff_file;
TCHAR tiffname[MAXSTR];
CDSCDOSEPS doseps;
int code;
FILE *tpsfile;
TCHAR tpsname[MAXSTR];
char *buffer;
unsigned int count;
CDSC *dsc = doc->dsc;

    if (dsc == NULL)
	return -1;

    /* Create TIFF file */
    if ((tiff_file = app_temp_file(doc->app, tiffname, 
	sizeof(tiffname)/sizeof(TCHAR), TEXT("wb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open temporary TIFF file \042%s\042\n"),
	    tiffname);
	return -1;
    }
    code = write_tiff(tiff_file, img, devbbox, xdpi, ydpi, 
	tiff4, use_packbits);
    fclose(tiff_file);
    if (code) {
	app_csmsgf(doc->app, 
	    TEXT("Failed to write temporary TIFF file \042%s\042\n"),
	    tiffname);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tiffname);
	return code;
    }

    /* Create temporary EPS file with updated headers */
    tpsfile = NULL;
    memset(tpsname, 0, sizeof(tpsname));
    if ((tpsfile = app_temp_file(doc->app, tpsname, 
	sizeof(tpsname)/sizeof(TCHAR), TEXT("wb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't create temporary EPS file \042%s\042\n"),
	    tpsname);
	csunlink(tiffname);
	return -1;
    }
    fclose(tpsfile);

    code = copy_eps(doc, tpsname, bbox, hires_bbox, 
	DOSEPS_HEADER_SIZE, FALSE); 
    if (code) {
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tiffname);
	return -1;
    }


    if ( (tpsfile = csfopen(tpsname, TEXT("rb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open temporary EPS file \042%s\042\n"),
	    tpsname);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tiffname);
	return -1;
    }

    /* Create DOS EPS output file */
    if (*epsname!='\0')
	epsfile = csfopen(epsname,TEXT("wb"));
    else
	epsfile = stdout;
    if (epsfile == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open output EPS file \042%s\042\n"),
	    epsname);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tiffname);
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    doseps.ps_begin = DOSEPS_HEADER_SIZE;
    fseek(tpsfile, 0, SEEK_END);
    doseps.ps_length = ftell(tpsfile);
    doseps.wmf_begin = 0;
    doseps.wmf_length = 0;
    doseps.tiff_begin = doseps.ps_begin + doseps.ps_length;

    buffer = (char *)malloc(COPY_BUF_SIZE);
    if (buffer == (char *)NULL) {
	if (epsname[0]) {
	    fclose(epsfile);
	    if (!(debug & DEBUG_GENERAL))
		csunlink(epsname);
	}
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tiffname);
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    if ((tiff_file = csfopen(tiffname,TEXT("rb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open temporary TIFF file \042%s\042\n"),
	    tiffname);
	free(buffer);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tiffname);
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	if (epsname[0]) {
	    fclose(epsfile);
	if (!(debug & DEBUG_GENERAL))
		csunlink(epsname);
	}
	return -1;
    }
    fseek(tiff_file, 0, SEEK_END);
    doseps.tiff_length = ftell(tiff_file);
    doseps.checksum = 0xffff;
    write_doseps_header(&doseps, epsfile);

    /* copy EPS file */
    rewind(tpsfile);
    while ((count = (int)fread(buffer, 1, COPY_BUF_SIZE, tpsfile)) != 0)
	fwrite(buffer, 1, count, epsfile);

    /* copy tiff file */
    rewind(tiff_file);
    while ((count = (int)fread(buffer, 1, COPY_BUF_SIZE, tiff_file)) != 0)
	fwrite(buffer, 1, count, epsfile);

    free(buffer);
    fclose(tiff_file);
    if (!(debug & DEBUG_GENERAL))
	csunlink(tiffname);
    fclose(tpsfile);
    if (!(debug & DEBUG_GENERAL))
	csunlink(tpsname);
    if (epsname[0])
       fclose(epsfile);
    return 0;
}

/*********************************************************/

static char hex[17] = "0123456789ABCDEF";

/* Write interchange preview to file f */
/* Does not copy the file itself */
int
write_interchange(FILE *f, IMAGE *img, CDSCBBOX devbbox)
{
    int i, j;
    unsigned char *preview;
    BYTE *line;
    int preview_width, bwidth;
    int lines_per_scan;
    int topfirst = ((img->format & DISPLAY_FIRSTROW_MASK) == DISPLAY_TOPFIRST);
    
    validate_devbbox(img, &devbbox);

    /* byte width of interchange preview with 1 bit/pixel */
    bwidth = (((devbbox.urx-devbbox.llx) + 7) & ~7) >> 3;
    /* byte width of intermediate preview with 1 bit/pixel */
    preview_width = ((img->width + 7) & ~7) >> 3;

    preview = (unsigned char *) malloc(preview_width);

    lines_per_scan = ((bwidth-1) / 32) + 1;
    fprintf(f,"%%%%BeginPreview: %u %u 1 %u",(devbbox.urx-devbbox.llx), (devbbox.ury-devbbox.lly), 
	(devbbox.ury-devbbox.lly)*lines_per_scan);
    fputs(EOLSTR, f);

    if (topfirst) 
	line = img->image + img->raster * (img->height - devbbox.ury);
    else
	line = img->image + img->raster * (devbbox.ury-1);

    /* process each line of bitmap */
    for (i = 0; i < (devbbox.ury-devbbox.lly); i++) {
	memset(preview,0xff,preview_width);
	image_to_mono(img, preview, line);
	if (devbbox.llx)
	    shift_preview(preview, preview_width, devbbox.llx);
	fputs("% ",f);
	for (j=0; j<bwidth; j++) {
	    if (j && ((j & 31) == 0)) {
		fputs(EOLSTR, f);
		fputs("% ",f);
	    }
	    fputc(hex[15-((preview[j]>>4)&15)],f);
	    fputc(hex[15-((preview[j])&15)],f);
	}
	fputs(EOLSTR, f);
	if (topfirst)
	    line += img->raster;
	else
	    line -= img->raster;

    }

    fputs("%%EndPreview",f);
    fputs(EOLSTR, f);
    free(preview);

    return 0;
}

/*********************************************************/

typedef enum PREVIEW_TYPE_e {
    PREVIEW_UNKNOWN = 0,
    PREVIEW_TIFF = 1,
    PREVIEW_WMF = 2
} PREVIEW_TYPE;

/* Make a DOS EPS file from a PS file and a user supplied preview.
 * Preview may be WMF or TIFF.
 * Returns 0 on success.
 */
int
make_eps_user(Doc *doc, LPCTSTR preview_name, LPCTSTR epsname)
{
FILE *epsfile;
FILE *preview_file;
unsigned long preview_length;
unsigned char id[4];
PREVIEW_TYPE type = PREVIEW_UNKNOWN;
CDSCDOSEPS doseps;
char *buffer;
unsigned int count;
FILE *tpsfile;
TCHAR tpsname[MAXSTR];

    if ((preview_name == NULL) || preview_name[0] == '\0')
	return -1;
    if (doc->dsc == NULL)
	return -1;

    /* open preview, determine length and type */
    preview_file = csfopen(preview_name, TEXT("rb"));
    if (preview_file == (FILE *)NULL) {
	app_csmsgf(doc->app, TEXT("Can't open preview file \042%s\042\n"),
	    preview_name);
	return -1;
    }

    /* Determine type of preview */
    id[0] = (unsigned char)fgetc(preview_file);
    id[1] = (unsigned char)fgetc(preview_file);
    id[2] = (unsigned char)fgetc(preview_file);
    id[3] = (unsigned char)fgetc(preview_file);
    fseek(preview_file, 0, SEEK_END);
    preview_length = ftell(preview_file);
    fseek(preview_file, 0, SEEK_SET);

    if ((id[0] == 'I') && (id[1] == 'I'))
	type = PREVIEW_TIFF;
    if ((id[0] == 'M') && (id[1] == 'M'))
	type = PREVIEW_TIFF;
    if ((id[0] == 0x01) && (id[1] == 0x00) && 
	(id[2] == 0x09) && (id[3] == 0x00))
	type = PREVIEW_WMF;
    if ((id[0] == 0xd7) && (id[1] == 0xcd) && 
	(id[2] == 0xc6) && (id[3] == 0x9a)) {
	type = PREVIEW_WMF;
	preview_length -= 22;	/* skip over placeable metafile header */
	fseek(preview_file, 22, SEEK_SET);
    }

    if (type == PREVIEW_UNKNOWN) {
	app_csmsgf(doc->app, 
	 TEXT("Preview file \042%s\042 is not TIFF or Windows Metafile\n"),
	 preview_name);
	fclose(preview_file);
	return -1;
    }

    /* Create temporary EPS file containing all that is needed
     * including updated header and trailer.
     */
    tpsfile = NULL;
    memset(tpsname, 0, sizeof(tpsname));
    if ((tpsfile = app_temp_file(doc->app, tpsname, 
	sizeof(tpsname)/sizeof(TCHAR), TEXT("wb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't create temporary EPS file \042%s\042\n"),
	    tpsname);
	fclose(preview_file);
	return -1;
    }

    if (copy_eps(doc, tpsname, doc->dsc->bbox, doc->dsc->hires_bbox, 
	DOSEPS_HEADER_SIZE, FALSE) < 0)
	return -1; 


    if ( (tpsfile = csfopen(tpsname, TEXT("rb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open temporary EPS file \042%s\042\n"),
	    tpsname);
	fclose(preview_file);
	return -1;
    }


    /* Create EPS output file */
    if (*epsname!='\0')
	epsfile = csfopen(epsname,TEXT("wb"));
    else
	epsfile = stdout;

    if (epsfile == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open EPS output file \042%s\042\n"),
	    epsname);
	fclose(preview_file);
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    /* write DOS EPS binary header */
    doseps.ps_begin = DOSEPS_HEADER_SIZE;
    fseek(tpsfile, 0, SEEK_END);
    doseps.ps_length = ftell(tpsfile);
    if (type == PREVIEW_WMF) {
	doseps.wmf_begin = doseps.ps_begin + doseps.ps_length;
	doseps.wmf_length = preview_length;
	doseps.tiff_begin = 0;
	doseps.tiff_length = 0;
    }
    else {
	doseps.wmf_begin = 0;
	doseps.wmf_length = 0;
	doseps.tiff_begin = doseps.ps_begin + doseps.ps_length;
	doseps.tiff_length = preview_length;
    }
    doseps.checksum = 0xffff;
    write_doseps_header(&doseps, epsfile);

    /* copy preview file */
    buffer = (char *)malloc(COPY_BUF_SIZE);
    if (buffer == (char *)NULL) {
	app_csmsgf(doc->app, TEXT("Out of memory in make_eps_user\n"));
	fclose(epsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(epsname);
	fclose(preview_file);
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    /* copy EPS file */
    rewind(tpsfile);
    while ((count = (int)fread(buffer, 1, COPY_BUF_SIZE, tpsfile)) != 0)
	fwrite(buffer, 1, count, epsfile);
    
    /* copy preview file */
    rewind(preview_file);
    while ( (count = (int)fread(buffer, 1, COPY_BUF_SIZE, preview_file)) != 0 )
	fwrite(buffer, 1, count, epsfile);

    free(buffer);
    fclose(tpsfile);
    if (!(debug & DEBUG_GENERAL))
	csunlink(tpsname);
    fclose(preview_file);
    if (*epsname!='\0')
       fclose(epsfile);
    return 0; /* success */
}


/*********************************************************/

typedef struct tagMFH {
    WORD type;
    WORD headersize;
    WORD version;
    DWORD size;
    WORD nobj;
    DWORD maxrec;
    WORD noparam;
} MFH;

int write_metafile(FILE *f, IMAGE *img, CDSCBBOX devbbox, 
    float xdpi, float ydpi, MFH *mf);
static int metafile_init(IMAGE *img, CDSCBBOX *pdevbbox, MFH* mf);

/* A metafile object must not be larger than 64k */
/* Metafile bitmap object contains metafile header, */
/* bitmap header, palette and bitmap bits */
#define MAX_METAFILE_BITMAP 64000L	/* max size of bitmap bits */

static int
metafile_init(IMAGE *img, CDSCBBOX *pdevbbox, MFH* mf)
{
int wx, wy;
int ny, nylast;
int complete, partial;
int bytewidth;
int palcount;
unsigned long size;
int depth = image_depth(img);

    switch (depth) {
	case 1:
	case 4:
	case 8:
	case 24:
	    break;
	default:
	    /* unsupported format */
	    return -1;
    }

    validate_devbbox(img, pdevbbox);

    wx = pdevbbox->urx - pdevbbox->llx;
    wy = pdevbbox->ury - pdevbbox->lly;
    bytewidth = (( wx * depth + 31) & ~31) >> 3;
    ny = (int)(MAX_METAFILE_BITMAP / bytewidth);
    if (depth == 24)
	palcount = 0;
    else
	palcount = 1<<depth;

    complete = wy / ny;
    nylast = wy % ny;
    partial = nylast ? 1 : 0;
    
    mf->type = 1;		/* metafile in file */
    mf->headersize = 9;		/* 9 WORDs */
    mf->version = 0x300;	/* Windows 3.0 */
    mf->size = 			/* sizes in WORDs */
	9UL +			/* header */
	5 +			/* SetWindowOrg */
	5; 			/* SetWindowExt */
    /* complete StretchDIBits */
    mf->size += 14*complete;
    size = (40L + palcount*4L + (unsigned long)ny*(unsigned long)bytewidth)/2L;
    mf->size += size * (unsigned long)complete;
    /* partial StretchDIBits */
    mf->size += 14*partial;
    size = (40L + palcount*4L + (unsigned long)nylast*(unsigned long)bytewidth)/2L;
    mf->size += size * (unsigned long)partial;
    mf->size += 3;			/* end marker */

    mf->nobj = 0;
    size = complete ? 
       (40L + palcount*4L + (unsigned long)ny*(unsigned long)bytewidth)/2L
     : (40L + palcount*4L + (unsigned long)nylast*(unsigned long)bytewidth)/2L;
    mf->maxrec = 14L + size;
    mf->noparam = 0;
    return 0;
}



/* convert the display bitmap to a metafile picture */
int
write_metafile(FILE *f, IMAGE *img, CDSCBBOX devbbox, 
    float xdpi, float ydpi, MFH *mf)
{
    int i;
    int wx;
    int ny, sy, dy, wy;
    BYTE *line;
    BYTE *line2;
    LPBITMAP2 pbmi;
    int bsize;
    int bitoffset;
    int bytewidth, activewidth;
    int palcount;
    unsigned long size;
    int depth = image_depth(img);
    int topfirst = ((img->format & DISPLAY_FIRSTROW_MASK) == DISPLAY_TOPFIRST);

    dy = 0;
    wx = devbbox.urx - devbbox.llx;
    sy = devbbox.lly;
    wy = devbbox.ury - devbbox.lly;
    bitoffset = (devbbox.llx * depth);
    bytewidth = (( wx * depth + 31) & ~31) >> 3;
    activewidth = (( wx * depth + 7) & ~7) >> 3;
    ny = (int)(MAX_METAFILE_BITMAP / bytewidth);
    if (depth == 24)
	palcount = 0;
    else
	palcount = 1<<depth;

    /* create and initialize a BITMAPINFO for the <64k bitmap */
    bsize = sizeof(BITMAP2) + palcount * RGB4_LENGTH; 
    pbmi = (LPBITMAP2)malloc(bsize);
    if (pbmi == NULL) {
       return -1;
    }
    memset((char *)pbmi, 0, bsize);
    make_bmp_info(pbmi, img, xdpi, ydpi);
    pbmi->biClrUsed = 0;		/* write out full palette */
    pbmi->biClrImportant = 0;

    line2 = (BYTE *)malloc(img->raster);
    if (line2 == (BYTE *)NULL)
       return -1;
    pbmi->biWidth = wx;

    if (topfirst) 
	line = img->image + img->raster * (img->height - devbbox.lly - 1);
    else
	line = img->image + img->raster * devbbox.lly;

    /* write metafile header */
    write_word(mf->type, f);
    write_word(mf->headersize, f);
    write_word(mf->version, f);
    write_dword(mf->size, f);
    write_word(mf->nobj, f);
    write_dword(mf->maxrec, f);
    write_word(mf->noparam, f);

    /* write SetWindowOrg */
    write_dword(5, f);
    write_word(0x20b, f);
    write_word(0, f);
    write_word(0, f);

    /* write SetWindowExt */
    write_dword(5, f);
    write_word(0x20c, f);
    write_word((WORD)wy, f);
    write_word((WORD)wx, f);

    /* copy in chunks < 64k */
    for ( ; wy > ny; dy += ny, wy -= ny, sy += ny ) {
	pbmi->biHeight = ny;

	size = bsize + (unsigned long)bytewidth * (unsigned long)ny;
	/* write StretchDIBits header */
	write_dword(14 + size/2L, f);
	write_word(0x0f43, f);
	write_dword(0x00cc0020L, f);	/* SRC_COPY */
	write_word(0, f);			/* DIB_RGB_COLORS */
	write_word((WORD)ny, f);		/* Source cy */
	write_word((WORD)wx, f);		/* Source cx */
	write_word(0, f);			/* Source y */
	write_word(0, f);			/* Source x */
	write_word((WORD)ny, f);		/* Dest   cy */
	write_word((WORD)wx, f);		/* Dest   cx */
	write_word((WORD)(devbbox.ury - devbbox.lly - ny - dy), f); 
						/* Dest   y */
	write_word(0, f);			/* Dest   x */

	/* write bitmap header */
	write_bitmap_info(img, pbmi, f);

	/* write bitmap rows */
	for (i=0; i<ny; i++) {
	    if (depth == 24)
		image_to_24BGR(img, line2, line);
	    else
		memmove(line2, line, img->raster);
	    shift_preview(line2, img->raster, bitoffset);
	    if (activewidth < bytewidth)
		memset(line2+activewidth, 0xff, bytewidth-activewidth);
	    fwrite(line2, 1, bytewidth, f);
	    if (topfirst)
		line -= img->raster;
	    else
		line += img->raster;
	}
	
    }

    /* write StretchDIBits header */
    pbmi->biHeight = wy;
    size = bsize + (unsigned long)bytewidth * (unsigned long)wy;
    write_dword(14 + size/2L, f);
    write_word(0x0f43, f);
    write_dword(0x00cc0020L, f);	/* SRC_COPY */
    write_word(0, f);		/* DIB_RGB_COLORS */
    write_word((WORD)wy, f);	/* Source cy */
    write_word((WORD)wx, f);	/* Source cx */
    write_word(0, f);		/* Source y */
    write_word(0, f);		/* Source x */
    write_word((WORD)wy, f);	/* Dest   cy */
    write_word((WORD)wx, f);	/* Dest   cx */
    write_word((WORD)(devbbox.ury - devbbox.lly - wy - dy), f);	/* Dest   y */
    write_word(0, f);		/* Dest   x */

    /* write bitmap header */
    write_bitmap_info(img, pbmi, f);

    /* copy last chunk */
    for (i=0; i<wy; i++) {
	if (depth == 24)
	    image_to_24BGR(img, line2, line);
	else
	    memmove(line2, line, img->raster);
	shift_preview(line2, img->raster, bitoffset);
	if (activewidth < bytewidth)
	    memset(line2+activewidth, 0xff, bytewidth-activewidth);
	fwrite(line2, 1, bytewidth, f);
	if (topfirst)
	    line -= img->raster;
	else
	    line += img->raster;
    }

    /* write end marker */
    write_dword(3, f);
    write_word(0, f);

    free((char *)pbmi);
    free(line2);

    return 0;
}


/*********************************************************/

/* Copy a DSC section, removing existing bounding boxes */
void
copy_nobbox(FILE *outfile, FILE *infile, 
    unsigned long begin, unsigned long end)
{
    const char bbox_str[] = "%%BoundingBox:";
    const char hiresbbox_str[] = "%%HiResBoundingBox:";
    char buf[DSC_LINE_LENGTH+1];
    int len;
    fseek(infile, begin, SEEK_SET);
    begin = ftell(infile);
    while (begin < end) {
	len = ps_fgets(buf, sizeof(buf)-1, infile);
	begin = ftell(infile);
	if (len == 0) {
	    return;	/* EOF on input file */
	}
	else if (strncmp(buf, bbox_str, strlen(bbox_str)) == 0) {
	    /* skip it */
	}
	else if (strncmp(buf, hiresbbox_str, strlen(hiresbbox_str)) == 0) {
	    /* skip it */
	}
	else
	    fwrite(buf, 1, len, outfile);
    }
}

/* Copy a DSC header, removing existing bounding boxes 
 * and adding new ones.
 */
void
copy_bbox_header(FILE *outfile, 
    FILE *infile, unsigned long begin, unsigned long end,
    CDSCBBOX *bbox, CDSCFBBOX *hiresbbox)
{
    char buf[DSC_LINE_LENGTH+1];
    int len;

    memset(buf, 0, sizeof(buf)-1);
    fseek(infile, begin, SEEK_SET);
    len = ps_fgets(buf, sizeof(buf)-1, infile);
    if (len)
	fwrite(buf, 1, len, outfile);	/* copy version line */
    /* Add bounding box lines */
    if (bbox) {
	fprintf(outfile, "%%%%BoundingBox: %d %d %d %d\n",
	    bbox->llx, bbox->lly, bbox->urx, bbox->ury);
    }
    if (hiresbbox) {
	fprintf(outfile, "%%%%HiResBoundingBox: %g %g %g %g\n",
	    hiresbbox->fllx, hiresbbox->flly, 
	    hiresbbox->furx, hiresbbox->fury);
    }

    begin = ftell(infile);
    copy_nobbox(outfile, infile, begin, end);
}

/* return the length of the line less the EOL characters */
static int
without_eol(const char *str, int length)
{
    int j;
    for (j=length-1; j>=0; j--) {
	if (!((str[j] == '\r') || (str[j] == '\n'))) {
	    j++;
	    break;
	}
    }
    if (j < 0)
	j = 0;
    return j;
}

static BOOL is_process_colour(const char *name)
{
    return ( 
	(dsc_stricmp(name, "Cyan")==0) ||
    	(dsc_stricmp(name, "Magenta")==0) ||
    	(dsc_stricmp(name, "Yellow")==0) ||
    	(dsc_stricmp(name, "Black")==0)
	);
}

static const char process_str[] = "%%DocumentProcessColors:";
static const char custom_str[] = "%%DocumentCustomColors:";

/* Find separation begin and end */
static int
dcs2_separation(CDSC *dsc, int pagenum, 
    const char **fname, unsigned long *pbegin, unsigned long *pend)
{
    FILE *f;
    *fname = dsc_find_platefile(dsc, pagenum);
    if (*fname) {
	if ((f = fopen(*fname, "rb")) != (FILE *)NULL) {
	    fseek(f, 0, SEEK_END);
	    *pbegin = 0;
	    *pend = ftell(f);
	    fclose(f);
	}
	else {
	    /* Separation file didn't exist */
	    *pbegin = *pend = 0;
	    return 1;
	}
    }
    else {
	*pbegin = dsc->page[pagenum].begin;
	*pend = dsc->page[pagenum].end;
    }
    return 0;
}


static int 
fix_custom(Doc *doc, char *buf, int buflen)
{
    /* Create a new %%DocumentCustomColors:
     * containing only those separations that exist
     */
    CDSC *dsc = doc->dsc;
    int i;
    int missing;
    int n = 0;
    unsigned long begin, end;
    const char *fname;
    int count = min((int)strlen(buf), buflen);
    if ((dsc == NULL) || !dsc->dcs2)
	return count;
    strncpy(buf, custom_str, buflen);
    count = strlen(buf);
    for (i=1; i<(int)dsc->page_count; i++) {
	missing = dcs2_separation(dsc, i, &fname, &begin, &end);
	if (!missing && !is_process_colour(dsc->page[i].label)) {
	    n++;
	    strncpy(buf+count, " (", buflen-count-1);
	    count = strlen(buf);
	    strncpy(buf+count, dsc->page[i].label, buflen-count-1);
	    count = strlen(buf);
	    strncpy(buf+count, ")", buflen-count-1);
	    count = strlen(buf);
	}
    }
    if (n == 0)
	count = 0;
    return count;
}

static int
fix_process(Doc *doc, char *buf, int buflen)
{
    /* Create a new %%DocumentProcessColors:
     * containing only those separations that exist
     */
    CDSC *dsc = doc->dsc;
    int i;
    int n = 0;
    int missing;
    unsigned long begin, end;
    const char *fname;
    int count = min((int)strlen(buf), buflen);
    if ((dsc == NULL) || !dsc->dcs2)
	return count;
    strncpy(buf, process_str, buflen);
    count = strlen(buf);
    for (i=1; i<(int)dsc->page_count; i++) {
	missing = dcs2_separation(dsc, i, &fname, &begin, &end);
	if (!missing && is_process_colour(dsc->page[i].label)) {
	    n++;
	    strncpy(buf+count, " ", buflen-count-1);
	    count++;
	    strncpy(buf+count, dsc->page[i].label, buflen-count-1);
	    count = strlen(buf);
	}
    }
    if (n == 0)
	count = 0;
    return count;
}

/* Copy a DCS 2.0 file.
 * DSC 2.0 as single file looks like concatenated EPS files.
 * DSC parser treats these as separate pages and the
 * entire first EPS file is contained in the first page.
 * That is, there is no header, prolog or trailer.
 * Don't update the bounding box.
 * Do update the %%PlateFile comments.
 * If missing_separations is true, remove the names of missing
 * separations from the DSC comments.
 * The length of the composite page is returned in complen.
 * If composite is not NULL, use this file as the new composite page
 * with the existing header.
 */
int
copy_dcs2(Doc *doc, FILE *docfile, FILE *epsfile, LPCTSTR epsname,
    int offset, BOOL dcs2_multi, BOOL write_all, BOOL missing_separations,
    unsigned long *complen, FILE *composite)
{
    const char platefile_str[] = "%%PlateFile:";
    const char cyanplate_str[] = "%%CyanPlate:";
    const char magentaplate_str[] = "%%MagentaPlate:";
    const char yellowplate_str[] = "%%YellowPlate:";
    const char blackplate_str[] = "%%BlackPlate:";
    const char endcomments_str[] = "%%EndComments";
    char buf[DSC_LINE_LENGTH+1];
    char platename[MAXSTR];
    CDSC *dsc = doc->dsc;
    int i;
    unsigned long len;
    unsigned long file_offset = *complen;
    unsigned long begin, end;
    unsigned long header_position;
    BOOL found_endheader = FALSE;
    BOOL ignore_continuation = FALSE;
    int count;
    FILE *f;
    
    if (dsc->page_count == 0)
	return -1;

    fseek(docfile, dsc->page[0].begin, SEEK_SET);
    memset(buf, 0, sizeof(buf)-1);
    count = ps_fgets(buf, sizeof(buf)-1, docfile);
    header_position = ftell(docfile);
    if (count) { 
	/* copy version line */
	fwrite(buf, 1, without_eol(buf, count), epsfile);
	fputs(EOLSTR, epsfile);		/* change EOL */
    }
    while ((count = ps_fgets(buf, sizeof(buf)-1, docfile)) != 0) {
	header_position = ftell(docfile);
	/* check if end of header */
        if (count < 2)
	    found_endheader = TRUE;
	if ((buf[0] != '%') || (buf[1] != '%'))
	    found_endheader = TRUE;
	if (strncmp(buf, endcomments_str, strlen(endcomments_str)) == 0)
	    found_endheader = TRUE;
	if (strncmp(buf, "%%Begin", 7) == 0)
	    found_endheader = TRUE;
	if (found_endheader)
	    break;	/* write out count characters from buf later */
	if ((buf[0] == '%') && (buf[1] == '%') && (buf[2] == '+') &&
	    ignore_continuation)
	    continue;
	else
	    ignore_continuation = FALSE;
	if ((strncmp(buf, platefile_str, strlen(platefile_str)) != 0) &&
	     (strncmp(buf, cyanplate_str, strlen(cyanplate_str)) != 0) &&
	     (strncmp(buf, magentaplate_str, strlen(magentaplate_str)) != 0) &&
	     (strncmp(buf, yellowplate_str, strlen(yellowplate_str)) != 0) &&
	     (strncmp(buf, blackplate_str, strlen(blackplate_str)) != 0)) {
	    /* Write all header lines except for DCS plate lines */
	    if ((strncmp(buf, custom_str, strlen(custom_str)) == 0)
	 	&& missing_separations) {
		count = fix_custom(doc, buf, sizeof(buf));
		ignore_continuation = TRUE;
	    }
	    else if ((strncmp(buf, process_str, strlen(process_str)) == 0)
	 	&& missing_separations) {
		count = fix_process(doc, buf, sizeof(buf));
		ignore_continuation = TRUE;
	    }
	    if (count == 0)
		continue;
	    fwrite(buf, 1, without_eol(buf, count), epsfile);
	    fputs(EOLSTR, epsfile);		/* change EOL */
	}
    }

    /* Now write the platefile comments */
    for (i=1; i<(int)dsc->page_count; i++) {
	/* First find length of separation */
	int missing;
	const char *fname = NULL;
	missing = dcs2_separation(dsc, i, &fname, &begin, &end);
	len = end - begin;
	/* Now write out platefile */
	if (missing && missing_separations) {
	    if (debug & DEBUG_GENERAL)
		app_msgf(doc->app, 
		    "Skipping missing separation page %d \042%s\042\n",
		    i, dsc->page[i].label);
	}
	else if (dcs2_multi) {
	    memset(platename, 0, sizeof(platename));
	    cs_to_narrow(platename, sizeof(platename)-1, epsname, 
		(int)cslen(epsname)+1);
	    strncat(platename, ".", sizeof(platename) - strlen(platename));
	    strncat(platename, dsc->page[i].label, 
		sizeof(platename) - strlen(platename));
	    fprintf(epsfile, "%%%%PlateFile: (%s) EPS Local %s",
		dsc->page[i].label, platename);
	    fputs(EOLSTR, epsfile);
	    if (write_all) {
		/* Write out multi file separations */
		FILE *pf = fopen(platename, "wb");
		if (pf != (FILE *)NULL) {
		    const char *fname = dsc_find_platefile(dsc, i);
		    if (fname) {
			if ((f = fopen(fname, "rb")) != (FILE *)NULL) {
			    ps_copy(pf, f, begin, end);
			    fclose(f);
			}
		    }
		    else {
			ps_copy(pf, docfile, begin, end);
		    }
		    fclose(pf);
		}
		else {
		    return -1;
		}
	    }
	}
	else {
	    fprintf(epsfile, "%%%%PlateFile: (%s) EPS #%010lu %010lu",
		dsc->page[i].label, offset+file_offset, len);
	    fputs(EOLSTR, epsfile);
	    file_offset += len;
	}
    }
    /* Copy last line of header */
    if (found_endheader) {
	fwrite(buf, 1, without_eol(buf, count), epsfile);
	fputs(EOLSTR, epsfile);		/* change EOL */
    }
    /* copy rest of composite */
    if (composite) {
	fseek(composite, 0, SEEK_END);
	end = ftell(composite);
	fseek(composite, 0, SEEK_SET);
	fputs("%%BeginDocument: composite", epsfile);
	fputs(EOLSTR, epsfile);
	ps_copy(epsfile, composite, 0, end);
	fputs("%%EndDocument", epsfile);
	fputs(EOLSTR, epsfile);
	fputs("%%Trailer", epsfile);
	fputs(EOLSTR, epsfile);
    }
    else {
	begin = header_position;
	end = dsc->page[0].end;
	ps_copy(epsfile, docfile, begin, end);
    }

    file_offset = ftell(epsfile);
    *complen = file_offset;

    /* Write out single file separations */
    if (write_all && !dcs2_multi) {
	for (i=1; i<(int)dsc->page_count; i++) {
	    int missing;
	    const char *fname = NULL;
	    missing = dcs2_separation(dsc, i, &fname, &begin, &end);
	    len = end - begin;
	    if (missing && missing_separations) {
		if (debug & DEBUG_GENERAL)
		    app_msgf(doc->app, 
			"Skipping missing separation page %d \042%s\042\n",
			i, dsc->page[i].label);
	    }
	    else if (fname) {
		if ((f = fopen(fname, "rb")) != (FILE *)NULL) {
		    begin = 0;
		    fseek(f, 0, SEEK_END);
		    end = ftell(f);
		    if (debug & DEBUG_GENERAL)
			app_msgf(doc->app, 
			    "Copying page %d from \042%s\042  %ld %ld\n",
			    i, fname, begin, end);
		    ps_copy(epsfile, f, begin, end);
		    fclose(f);
		}
	    }
	    else {
		begin = dsc->page[i].begin;
		end = dsc->page[i].end;
		if (debug & DEBUG_GENERAL)
		    app_msgf(doc->app, 
			"Copying page %d  %ld %ld\n",
			i, fname, begin, end);
		ps_copy(epsfile, docfile, begin, end);
	    }
	}
    }
    return 0;
}


/* Copy an EPS file.
 * %%BoundingBox and %%HiResBoundingBox will be brought to
 * the start of the header.
 * The new EPS file will have a prolog of "offset" bytes,
 * so update DCS 2.0 offsets accordingly.
 */
int
copy_eps(Doc *doc, LPCTSTR epsname, CDSCBBOX *bbox, CDSCFBBOX *hires_bbox, 
    int offset, BOOL dcs2_multi)
{
    FILE *docfile;
    FILE *f = NULL;
    CDSC *dsc = doc->dsc;
    int code = 0;

    docfile = csfopen(doc_name(doc), TEXT("rb"));
    if (docfile == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open document file \042%s\042\n"), doc_name(doc));
	return -1;
    }

    f = csfopen(epsname, TEXT("wb"));
    if (f == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open EPS file \042%s\042\n"), epsname);
	fclose(docfile);
	return -1;
    }

    if (dsc->dcs2) {
	/* Write DCS 2.0, updating the %%PlateFile, but don't update
	 * %%BoundingBox.
	 */
	unsigned long complen = 0;	/* length of composite page */
	/* Write once to calculate the offsets */
	code = copy_dcs2(doc, docfile, f, epsname, 
	    offset, dcs2_multi, FALSE, FALSE, &complen, NULL);
	rewind(docfile);
	fclose(f);
	f = NULL;
	if (code == 0) {
	    f = csfopen(epsname, TEXT("wb"));
	    if (f == (FILE *)NULL) {
		app_csmsgf(doc->app, 
		    TEXT("Can't open EPS file \042%s\042\n"), epsname);
		fclose(docfile);
		return -1;
	    }
	}
	if (code == 0) {
	    /* Then again with the correct offsets */
	    rewind(f);
	    code = copy_dcs2(doc, docfile, f, epsname,
		offset, dcs2_multi, TRUE, FALSE, &complen, NULL);
	}
    }
    else {
	/* Update the bounding box in the header and remove it from
	 * the trailer
	 */
	copy_bbox_header(f, docfile, 
	    dsc->begincomments, dsc->endcomments,
	    bbox, hires_bbox);
	ps_copy(f, docfile, dsc->begindefaults, dsc->enddefaults);
	ps_copy(f, docfile, dsc->beginprolog, dsc->endprolog);
	ps_copy(f, docfile, dsc->beginsetup, dsc->endsetup);
	if (dsc->page_count)
	    ps_copy(f, docfile, dsc->page[0].begin, dsc->page[0].end);
	copy_nobbox(f, docfile, dsc->begintrailer, dsc->endtrailer);
    }
    if (f)
	fclose(f);
    fclose(docfile);
    return code;
}



/*********************************************************/

/* make an EPSI file with an Interchange Preview */
/* from a PS file and a bitmap */
int
make_eps_interchange(Doc *doc, IMAGE *img, CDSCBBOX devbbox, 
    CDSCBBOX *bbox, CDSCFBBOX *hires_bbox, LPCTSTR epsname)
{
    FILE *epsfile;
    FILE *docfile;
    int code;
    CDSC *dsc = doc->dsc;

    if (dsc == NULL)
	return -1;

    if ( (docfile = csfopen(doc_name(doc), TEXT("rb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open EPS file \042%s\042\n"),
	    doc_name(doc));
	return -1;
    }

    if (*epsname!='\0')
	epsfile = csfopen(epsname, TEXT("wb"));
    else
	epsfile = stdout;

    if (epsfile == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open output EPS file \042%s\042\n"),
	    epsname);
	fclose(docfile);
	return -1;
    }

    /* adjust %%BoundingBox: and %%HiResBoundingBox: comments */
    copy_bbox_header(epsfile, docfile, dsc->begincomments, dsc->endcomments, 
	bbox, hires_bbox);

    code = write_interchange(epsfile, img, devbbox);

    ps_copy(epsfile, docfile, dsc->begindefaults, dsc->enddefaults);
    ps_copy(epsfile, docfile, dsc->beginprolog, dsc->endprolog);
    ps_copy(epsfile, docfile, dsc->beginsetup, dsc->endsetup);
    if (dsc->page_count)
	ps_copy(epsfile, docfile, dsc->page[0].begin, dsc->page[0].end);
    copy_nobbox(epsfile, docfile, dsc->begintrailer, dsc->endtrailer);
    fclose(docfile);
    if (*epsname!='\0') {
	fclose(epsfile);
	if (code && (!(debug & DEBUG_GENERAL)))
	    csunlink(epsname);
    }
    return code;
}


/*********************************************************/


/* make a PC EPS file with a Windows Metafile Preview */
/* from a PS file and a bitmap */
int
make_eps_metafile(Doc *doc, IMAGE *img, CDSCBBOX devbbox, CDSCBBOX *bbox, 
    CDSCFBBOX *hires_bbox, float xdpi, float ydpi, LPCTSTR epsname)
{
    MFH mf;
    FILE *epsfile;
    FILE *tpsfile;
    TCHAR tpsname[MAXSTR];
    int code;
    int count;
    char *buffer;
    CDSCDOSEPS doseps;

    /* prepare metafile header and calculate length */
    code = metafile_init(img, &devbbox, &mf);

    /* Create temporary EPS file with updated headers */
    tpsfile = NULL;
    memset(tpsname, 0, sizeof(tpsname));
    if ((tpsfile = app_temp_file(doc->app, tpsname, 
	sizeof(tpsname)/sizeof(TCHAR), TEXT("wb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't create temporary EPS file \042%s\042\n"),
	    tpsname);
	return -1;
    }
    fclose(tpsfile);

    code = copy_eps(doc, tpsname, bbox, hires_bbox, DOSEPS_HEADER_SIZE, FALSE); 
    if (code)
	return -1;

    if ( (tpsfile = csfopen(tpsname, TEXT("rb"))) == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open temporary EPS file \042%s\042\n"),
	    tpsname);
	return -1;
    }

    /* Create DOS EPS output file */
    if (*epsname!='\0')
	epsfile = csfopen(epsname, TEXT("wb"));
    else
	epsfile = stdout;
    if (epsfile == (FILE *)NULL) {
	app_csmsgf(doc->app, 
	    TEXT("Can't open output EPS file \042%s\042\n"),
	    epsname);
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    doseps.ps_begin = DOSEPS_HEADER_SIZE;
    fseek(tpsfile, 0, SEEK_END);
    doseps.ps_length = ftell(tpsfile);
    doseps.wmf_begin = doseps.ps_begin + doseps.ps_length;
    doseps.wmf_length = mf.size * 2;
    doseps.tiff_begin = 0;
    doseps.tiff_length = 0;
    doseps.checksum = 0xffff;
    write_doseps_header(&doseps, epsfile);

    buffer = (char *)malloc(COPY_BUF_SIZE);
    if (buffer == (char *)NULL) {
	if (epsname[0]) {
	    fclose(epsfile);
	    if (!(debug & DEBUG_GENERAL))
		csunlink(epsname);
	}
	fclose(tpsfile);
	if (!(debug & DEBUG_GENERAL))
	    csunlink(tpsname);
	return -1;
    }

    /* copy EPS file */
    rewind(tpsfile);
    while ((count = (int)fread(buffer, 1, COPY_BUF_SIZE, tpsfile)) != 0)
	fwrite(buffer, 1, count, epsfile);

    /* copy metafile */
    code = write_metafile(epsfile, img, devbbox, xdpi, ydpi, &mf);

    free(buffer);
    fclose(tpsfile);
    if (!(debug & DEBUG_GENERAL))
	csunlink(tpsname);
    if (epsname[0])
       fclose(epsfile);
    return 0;
}

/*********************************************************/
/* Copy one page to a file. 
 * This will be used for feeding directly to Ghostscript,
 * so don't bother updating header comments.
 * Add code to force one and one only showpage.
 */
int
copy_page_temp(Doc *doc, FILE *f, int page)
{
    CDSC *dsc = doc->dsc;
    FILE *docfile;
    const char *fname;
    if (dsc == NULL)
	return -1;
    fprintf(f, "%%!\nsave /GSview_save exch def\n/showpage {} def\n");

   
    fname = dsc_find_platefile(dsc, page);
    if (fname) {
	/* A separation in a separate file */
	unsigned long end;
	if ((docfile = fopen(fname, "rb")) != (FILE *)NULL) {
	    fseek(docfile, 0, SEEK_END);
	    end = ftell(docfile);
	    ps_copy(f, docfile, 0, end);
	    fclose(docfile);
	}
	/* else separation is missing, don't flag an error */
    }
    else {
	/* ordinary file */
	if ((docfile = csfopen(doc_name(doc),TEXT("rb"))) == (FILE *)NULL) {
	    app_csmsgf(doc->app, 
		TEXT("Can't open document file \042%s\042\n"),
		doc_name(doc));
	    return -1;
	}
	ps_copy(f, docfile, dsc->begincomments, dsc->endcomments);
	ps_copy(f, docfile, dsc->begindefaults, dsc->enddefaults);
	ps_copy(f, docfile, dsc->beginprolog, dsc->endprolog);
	ps_copy(f, docfile, dsc->beginsetup, dsc->endsetup);
	if (dsc->page_count && (page >= 0) && (page < (int)dsc->page_count))
	    ps_copy(f, docfile, dsc->page[page].begin, dsc->page[page].end);
	ps_copy(f, docfile, dsc->begintrailer, dsc->endtrailer);
        fclose(docfile);
    }

    fprintf(f, "\nclear cleardictstack GSview_save restore\nshowpage\n");

    return 0;
}


/*********************************************************/

/* 
Desktop Color Separations 2.0

Single file version:
%%PlateFile contain offsets from the beginning of the file,
not the beginning of the PS section.  When adding or removing
a preview, the platefile offsets need to be adjusted.
The end of the composite section is the lowest of the separation
start offsets.

Multiple file version:
Nothing special needed for EPS handling.
For displaying:
  If file is local, open it.
  If file is not local, ignore it.

Need to do:
- Display multiple file DCS 2.0 as multiple pages.
- Extract page from single file DCS 2.0.
    Don't do - use split.
Done:
- Split single file DCS 2.0 to multi file DCS 2.0.
- Combine multifile DCS 2.0 to single file DCS 2.0.
- Create preview from composite image.
- Add preview to single file DCS 2.0.
- Add preview to multiple file DCS 2.0 (trivial).
- Display single file DCS 2.0 as multiple pages.


*/



/*********************************************************/

#ifdef NOTUSED


/******************************************************************/


/******************************************************************/


#endif

