/* Copyright (C) 2001-2002 Ghostgum Software Pty Ltd.  All rights reserved.

  This software is provided AS-IS with no warranty, either express or
  implied.

  This software is distributed under licence and may not be copied,
  modified or distributed except as expressly authorised under the terms
  of the licence contained in the file LICENCE in this distribution.

  For more information about licensing, please refer to
  http://www.ghostgum.com.au/ or contact Ghostsgum Software Pty Ltd, 
  218 Gallaghers Rd, Glen Waverley VIC 3150, AUSTRALIA, 
  Fax +61 3 9886 6616.
*/

/* $Id: cbmp.c,v 1.7 2002/08/24 05:33:31 ghostgum Exp $ */
/* BMP and PBM to IMAGE conversion */
/* IMAGE to BMP, PBM, TIFF conversion */

/* These functions convert between a BMP or PBM file
 * and an IMAGE in the format as used by the Ghostscript
 * display device.
 */

#include "common.h"
#include "dscparse.h"
#include "gdevdsp.h"
#include "cbmp.h"
#include "ceps.h"	/* FIX: should put write_tiff in this file */
#include "cimg.h"

#ifdef HAVE_LIBPNG
#include "png.h"
#endif

/* local prototypes */
static int read_pbm_bits(unsigned char *pbitmap, 
    unsigned int width, unsigned int height, FILE *f);
static int read_pgnm_bytes(unsigned char *pbitmap, 
    unsigned int length, FILE *f);
static int image_bmp2init(IMAGE *img, BITMAP2 *bmp2);


/* read/write word and dword in a portable way */

DWORD
get_dword(const unsigned char *buf)
{
    DWORD dw;
    dw = (DWORD)buf[0];
    dw += ((DWORD)buf[1])<<8;
    dw += ((DWORD)buf[2])<<16;
    dw += ((DWORD)buf[3])<<24;
    return dw;
}

WORD
get_word(const unsigned char *buf)
{
    WORD w;
    w = (WORD)buf[0];
    w |= (WORD)(buf[1]<<8);
    return w;
}

/* write DWORD as DWORD */
void
write_dword(DWORD val, FILE *f)
{
    fputc((unsigned char)( val      & 0xff), f);
    fputc((unsigned char)((val>>8)  & 0xff), f);
    fputc((unsigned char)((val>>16) & 0xff), f);
    fputc((unsigned char)((val>>24) & 0xff), f);
}

/* write WORD as DWORD */
void
write_word_as_dword(WORD val, FILE *f)
{
    fputc((unsigned char)( val      & 0xff), f);
    fputc((unsigned char)((val>>8)  & 0xff), f);
    fputc('\0', f);
    fputc('\0', f);
}

/* write WORD as WORD */
void
write_word(WORD val, FILE *f)
{
    fputc((unsigned char)( val      & 0xff), f);
    fputc((unsigned char)((val>>8)  & 0xff), f);
}


/* Load a Windows bitmap and return an image.
 * Because we are forcing this into a format that 
 * could be written by the Ghostscript "display"
 * device, we can't write palette images since
 * the palette in the display device is fixed.
 * We convert 4 and 8-bit palette bitmaps to
 * 24-bit colour.
 */
IMAGE *
bmpfile_to_image(LPCTSTR filename)
{
    FILE *f = csfopen(filename, TEXT("rb"));
    IMAGE *img;
    unsigned char bmf_buf[BITMAPFILE_LENGTH];
    unsigned char *pbitmap;
    unsigned int length;
    unsigned int count;
    BITMAPFILE bmf;
    if (f == (FILE *)NULL)
	return NULL;
    fread(bmf_buf, 1, sizeof(bmf_buf), f);
    if ((bmf_buf[0] != 'B') || (bmf_buf[1] != 'M')) {
	/* Not a Windows bitmap */
	fclose(f);
	return NULL;
    }
    bmf.bfType = get_word(bmf_buf);
    bmf.bfSize = get_dword(bmf_buf+2);
    bmf.bfReserved1 = get_word(bmf_buf+6);
    bmf.bfReserved1 = get_word(bmf_buf+8);
    bmf.bfOffBits = get_dword(bmf_buf+10);
    length = bmf.bfSize - BITMAPFILE_LENGTH;

    pbitmap = (unsigned char *)malloc(length);
    if (pbitmap == NULL) {
	fclose(f);
	return NULL;
    }
    
    count = (int)fread(pbitmap, 1, length, f);
    fclose(f);

    img = bmp_to_image(pbitmap, length);
    free(pbitmap);

    return img;
}

static const unsigned char clr555[] = {
    0x1f, 0x00, 0x00, 0x00, /* blue */
    0xe0, 0x03, 0x00, 0x00, /* green */
    0x00, 0x7c, 0x00, 0x00 /* red */
};
static const unsigned char clr565[] = {
    0x1f, 0x00, 0x00, 0x00, /* blue */
    0xe0, 0x07, 0x00, 0x00, /* green */
    0x00, 0xf8, 0x00, 0x00 /* red */
};

IMAGE *
bmp_to_image(unsigned char *pbitmap, unsigned int length)
{
    BITMAP2 bmp2;
    RGB4 colour[256];
    int depth;
    int palcount;
    int pallength;
    int bytewidth;
    BOOL convert;
    int i;
    int x, y;
    IMAGE img;
    IMAGE *pimage;
    unsigned char *bits;
    unsigned char *dest;
    memset(&img, 0, sizeof(img));
    if (length < BITMAP2_LENGTH)
	return NULL;
    /* Read the BITMAPINFOHEADER in a portable way. */
    bmp2.biSize = get_dword(pbitmap);
    pbitmap += 4;
    if (bmp2.biSize < BITMAP2_LENGTH)
	return NULL;	/* we don't read OS/2 BMP format */
    bmp2.biWidth = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biHeight = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biPlanes = get_word(pbitmap);
    pbitmap += 2;
    bmp2.biBitCount = get_word(pbitmap);
    pbitmap += 2;
    bmp2.biCompression = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biSizeImage = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biXPelsPerMeter = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biYPelsPerMeter = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biClrUsed = get_dword(pbitmap);
    pbitmap += 4;
    bmp2.biClrImportant = get_dword(pbitmap);
    pbitmap += 4;
    pbitmap += bmp2.biSize - BITMAP2_LENGTH;

    /* Calculate the raster size, depth, palette length etc. */
    depth = bmp2.biPlanes * bmp2.biBitCount;
    bytewidth = ((bmp2.biWidth * depth + 31) & ~31) >> 3;
    palcount = 0;
    if (depth <= 8)
	palcount = bmp2.biClrUsed > 0 ? bmp2.biClrUsed : 1 << depth;
    pallength = 0;
    if ((depth == 16) || (depth == 32)) {
	if (bmp2.biCompression == BI_BITFIELDS)
	    pallength = 12;
    }
    else
	pallength = palcount * RGB4_LENGTH;
    for (i=0; i<palcount; i++) {
	colour[i].rgbBlue  = pbitmap[i*4+RGB4_BLUE];
	colour[i].rgbGreen = pbitmap[i*4+RGB4_GREEN];
	colour[i].rgbRed   = pbitmap[i*4+RGB4_RED];
    }

    if (length < bmp2.biSize + pallength + bmp2.biHeight * bytewidth)
	return NULL;

    /* Now find out which format to use */
    /* Default is 24-bit BGR */
    img.width = bmp2.biWidth;
    img.height = bmp2.biHeight;
    img.raster = img.width * 3;
    img.format = DISPLAY_COLORS_RGB | DISPLAY_ALPHA_NONE |
       DISPLAY_DEPTH_8 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST;
    convert = FALSE;

    /* We will save it as either 1-bit/pixel, 8-bit/pixel grey,
     * or 24-bit/pixel RGB.
     */
    if (depth == 1) {
	if ((colour[0].rgbBlue == 0) && 
	    (colour[0].rgbGreen == 0) && 
	    (colour[0].rgbRed == 0) &&
	    (colour[1].rgbBlue == 0xff) && 
	    (colour[1].rgbGreen == 0xff) && 
	    (colour[1].rgbRed == 0xff)) {
	    /* black and white */
	    img.format = DISPLAY_COLORS_GRAY | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_1 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST;
	    img.raster = (img.width + 7) >> 3;
	}
	else if ((colour[0].rgbBlue == 0xff) && 
	    (colour[0].rgbGreen == 0xff) && 
	    (colour[0].rgbRed == 0xff) &&
	    (colour[1].rgbBlue == 0) && 
	    (colour[1].rgbGreen == 0) && 
	    (colour[1].rgbRed == 0)) {
	    /* black and white */
	    img.format = DISPLAY_COLORS_NATIVE | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_1 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST;
	    img.raster = (img.width + 7) >> 3;
	}
	else if ( (colour[0].rgbBlue == colour[0].rgbGreen) && 
	    (colour[0].rgbRed == colour[0].rgbGreen) &&
	    (colour[1].rgbBlue == colour[1].rgbGreen) && 
	    (colour[1].rgbRed == colour[1].rgbGreen)) {
	    /* convert to greyscale */
	    img.format = DISPLAY_COLORS_GRAY | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_8 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST;
	    img.raster = img.width;
	    convert = TRUE;
	}
	else
	    /* convert to colour */
	    convert = TRUE;
    }
    else if ((depth == 4) || (depth == 8)) {
	BOOL grey = TRUE;
	for (i=0; i<palcount; i++) {
	    if ((colour[i].rgbBlue != colour[i].rgbGreen) ||  
	    (colour[i].rgbRed != colour[i].rgbGreen))
		grey = FALSE;
	}
	if (grey) {
	    img.format = DISPLAY_COLORS_GRAY | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_8 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST;
	    img.raster = img.width;
	}
	convert = TRUE;
	
    }
    else if (depth == 16) {
	if (pallength == 0) {
	    img.format = DISPLAY_COLORS_NATIVE | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_16 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST |
	        DISPLAY_NATIVE_555;
	    img.raster = img.width * 2;
	}
	else if ((pallength == 12) &&
	    (memcmp(pbitmap, clr555, sizeof(clr555)) == 0)) {
	    img.format = DISPLAY_COLORS_NATIVE | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_16 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST |
	        DISPLAY_NATIVE_555;
	    img.raster = img.width * 2;
	}
	else if ((pallength == 12) &&
		(memcmp(pbitmap, clr565, sizeof(clr565)) == 0)) {
	    img.format = DISPLAY_COLORS_NATIVE | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_16 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST |
	        DISPLAY_NATIVE_565;
	    img.raster = img.width * 2;
	}
	else
	    return NULL;	/* unrecognised format */
    }
    else if (depth == 24) {
	/* already in correct format */
    }
    else if (depth == 32) {
	unsigned char clr888[] = {
	    0xff, 0x00, 0x00, 0x00, /* blue */
	    0x00, 0xff, 0x00, 0x00, /* green */
	    0x00, 0x00, 0xff, 0x00 /* red */
	};
	if ( (pallength == 0) || 
	    ((pallength == 12) &&
	     (memcmp(pbitmap, clr888, sizeof(clr888)) == 0)) ) {
	    img.format = DISPLAY_COLORS_RGB | DISPLAY_UNUSED_LAST |
	       DISPLAY_DEPTH_8 | DISPLAY_LITTLEENDIAN | DISPLAY_BOTTOMFIRST;
	    img.raster = img.width * 4;
	}
	else
	    return NULL;	/* unrecognised format */
    }
    else
	return NULL;	/* unrecognised format */

    pbitmap += pallength;

    img.raster = (img.raster + 3) & ~3;
    bits = (unsigned char *)malloc(img.raster * img.height);
    if (bits == NULL)
	return NULL;

    for (y=0; y<(int)img.height; y++) {
        dest = bits + y * img.raster;
	if (convert) {
	    int idx;
	    int shift = 7;
	    if (depth == 1) {
		for (x=0; x<bmp2.biWidth; x++) {
		    idx = pbitmap[x >> 3];
		    idx = (idx >> shift) & 1;
		    if ((img.format & DISPLAY_COLORS_MASK) == DISPLAY_COLORS_GRAY)
			*dest++ = colour[idx].rgbBlue;
		    else {
			/* colour */
			*dest++ = colour[idx].rgbBlue;
			*dest++ = colour[idx].rgbGreen;
			*dest++ = colour[idx].rgbRed;
		    }
		    shift--;
		    if (shift < 0)
			shift = 7;
		}
	    }
	    else if (depth == 4) {
		for (x=0; x<bmp2.biWidth; x++) {
		    idx = pbitmap[x/2];
		    if (x & 0)
			idx &= 0xf;
		    else
			idx = (idx >> 4) & 0xf;
		    if ((img.format & DISPLAY_COLORS_MASK) == DISPLAY_COLORS_GRAY)
			*dest++ = colour[idx].rgbBlue;
		    else {
			/* colour */
			*dest++ = colour[idx].rgbBlue;
			*dest++ = colour[idx].rgbGreen;
			*dest++ = colour[idx].rgbRed;
		    }
		}
	    }
	    else if (depth == 8) {
		for (x=0; x<bmp2.biWidth; x++) {
		    idx = pbitmap[x];
		    if ((img.format & DISPLAY_COLORS_MASK) == DISPLAY_COLORS_GRAY)
			*dest++ = colour[idx].rgbBlue;
		    else {
			/* colour */
			*dest++ = colour[idx].rgbBlue;
			*dest++ = colour[idx].rgbGreen;
			*dest++ = colour[idx].rgbRed;
		    }
		}
	    }
	    else {
		free(bits);
		return NULL;
	    }
	}
	else {
	    memcpy(dest, pbitmap, img.raster);
	}
	pbitmap += bytewidth;
    }

    pimage = (IMAGE *)malloc(sizeof(IMAGE));
    if (pimage == NULL) {
	free(bits);
	return NULL;
    }
    memcpy(pimage, &img, sizeof(IMAGE));
    pimage->image = bits;
    return pimage;
}


/* read width * height "bits" from file */
static int
read_pbm_bits(unsigned char *pbitmap, unsigned int width, unsigned int height,
    FILE *f)
{
    int count = 0;
    int ch;
    int mask = 0x80;
    int x, y;

    for (y=0; y<(int)height; y++) {
	mask = 0x80;
	for (x=0; x<(int)width; x++) {
	    if (mask == 0) {
		mask = 0x80;
		pbitmap++;
		count++;
	    }
	    ch = 0;
	    while ((ch != '0') && (ch != '1')) {
		ch = fgetc(f);
		if (ch == EOF)
		    return -1;
	    }
	    *pbitmap = (unsigned char)
		((ch == '1') ? (*pbitmap | mask) : (*pbitmap & (~mask)));
	    mask >>= 1;
	}
        pbitmap++;
	count++;
    }
    return count;
}

/* read length "bytes" from file */
static int
read_pgnm_bytes(unsigned char *pbitmap, unsigned int length, FILE *f)
{
    int count = 0;
    int ch = 0;
    int val;

    for (count=0; count < (int)length; count++) {
	val = 0;
	while (!((ch >= '0') && (ch <= '9'))) {
	    ch = fgetc(f);
	    if (ch == EOF)
		return -1;
	}
	while ((ch >= '0') && (ch <= '9')) {
	    val = (val*10) + ch - '0';
	    ch = fgetc(f);
	    if (ch == EOF)
		return -1;
	}
	*pbitmap++ = (unsigned char)val;
    }
    return count;
}

/* Load a PBMPLUS bitmap and return an image.
 * Supported formats are pbmraw, pgmraw and ppmraw as written
 * by Ghostscript.
 */
IMAGE *
pnmfile_to_image(LPCTSTR filename)
{
    FILE *f = csfopen(filename, TEXT("rb"));
    IMAGE img;
    IMAGE *pimage;
    int code;
    char typeline[256];
    char sizeline[256];
    char maxvalline[256];
    char hdrline[256];
    char tupltype[256];
    int width = 0;
    int height = 0;
    int maxval = 255;
    int depth = 0;
    int pam = 0;
    int pbm = 0;
    int pgm = 0;
    int ppm = 0;
    int raw = 0;
    int cmyk = 0;
    unsigned int length;
    unsigned char *pbitmap;
    int endhdr;
    char *t1;
    char *t2;

    unsigned int count;

    if (f == (FILE *)NULL)
	return NULL;
    memset(&img, 0, sizeof(img));
    memset(typeline, 0, sizeof(typeline));
    memset(sizeline, 0, sizeof(sizeline));
    memset(maxvalline, 0, sizeof(maxvalline));
    code = fgets(typeline, sizeof(typeline)-1, f) == NULL;
    if (typeline[0] != 'P')
	code = 1;
    switch (typeline[1]) {
	case '1':
	    pbm = 1;
	    raw = 0;
	    break;
	case '2':
	    pgm = 1;
	    raw = 0;
	    break;
	case '3':
	    ppm = 1;
	    raw = 0;
	    break;
	case '4':
	    pbm = 1;
	    raw = 1;
	    break;
	case '5':
	    pgm = 1;
	    raw = 1;
	    break;
	case '6':
	    ppm = 1;
	    raw = 1;
	    break;
	case '7':
	    pam = 1;
	    raw = 1;
	    break;
	default:
	    code = 1;
    }

    if (pam) {
	/* Portable Arbitrary Map */
	endhdr = 0;
	while (!endhdr && !code) {
	    if (!code)
		code = fgets(hdrline, sizeof(hdrline)-1, f) == NULL;
	    while (!code && (hdrline[0] == '#'))
		/* skip comments */
		code = fgets(hdrline, sizeof(hdrline)-1, f) == NULL;
	    if (code)
		break;
	    t1 = hdrline;
	    while (*t1 && ((*t1==' ') || (*t1=='\t')))
		t1++;	/* skip whitespace */
	    t1 = strtok(t1, " \t\r\n");
	    if (t1 == NULL)
		break;
	    t2 = strtok(NULL, " \t\r\n");
	    if (strcmp(t1, "ENDHDR")==0) {
		endhdr = 1;
		continue;
	    }
	    else if (strcmp(t1, "WIDTH")==0) {
		if (t2)
		    code = sscanf(t2, "%u", &width) != 1;
	    }
	    else if (strcmp(t1, "HEIGHT")==0) {
		if (t2)
		    code = sscanf(t2, "%u", &height) != 1;
	    }
	    else if (strcmp(t1, "DEPTH")==0) {
		if (t2)
		    code = sscanf(t2, "%u", &depth) != 1;
	    }
	    else if (strcmp(t1, "MAXVAL")==0) {
		if (t2)
		    code = sscanf(t2, "%u", &maxval) != 1;
	    }
	    else if (strcmp(t1, "TUPLTYPE")==0) {
		if (t2)
		    strncpy(tupltype, t2, sizeof(tupltype)-1);
	    }
	}
	if (!endhdr)
	    code = 1;
	if ((width == 0) || (height == 0) || (depth == 0) || (maxval == 0))
	    code = 1;
	if ((strcmp(tupltype, "BLACKANDWHITE")==0) &&
	    (depth == 1) && (maxval == 1))
	    pbm = 1;
	if ((strcmp(tupltype, "GRAYSCALE")==0) &&
	    (depth == 1) && (maxval == 255))
	    pgm = 1;
	if ((strcmp(tupltype, "RGB")==0) && 
	    (depth == 3) && (maxval == 255))
	    ppm = 1;
	if ((strcmp(tupltype, "CMYK")==0) &&
	    (depth == 4) && (maxval == 255))
	    cmyk = 1;
    }
    else {
	if (!code)
	    code = fgets(sizeline, sizeof(sizeline)-1, f) == NULL;
	while (!code && (sizeline[0] == '#')) /* skip comments */
	    code = fgets(sizeline, sizeof(sizeline)-1, f) == NULL;

	if (!code)
	    code = sscanf(sizeline, "%u %u", &width, &height) != 2;
	if ((width == 0) || (height == 0))
	    code = 1;

	if (!code && (pgm || ppm)) {
	    code = fgets(maxvalline, sizeof(maxvalline)-1, f) == NULL;
	    while (!code && (maxvalline[0] == '#'))
		code = fgets(maxvalline, sizeof(maxvalline)-1, f) == NULL;
	    if (!code)
		code = sscanf(maxvalline, "%u", &maxval) != 1;
	}
	if (maxval != 255)
	    code = 1;
    }

    img.width = width;
    img.height = height;
    if (pbm) {
	img.format = DISPLAY_COLORS_NATIVE | DISPLAY_ALPHA_NONE |
	   DISPLAY_DEPTH_1 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST;
	img.raster = (img.width + 7) >> 3;
    }
    else if (pgm) {
	img.format = DISPLAY_COLORS_GRAY | DISPLAY_ALPHA_NONE |
	   DISPLAY_DEPTH_8 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST;
	img.raster = img.width;
    }
    else if (ppm) {
	img.format = DISPLAY_COLORS_RGB | DISPLAY_ALPHA_NONE |
	   DISPLAY_DEPTH_8 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST;
	img.raster = img.width * 3;
    }
    else if (cmyk) {
	img.format = DISPLAY_COLORS_CMYK | DISPLAY_ALPHA_NONE |
	   DISPLAY_DEPTH_8 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST;
	img.raster = img.width * 4;
    }
    else
	code = 1;

    length = img.raster * img.height;

    if (code) {
	fclose(f);
	return NULL;
    }

    pbitmap = (unsigned char *)malloc(length);
    if (pbitmap == NULL) {
	fclose(f);
	return NULL;
    }

    if (raw)
	count = (int)fread(pbitmap, 1, length, f);
    else if (pbm)
	count = read_pbm_bits(pbitmap, img.width, img.height, f);
    else
	count = read_pgnm_bytes(pbitmap, length, f);
    fclose(f);

    if (count != length) {
	free(pbitmap);
	return NULL;
    }

    pimage = (IMAGE *)malloc(sizeof(IMAGE));
    if (pimage == NULL) {
	free(pbitmap);
	return NULL;
    }
    memcpy(pimage, &img, sizeof(IMAGE));
    pimage->image = pbitmap;
    return pimage;
}

/* Free an image created by bmpfile_to_image or pnmfile_to_image */
void
bitmap_image_free(IMAGE *img)
{
    if (img && img->image) {
	free(img->image);
	memset(img, 0, sizeof(IMAGE));
	free(img);
    }
}

static int
image_bmp2init(IMAGE *img, BITMAP2 *bmp2)
{
    /* Create a BMP header from the IMAGE */
    /* If we need to convert the IMAGE before writing to a BMP file, 
     * the BMP header will not exactly match the image.
     */
    bmp2->biSize = BITMAP2_LENGTH;
    bmp2->biWidth = img->width;
    bmp2->biHeight = img->height;
    bmp2->biPlanes = 1;
    bmp2->biCompression = 0;
    switch (img->format & DISPLAY_COLORS_MASK) {
	case DISPLAY_COLORS_NATIVE:
	    switch (img->format & DISPLAY_DEPTH_MASK) {
		case DISPLAY_DEPTH_1:
		    bmp2->biBitCount = 1;
		    bmp2->biClrUsed = 2;
		    bmp2->biClrImportant = 2;
		    break;
		case DISPLAY_DEPTH_4:
		    /* Fixed color palette */
		    bmp2->biBitCount = 4;
		    bmp2->biClrUsed = 16;
		    bmp2->biClrImportant = 16;
		    break;
		case DISPLAY_DEPTH_8:
		    /* Fixed color palette */
		    bmp2->biBitCount = 8;
		    bmp2->biClrUsed = 96;
		    bmp2->biClrImportant = 96;
		    break;
		case DISPLAY_DEPTH_16:
		    /* RGB bitfields */
		    /* Bit fields */
		    if ((img->format & DISPLAY_ENDIAN_MASK)
			== DISPLAY_BIGENDIAN) {
			/* convert */
			bmp2->biBitCount = 24;
			bmp2->biClrUsed = 0;
			bmp2->biClrImportant = 0;
		    }
		    else {
			bmp2->biBitCount = 16;
			bmp2->biCompression = BI_BITFIELDS;
			bmp2->biClrUsed = 0;
			bmp2->biClrImportant = 0;
		    }
		    break;
		default:
		    return_error(-1);
	    }
	    break;
	case DISPLAY_COLORS_GRAY:
	    switch (img->format & DISPLAY_DEPTH_MASK) {
		case DISPLAY_DEPTH_1:
		    bmp2->biBitCount = 1;
		    bmp2->biClrUsed = 2;
		    bmp2->biClrImportant = 2;
		    break;
		case DISPLAY_DEPTH_4:
		    /* Fixed gray palette */
		    bmp2->biBitCount = 4;
		    bmp2->biClrUsed = 16;
		    bmp2->biClrImportant = 16;
		    break;
		case DISPLAY_DEPTH_8:
		    /* Fixed gray palette */
		    bmp2->biBitCount = 8;
		    bmp2->biClrUsed = 256;
		    bmp2->biClrImportant = 256;
		    break;
		default:
		    return_error(-1);
	    }
	    break;
	case DISPLAY_COLORS_RGB:
	    if ((img->format & DISPLAY_DEPTH_MASK) != DISPLAY_DEPTH_8)
		return_error(-1);
	    /* either native BGR, or we need to convert it */
	    bmp2->biBitCount = 24;
	    bmp2->biClrUsed = 0;
	    bmp2->biClrImportant = 0;
	    break;
	case DISPLAY_COLORS_CMYK:
	    /* convert */
	    bmp2->biBitCount = 24;
	    bmp2->biClrUsed = 0;
	    bmp2->biClrImportant = 0;
	    break;
    }

    bmp2->biSizeImage = 0;
    bmp2->biXPelsPerMeter = 0;
    bmp2->biYPelsPerMeter = 0;
    return 0;
}


/* Write an IMAGE as a Windows BMP file */
/* This is typically used to copy the display bitmap to a file */
int
image_to_bmpfile(IMAGE* img, LPCTSTR filename, float xdpi, float ydpi)
{
    BITMAP2 bmp2;
    BITMAPFILE bmf;
    int bytewidth;
    int depth;
    int palcount;
    int pallength;
    FILE *f;
    unsigned char r, g, b;
    int i;
    unsigned char *bits;
    unsigned char *row;
    int topfirst;

    if ((img == NULL) || (img->image == NULL))
	return -1;
    if (image_bmp2init(img, &bmp2) < 0)
	return -1;

    if ((xdpi > 0.0) && (ydpi > 0.0)) {
	bmp2.biXPelsPerMeter = (int)(xdpi * 1000.0 / 25.4 + 0.5);
	bmp2.biYPelsPerMeter = (int)(ydpi * 1000.0 / 25.4 + 0.5);
    }

    depth = bmp2.biPlanes * bmp2.biBitCount;
    bytewidth = ((bmp2.biWidth * depth + 31) & ~31) >> 3;
    switch (depth) {
	case 1:
	case 4:
	case 8:
	    palcount = 1 << depth;
	    pallength = palcount * RGB4_LENGTH;
	    break;
	case 16:
	    palcount = 0;
	    pallength = 12;
	default:
	    palcount = 0;
	    pallength = 0;
    }

    bmf.bfType = get_word((const unsigned char *)"BM");
    bmf.bfReserved1 = 0;
    bmf.bfReserved2 = 0;;
    bmf.bfOffBits = BITMAPFILE_LENGTH + BITMAP2_LENGTH + palcount;
    bmf.bfSize = bmf.bfOffBits + bytewidth * bmp2.biHeight;

    row = (unsigned char *)malloc(bytewidth);
    if (row == NULL)
	return -1;
    
    f = csfopen(filename, TEXT("wb"));
    if (f == NULL) {
	free(row);
	return -1;
    }

    /* Write BITMAPFILEHEADER */
    write_word(bmf.bfType, f);
    write_dword(bmf.bfSize, f);
    write_word(bmf.bfReserved1, f);
    write_word(bmf.bfReserved2, f);
    write_dword(bmf.bfOffBits, f);

    /* Write BITMAPINFOHEADER */
    write_dword(bmp2.biSize, f);
    write_dword(bmp2.biWidth, f);
    write_dword(bmp2.biHeight, f);
    write_word(bmp2.biPlanes, f);
    write_word(bmp2.biBitCount, f);
    write_dword(bmp2.biCompression, f);
    write_dword(bmp2.biSizeImage, f);
    write_dword(bmp2.biXPelsPerMeter, f);
    write_dword(bmp2.biYPelsPerMeter, f);
    write_dword(bmp2.biClrUsed, f);
    write_dword(bmp2.biClrImportant, f);

    /* Write palette or bitmasks */
    for (i=0; i<palcount; i++) {
	image_colour(img->format, i, &r, &g, &b);
	fputc(b, f);
	fputc(g, f);
	fputc(r, f);
	fputc('\0', f);
    }
    if (bmp2.biCompression == BI_BITFIELDS) {
	if ((img->format & DISPLAY_555_MASK) == DISPLAY_NATIVE_555)
	    fwrite(clr555, 1, sizeof(clr555), f);
	else 
	    fwrite(clr565, 1, sizeof(clr565), f);
    }

    /* Write the bits */
    topfirst = ((img->format & DISPLAY_FIRSTROW_MASK) == DISPLAY_TOPFIRST);
    for (i=0; i<bmp2.biHeight; i++) {
	if (topfirst)
	    bits = img->image + img->raster * (img->height - i - 1);
	else
	    bits = img->image + img->raster * i;
	if (depth == 24) {
	    image_to_24BGR(img, row, bits);
	    fwrite(row, 1, bytewidth, f);
	}
	else {
	    if ((int)img->raster < bytewidth) {
		int j;
	        fwrite(bits, 1, img->raster, f);
		for (j=bytewidth-img->raster; j>0; j--)
		    fputc('\0', f);
	    }
	    else
	        fwrite(bits, 1, bytewidth, f);
	}
    }

    free(row);
    fclose(f);
    return 0;
}

/* Write an IMAGE as a TIFF file */
/* This is typically used to copy the display bitmap to a file */
int
image_to_tifffile(IMAGE* img, LPCTSTR filename, float xdpi, float ydpi)
{
    FILE *f;
    CDSCBBOX devbbox;
    int code = 0;
    BOOL tiff4 = ((img->format & DISPLAY_DEPTH_MASK) == DISPLAY_DEPTH_1);

    if ((img == NULL) || (img->image == NULL))
	return -1;
    
    f = csfopen(filename, TEXT("wb"));
    if (f == NULL)
	return -1;

    devbbox.llx = devbbox.lly = 0;
    devbbox.urx = img->width;
    devbbox.ury = img->height;
    code = write_tiff(f, img, devbbox, xdpi, ydpi, tiff4, !tiff4);

    fclose(f);
    return 0;
}

int
image_to_pnmfile(IMAGE* img, LPCTSTR filename, PNM_FORMAT pnm_format)
{
    PNM_FORMAT format = pnm_format;
    FILE *f;
    int bytewidth;
    unsigned char *row;
    unsigned char *bits;
    int topfirst;
    int i;
    if ((img == NULL) || (img->image == NULL))
	return -1;
    
    /* check if mono, grey or colour */
    if ((format != PBMRAW) && (format != PGMRAW) && (format != PPMRAW)) {
      switch (img->format & DISPLAY_COLORS_MASK) {
	case DISPLAY_COLORS_NATIVE:
	    switch (img->format & DISPLAY_DEPTH_MASK) {
		case DISPLAY_DEPTH_1:
		    format = PBMRAW;
		    break;
		case DISPLAY_DEPTH_4:
		case DISPLAY_DEPTH_8:
		case DISPLAY_DEPTH_16:
		    format = PPMRAW;
		    break;
		default:
		    return_error(-1);
	    }
	    break;
	case DISPLAY_COLORS_GRAY:
	    switch (img->format & DISPLAY_DEPTH_MASK) {
		case DISPLAY_DEPTH_1:
		    format = PBMRAW;
		    break;
		case DISPLAY_DEPTH_4:
		case DISPLAY_DEPTH_8:
		    /* Fixed gray palette */
		    format = PGMRAW;
		    break;
		default:
		    return_error(-1);
	    }
	    break;
	case DISPLAY_COLORS_RGB:
	    if ((img->format & DISPLAY_DEPTH_MASK) != DISPLAY_DEPTH_8)
		return_error(-1);
	    format = PPMRAW;
	    break;
	case DISPLAY_COLORS_CMYK:
	    /* convert */
	    format = PPMRAW;
	    break;
      }
    }
    if (format == PPMRAW)
	bytewidth = img->width * 3;
    else if (format == PGMRAW)
	bytewidth = img->width;
    else
	bytewidth = (img->width + 7) >> 3;
    row = (unsigned char *)malloc(bytewidth);
    if (row == NULL)
	return -1;
    topfirst = ((img->format & DISPLAY_FIRSTROW_MASK) == DISPLAY_TOPFIRST);

    f = csfopen(filename, TEXT("wb"));
    if (f == NULL) {
	free(row);
	return -1;
    }

    fprintf(f, "P%c\n", (int)(format+'0'));
    fprintf(f, "# Created by GSview\n");
    fprintf(f, "%d %d\n", img->width, img->height);
    if ((format == PGMRAW) || (format == PPMRAW))
        fprintf(f, "255\n");	/* max value */
    for (i=0; i<(int)img->height; i++) {
	if (topfirst)
	    bits = img->image + img->raster * i;
	else
	    bits = img->image + img->raster * (img->height - i - 1);
	if (format == PPMRAW)
	    image_to_24RGB(img, row, bits);
	else if (format == PGMRAW)
	    image_to_grey(img, row, bits);
	else
	    image_to_mono(img, row, bits);
        fwrite(row, 1, bytewidth, f);
    }
    
    free(row);
    fclose(f);
    return 0;
}


#ifdef HAVE_LIBPNG

static void
image_png_flush(png_structp png_ptr)
{
    png_FILE_p io_ptr;
    io_ptr = (png_FILE_p)CVT_PTR((png_ptr->io_ptr));
    if (io_ptr != NULL)
	fflush(io_ptr);
}

static void
image_png_write_data(png_structp png_ptr, png_bytep data, png_size_t length)
{
   int count = fwrite(data, 1, length, (png_FILE_p)png_ptr->io_ptr);
   if (count != (int)length)
      png_error(png_ptr, "Write Error");
}

static void
image_png_read_data(png_structp png_ptr, png_bytep data, png_size_t length)
{
   int count;
   count = fread(data, 1, length, (png_FILE_p)png_ptr->io_ptr);
   if (count != (int)length)
      png_error(png_ptr, "Read Error!");
}

int
image_to_pngfile(IMAGE* img, LPCTSTR filename)
{
    FILE *f;
    png_structp png_ptr;
    png_infop info_ptr;
    int colour_type;
    int bit_depth;
    int num_palette = 0;
    png_color palette[96];
    BOOL invert_mono = FALSE;
    BOOL topfirst;
    unsigned char *bits;
    unsigned char *row = NULL;
    int i;

    if ((img == NULL) || (img->image == NULL))
	return -1;

    switch (img->format & DISPLAY_COLORS_MASK) {
	case DISPLAY_COLORS_NATIVE:
	    switch (img->format & DISPLAY_DEPTH_MASK) {
		case DISPLAY_DEPTH_1:
		    colour_type = PNG_COLOR_TYPE_GRAY;
		    bit_depth = 1;
		    invert_mono = TRUE;
		    break;
		case DISPLAY_DEPTH_4:
		    colour_type = PNG_COLOR_TYPE_PALETTE;
		    bit_depth = 4;
		    num_palette = 16;
		    break;
		case DISPLAY_DEPTH_8:
		    colour_type = PNG_COLOR_TYPE_PALETTE;
		    num_palette = 96;
		    bit_depth = 8;
		    break;
		case DISPLAY_DEPTH_16:
		    /* convert */
		    colour_type = PNG_COLOR_TYPE_RGB;
		    bit_depth = 8;
		    break;
		default:
		    return_error(-1);
	    }
	    break;
	case DISPLAY_COLORS_GRAY:
	    switch (img->format & DISPLAY_DEPTH_MASK) {
		case DISPLAY_DEPTH_1:
		    colour_type = PNG_COLOR_TYPE_GRAY;
		    bit_depth = 1;
		    break;
		case DISPLAY_DEPTH_4:
		    colour_type = PNG_COLOR_TYPE_GRAY;
		    bit_depth = 4;
		    break;
		case DISPLAY_DEPTH_8:
		    colour_type = PNG_COLOR_TYPE_GRAY;
		    bit_depth = 8;
		    break;
		default:
		    return_error(-1);
	    }
	    break;
	case DISPLAY_COLORS_RGB:
	    if ((img->format & DISPLAY_DEPTH_MASK) != DISPLAY_DEPTH_8)
		return_error(-1);
	    colour_type = PNG_COLOR_TYPE_RGB;
	    bit_depth = 8;
	    break;
	case DISPLAY_COLORS_CMYK:
	    /* convert */
	    colour_type = PNG_COLOR_TYPE_RGB;
	    bit_depth = 8;
	    break;
	default:
	    return -1;
    }

    if (num_palette > sizeof(palette)/sizeof(palette[0]))
	return -1;
    if (colour_type == PNG_COLOR_TYPE_RGB) {
	row = (unsigned char *)malloc(img->width * 3);
	if (row == NULL)
	    return -1;
    }

    f = csfopen(filename, TEXT("wb"));
    if (f == NULL) {
	if (row)
	    free(row);
	return -1;
    }

    png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
	(png_voidp)NULL, NULL, NULL);
    if (png_ptr == NULL) {
	fclose(f);
	if (row)
	    free(row);
	return -1;
    }

    info_ptr = png_create_info_struct(png_ptr);
    if (info_ptr == NULL) {
	png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
	fclose(f);
	return -1;
    }

    if (setjmp(png_jmpbuf(png_ptr))) {
	png_destroy_write_struct(&png_ptr, &info_ptr);
	fclose(f);
	if (row)
	    free(row);
	return -1;
    }

    png_set_write_fn(png_ptr, (png_voidp)f, image_png_write_data,
	image_png_flush);

    png_set_IHDR(png_ptr, info_ptr, img->width, img->height,
	bit_depth, colour_type, PNG_INTERLACE_NONE,
	PNG_COMPRESSION_TYPE_DEFAULT,
	PNG_FILTER_TYPE_DEFAULT);

    if (colour_type == PNG_COLOR_TYPE_PALETTE) {
	for (i=0; i<num_palette; i++)
	    image_colour(img->format, i, 
		&palette[i].red, &palette[i].green, &palette[i].blue);
	png_set_PLTE(png_ptr, info_ptr, palette, num_palette);
    }
    if (invert_mono)
	png_set_invert_mono(png_ptr);

    png_write_info(png_ptr, info_ptr);

    topfirst = ((img->format & DISPLAY_FIRSTROW_MASK) == DISPLAY_TOPFIRST);
    for (i=0; i<(int)img->height; i++) {
	if (topfirst)
	    bits = img->image + img->raster * i;
	else
	    bits = img->image + img->raster * (img->height - i - 1);
	if (colour_type == PNG_COLOR_TYPE_RGB) {
	    image_to_24RGB(img, row, bits);
	    bits = row;
	}
	png_write_row(png_ptr, bits);
    }
    png_write_end(png_ptr, info_ptr);
    png_destroy_write_struct(&png_ptr, &info_ptr);
    fclose(f);
    if (row)
	free(row);
    return 0;
}

IMAGE *
pngfile_to_image(LPCTSTR filename)
{
    FILE *f;
    png_structp png_ptr;
    png_infop info_ptr;
    png_infop end_info;
    unsigned char png_buf[8];
    png_uint_32 width;
    png_uint_32 height;
    int bit_depth;
    int color_type;
    int interlace_type;
    int compression_method;
    int filter_method;
    unsigned char **rows = NULL;
    unsigned char *bits = NULL;
    int nbytes;
    int raster;
    int i;
    IMAGE *img = NULL;

    f = csfopen(filename, TEXT("rb"));
    if (f == (FILE *)NULL)
	return NULL;
    fread(png_buf, 1, sizeof(png_buf), f);
    if (png_sig_cmp(png_buf, 0, sizeof(png_buf))!=0) {
	/* Not a PNG file */
	fclose(f);
	return NULL;
    }

    png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
	(png_voidp)NULL, NULL, NULL);
    if (png_ptr == NULL) {
	fclose(f);
	return NULL;
    }
    png_set_read_fn(png_ptr, (png_voidp)f, image_png_read_data);

    info_ptr = png_create_info_struct(png_ptr);
    if (info_ptr == NULL) {
	png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
	fclose(f);
	return NULL;
    }

    end_info = png_create_info_struct(png_ptr);
    if (end_info == NULL) {
	png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
	fclose(f);
	return NULL;
    }

    if (setjmp(png_jmpbuf(png_ptr))) {
	png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
	fclose(f);
	if (rows)
	    free(rows);
	if (bits)
	    free(bits);
	if (img)
	    free(img);
	return NULL;
    }

    png_set_sig_bytes(png_ptr, sizeof(png_buf));

    png_read_info(png_ptr, info_ptr);

    png_get_IHDR(png_ptr, info_ptr, &width, &height, 
	&bit_depth, &color_type, &interlace_type,
	&compression_method, &filter_method);


    if (color_type == PNG_COLOR_TYPE_PALETTE)
	png_set_palette_to_rgb(png_ptr);

    if (color_type == PNG_COLOR_TYPE_GRAY)
	nbytes = 1;
    else
	nbytes = 3;

    if ((color_type == PNG_COLOR_TYPE_GRAY) &&
	(bit_depth < 8))
	png_set_gray_1_2_4_to_8(png_ptr);

    if (bit_depth == 16)
	png_set_strip_16(png_ptr);

    if (color_type & PNG_COLOR_MASK_ALPHA)
	png_set_strip_alpha(png_ptr);


    /* Allocate memory, and set row pointers */
    raster = (width * nbytes + 3) & ~3;
    rows = (unsigned char **)malloc(height * sizeof(unsigned char *));
    if (rows == NULL)
	longjmp(png_jmpbuf(png_ptr),-1);
    bits = (unsigned char *)malloc(raster * height);
    if (bits == NULL)
	longjmp(png_jmpbuf(png_ptr),-1);
    for (i=0; i<(int)height; i++)
	rows[i] = bits + i * raster;

    png_read_rows(png_ptr, rows, NULL, height);

    png_read_end(png_ptr, end_info);

    png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
    fclose(f);
    if (rows)
	free(rows);

    img = (IMAGE *)malloc(sizeof(IMAGE));
    if (img == NULL)
	free(bits);
    else {
	img->width = width;
	img->height = height;
	img->raster = raster;
	img->format = (nbytes == 1) 
	    ? (DISPLAY_COLORS_GRAY | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_8 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST)
	    : (DISPLAY_COLORS_RGB | DISPLAY_ALPHA_NONE |
	       DISPLAY_DEPTH_8 | DISPLAY_BIGENDIAN | DISPLAY_TOPFIRST);
	img->image = bits;
    }

    return img;
}

#else /* HAVE_LIBPNG */
int
image_to_pngfile(IMAGE* img, LPCTSTR filename)
{
    /* not supported */
    return -1;
}

IMAGE *
pngfile_to_image(LPCTSTR filename)
{
    /* not supported */
    return NULL;
}
#endif /* HAVE_LIBPNG */
