/*
 * GNU m4 -- A simple macro processor
 * Copyright (C) 1989, 1990, 1991 Free Software Foundation, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * printf like formatting for m4.
 */

#include "m4.h"

/*
 * Various constants for floating point formatting.
 */
#define MAXFIELD	128		/* size of buffer for formatted text */
    /* THE FOLLOWING TWO ARE HARDWARE DEPENDANT */
#define ECVTMAX		18		/* max number of significant digits for %e */
#define FCVTMAX		(18+38+4)	/* max number of significant digits for %f */

/*
 * Externs used herin.
 */
extern char *ecvt(), *fcvt(), *gcvt();
extern int atoi();
extern long atol();
extern double atof();

/*
 * Simple varargs substitute.
 */

#define ARG_INT(argc, argv)			\
	((argc == 0) ? 0 : (--argc, argv++, atoi(TOKEN_DATA_TEXT(argv[-1]))))

#define ARG_UINT(argc, argv)			\
	((argc == 0) ? 0 : (--argc, argv++, (unsigned int)atoi(TOKEN_DATA_TEXT(argv[-1]))))

#define ARG_LONG(argc, argv)			\
	((argc == 0) ? 0 : (--argc, argv++, atol(TOKEN_DATA_TEXT(argv[-1]))))

#define ARG_ULONG(argc, argv)			\
	((argc == 0) ? 0 : (--argc, argv++, (unsigned long)atol(TOKEN_DATA_TEXT(argv[-1]))))

#define ARG_STR(argc, argv)			\
	((argc == 0) ? "" : (--argc, argv++, TOKEN_DATA_TEXT(argv[-1])))

#define ARG_DOUBLE(argc, argv)			\
	((argc == 0) ? 0 : (--argc, argv++, atof(TOKEN_DATA_TEXT(argv[-1]))))
#define min(a, b)	((a) < (b) ? (a) : (b))

static char digits[] = "0123456789abcdef";
static char Digits[] = "0123456789ABCDEF";

static char *
ltoa(val, str, base, digits)
    register unsigned long val;
    char str[MAXFIELD];
    int base;
    char *digits;
{
    register char *s = &str[MAXFIELD];

    *--s = '\0';
    do {
	*--s = digits[val%base];
	val /= base;
    } while (val > 0);

    return s;
}

static char *
clr0(s)				/* clear trailing zeroes, return argument */
    char *s;
{
    register char *t;

    for (t = s + strlen(s); *--t == '0' && t > s;)
	*t = '\0';
    return s;
}



/*
 * The main formatting function.  Output is placed on the obstack OBS,
 * the first argument in ARGV is the formatting string, and the rest is
 * arguments for the string.
 */
void
format(obs, argc, argv)
    struct obstack *obs;
    int argc;
    token_data **argv;
{
    char *fmt;
    int c;				/* A simple character */
    char fc;				/* Format code */

    /* flags */
    char flags;				/* 1 iff treating flags */
    char ljust;				/* Left justification */
    char mandsign;			/* Mandatory sign */
    char noplus;			/* Use space if no sign */
    char alternate;			/* Use alternate form */
    char zeropad;			/* Do zero padding */
    char plus;				/* Plus-sign, according to mandatory and noplus */

    /* precision specifiers */
    int width;				/* Minimum field width */
    int prec;				/* Precision */
    int maxch;				/* Maximum no. of chars to print */
    char lflag;				/* Long flag */
    char hflag;				/* Short flag */

    /* Different parts of each specification */
    char sign;				/* Wanted sign, iff any */
    int ppad;				/* Pre-prefix zero padding */
    char *prefix;			/* Value prefix */
    int lpad;				/* Zero padding on the left */
    register char *s;			/* Ptr to formatted text */
    int rpad;				/* Zero padding on the rigth*/
    char *suffix;			/* Value suffix */

    /* Buffer and stuff */
    char str[MAXFIELD];			/* Buffer for formatted text */
    int length;				/* Length of str */
    int padding;			/* Padding at the left or rigth */
    register int i;			/* An index */

#define LENGTH(s)	(&str[MAXFIELD-1] - (s)) /* length of trailing string in str. */
#define HAS_SIGN	(sign != '\0')

    fmt = ARG_STR(argc, argv);
    for (;;) {
	while ((c = *fmt++) != '%') {
	    if (c == 0)
		return;
	    obstack_1grow(obs, c);
	}
	if (*fmt == '%') {
	    obstack_1grow(obs, '%');
	    fmt++;
	    continue;
	}

	/* Parse flags */
	flags = 1;
	ljust = mandsign = noplus = alternate = zeropad = 0;
	do {
	    switch (*fmt) {
	    case '-':			/* left justification */
		ljust = 1;
		break;
	    case '+':			/* mandatory sign */
		mandsign = 1;
		break;
	    case ' ':			/* space instead of positive sign */
		noplus = 1;
		break;
	    case '0':			/* zero padding */
		zeropad = 1;
		break;
	    case '#':			/* alternate output */
		alternate = 1;
		break;
	    default:
		flags = 0;
		break;
	    }
	} while (flags && fmt++);

	plus = '\0';			/* what to use as a plus ??? */
	if (mandsign)
	    plus = '+';
	else if (noplus)
	    plus = ' ';

	if (ljust)
	    zeropad = 0;

	/* Minimum field width */
	width = -1;
	if (*fmt == '*') {
	    width = ARG_INT(argc, argv);
	    fmt++;
	} else if (isdigit(*fmt)) {
	    width = 0;
	    do {
		width = width * 10 + *fmt++ - '0';
	    } while (isdigit(*fmt));
	}

	/* Maximum precision */
	prec = -1;
	if (*fmt == '.') {
	    if (*(++fmt) == '*') {
		prec = ARG_INT(argc, argv);
		++fmt;
	    } else if (isdigit(*fmt)) {
		prec = 0;
		do {
		    prec = prec * 10 + *fmt++ - '0';
		} while (isdigit(*fmt));
	    }
	}

	/* Length modifiers */
	lflag = (*fmt == 'l');
	hflag = (*fmt == 'h');
	if (lflag || hflag)
	    fmt++;

	sign = '\0';
	ppad = lpad = rpad = 0;
	maxch = -1;
	prefix = suffix = "";

	switch (fc = *fmt++) {

	case '\0':
	    return;

	case 'c':
	    c = ARG_INT(argc, argv);
	    str[0] = (unsigned char)c;
	    str[1] = '\0';
	    s = str;
	    break;

	case 's':
	    s = ARG_STR(argc, argv);
	    maxch = prec;
	    break;

	case 'd':
	case 'i':
	    if (lflag) {
		long val = ARG_LONG(argc, argv);
		if (val < 0) {
		    val = -val;		/* doesn't work for MINLONG */
		    sign = '-';
		} else
		    sign = plus;
		s = ltoa((unsigned long)val, str, 10, digits);
	    } else {
		int val = ARG_INT(argc, argv);
		if (hflag)
		    val = (short)val;
		if (val < 0) {
		    val = -val;		/* doesn't work for MININT */
		    sign = '-';
		} else
		    sign = plus;
		s = ltoa((unsigned long)val, str, 10, digits);
	    }
	    if (zeropad)
		lpad = width - LENGTH(s) - HAS_SIGN;
	    break;

	case 'o':
	    if (lflag) {
		unsigned long val = ARG_ULONG(argc, argv);
		s = ltoa((unsigned long)val, str, 8, digits);
	    } else {
		unsigned int val = ARG_UINT(argc, argv);
		if (hflag)
		    val = (unsigned short)val;
		s = ltoa((unsigned long)val, str, 8, digits);
	    }
	    if (alternate)
		prefix = "0";
	    if (zeropad)
		lpad = width - LENGTH(s) - alternate;
	    break;

	case 'x':
	case 'X':
	    if (lflag) {
		unsigned long val = ARG_ULONG(argc, argv);
		s = ltoa((unsigned long)val, str, 16,
			 (fc == 'x') ? digits : Digits);
	    } else {
		unsigned int val = ARG_UINT(argc, argv);
		if (hflag)
		    val = (unsigned short)val;
		s = ltoa((unsigned long)val, str, 16,
			 (fc == 'x') ? digits : Digits);
	    }
	    if (alternate)
		prefix = (fc == 'X') ? "0X" : "0x";
	    if (zeropad)
		lpad = width - LENGTH(s) - 2*alternate;
	    break;

	case 'u':
	    if (lflag) {
		unsigned long val = ARG_ULONG(argc, argv);
		s = ltoa((unsigned long)val, str, 10, digits);
	    } else {
		unsigned int val = ARG_UINT(argc, argv);
		if (hflag)
		    val = (unsigned short)val;
		s = ltoa((unsigned long)val, str, 10, digits);
	    }
	    if (zeropad)
		lpad = width - LENGTH(s);
	    break;

	case 'e':
	case 'E':
	    ;{
		char *t;
		int sgn, decpt, exp, n;
		double val = ARG_DOUBLE(argc, argv);

		if (prec < 0)
		    prec = 6;
		t = clr0(ecvt(val, min(prec+1, ECVTMAX), &decpt, &sgn));
		sign = sgn ? '-' : plus;

		n = prec;
		s = str;
		exp = (t[0] == '0' && t[1] == '\0') ? 0 : decpt - 1;

		*s++ = *t++;
		if (n > 0 || alternate)
		    *s++ = '.';
		while (*t != '\0' && --n >= 0)
		    *s++ = *t++;
		*s = '\0';
		rpad = n;

		sgn = 0;
		if (exp < 0) {
		    exp = -exp;
		    sgn = 1;
		}
		t = ltoa((unsigned long)exp, str, 10, digits);
		if (exp < 10)
		    *--t = '0';		/* always at least two digits */
		*--t = sgn ? '-' : '+';
		*--t = fc;

		if (zeropad) {
		    lpad = width - HAS_SIGN - (s - str) - LENGTH(t);
		    if (rpad > 0)
			lpad -= rpad;
		}

		suffix = t;
		s = str;
	    }
	    break;

	case 'f':
	    ;{
		char *t;
		int sgn, decpt, n;
		double val = ARG_DOUBLE(argc, argv);

		if (prec < 0)
		    prec = 6;

		t = clr0(fcvt(val, min(prec,FCVTMAX), &decpt, &sgn));

		sign = sgn ? '-' : plus;

		n = prec;
		s = str;

		if (decpt <= 0) {
		    prefix = (n > 0 || alternate) ? "0." : "0";
		    lpad = min(-decpt, prec);
		    n -= lpad;
		} else {
		    while (--decpt >= 0)
			*s++ = *t++;
		    if (n > 0 || alternate)
			*s++ = '.';
		}
		while (*t && --n >= 0)
		    *s++ = *t++;

		*s = '\0';
		rpad = n;

		if (zeropad)
		    ppad = width - HAS_SIGN - (prefix[1] ? 2 : 1) - lpad -
			(s - str) - rpad;

		s = str;
	    }
	    break;

	default:
	    continue;
	}

	if (lpad < 0)
	    lpad = 0;
	if (rpad < 0)
	    rpad = 0;
	if (width < 0)
	    width = 0;

	i = strlen(s);
	if (maxch <= 0 || maxch > i)
	    maxch = i;

	length = HAS_SIGN + ppad + strlen(prefix) + lpad + maxch + rpad + strlen(suffix);
	padding = 0;
	if (width != 0) {
	    padding = width - length;
	}

	if (ljust == 0)			/* left padding */
	    for (i = padding; --i >= 0;)
		obstack_1grow(obs, ' ');
	if (HAS_SIGN)			/* sign */
	    obstack_1grow(obs, sign);
	for (i = ppad; --i >= 0;)	/* pre-prefix zero padding */
	    obstack_1grow(obs, '0');
	for (; *prefix; ++prefix)	/* prefix */
	    obstack_1grow(obs, *prefix);
	for (i = lpad; --i >= 0;)	/* left zero padding */
	    obstack_1grow(obs, '0');
	for (i = maxch; --i >= 0; ++s)	/* actual text */
	    obstack_1grow(obs, *s);
	for (i = rpad; --i >= 0;)	/* right zero padding */
	    obstack_1grow(obs, '0');
	for (; *suffix; ++suffix)	/* suffix */
	    obstack_1grow(obs, *suffix);
	if (ljust != 0)			/* right padding */
	    for (i = padding; --i >= 0;)
		obstack_1grow(obs, ' ');
    }
    /* NOTREACHED */
}
