/* pc.c - terminal I/O calls for GNU info for OS/2 and DOS
   and other PC support functions
   Kai Uwe Rommel, 1990-1991
 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include <io.h>
#include <conio.h>
#include <fcntl.h>
#include <string.h>

#define INCL_NOPM
#define INCL_VIO
#define INCL_DOSPROCESS
#include <os2.h>

#include "pc.h"

#define OUTBUF 1024
char outbuf[OUTBUF + 1];
int outlen = 0;

void
pc_initterm (int *terminal_rows, int *terminal_columns)
{
  VIOMODEINFO vi;

  if ( _osmode == DOS_MODE )
  {
    *terminal_columns  = * (char _far *) 0x0040004A;
    *terminal_rows     = * (char _far *) 0x00400084 + 1;

    if ( *terminal_rows == 0 )
      *terminal_rows = 25;
  }
  else
  {
    vi.cb = sizeof(vi);
    VioGetMode(&vi, 0);
    *terminal_columns  = vi.col;
    *terminal_rows     = vi.row;
  }

  setmode(1, O_BINARY);
}

void pc_flush(void)
{
  if ( outlen )
  {
    write(1, outbuf, outlen);
    outlen = 0;
  }
}

void pc_putchar (int chr)
{
  if ( outlen == OUTBUF )
    pc_flush();

  outbuf[outlen++] = (char) chr;
}

int pc_getc (void)
{
  int c;

  pc_flush();
  c = getch ();

  if (c != 0x00 && c != 0xE0)
    return c;
  else
    switch (c = getch ())
      {
      case 25:
        /* Alt-P -> Ctrl-P, easier to read */
        return 'P'-64;

      case 59:
        /* F1 */
        return 'h';

      case 77:
        /* Right Arrow */
        return 'n';
      case 75:
        /* Left Arrow */
        return 'p';

      case 71:
	/* Home */
	return 'b';
      case 79:
        /* End */
        return 'e';

      case 119:
        /* Ctrl-Home */
        return 'd';
      case 145:
        /* Ctrl-Down Arrow */
        return 'f';
      case 141:
        /* Ctrl-Up Arrow */
        return 'u';

      case 80:
        /* Down Arrow */
        return '\t';
      case 72:
        /* Up Arrow */
        return '\v';

      case 83:
	/* Del */
      case 73:
	/* Page up */
	return 0177;
      case 81:
	/* Page down */
        return ' ';

      default:
	return ' ';
      }
}

typedef enum { unopened = 0, reading, writing } pipemode;

static struct
{
    char *name;
    char *command;
    pipemode pmode;
}
pipes[_NFILE];

static FILE *dos_popen(char *command, char *mode)
{
    FILE *current;
    char name[128];
    char *tmp = getenv("TMP");
    int cur;
    pipemode curmode;

    /*
    ** decide on mode.
    */
    if(strchr(mode, 'r') != NULL)
        curmode = reading;
    else if(strchr(mode, 'w') != NULL)
        curmode = writing;
    else
        return NULL;

    /*
    ** get a name to use.
    */

    strcpy(name, tmp ? tmp : "\\");
    if ( name[strlen(name) - 1] != '\\' )
      strcat(name, "\\");
    strcat(name, "piXXXXXX");
    mktemp(name);

    /*
    ** If we're reading, just call system to get a file filled with
    ** output.
    */
    if(curmode == reading)
    {
        char cmd[256];
        sprintf(cmd,"%s > %s", command, name);
        system(cmd);

        if((current = fopen(name, mode)) == NULL)
            return NULL;
    }
    else
    {
        if((current = fopen(name, mode)) == NULL)
            return NULL;
    }

    cur = fileno(current);
    pipes[cur].name = strdup(name);
    pipes[cur].command = strdup(command);
    pipes[cur].pmode = curmode;

    return current;
}

static int dos_pclose(FILE * current)
{
    int cur = fileno(current), rval;
    char command[256];

    /*
    ** check for an open file.
    */
    if(pipes[cur].pmode == unopened)
        return -1;

    if(pipes[cur].pmode == reading)
    {
        /*
        ** input pipes are just files we're done with.
        */
        rval = fclose(current);
        unlink(pipes[cur].name);
    }
    else
    {
        /*
        ** output pipes are temporary files we have
        ** to cram down the throats of programs.
        */
        fclose(current);
        sprintf(command,"%s < %s", pipes[cur].command, pipes[cur].name);
        rval = system(command);
        unlink(pipes[cur].name);
    }

    /*
    ** clean up current pipe.
    */
    free(pipes[cur].name);
    free(pipes[cur].command);
    pipes[cur].pmode = unopened;

    return rval;
}

FILE *popen(char *cmd, char *mode)
{
  return (_osmode == DOS_MODE) ? dos_popen(cmd, mode) : _popen(cmd, mode);
}

int pclose(FILE *ptr)
{
  return (_osmode == DOS_MODE) ? dos_pclose(ptr) : _pclose(ptr);
}

void sleep(int sec)
{
  DosSleep(1000L * sec);
}

long hread(int fd, char huge * buffer, long bytes)
{
  long bytes_read = 0L;
  unsigned n;
  char *local;

  if ( (local = malloc(0x4001)) == NULL )
  {
    perror("virtual memory exhausted");
    exit (4);
  }

  while ( 1 )
  {
    n = read(fd, local, (unsigned int) min(0x4000L, bytes));

    if ( n > 0 )
    {
      memcpy(buffer, local, n);
      buffer += n;
      bytes_read += (long) n;
      bytes -= (long) n;
    }
    else
    {
      free(local);
      return (n == -1) ? -1L : bytes_read;
    }
  }
}

long hwrite(int fd, char huge *buffer, long bytes)
{
  long bytes_written = 0L;
  unsigned n, m;
  char *local;

  if ( (local = malloc(0x4001)) == NULL )
  {
    perror("virtual memory exhausted");
    exit (4);
  }

  while ( 1 )
  {
    n = (unsigned int) min(0x4000L, bytes);
    memcpy(local, buffer, n);
    buffer += n;

    m = write(fd, local, n);

    if ( m > 0  )
    {
      bytes_written += (long) m;
      bytes -= (long) m;

      if ( bytes <= 0 )
      {
	free(local);
	return bytes_written;
      }
    }
    else
    {
      free(local);
      return (m == -1) ? -1L : bytes_written;
    }
  }
}
