/* scpdos.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date: 2002/06/19 14:09:35 $
 * $Revision: 1.11 $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <time.h>
#include <stdarg.h>
#include <dir.h>
#include <dos.h>
#include <utime.h>
#include <sys/stat.h>

#if defined (__DJGPP__)
 #include <unistd.h>
 #include <crt0.h>
 #include "tcp_djgp.h"
#elif defined (__TURBOC__)
 #include "tcp.h"
 #include <io.h>
#endif

#include "ssh.h"
#include "protocol.h"
#include "cipher.h"
#include "version.h"
#include "xmalloc.h"
#include "config.h"

#ifdef __DJGPP__
/*
 * we don't want DJGPP to extend wildcards, so include
 * this dummy function according to the official FAQ
 */
char **__crt0_glob_function (char *arg)
{
  return 0;
}
#endif

struct scp_sink_action {	/* SCP control block */
   char action;			/* FILE, DIR, ENDDIR */
   char *buf;			/* will need freeing after use */
   char *name;			/* filename or dirname (not ENDDIR) */
   int mode;			/* access mode (not ENDDIR) */
   unsigned long size;		/* file size (not ENDDIR) */
   char settime;		/* 1 if atime and mtime are filled */
   unsigned long atime, mtime;	/* access times for the file */
};

/* external functions */
SendFuncPtr SendPacket;
extern SessKeyInitPtr SessKeyInit;
extern CryptPtr EncryptPacket;
extern CryptPtr DecryptPacket;

/* external structures, variables */
extern struct Packet pktin;	/* incoming SSH packet */
extern struct Packet pktout;	/* outgoing SSH packet */

/* global variables */
Config GlobalConfig;		/* global configuration structure */
unsigned short Configuration = 0;	/* Configuration bits */

/* local variables */
static short firstarg;		/* first argument, which is not a switch */
static char command[256];	/* buffer for the command to send */
static char TargetShouldBeDirectory = 0; /* nomen est omen */
static char IsDir = 0;		/* is it really a directory? */

static FILE *fr;		/* File handle */
static char *localfile;		/* pointer to local filename */
static char *remotefile;	/* pointer to remote filename */
static char *transbuffer;	/* buffer for sending files */
static unsigned pendlen = 0;	/* we have unused SCP control bytes */
static char *pendbuf = NULL;	/* buffer for unused SCP control bytes */
static char local;		/* decide copying direction */

/* local functions begin here */
static void rsource(char *);

/*
 * Get the size of a file
 */
static unsigned long Getfilesize(char *s)
{
struct stat ss;

   stat(s, &ss);
   return(ss.st_size);
}

/*
 * Get the creation time of a file
 */
static unsigned long Getfiletime(char *s)
{
struct stat ss;
   stat(s, &ss);
   return(ss.st_atime);
}

/*
 * Get attributes of a file, return error if doesn't exist
 */
static int Getfileattr(char *s)
{
struct stat ss;
int j;

   j = stat(s, &ss);
   if(j)
	return(j);
   else
	return(ss.st_mode);
}

/*
 * Determine whether a string is entirely composed of dots
 */
static int is_dots(char *str)
{
   return str[strspn(str, ".")] == '\0';
}

/*
 * Return a pointer to the portion of str that comes after the last
 * slash (or backslash or colon, if `local' is TRUE).
 */
static char *stripslashes(char *str, int local)
{
char *p;

   if(local){
	p = strchr(str, ':');
	if (p) str = p+1;
   }

   p = strrchr(str, '/');
   if(p)
	str = p+1;

   if(local){
	p = strrchr(str, '\\');
	if (p) str = p+1;
   }

   return str;
}

/*
 * Allocate the concatenation of N strings. Terminate arg list with NULL
 */
static char *dupcat(char *s1, ...)
{
int len;
char *p, *q, *sn;
va_list ap;

   len = strlen(s1);
   va_start(ap, s1);
   while(1){
	sn = va_arg(ap, char *);
	if(!sn)
	   break;
	len += strlen(sn);
   }
   va_end(ap);

   p = xmalloc(len + 1);
   strcpy(p, s1);
   q = p + strlen(p);

   va_start(ap, s1);
   while(1){
	sn = va_arg(ap, char *);
	if(!sn)
	   break;
	strcpy(q, sn);
	q += strlen(q);
   }
   va_end(ap);

   return p;
}

/*
 * fatal error handler
 */
void fatal(const char *fmt, ...)
{
va_list ap;
char buf[256];

   va_start(ap, fmt);
   vsprintf(buf, fmt, ap);
   va_end(ap);
   printf("%s\n", buf);
   if(GlobalConfig.debugfile)
	fclose(GlobalConfig.debugfile);
   if(fr != NULL)
	fclose(fr);
   sock_close(&GlobalConfig.s);
   exit(255);
}

/*
 * Initialize global variables
 */
static void Config_Init(void)
{
   GlobalConfig.username = NULL;
   GlobalConfig.password = NULL;
   GlobalConfig.remotehost = NULL;
   GlobalConfig.RemotePort = SSH_PORT;
   GlobalConfig.identity = NULL;
   GlobalConfig.CipherType = SSH_CIPHER_BLOWFISH;
   GlobalConfig.debugfile = NULL;

   SendPacket = SendSSHPacket;
   SessKeyInit = blowfish_sesskey;
   EncryptPacket = blowfish_encrypt_blk;
   DecryptPacket = blowfish_decrypt_blk;
}

/*
 * Send a command to be executed on the remote host
 */
void sendcommand(void)
{
unsigned int len;

   s_wrpkt_start(SSH_CMSG_EXEC_CMD, (len = strlen(command)) + 4);
   pktout.body[0] = pktout.body[1] = 0;
   pktout.body[2] = (len >> 8) & 0xFF;
   pktout.body[3] = len & 0xFF;            /* up to 256 bytes command */
   memcpy(pktout.body+4, command, len);
   s_wrpkt();
}

/*
 * Get command line arguments
 */
static void getargs(int argc, char *argv[])
{
int i, j, remote = 0;
struct ffblk ffblk;
char *s;
char *usage="Usage: scpdos [options] from to\n"
            "from = <localfile | username@remotehost:remotefile>\n"
            "  to = <localfile | username@remotehost:remotefile>\n"
	    "Wildcards are accepted.\n"
	    "Options:\n"
	    "-c <3des|blowfish>     - cipher type\n"
	    "-i <identity file>     - identity file\n"
	    "-p                     - preserve file attributes\n"
	    "-r                     - recursively copy directories\n"
	    "-l                     - convert sent filenames to lowercase\n"
	    "-v                     - verbose output\n"
	    "-q                     - disable progess meter\n"
	    "-C                     - enable compression\n"
	    "-P <port>              - remote port number\n"
	    "-s                     - remote password\n"
	    "-d                     - save SSH packets to debug.pkt\n"
	    "Default is blowfish cipher.";

   for (i = 1; i < argc; ++i){
	s = argv[i];
	if(*s != '-')
	   break;
	switch (*++s){
	   case '\0':
	    fatal(usage);
	    return;

	   case 'c':
		if(*++s){
			if(!strcmp(s,"blowfish"))
				break;
			else if(!strcmp(s,"3des")){
				GlobalConfig.CipherType = SSH_CIPHER_3DES;
				SessKeyInit = des3_sesskey;
				EncryptPacket = des3_encrypt_blk;
				DecryptPacket = des3_decrypt_blk;
			}
			else fatal(usage);
		}
		else if(++i < argc){
			if(!strcmp(argv[i],"blowfish"))
				break;
			else if(!strcmp(argv[i],"3des")){
				GlobalConfig.CipherType = SSH_CIPHER_3DES;
				SessKeyInit = des3_sesskey;
				EncryptPacket = des3_encrypt_blk;
				DecryptPacket = des3_decrypt_blk;
			}
			else fatal(usage);
		}
		else fatal(usage);
		continue;

	   case 'i':
		if (*++s)
		   GlobalConfig.identity = s;
		else if (++i < argc)
		   GlobalConfig.identity = argv[i];
		else
		   fatal(usage);
		continue;

	   case 's':
		if(*++s)
		   GlobalConfig.password = strdup(s);
	 	else if(++i < argc)
		   GlobalConfig.password = strdup(argv[i]);
	 	else
		   fatal(usage);
		GlobalConfig.password[MAX_PASSWORD_LENGTH] = '\0';
	 	continue;

	   case 'p':
		Configuration |= PRESERVE_ATTRIBUTES;
		continue;

	   case 'l':
		Configuration |= CONVERT_LOWERCASE;
		continue;

	   case 'r':
		Configuration |= RECURSIVE_COPY;
		continue;

	   case 'v':
		Configuration |= VERBOSE_MODE;
		continue;

	   case 'q':
		Configuration |= QUIET_MODE;
		continue;

	   case 'C':
		Configuration |= COMPRESSION_REQUESTED;
		continue;

	   case 'P':
		if(*++s)
		   GlobalConfig.RemotePort = atoi(s);
		else if(++i < argc)
		   GlobalConfig.RemotePort = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'd':
		if((GlobalConfig.debugfile = fopen("debug.pkt","w+")) == NULL)
		   fatal("Cannot create debug file");
		else
		   fputs("\n-------------------\n",GlobalConfig.debugfile);
		continue;

	   default:
		fatal(usage);
	} /* end switch */
   } /* end for */

/* no_more_options */
   if(i + 2 > argc)
	fatal(usage);

/*
 * Try and work out which file is remote and which file is local 
 * 
 * Works on the assumption that the "remote file" has to have a
 * "@" and a":" character.
 */

   if(strchr(argv[i],'@')!=NULL && strchr(argv[i],':')!=NULL)
	local = 1;
   if(strchr(argv[argc - 1],'@')!=NULL && strchr(argv[argc - 1],':')!=NULL)
	remote = 1;

   if((local == 1) && (remote == 1))
	fatal("Error - both files are remote");
   else if((local == 0) && (remote == 0))
	fatal("Error - both files are local");

   if(local){
	GlobalConfig.username = argv[i];
	GlobalConfig.remotehost = argv[i];
        localfile = argv[i+1];
   }
   else{
	GlobalConfig.username = argv[argc - 1];
	GlobalConfig.remotehost = argv[argc - 1];
   }

   firstarg = i;
   GlobalConfig.remotehost = strchr(GlobalConfig.username, '@');
   *GlobalConfig.remotehost++ = '\0';	/* kill '@' after username */

   remotefile = strchr(GlobalConfig.remotehost, ':');
   *remotefile++ = '\0';		/* kill ':' after hostname */

/*
 * Check if the specified thing is a directory,
 * more files or wildcards, and exists or not
 */

   if(local){ /* from remote to local */
	if(firstarg + 2 != argc)
	   fatal("More than one remote source not supported");
        j = Getfileattr(localfile);
	if((j != -1) && (j & S_IFDIR)) /* does it exist and is it a dir */
		IsDir = 1;
	if(strchr(remotefile, '*') || strchr(remotefile, '?')){
	   TargetShouldBeDirectory = 1;
           if(!IsDir) /* if local must be directory but it isn't, bomb out */
		fatal("Error - must specify a local directory");
	}
   } else { /* from local to remote */
	/* More local files, recursive mode or wildcards specified? */
	if((Configuration & RECURSIVE_COPY) || (firstarg + 2 < argc) ||
	    strchr(argv[firstarg], '*') || strchr(argv[firstarg], '?'))
		TargetShouldBeDirectory = 1;

	/* Examine local file(s) for existence */
	for(i = firstarg; i < argc - 1; i++)
           if(findfirst(argv[i], &ffblk, FA_DIREC))
		fatal("Error - %s not found", argv[i]);

	/* if no remote file specified, let it be a '.' */
	if(*remotefile <= ' ')
	   remotefile = ".";
   } /* else */
}

/*
 * Progress meter
 */
static void progress(unsigned long fsize, unsigned long cursize)
{
unsigned int ratio;
int x, y, i;

   x = wherex();
   y = wherey();

   if(fsize != 0) {
	ratio = 100 * cursize / fsize;
   } else
	ratio = 100;
   i = 30 * ratio / 100;
   cprintf("|%.*s%*s|", i, "***********************************", 30 - i, "");
   gotoxy(x, y);
}

/*
 * Send a file to the host
 */
static void source(char *file)
{
unsigned long FileSize, FileTime;
unsigned long len;
unsigned long sent;
unsigned long i;
char cmdbuf[80];
char *p;

   if(Configuration & CONVERT_LOWERCASE)
	strlwr(file); /* if we're asked to convert it to lowercase */
   /*
    * If we have a directory name here, we must check if recursive
    * mode enabled or not. If yes, do it, else warn
    */
   i = Getfileattr(file);
   if(i & S_IFDIR){
	if(Configuration & RECURSIVE_COPY){
	   p = strrchr(file, '\\');
	   if(!p)
		p = file;
	   else
		p++;
	   if(!strcmp(p, ".") || !strcmp(p, ".."))
		/* skip . and .. */ ;
	   else
		rsource(file);
	} else {
	   printf("%s: not a regular file\n", file);
        }
        return;
   }

   /*
    * Now it's sure that we have a regular file here.
    * Get it's size and time, open it and send
    */
   FileSize = Getfilesize(file);
   FileTime = Getfiletime(file);

   if((fr = fopen(file, "rb")) == NULL)
	fatal("Error - cannot open %s", file);

   /* preserve attributes if configured */
   if(Configuration & PRESERVE_ATTRIBUTES){
	sprintf(cmdbuf, "T%lu 0 %lu 0\n", FileTime, FileTime);
	SendPacket(cmdbuf, strlen(cmdbuf));
	packet_read_type();
   }

   /* change '\' to '/' */
   for(i = 0; i < strlen(file); i++)
        if(file[i] == '\\')
	   file[i] = '/';

   /* we only need the file name without directory name */
   if((p = strrchr(file, '/')) == NULL)
	p = file;
   else
	p++;

   printf("%s (%lu bytes): ",p , FileSize);

   /* send filename and size */
   sprintf(cmdbuf, "C%04o %lu %s\n", 0644, FileSize, p);
   SendPacket(cmdbuf, strlen(cmdbuf));
   packet_read_type();

   /* send file itself */
   sent = 0;
   transbuffer=(char *)xmalloc(TRANSBUF_SIZE * sizeof(char));
   for (i = 0; i < FileSize; i += TRANSBUF_SIZE){
	len = fread(transbuffer, 1, TRANSBUF_SIZE, fr);
	SendPacket(transbuffer, len);
	sent += len;
	if(!(Configuration & QUIET_MODE))
           progress(FileSize, sent);
   } /* for */

   /* finish SCP send */
   xfree(transbuffer);
   fclose(fr);
   SendPacket("", 1);
   packet_read_type();
   puts("");
}

/*
 *  Recursively send the contents of a directory.
 */
static void rsource(char *src)
{
char *findfile, *foundfile, *last;
char cmdbuf[80];
struct ffblk ffblk;

   if((last = strrchr(src, '\\')) == NULL)
	last = src;
   else
	last++;
   if(strrchr(last, '\\'))
	last = strrchr(last, '\\') + 1;
   if(last == src && strchr(src, ':'))
	last = strchr(src, ':') + 1;

   if(Configuration & VERBOSE_MODE)
	printf("Entering directory: %s\n", last);
   sprintf(cmdbuf, "D%04o 0 %s\n", 0755, last);
   SendPacket(cmdbuf, strlen(cmdbuf));
   packet_read_type();

   findfile = dupcat(src, "\\*.*", NULL);
   findfirst(findfile, &ffblk, FA_DIREC);
   xfree(findfile);
   do{
	if(!strcmp(ffblk.ff_name, ".") ||
	   !strcmp(ffblk.ff_name, ".."))	/* ignore . and .. */
		continue;
	foundfile = dupcat(src, "\\", ffblk.ff_name, NULL);
	source(foundfile);
	xfree(foundfile);
   } while(!findnext(&ffblk));
   if(Configuration & VERBOSE_MODE)
	printf("Leaving directory: %s\n", last);
   SendPacket("E\n", 2);
   packet_read_type();
}

/*
 * Get data from the SSH layer to the SCP layer
 */
static int ssh_scp_recv(char *buf, int len)
{
   /*
    * If we have enough pending data, no problem. However,
    * if the SCP layer needs more than we have, we must
    * get enough data with blocking from the SSH layer
    */
restart:
   if (pendlen >= len) { /* we have enough? */
	memcpy(buf, pendbuf, len);
	memmove(pendbuf, pendbuf + len, pendlen - len);
	pendlen -= len;
	if(pendlen == 0){ /* if we have no more, free the buffer */
	   xfree(pendbuf);
	   pendbuf = NULL;
	} else
	   xrealloc(pendbuf, pendlen);
	return len;
    } else { /* we must wait for more input from the SSH layer */
	packet_read_block();
	pktin.length -= 5;
	if(!pktin.length)
	   return(0);
	pendlen += pktin.length;
	pendbuf = (pendbuf ? xrealloc(pendbuf, pendlen) :
		xmalloc(pendlen));
        memcpy(pendbuf + pendlen - pktin.length, pktin.body + 4, pktin.length);
        xfree(pktin.whole);
        goto restart;
    }
}

/*
 * Get the next SCP control packet and decide what to do
 */
static int scp_get_sink_action(struct scp_sink_action *act)
{
int i, done, bufsize, action;
char ch;

   act->settime = done = bufsize = 0;
   act->buf = NULL;

   while (!done) {
	if(ssh_scp_recv(&ch, 1) <= 0) /* get the command byte */
	   return(1);
        if (ch == '\n')
	   fatal("Protocol error: Unexpected newline");
	i = 0;
	action = ch;
	do{ /* get the remaining command string */
	   if(ssh_scp_recv(&ch, 1) != 1)
		fatal("Lost connection");
	   if(i >= bufsize){
		bufsize = i + 128;
		act->buf = xrealloc(act->buf, bufsize);
	   }
	   act->buf[i++] = ch;
	} while (ch != '\n');
	act->buf[i - 1] = '\0';

        switch(action){
	   case '\01':	/* warning message */
	        puts(act->buf);
	        continue;
	   case '\0':	/* fatal error */
	        fatal("%s", act->buf);
	   case 'E':	/* end of directory */
	        SendPacket("", 1);
	        act->action = SCP_SINK_ENDDIR;
	        return(0);
	   case 'T':	/* file time */
                if(sscanf(act->buf, "%ld %*d %ld %*d",
			   &act->mtime, &act->atime) == 2){
                   act->settime = 1;
		   SendPacket("", 1);
		   continue;	       /* go round again */
	        }
	        fatal("Protocol error: Illegal time format");
	   case 'C':	/* create file */
	   case 'D':	/* create directory */
	        act->action = (action == 'C' ? SCP_SINK_FILE : SCP_SINK_DIR);
	        break;
	   default:
	        fatal("Protocol error: Expected control record");
        }
        done = 1;
   }
   if(sscanf(act->buf, "%o %lu %n", &act->mode, &act->size, &i) !=2)
        fatal("Protocol error: Illegal file descriptor format");
   act->name = act->buf + i;
   return(0);
}

/*
 * Receive a file from the host
 */
static void sink(char *targ)
{
unsigned long received;
unsigned long gotbytes;
int attr, exists;
char *striptarget, *destfname;
struct scp_sink_action act;
struct utimbuf times;
short error;

   SendPacket("", 1);
   while(1){
	if(scp_get_sink_action(&act)) /* get command string from host */
	   return;
	if(act.action == SCP_SINK_ENDDIR)
	   return;
	if(act.action == SCP_SINK_RETRY)
	   continue;

        if(IsDir){
	    /*
	     * Prevent the remote side from maliciously writing to
	     * files outside the target area by sending a filename
	     * containing `../'. In fact, it shouldn't be sending
	     * filenames with any slashes or colons in at all; so
	     * we'll find the last slash, backslash or colon in the
	     * filename and use only the part after that. (And
	     * warn!)
	     */
            striptarget = stripslashes(act.name, 1);
	    if(striptarget != act.name){
		printf("Warning: remote host sent a compound pathname '%s'",
                   act.name);
		printf("         renaming local file to '%s'", striptarget);
	    }

	    /*
	     * Also check to see if the target filename is '.' or
	     * '..', or indeed '...'
	     */
	    if (is_dots(striptarget)) {
		fatal("Security violation: remote host attempted to write to"
		     " a '.' or '..' path!");
	    }

	    if (targ[0] != '\0')
		destfname = dupcat(targ, "\\", striptarget, NULL);
	    else
		destfname = strdup(striptarget);
        } else { /* plain file */
	    destfname = strdup(targ);
        }

	attr = Getfileattr(destfname);
	exists = (attr != -1);

	if(act.action == SCP_SINK_DIR){ /* create that directory */
           if(!exists)
#if defined (__TURBOC__)
		if(mkdir(destfname))
#elif defined (__DJGPP__)
		if(mkdir(destfname, NULL))
#endif
		   fatal("Cannot create directory %s", destfname);
	   sink(destfname);
	   continue;
        }

	/* It's sure that we have a regular filename here */
	error = 0;
	if((fr = fopen(destfname, "wb"))==NULL){
	   printf("Cannot create %s\n", destfname);
	   error++;
	}
        SendPacket("", 1);

	printf("%s (%lu bytes): ",destfname, act.size);

	received = 0;
	while (received < act.size) {
	   packet_read_block();
	   gotbytes = pktin.length - 5;
	   received += gotbytes;
	   if(gotbytes > act.size)
		gotbytes = act.size;
	   else if(gotbytes > received - act.size)
		gotbytes -= received - act.size;
	   if(!error)	/* write only if we could create this file */
		fwrite(pktin.body + 4, 1, gotbytes, fr);
	   xfree(pktin.whole);
	   if(!(Configuration & QUIET_MODE))
		progress(act.size, received);
	} /* while */

	if(!error)
	   fclose(fr);
	xfree(act.buf);
	if(act.settime){
	   times.modtime = act.mtime;
	   utime(destfname, &times);
	}
        xfree(destfname);
	SendPacket("", 1);
	puts("");
   } /* while */
}

/*
 * Receive files
 */
static void tolocal(void)
{
   sprintf(command, "scp%s%s%s -f %s",
        TargetShouldBeDirectory ? " -d" : "",
        Configuration & PRESERVE_ATTRIBUTES ? " -p" : "",
        Configuration & RECURSIVE_COPY ? " -r" : "",
	remotefile);
   sendcommand();
   puts("Receiving:");
   sink(localfile);
}

/*
 * Send files
 */
static void toremote(int argc, char *argv[])
{
struct ffblk ffblk;
int i;
char *last, *srcpath, *filename;

   /* send SCP command to the server */
   sprintf(command, "scp%s%s%s -t %s",
        TargetShouldBeDirectory ? " -d" : "",
	Configuration & PRESERVE_ATTRIBUTES ? " -p" : "",
	Configuration & RECURSIVE_COPY ? " -r" : "",
	remotefile);
   sendcommand();
   packet_read_type();

   puts("Sending:");

   /* Process all local file arguments */
   for(i = firstarg; i < argc - 1; i++){
	/*
	 * Trim off the last pathname component of `src', to
	 * provide the base pathname which will be prepended to
	 * filenames returned from Find{First,Next}File.
	 */
	srcpath = strdup(argv[i]);
	last = stripslashes(srcpath, 1);
	*last = '\0';

	/* sure it exists, we checked in getargs */
	findfirst(argv[i], &ffblk, FA_DIREC);
	do{
	   if(!strcmp(ffblk.ff_name, ".") || /* don't bother with '.' */
	      !strcmp(ffblk.ff_name, ".."))  /* and '..' */
                continue;
	   filename = dupcat(srcpath, ffblk.ff_name, NULL);
	   source(filename);
	   xfree(filename);
        } while(!findnext(&ffblk));
        xfree(srcpath);
   } /* for */
}

/*
 * Main program starts here
 */
int main(int argc, char **argv)
{
   printf("SCPDOS v%s\n\r", SSH_VERSION);

   Config_Init();	/* Initialize global variables */
   srand(time(NULL));	/* Initialize random number generator */

   getargs(argc, argv);	/* Process command line */

   if(TCPConnect(GlobalConfig.remotehost))	/* Connect to server */
	fatal("TCP connection error");
   if(SSHConnect())				/* begin SSH negotiation */
	fatal("SSH connection error");

   /* Check which way we are transferring */
   if(!local){
        toremote(argc, argv);		/* from local to remote */
	s_wrpkt_start(SSH_CMSG_EOF, 0);	/* send EOF when ready */
	s_wrpkt();
	packet_read_expect(SSH_SMSG_EXITSTATUS);
	xfree(pktin.whole);
   }
   else
        tolocal();	/* from remote to local */

   s_wrpkt_start(SSH_CMSG_EXIT_CONFIRMATION, 0);
   s_wrpkt();

#ifdef __TURBOC__
   if(Configuration & COMPRESSION_ENABLED)
	Disable_Compression();
#endif

   sock_close(&GlobalConfig.s);   /* Close TCP socket */

   /* Close open file */
   if(GlobalConfig.debugfile)
	fclose(GlobalConfig.debugfile);

   return(0);
}
