{ ****************************************************************************
  `SBMIX' - SBPro mixer control
    Written by Michal H. Tyc

  Copyright (c) 1997-2005 BTTR Software
  All rights reserved.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the `MODIFIED' BSD LICENSE.

  See `legal.txt' for details.
  **************************************************************************** }

{.$DEFINE DEBUG}  { remove `.' to activate }

{$IFDEF DEBUG}
  {$A+,B-,D+,E-,F-,G-,I+,L+,N-,O-,R+,S+,V-,X+}  { TP 6.x compiler options }
  {$IFDEF VER70}                                { TP 7.x specific options }
    {$P-,Q+,T-}
  {$ENDIF}
{$ELSE}
  {$A+,B-,D-,E-,F-,G-,I-,L-,N-,O-,R-,S-,V-,X+}  { TP 6.x compiler options }
  {$IFDEF VER70}                                { TP 7.x specific options }
    {$P-,Q-,T-}
  {$ENDIF}
{$ENDIF}
{$M 4096,0,0}

program sbmix;

uses
  Dos;

const
  SB_MIXIDX  = $04;   { Mixer index port, relative to base I/O address }
  SB_DSPRST  = $06;   { DSP reset port, base-relative }
  SB_DSPRD   = $0A;   { DSP read data port, base-relative }
  SB_DSPWR   = $0C;   { DSP write data/command port, base-relative }
  SB_DSPAV   = $0E;   { DSP data available status port, base-relative }
  MIX_MASTER = $22;   { Master volume port index }
  MIX_WAVE   = $04;   { Wave volume port index }
  MIX_FM     = $26;   { FM synth volume port index }
  MIX_LINE   = $2E;   { Line input volume port index }
  MIX_CD     = $28;   { CD input volume port index }
  MIX_MIC    = $0A;   { Mic input volume port index }
  DSP_GETVER = $E1;   { Get DSP Version command }
  DSPTIMEOUT = $4FF;  { Timeout for DSP }

var
  base,     { base I/O port address }
  i: Word;  { loop counter }

  model: Integer;  { SB card type }

  vol,        { Master volume (bits 7..4 = left, 3..0 = right channel) }
  wave,       { Wave volume (as above) }
  fm,         { FM synth volume (as above) }
  cd,         { CD input volume (as above) }
  line,       { Line input volume (as above) }
  mic,        { Mic input volume (mono, range = 0..7) }
  chg: Byte;  { flag: the mixer needs re-programming }

  flag1, flag2: Integer;  { flags for string to integer conversion }
  vl, vr: Longint;        { temporary variables for the same }
  par: string[31];        { currently analysed command-line parameter }
  blastenv: string[31];   { a copy of `BLASTER' environment variable }
  opt: Char;              { currently analysed command-line option }

{ Read option value from `BLASTER' environment variable mirrored
  in blastenv. Returns $ffff if option not found or malformed.   }
function getblastopt(opt, hex: Char): Word;
const
  STRLEN = 31;
type
  STRLESS = string[STRLEN - 2];
var
  i: Integer;
  p: Word;
  b: string[STRLEN];
const
  opt2: string[2] = '  ';
begin
  getblastopt := $FFFF;             { value to be returned on error }
  STRLESS((@b[1])^) := blastenv;    { option 'Oxxxx' }
  b[0] := Succ(b[1]); b[1] := ' ';  { always preceded with space }
  opt2[2] := opt;
  i := Pos(opt2, b);                { find the option }
  if (i <> 0) then
  begin
    Delete(b, 1, i);                { remove the part before 'Oxxxx' }
    i := Pos(' ', b) - 1;
    if (i >= 0) then
      b[0] := Chr(i);               { remove the part after 'Oxxxx' }
    b[1] := hex;                    { replace 'O' with '$' or space }
    Val(b, p, i);
    if (i = 0) then
      getblastopt := p;             { return the number if valid }
  end;
end;

{ Set SBPro mixer register }
procedure setmixreg(index, value: Byte); assembler;
asm
  mov dx, [base]
  add dx, SB_MIXIDX  { mixer index port }
  mov al, [index]
  out dx, al
  inc dx             { mixer data port }
  mov al, [value]
  out dx, al
end;

{ Read SBPro mixer register }
function getmixreg(index: Byte): Byte; assembler;
asm
  mov dx, [base]
  add dx, SB_MIXIDX  { mixer index port }
  mov al, [index]
  out dx, al
  inc dx             { mixer data port }
  in al, dx
end;

{ Write SBPro DSP Data/Command register. Return nonzero if timeout. }
function writedspreg(value: Byte): Byte; assembler;
asm
  mov dx, [base]
  add dx, SB_DSPWR    { DSP Data/Command (W) and Write Buffer Status (R) }
  mov cx, DSPTIMEOUT
@wait:
  in al, dx
  test al, 80h
  loopnz @wait        { wait until bit 7 = 0 }
  jnz @quit           { timeout, return nonzero, flags set for caller }
  mov al, [value]
  out dx, al
  xor al, al          { return zero, set flags for caller }
@quit:
end;

{ Read SBPro DSP Data (Lo) and Data Available (Hi) register.
  Return negative if timeout. }
function readdspreg: Integer; assembler;
asm
  mov dx, [base]
  add dx, SB_DSPAV               { DSP Data Available }
  mov cx, DSPTIMEOUT
  mov ah, 80h                    { bit mask }
@wait:
  in al, dx
  test al, ah
  loopz @wait                    { wait until bit 7 = 1 }
  mov ah, al                     { return also status byte }
  jz @quit                       { timeout, return nonzero }
  sub dx, (SB_DSPAV - SB_DSPRD)  { DSP Data Read }
  in al, dx
@quit:
  neg ah                         { set flags for caller }
end;

{ Reset DSP version, return negative if failed }
function resetdsp: Integer; assembler;
asm
  mov dx, [base]
  add dx, 6
  mov al, 1
  out dx, al       { write 1 to DSP Reset port }
  in al, dx
  in al, dx
  in al, dx
  in al, dx        { wait a bit (3.3 microseconds) }
  dec ax
  out dx, al       { then write 0 }
  call readdspreg  { read the answer }
  js @quit         { error }
  cmp al, 0AAh
  je @quit         { ok }
  neg ah           { error, set flags }
@quit:
end;

{ Return DSP version, negative if failed }
function dspver: Integer; assembler;
asm
  mov al, DSP_GETVER
  push ax
  call writedspreg    { send command }
  js @err
  call readdspreg     { read version major }
  xchg ax, bx         { save }
  js @err             { timeout }
  call readdspreg     { read version minor }
  jns @done
@err:
  mov bl, -1          { return negative }
@done:
  mov ah, bl          { return major:minor in ax }
end;

{ Reset SBPro mixer }
procedure resetmix; assembler;
asm
  mov dx, [base]
  add dx, 4
  mov al, 0       { mixer Reset port is index 0 }
  out dx, al      { mixer index port = base + 4 }
  inc dx
  inc ax          { 1 to reset }
  out dx, al      { mixer data port = base + 5 }
  dec ax          { then 0 }
  out dx, al
end;

{ Get current settings }
procedure getallmix;
begin
  vol  := getmixreg(MIX_MASTER);
  wave := getmixreg(MIX_WAVE);
  fm   := getmixreg(MIX_FM);
  line := getmixreg(MIX_LINE);
  cd   := getmixreg(MIX_CD);
  mic  := getmixreg(MIX_MIC);
end;

{ Set all corresponding mixer registers }
procedure setallmix;
begin
  resetmix;
  setmixreg(MIX_MASTER, vol);
  setmixreg(MIX_WAVE,   wave);
  setmixreg(MIX_FM,     fm);
  setmixreg(MIX_LINE,   line);
  setmixreg(MIX_CD,     cd);
  setmixreg(MIX_MIC,    mic);
end;

{ Display current mixer settings }
procedure showmix;
begin
  Writeln(' BLASTER environment variable... ', blastenv);
  Writeln(' DSP version.................... ',
          Hi(model), '.', Lo(model) div 10, Lo(model) mod 10);
  Writeln(' Master volume (Left, Right)....',
          vol  shr 4: 3, ',', vol  and 15: 3);
  Writeln(' Wave volume (Left, Right)......',
          wave shr 4: 3, ',', wave and 15: 3);
  Writeln(' FM volume (Left, Right)........',
          fm   shr 4: 3, ',', fm   and 15: 3);
  Writeln(' Line volume (Left, Right)......',
          line shr 4: 3, ',', line and 15: 3);
  Writeln(' CD volume (Left, Right)........',
          cd   shr 4: 3, ',', cd   and 15: 3);
  Writeln(' Mic volume.....................', mic  and 7: 3);
end;

{ Display help message }
procedure help;
begin
  Writeln('SBMIX Version 09-FEB-2005  SBPro mixer control'^M^J +
          'Copyright (c) 1997-2005 BTTR Software'^M^J +
          '[Under `MODIFIED'' BSD LICENSE]'^M^J +
          ^M^J,
          'Usage: SBMIX [<master>] [/W<wave>] [/F<fm>] [/C<cd>]' +
          ' [/L<line>] [/X<mic>]'^M^J +
          '       SBMIX [/Help|/?]'^M^J +
          ^M^J +
          '  <master> = <both> | <left>,<right> = master volume,     0..15'^M^J +
          '  <wave>   = <both> | <left>,<right> = wave (DSP) volume, 0..15'^M^J,
          '  <fm>     = <both> | <left>,<right> = FM synth volume,   0..15'^M^J +
          '  <cd>     = <both> | <left>,<right> = CD input volume,   0..15'^M^J +
          '  <line>   = <both> | <left>,<right> = Line input volume, 0..15'^M^J,
          '  <mic>    = <mono>                  = Mic input volume,  0..7'^M^J +
          ^M^J +
          'Note: `BLASTER'' environment variable must be set correctly!'^M^J +
          ^M^J +
          'Examples:'^M^J +
          '  SBMIX 3,5'^M^J +
          '  SBMIX 9 /W15,5 /F15,5 /C5,15 /L1 /X1'^M^J +
          '  SBMIX');
  Halt;
end;

{ Halt with error message }
procedure errhalt;
begin
  Writeln('SBMIX: invalid parameters! (SBMIX /? for help.)');
  Halt(1);
end;

{ Get two comma-separated integers from range 0..15;
  report error and halt if needed                    }
procedure getval2(var v: Byte);
var
  j: Integer;
begin
  j := pos(',', par);
  if (j <> 0) then
  begin
    Val(Copy(par, 1, j - 1), vl, flag1);
    Val(Copy(par, j + 1, 255), vr, flag2);
  end
  else
  begin
    val(par, vl, flag1); flag2 := 0; vr := vl;
  end;
  if (((flag1 or flag2) = 0)
      and (vl >= 0) and (vl <= 15) and (vr >= 0) and (vr <= 15)) then
  begin
    v := (Lo(vl) shl 4) + Lo(vr); Inc(chg);
  end
  else
    errhalt;
end;

{ Main program }
label
  DSPFAIL;
begin
{$IFDEF VER60}  { TP 6.x RTL does not initialize global variables }
  show := 0;
  chg := 0;
{$ENDIF}
  blastenv := GetEnv('BLASTER');
  base := getblastopt('A', '$');  { get base I/O address, hex }
  if (base = $FFFF) then
  begin
    Writeln('SBMIX: `BLASTER'' environment variable not set correctly!');
    Halt(1);
  end;
  if (getblastopt('T', ' ') or 1 <> 3) then  { `BLASTER' says SBPro+ }
  begin                                      { or says nothing }
    if (resetdsp < 0) then                   { reset the card }
      goto DSPFAIL;                          { failed }
    model := dspver;                         { check hardware }
    if ((model < 0)                          { failed }
        or (model <> dspver)) then           { conflict with another card }
    begin
DSPFAIL:
      Writeln('SBMIX: Hardware I/O error!');
      Halt(1);
    end;
  end;
  if (Hi(model) < 3) then
  begin
    Writeln('SBMIX: SoundBlaster Pro or compatible required!');
    Halt(1);
  end;

  getallmix;  { read current settings from SBPro registers }

  for i := 1 to ParamCount do
  begin
    par := ParamStr(i);
    if (par[1] = '/') then
    begin
      opt := UpCase(par[2]);
      Delete(par, 1, 2);      { remove the switch `/x', leave the value }
      case opt of
      'W': getval2(wave);
      'F': getval2(fm);
      'C': getval2(cd);
      'L': getval2(line);
      'X':
        begin
          Val(par, vl, flag1);
          if ((flag1 = 0) and (vl >= 0) and (vl <= 7)) then
          begin
            mic := vl; Inc(chg);  { Mic volume in range, set }
          end
          else
            errhalt;              { wrong Mic volume }
        end;
      'H',
      '?': help;
      else
        errhalt;
      end;
    end
    else
      getval2(vol);  { no preceding switch, this should be Master volume }
  end;

  if (chg <> 0) then  { some value was changed }
  begin
    setallmix;        { set new values }
    getallmix;        { re-read settings from SBPro registers }
  end;
  showmix;            { display all values }
end.