{
* sdlgrph (unit)
* some graph functions with SDL
*
* Copyright (c) 2005-2006 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: FreePascal and SDL4FreePascal
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*}

{$X+}

unit sdlgrph;

interface
uses uakfquiz, sdl, sdl_video, sdl_events, sdl_keyboard, sdl_mouse;

{$I hginfo.inc}

type TscreenPos = Integer;

const MaxAnswers = 35;

procedure initializeGraphicMode(const title, short: string; 
                                fullscreen: boolean);
procedure endGraphics;
procedure setColors(foreground, background: word);
procedure PutPixel(x, y: TscreenPos; color: word);
function  GetPixel(x, y: TscreenPos): word;
procedure showimage(x, y: Integer; var img);
procedure drawBackground(var img);
procedure MoveTo(x, y: TscreenPos);
function  GetX: TscreenPos;
function  GetY: TscreenPos;
function  GetMaxX: TscreenPos;
function  GetMaxY: TscreenPos;
function  GetRGBColor(r, g, b: byte): word;

procedure DefineTextArea(x1, y1, x2, y2: TscreenPos; useTextArea: boolean);
procedure ClearTextArea;

procedure setExitKey(c: char);
function GetKey: char;

procedure showmouse(on: boolean);
procedure answerStarts(ans: word);
procedure answerEnds(ans: word);


implementation

type TScrMap = array[0..ScreenHeight-1, 0..ScreenWidth-1] of word;

type 
  Tgrfimage = record
              Width    : longint;
              Height   : longint;
              reserved : longint;
              Image    : array[0..ScreenWidth*ScreenHeight] of word
              end;

var answerposition: array[1..MaxAnswers] of record f, t: TscreenPos end;

var screen, textarea: pSDL_SURFACE;
var tx1, ty1, tx2, ty2, tw, th : TscreenPos;
var xPos, yPos : TscreenPos;
var mode : LongInt;
var ExitKey : char;
var mouseactive, mouseshown: boolean;

procedure resetAnswerPositions;
var i: integer;
begin
for i:=1 to MaxAnswers do
    begin
    answerposition[i].f := -1;
    answerposition[i].t := -1
    end
end;

procedure answerStarts(ans: word);
begin
answerposition[ans].f := ypos
end;

procedure answerEnds(ans: word);
begin 
answerposition[ans].t := ypos
end;

procedure showmouse(on: boolean);
begin
mouseshown := on;
if mouseshown 
  then begin SDL_ShowCursor(SDL_ENABLE); mouseactive := true end
  else SDL_ShowCursor(SDL_DISABLE)
end;

procedure initializeGraphicMode(const title, short: string; 
                                fullscreen: boolean);
begin
xPos   := 0;
yPos   := 0;
tx1    := 0;
ty1    := 0;
tx2    := ScreenWidth-1;
ty2    := ScreenHeight-1;
textarea := NIL;

SDL_INIT(SDL_INIT_VIDEO);
SDL_WM_SetCaption(PChar(title + ' (SDL)'), PChar(short));

mode := SDL_SWSURFACE;
if fullscreen then mode := mode or SDL_FULLSCREEN;

screen := SDL_SETVIDEOMODE(ScreenWidth, ScreenHeight, 
                           ScreenDepth, mode);
if screen=NIL then 
   begin
   WriteLn(stderr, 'grquiz error: graphic mode not supported');
   Halt(1)
   end;

showmouse(true);

{ ignore some events }
SDL_EventState(SDL_MOUSEMOTION,   SDL_IGNORE);
SDL_EventState(SDL_MOUSEBUTTONUP, SDL_IGNORE);
SDL_EventState(SDL_KEYUP,         SDL_IGNORE);
SDL_EventState(SDL_EVENTACTIVE,   SDL_IGNORE);

{ enable unicode handling }
SDL_EnableUNICODE(1)
end;

procedure setExitKey(c: char);
begin
ExitKey := c
end;

procedure endGraphics;
begin
if textarea<>NIL then SDL_FREESURFACE(textarea);
SDL_FREESURFACE(screen);
SDL_QUIT
end;

procedure setColors(foreground, background: word);
begin end;

function GetRGBColor(r, g, b: byte): word;
begin
GetRGBColor := SDL_MapRGB(screen^.format, r, g, b)
end;

procedure PutPixel(x, y: TscreenPos; color: word);
begin
x := x + tx1;
y := y + ty1;
if (x<=tx2) and (y<=ty2) then
  TscrMap(screen^.pixels^)[y, x] := color
end;

function GetPixel(x, y: TscreenPos): word;
begin
GetPixel := TscrMap(screen^.pixels^)[y+ty1, x+tx1]
end;

procedure showimage(x, y: TScreenPos; var img);
var i : TScreenPos;
begin
for i := 0 to Tgrfimage(img).Height-1 do
  Move(Tgrfimage(img).Image[i*Tgrfimage(img).Width], 
       TscrMap(screen^.pixels^)[y+ty1+i, x+tx1], 
       Tgrfimage(img).Width*SizeOf(word))
end;

procedure drawBackground(var img);
begin
Move(Tgrfimage(img).Image, screen^.pixels^, 
      ScreenHeight*ScreenWidth*2);
SDL_UPDATERECT(screen, 0, 0, 0, 0)
end;

procedure MoveTo(x, y: TscreenPos);
begin
xPos := x;
yPos := y
end;

function GetX: TscreenPos;
begin
GetX := xPos
end;

function  GetY: TscreenPos;
begin
GetY := yPos
end;

function GetMaxX: TscreenPos;
begin
GetMaxX := ScreenWidth
end;

function  GetMaxY: TscreenPos;
begin
GetMaxY := ScreenHeight
end;

procedure defineTextArea(x1, y1, x2, y2: TscreenPos; useTextArea: boolean);
var rect: SDL_Rect;
begin
tx1 := x1;
ty1 := y1;
tx2 := x2;
ty2 := y2;
tw  := tx2-tx1+1;
th  := ty2-ty1+1;
if textarea<>NIL then SDL_FREESURFACE(textarea);
textarea := SDL_CreateRGBSurface(SDL_SWSURFACE, tw, th,
                                 ScreenDepth,
                                 screen^.format^.rmask,
                                 screen^.format^.gmask,
                                 screen^.format^.bmask,
                                 screen^.format^.amask);
rect.x := tx1;
rect.y := ty1;
rect.w := tw;
rect.h := th;
SDL_BlitSurface(screen, @rect, textarea, NIL);
resetAnswerPositions
end;

procedure ClearTextArea;
var rect : SDL_Rect;
begin
xPos := 0;
yPos := 0;
rect.x := tx1;
rect.y := ty1;
SDL_BlitSurface(textarea, NIL, screen, @rect);
resetAnswerPositions
end;

function GetMouseAnswer(const mouse): char;
const defaultanswer = #06; { #06 = ACK: no answer - but it's "any key" }
var 
  x, y : LongInt;
  button: byte;
  i : integer;
  answer: char;
begin
x := SDL_MouseButtonEvent(mouse).x - tx1;
y := SDL_MouseButtonEvent(mouse).y - ty1;
button := SDL_MouseButtonEvent(mouse).button;
answer := defaultanswer;

{ left mouse button in textarea? }
if (button=SDL_BUTTON_LEFT) and 
   (x>=0) and (x<=tw) and (y>=0) and (y<=th) then
     begin
     i := 1;
     repeat
       if (y>=answerposition[i].f) and (y<=answerposition[i].t)
          then answer := ValueToKey(i);
       inc(i)
     until (answer<>defaultanswer) or (i>MaxAnswers);
     end;

{ right mouse button = Enter }
if button=SDL_BUTTON_RIGHT then answer := #13;

GetMouseAnswer := answer
end;

function GetKey: char;
var 
 event: SDL_Event;
 c: char;
begin
{ the textarea is only updated here, when the program waits for events }
SDL_UPDATERECT(screen, tx1, ty1, tw, th);

c := #0;
repeat
  SDL_WaitEvent(@event);
  case event.eventtype of
    SDL_EventQuit: c:=ExitKey;
    SDL_MouseButtonDown: if mouseactive then c := GetMouseAnswer(event);
    SDL_KeyDown: begin
                 c := chr(SDL_KeyboardEvent(event).keysym.unicode and $FF);
		 if c=#0 then
		   Case SDL_KeyboardEvent(event).keysym.sym of
		     SDLK_KP0 : c := '0';
		     SDLK_KP1 : c := '1';
		     SDLK_KP2 : c := '2';
		     SDLK_KP3 : c := '3';
		     SDLK_KP4 : c := '4';
		     SDLK_KP5 : c := '5';
		     SDLK_KP6 : c := '6';
		     SDLK_KP7 : c := '7';
		     SDLK_KP8 : c := '8';
		     SDLK_KP9 : c := '9';
		     SDLK_KP_Enter : c := chr(13);
		     SDLK_EURO : c:=chr(164); { ISO-8859-15 }
		     end
		 end;
    end
until c<>#0;
GetKey := c
end;

end.
