{
* uakfquiz (unit)
* Copyright (c) 2003-2005 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: FreePascal or GNU-Pascal
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*}

{ 
  Conditional Defines:

  NoBuffer
     don't Buffer the input
  Buffer
     enforce Buffering
  NoProjectLink
    disable Link to the Projects Homepage
}

{$IfDef FPC}
  {$Mode Delphi}
  {$Smartlink on}
{$EndIf}

{ compatiblity definition }
{$IfDef _WIN32} {$Define Win32} {$EndIf}

{$IfNDef NoBuffer}
  {$IfNDef __GPC__} { GPC doesn't need no additional buffer }
    {$Define Buffer}
  {$EndIf}
{$EndIf}

{$I-} { no automatic I/O-Check }
{$X+} { function call as procedure allowed }


unit uakfquiz;    { "All unit is love" (the Beatles) ;-) }

interface
uses qsys, qmsgs;

{ please add your name, when you make changes }
const AKFQuizCopyright = 
   'Copyright (c) 2003-2005 AKFoerster';

{ don't change these here, but in the script "configure" }
const AKFQuizName = 'AKFQuiz';
const AKFQuizVersion = '4.1.1';

{ some default values }
const
   def_title          = 'Quiz';
   def_author         = '';
   def_language       = 'en';   { see also Takfquiz.resetQuiz }
   def_charset        = sys_charset;
   def_baseURI        = '';
   def_javascript     = 'akfquiz5.js';
   def_CSS            = '';
   def_defanswer      = '';
   def_neutral        = false;
   def_HTMLcode       = false;  { keep that false as default! }
   def_noindex        = false;
   def_rtl            = false;
   def_assessmentURI  = '';

{ strings are implemented differently in Pascal dialects. }
type string75  = string[75];
type string80  = string[80];
type string255 = string[255];

type pointsType = LongInt;

type TquestionType = (radio, checkbox, textfield);

type TTypes = 'A'..'Z';

type 
  Ptypetest = ^TTypeTest;
  TTypeTest = object
     Types : array[TTypes] of pointsType;
     
     constructor Init;
     destructor Done; virtual;
     end;
     
type 
  Pakfquiz = ^Takfquiz;
  Takfquiz = 
    { abstract } object
      private
	htmltagskip: boolean; { skip over html Tags? }
	convert : Tconverter;
        inp : text;
	
        {$IfDef Buffer}
          Buffer : array[ 1..10240 ] of char;
        {$EndIf}

      protected
        { anything that can be defined in the input file belongs here! }
        title, author, copyright, translator, language, charset, defanswer, 
        keywords, baseURI, javascript, CSS : string255; 
	assessmentURI: string255;
	
	{ the "language" here may be an unsupported language,
	  "lang" in qmsgs.pas is always supported }

        HTMLcode, noindex, neutral, rtl: boolean;

        { internal stuff: }
	qType: TquestionType;        { type of actual question }
        questionNr: integer;         { number of actual question }
	answerNr: integer;           { number of answer for actual question }
        started: boolean;            { already started? }
	evaluated: boolean;          { already ealuated? }
        quit: boolean;               { quit from interactive apps }

        Points,
	MaxPoints,                   { maximum points for quiz }
	thisMaxPoints : pointsType;  { maximum points for this question }

      public
        constructor Init(infile: string);
	constructor InitCfg(cfgfile, infile: string); { experimental }
        destructor Done;			    virtual;
	
        procedure resetQuiz;                        virtual;
        procedure process;
        procedure StartQuiz;                        virtual;
	procedure ignoreBlock;
        procedure processComment;                   virtual;
	procedure processHint;                      virtual;
        procedure processQuestion;                  virtual;
        procedure processMulti;                     virtual;
	procedure processFitBest;                   virtual;
        procedure processAssessment;                virtual;
	function  readAssessmentPercent: mystring;  
        procedure processAssessmentPercent;         virtual;
	procedure setcharset(c: string);            virtual;
	procedure setconverter(p: Tconverter);
	{ called from StarQuiz }
	procedure convertsettings;                  virtual;
        procedure EndQuiz;                          virtual; { empty }
        procedure evaluate;                         virtual;
        function readLine: mystring;                virtual;
        procedure readAnswer(var p: pointsType; 
	                     var s: mystring);      virtual;
        procedure error;  			    virtual;
	function htmlconvert(const x: mystring): mystring; virtual;

        function checkEOF: boolean;
        procedure gotoEOF;
	function getTitle: mystring;
	function getAuthor: mystring;
	function getCopyright: mystring;
	function getTranslator: mystring;
	function getLanguage: mystring;
	function getCharset: mystring;
	function getDefAnswer: mystring;
	function getbaseURI: mystring;
	function getJavascript: mystring;
	function getCSS: mystring;
	function getneutral: boolean;
	function getHTMLcode: boolean;
	function getNoindex: boolean;
	function getPoints: pointsType;
	function getMaxPoints: pointsType;
	function getPercentage: integer;
        function stopQuiz: boolean;
	function quizstarted: boolean;
      private
      	procedure useQuizLanguage;
        function testStarted: boolean;
    end;

type 
  Phtmlquiz = ^Thtmlquiz;
  Thtmlquiz = 
    object(Takfquiz)
      protected
        outp: text;
        errorcode : integer;
	cet : string[3];              { close sequence for empty tags }
	BR  : string[6];              { <br> or <br /> }

      public
        constructor Init(infile, outfile: string);
        destructor Done;                            virtual;
        procedure newoutput;                        virtual;
	function  handleURIs(const x: string): mystring;
        function  htmlconvert(const x: mystring): mystring; virtual;
        procedure headdata;                         virtual;
        procedure StartQuiz;                        virtual;
        procedure putgraphic;                       virtual;
	procedure showanswer(value: pointsType;
	                     const ans: string);    virtual;
	procedure processBlock;
        procedure processComment;                   virtual;
        procedure processQuestion;                  virtual;
        procedure processAnswer;                    virtual;
        procedure processMulti;                     virtual;
	procedure processDefaultAnswer;		    virtual;
        procedure processAssessment;                virtual;
        procedure handleSettingsURIs;
        procedure EndQuiz;                          virtual;
        procedure error;			    virtual;
        function  geterror: integer;

    protected
        function getAssessmentURI: mystring;
    private
        procedure newparagraph;
    end;

function getQuizTitle(const x: string): MyString;
function getvalue(x: string): mystring;
function getvaluefb(x, fallback: string): mystring;
function getboolvalue(x: string): boolean;
function getPointsFromLine(var p: pointsType; var s: mystring):boolean;

{ for the interactive AKFQuiz interpreters }
function KeyToValue(c: char): integer;
function ValueToKey(v: integer): char;

{ some functions, not directly related to AKFQuiz }
function format(var s: mystring; len, maxlen: integer): mystring;
function qTypeStr(qType: TquestionType): mystring; 
procedure useSystemLanguage;

{ --------------------------------------------------------------------- }

implementation
{$IfDef FPC} 
  uses math; { for "min" }
{$EndIf}

{ for the interactive AKFQuiz interpreters }
function KeyToValue(c: char): integer;
begin
c := Upcase(c);
if (c>='0') and (c<='9') 
   then KeyToValue := ord(c)-48
   else if (c>='A') and (c<='Z')
           then KeyToValue := ord(c)-65+10
	   else KeyToValue := -1
end;

function ValueToKey(v: integer): char;
begin
if v < 10 
   then ValueToKey := chr(v+48)
   else ValueToKey := chr(v-10+65)
end;

{ "format" returns as much as fits and leaves the rest in the parameter }
function format(var s: mystring; len, maxlen: integer): mystring;
begin
if length(s) < len
  then begin format := s; s := '' end
  else
    begin
    while (len>0) and (s[len]<>' ') and (s[len]<>#9) do dec(len);
    if len<>0
       then begin { line can be splitted }
            format := copy(s, 1, len-1); 
            delete(s, 1, len)
	    end
       else begin { single word left @@@ }
	    if (length(s)>MaxLen) and (pos(' ',s)=0) 
	       then begin { single word must be splitted :-( }
	            format := copy(s, 1, MaxLen); 
	            delete(s, 1, MaxLen)
	            end
               else format := ''
	       { word fits into a new line -> nothing to be done }
            end
    end
end;


function qTypeStr(qType: TquestionType): mystring;
begin
case qType of
  radio    : qTypeStr := 'radio';
  checkbox : qTypeStr := 'checkbox';
  end
end;

{ gets string after colon, leading spaces stripped }
function getvalue(x: string): mystring;
var i: integer;
begin
i := succ(pos(':', x));
while (x[i]=' ') or (x[i]=TAB) do inc(i);
getvalue := copy(x, i, length(x)-i+1);
end;

{ getvalue with fallback }
function getvaluefb(x, fallback: string): mystring;
begin
x := getvalue(x);
if x='' then x := fallback;
getvaluefb := x
end;

{ get boolean value }
function getboolvalue(x: string): boolean;
begin
x := getvalue(x);
getboolvalue := 
  (x='1') or ((length(x)>=1) and (UpCase(x[1]) in ['T','Y','J','O']))
end;

procedure useSystemLanguage;
var l: string[2];
begin
l := copy(getsystemlanguage, 1, 2); { first 2 chars }
l := makeUpcase(l);
if l='EN' then lang := english;
if l='DE' then lang := deutsch;
if l='DA' then lang := dansk;
if l='IT' then lang := italiano
{ else unchanged }
end;

{ --------------------------------------------------------------------- }

constructor TTypeTest.Init;
var t: TTypes;
begin
for t := 'A' to 'Z' do Types[t] := 0
end;

destructor TTypeTest.Done;
begin
end;

{ --------------------------------------------------------------------- }

constructor Takfquiz.Init(infile: string);
begin
assign(inp, infile);
{$IfDef Buffer}
  SetTextBuf(inp, Buffer);
{$EndIf}

resetQuiz;
if (IOResult<>0) or checkEOF then fail
end;

{ Init with configuration file }
constructor Takfquiz.InitCfg(cfgfile, infile: string);
begin
assign(inp, cfgfile);
{$IfDef Buffer}
  SetTextBuf(inp, Buffer);
{$EndIf}

resetQuiz;
process;
close(inp);
if IOResult<>0 then fail;
if started then fail; { mustn't be started in config file! }

assign(inp, infile);
{$IfDef Buffer}
  SetTextBuf(inp, Buffer);
{$EndIf}
reset(inp);
if (IOResult<>0) or CheckEOF then fail
end;

destructor Takfquiz.Done;
begin close(inp) end;

procedure Takfquiz.resetQuiz;
begin
title         := def_title;
author        := def_author;
language      := def_language;
charset       := def_charset;
baseURI       := def_baseURI;
javascript    := def_javascript;
neutral       := def_neutral;
HTMLcode      := def_HTMLcode;
CSS           := def_CSS;
noindex       := def_noindex;
rtl           := def_rtl;
defanswer     := def_defanswer;
assessmentURI := def_assessmentURI;

quit          := false;
started       := false;
evaluated     := false;
qType         := radio;   { just to have something meaningful }

Points        := 0;
MaxPoints     := 0;
thisMaxPoints := 0;
htmltagskip   := false;
convert       := noconversion;

reset(inp)
end;

procedure Takfquiz.useQuizLanguage;
var l: string[2];
begin
l := makeUpcase(copy(Language,1,2));
if l='EN' then lang := english;
if l='DE' then lang := deutsch;
if l='DA' then lang := dansk;
if l='IT' then lang := italiano
{ else unchanged }
end;

function Takfquiz.readLine: mystring;
var 
  res, s: mystring;
  okay : boolean;
begin
res := '';

if not eof(inp) then
  repeat
    okay := true; { assume line ends here }
    ReadLn(inp, s);
    if IOResult<>0 then error;
    s := stripWhitespace(htmlconvert(s));
    if htmltagskip and (s='') then okay := false;
    if s='' then continue;
    if pos('#',s)=1 then
       begin
       okay := false;
       continue
       end;
    if pos('\', s) <> length(s) 
       then res := res + s
       else begin
            res := res + stripWhitespace(copy(s, 1, length(s)-1)) + ' ';
  	    okay := false
	    end
  until okay;
if started then res := convert(res);
readLine := res
end;

{ strips points from line }
function getPointsFromLine(var p: pointsType; var s: mystring):boolean;
var 
  i: integer;
  e: integer;
begin
getPointsFromLine := false; { assumption }

if s='' then exit;

{ correction for GPG clearsigned quiz-files with dash-escape }
if (s[1]='-') and (s[2]=' ') then Delete(s, 1, 2);

{ search for whitespace: }
i:=1;
while (i<length(s)) and (s[i]<>' ') and (s[i]<>TAB) do inc(i);

{ no whitespace? }
if i=length(s) then exit;

val(copy(s, 1, i-1), p, e);
if e<>0 then exit;

Delete(s, 1, i);
s := stripWhitespace(s);
getPointsFromLine := true
end;

procedure Takfquiz.readAnswer(var p: pointsType; var s: mystring);
var e : integer;
begin
p := 0;
s := readline;

if s<>'' then
   begin
   if not getPointsFromLine(p, s) then
      begin { try old format }
      val(readline, p, e);
      if e<>0 then error
      end;
   
   if qType = radio
      then thisMaxPoints := Max(thisMaxPoints, p)
      else if p > 0 then inc(thisMaxPoints, p)
   end
end;

procedure Takfquiz.evaluate;
begin 
evaluated := true
end;

procedure Takfquiz.StartQuiz;
begin
started := true;
convertsettings
end;

function Takfquiz.htmlconvert(const x: mystring): mystring;
var
 i: integer;
 e: mystring;
begin
if not HTMLcode
   then htmlconvert := x
   else begin
        e := '';
        i := 1;
        while i <= length(x) do
	  begin
	  if x[i]='<' then htmltagskip := true;
	  if not htmltagskip then 
	    if x[i]='&' 
	       then begin
	            if copy(x, i, 4)='&lt;' then 
		       begin e := e + '<'; inc(i, 3) end;
	            if copy(x, i, 4)='&gt;' then 
		       begin e := e + '>'; inc(i, 3) end;
	            if copy(x, i, 5)='&amp;' then 
		       begin e := e + '&'; inc(i, 4) end;
	            if copy(x, i, 6)='&quot;' then 
		       begin e := e + '"'; inc(i, 5) end;
	            if copy(x, i, 6)='&euro;' then 
		       begin e := e + 'Euro'; inc(i, 5) end;
	            end
	       else e := e + x[i];
	  if x[i]='>' then htmltagskip := false;
          inc(i)
	  end; { while }
	htmlconvert := e
        end { if not HTMLcode }
end;

function Takfquiz.testStarted: boolean; {@@@}
begin
if not started then StartQuiz;
testStarted := started and not quit
end;

procedure Takfquiz.setconverter(p: Tconverter);
begin
convert := p
end;

procedure Takfquiz.convertsettings;
begin
title      := convert(title); 
author     := convert(author); 
copyright  := convert(copyright);
translator := convert(translator);
defanswer  := convert(defanswer);
keywords   := convert(keywords)
end;

procedure Takfquiz.setcharset(c: string);
begin
charset := c
end;

procedure Takfquiz.ignoreblock;
var s: mystring;
begin
s := readLine;
while (s<>'') and not checkEOF do s := readLine
end;

{ simply ignore as default }
procedure Takfquiz.processComment;
begin ignoreblock end;

procedure Takfquiz.processHint;
begin ignoreblock end;

{ simply ignore as default }
procedure Takfquiz.processAssessment;
begin 
ignoreblock 
end;

function Takfquiz.readAssessmentPercent: mystring;
var 
  s: mystring;
  value, oldvalue: pointsType;
  reached: integer;
  found: boolean;
begin
found := false;
oldvalue := 101; { more than 100 }
reached  := getPercentage;
readAssessmentPercent := '';

s := readLine;
while s<>'' do
  begin
  if not getPointsFromLine(value, s) then error;
  if value >= oldvalue then error; { enforce descending order }
  if (not found) and (reached >= value) and (MaxPoints<>0) then 
     begin
     readAssessmentPercent := s;
     found := true
     end;
  s := readLine;
  oldvalue := value
  end;

{ last value must be 0! }
if value <> 0 then error
end;


{ ignore as default, but check syntax }
procedure Takfquiz.processAssessmentPercent;
begin 
readAssessmentPercent
end;

procedure Takfquiz.processQuestion;
begin 
qType := radio;
inc(questionNr);
answerNr := 0;
thisMaxPoints := 0
end;

procedure Takfquiz.processMulti;
begin 
qType := checkbox;
inc(questionNr);
answerNr := 0;
thisMaxPoints := 0
end;

procedure Takfquiz.processFitBest;
begin 
qType := textfield;
inc(questionNr);
answerNr := 0;
thisMaxPoints := 0
end;

procedure Takfquiz.process;
var s, e: mystring;
begin
started       := false;
evaluated     := false;
quit          := false;
questionNr    := 0;
answerNr      := 0;
Points        := 0;
MaxPoints     := 0;
thisMaxPoints := 0;

repeat
  s := makeUpcase(readLine)
until ( pos('AKFQUIZ', s)=1 ) or checkEOF;

if checkEOF then error; { no AKFQuiz header found }

while not checkEOF and not quit do
   begin
   s := readLine;
   e := makeUpcase(s);

   if (s<>'') and 
      (e<>'END') and 
      (e<>'ENDE') and 
      (pos(':',s)=0) then
     begin { if no keyword, no comment }
     error;
     { there is normally more than one line with this error }
     while not checkEOF and (s<>'') and (pos(':',s)=0) do s := readLine
     end;

   if (pos('TITLE:',e)=1) or
      (pos('TITEL:',e)=1)
          then title := getvaluefb(s, def_title);
   if (pos('AUTHOR:',e)=1) or
      (pos('AUTOR:',e)=1)
          then author:=getvalue(s);
   if (pos('COPYRIGHT:',e)=1)   
          then copyright:=getvalue(s);
   if (pos('TRANSLATOR:',e)=1) or
      (pos('UEBERSETZER:',e)=1) 
          then translator:=getvalue(s);
   if (pos('CHARSET:',e)=1) or
      (pos('ZEICHENSATZ:',e)=1) 
          then setcharset(getvaluefb(s, def_charset));
   if (pos('RTL:',e)=1) 
          then rtl := getboolvalue(s);
   if (pos('LANGUAGE:',e)=1) or
      (pos('SPRACHE:',e)=1)
          then begin language := getvalue(s); useQuizLanguage end;
   if pos('NEUTRAL:',e)=1
          then neutral := getboolvalue(s);
   if pos('HTMLCODE:',e)=1
          then HTMLcode:=getboolvalue(s);
   if (pos('BASEURI:',e)=1) or
      (pos('BASEURL:',e)=1) or
      (pos('HAUPTURI:',e)=1) or
      (pos('HAUPTURL:',e)=1) 
          then begin baseURI := getvalue(s);
                     { make sure, that it ends with a / }
	             if baseURI<>'' then 
                        if baseURI[length(baseURI)]<>'/'
			   then baseURI := baseURI+'/'
               end;
   if (pos('ASSESSMENTURI:',e)=1) or
      (pos('ASSESSMENTURL:',e)=1) or
      (pos('AUSWERTUNGSURI:',e)=1) or
      (pos('AUSWERTUNGSURL:',e)=1) or
      (pos('AUSWERTUNGURI:',e)=1) or
      (pos('AUSWERTUNGURL:',e)=1) then assessmentURI:=getvalue(s);
   if (pos('CSS:',e)=1) or
      (pos('LAYOUT:',e)=1)
          then CSS:=getvalue(s);
   if (pos('NOINDEX:',e)=1)  
          then noindex := getboolvalue(s);
   if (pos('KEYWORDS:',e)=1)  or
      (pos('STICHWORTE:',e)=1)
          then keywords := getvalue(s);
   if (pos('JAVASCRIPT:',e)=1) 
          then javascript := getvaluefb(s, def_javascript);
   if (pos('DEFAULT:',e)=1) or
      (pos('STANDARDANTWORT:',e)=1) 
          then defanswer:=getvalue(s);
   if (e='INTRO:') or
      (e='COMMENT:') or
      (e='KOMMENTAR:')
          then if testStarted then processComment;
   if (e='HINT:') or
      (e='REMARK:') or
      (e='HINWEIS:') or
      (e='ANMERKUNG:')
          then if testStarted then processHint;
   if (e='QUESTION:') or
      (e='FRAGE:') or
      (e='MC:')
          then begin
               if testStarted then processQuestion;
	       inc(MaxPoints, thisMaxPoints)
	       end;
   if (e='MULTI:') or
      (e='QUERY:') or
      (e='ANFRAGE:') or
      (e='MCMA:')
          then begin
               if testStarted then processMulti;
	       inc(MaxPoints, thisMaxPoints)
	       end;
   if (e='FITBEST:') or
      (e='FREEFORM:') or
      (e='FREIFORM:') 
          then begin
               if testStarted then processFitBest;
	       inc(MaxPoints, thisMaxPoints)
	       end;
   if (e='ASSESSMENT:') or
      (e='AUSWERTUNG:')
          then if teststarted then 
	          begin
		  if not evaluated then evaluate;
		  processAssessment
		  end;
   if (e='ASSESSMENT%:') or
      (e='AUSWERTUNG%:')
          then if teststarted then
	          begin
		  if not evaluated then evaluate;
		  processAssessmentPercent
		  end;
   if (e='END') or
      (e='ENDE') then gotoEOF
   end;
if started then EndQuiz
end;

function Takfquiz.getPoints: pointsType;
begin
getPoints := Points
end;

function Takfquiz.getMaxPoints: pointsType;
begin
getMaxPoints := MaxPoints
end;

function Takfquiz.getPercentage: integer;
begin
getpercentage := round(max(Points,0)*100/MaxPoints)
end;

function Takfquiz.getTitle: mystring;
begin getTitle := title end;

function Takfquiz.getAuthor: mystring;
begin getAuthor := Author end;

function Takfquiz.getCopyright: mystring;
begin getCopyright := Copyright end;

function Takfquiz.getTranslator: mystring;
begin getTranslator := translator end;

function Takfquiz.getLanguage: mystring;
begin getLanguage := Language end;

function Takfquiz.getCharset: mystring;
begin getCharset := charset end;

function Takfquiz.getDefAnswer: mystring;
begin getDefAnswer := defAnswer end;

function Takfquiz.getJavascript: mystring;
begin getJavascript := Javascript end;

function Takfquiz.getbaseURI: mystring;
begin getbaseURI := baseURI end;

function Takfquiz.getCSS: mystring;
begin getCSS := CSS end;

function Takfquiz.getneutral: boolean;
begin getneutral := neutral end;

function Takfquiz.getHTMLcode: boolean;
begin getHTMLcode := HTMLcode end;

function Takfquiz.getNoIndex: boolean;
begin getNoIndex := noindex end;

function Takfquiz.stopQuiz: boolean;
begin stopQuiz := quit end;

function Takfquiz.quizstarted: boolean;
begin quizstarted := started end;

function Takfquiz.checkEOF: boolean;
begin
checkEOF := EOF(inp)
end;

procedure Takfquiz.gotoEOF;
begin
while not EOF(inp) do ReadLn(inp)
end;

procedure Takfquiz.error;
begin quit := true end;

procedure Takfquiz.EndQuiz;
begin end;


{ --------------------------------------------------------------------- }

{ ATTENTION: never use non-ascii characters for the output here!
  The charset-encoding might be set to a different one in the 
  input-script. Instead use HTML-entities, like &copy; or &euro; }

constructor Thtmlquiz.Init(infile, outfile: string);
begin
inherited Init(infile);
errorcode := 0;
cet := '>'; { for xhtml: ' />' }
BR  := '<br' + cet;

{ use HTML-entities in messages, so they are independent of the 
  charset set in the quizfile }
setmsgconverter(UTF8toHTML);

assign(outp, outfile);
newoutput;
if IOResult<>0 then fail
end;

destructor Thtmlquiz.Done;
begin
inherited Done;
close(outp)
end;

procedure Thtmlquiz.newoutput;
begin
Rewrite(outp)
end;

function Thtmlquiz.handleURIs(const x: string): mystring;
var
  s, URI, rest: mystring;
  img, URIprefix, f, t, p : LongInt; { image, from, to, position }
  blank: boolean; { on a blank page? }
begin
s := '';
rest := x;

repeat
  f := 0;

  { prefixes: }
  img :=  pos('image:', rest);
  
  URIprefix := pos('URI:', rest);
  if URIprefix=0 then URIprefix := pos('URL:', rest);
  { URN: is handled later }
  if URIprefix<>0 then f := URIprefix + length('UR?:');

  { browser internal URIs }
  if f=0 then f := pos('http://', rest);   { RFC2616 }
  if f=0 then f := pos('https://', rest);  { RFC2818 }
  if f=0 then f := pos('ftp://', rest);    { RFC1738 }
  if f=0 then f := pos('tftp://', rest);   { RFC3617 }
  if f=0 then f := pos('gopher://', rest); { RFC-hoffman-gopher-uri-03.txt }
  if f=0 then f := pos('wais://', rest);   { RFC4156 }
  if f=0 then f := pos('file://localhost/', rest); { RFC1738 [*] }
  if f=0 then f := pos('file:///', rest);   { RFC1738 [*] }
  
  { [*] I'm delibertately strict with the file-URI syntax,
        because it's often misused }
  
  blank := (f<>0); { if one of the above is found, use a blank page }
  
  { only the above URIs can be handles as images }
  if (img<>0) and (f=0) then img := 0;
  
  { if the URI isn't directly after "image:", then it's not an image }
  if (img<>0) and (f<>img+length('image:')) 
                  and (URIprefix<>img+length('image:'))
	                 then img := 0;

  { URIs for external programs }
  if f=0 then f := pos('mailto:', rest);   { RFC2368 }
  if f=0 then f := pos('telnet:', rest);   { RFC4248 } { officially with // }
  if f=0 then f := pos('tn3270:', rest);   { ??? }
  if f=0 then f := pos('news:', rest);     { RFC1738 }
  if f=0 then f := pos('nntp://', rest);   { RFC1738 }
  if f=0 then f := pos('tel:', rest);      { RFC2806 }
  if f=0 then f := pos('fax:', rest);      { RFC2806 }
  if f=0 then f := pos('modem:', rest);    { RFC2806 }
  if f=0 then f := pos('nfs://', rest);    { RFC2224 }
  if f=0 then f := pos('ldap://', rest);   { RFC-ietf-ldapbis-url-09.txt }
  if f=0 then f := pos('prospero://', rest); { RFC4157 }
  if f=0 then f := pos('urn:', rest);      { RFC2141 }
  if f=0 then f := pos('URN:', rest);      { RFC2141 }

  { not official }
{ if f=0 then f := pos('fish:', rest); }

  if f<>0 then
    begin
    { start up to the found URI }
    if img=0 
      then s := s + copy(rest, 1, f-1)
      else s := s + copy(rest, 1, img-1);

    { extrapolate URI }
    URI := rest;
    Delete(URI, 1, f-1);
    
    { search end of URI: }
    t := length(URI);
    p := pos('&gt;', URI);
    if p<>0 then t := min(t, p);
    p := pos(' ', URI);
    if p<>0 then t := min(t, p);
    p := pos(TAB, URI);
    if p<>0 then t := min(t, p);
    p := pos('&quot;', URI);
    if p<>0 then t := min(t, p);
    if t<>length(URI) 
      then begin dec(t); URI := copy(URI, 1, t) end;
    
    delete(rest, 1, f+t-1);

    if img<>0 
      then s := s+'<img src="'+URI+'" alt="['+URI+
                ']" style="vertical-align:middle; float:right">'
      else if blank
             then s := s+'<a href="'+URI+'" target="_blank">'+URI+'</a>'
             else s := s+'<a href="'+URI+'">'+URI+'</a>'
    end
until (rest='') or (f=0);

handleURIs := s + rest
end;

{ quotes characters, that have special meanings in HTML }
function Thtmlquiz.htmlconvert(const x: mystring): mystring;
var 
   i: integer;
   e: mystring;
begin
if HTMLcode 
   then htmlconvert := x
   else begin
        e:='';
        for i:=1 to length(x) do
            case x[i] of
                 '<': e := e + '&lt;';
                 '>': e := e + '&gt;';
                 '&': e := e + '&amp;';
                 '"': e := e + '&quot;';
                 otherwise 
		   if (x[i] > chr(31)) or (x[i] = TAB) then e := e + x[i]
                 end;
	if started 
	   then htmlconvert := handleURIs(e)
	   else htmlconvert := e
        end
end;

procedure Thtmlquiz.error; {@@@}
begin
errorcode := 1;

if not started then {@@@}
  begin
  StartQuiz;
  quit := true
  end;

{ don't use no block-elements! It would break an existing block }
case lang of
  deutsch:
       begin
       Write(outp, '<strong class="error"');
       if rtl then Write(outp, ' dir="ltr"'); { switch back to ltr }
       WriteLn(outp, '>');
       WriteLn(outp, 'Fehler in den Eingabedaten</strong>', BR);
       WriteLn(stderr,'Fehler in den Eingabedaten - '+
                      'Einzeilheiten im HTML-Code')
      end
  otherwise
       begin
       Write(outp, '<strong class="error" lang="en"');
       if rtl then Write(outp, ' dir="ltr"'); { switch back to ltr }
       WriteLn(outp, '>');
       WriteLn(outp, 'fault in input data</strong>', BR);
       WriteLn(stderr, 'fault in input data - look at the HTML code')
       end
  end { case }
end;

procedure Thtmlquiz.headdata;
begin
if noindex then
   WriteLn(outp, '<meta name="robots" content="noindex"', cet);
WriteLn(outp, '<meta http-equiv="Content-Type" content="text/html; charset=',
        charset,'"', cet);

{$IfDef Win32}
WriteLn(outp, '<!-- AKFoerster: I don''t really like Windows, '
            + 'but I have to support it -->');
{$EndIf}

if language<>'' then
   WriteLn(outp, '<meta http-equiv="Content-Language" content="',language,
                 '"', cet);
if author<>'' then
   WriteLn(outp, '<meta name="author" content="', author, '"', cet);
if copyright<>'' then
   WriteLn(outp, '<meta name="copyright" content="', copyright, '"', cet);
if keywords<>'' then
   WriteLn(outp, '<meta name="keywords" content="', keywords, '"', cet);

WriteLn(outp, '<style type="text/css">');
WriteLn(outp, '<!-- @media print { .buttons, .defanswer, .resultlink, noscript {display:none} } -->');
WriteLn(outp, '</style>');

if CSS<>'' then
   WriteLn(outp, '<link rel="stylesheet" type="text/css" href="',CSS,'"', cet)
end;

procedure Thtmlquiz.StartQuiz;
begin
inherited StartQuiz;

WriteLn(outp, HTMLDocType);
WriteLn(outp);

Write(outp, '<html');
if language<>''
   then Write(outp, ' lang="',language,'"');
if RTL
   then Write(outp, ' dir="rtl"');
WriteLn(outp, '>');
WriteLn(outp);
WriteLn(outp, '<head>');
WriteLn(outp);
WriteLn(outp, '<meta name="generator" content="'+AKFQuizName+' '
              +AKFQuizVersion+'"', cet);

headdata;
if not HTMLcode then handleSettingsURIs;

WriteLn(outp);
WriteLn(outp, '<title>', title, '</title>');
WriteLn(outp, '</head>');
WriteLn(outp);
WriteLn(outp);
WriteLn(outp, '<body>');
WriteLn(outp);
WriteLn(outp, '<h1>', title, '</h1>');
WriteLn(outp)
end;

procedure Thtmlquiz.handleSettingsURIs;
begin
title      := handleURIs(title);
author     := handleURIs(author);
copyright  := handleURIs(copyright);
translator := handleURIs(translator);
defanswer  := handleURIs(defanswer);
keywords   := handleURIs(keywords)
end;

procedure Thtmlquiz.EndQuiz;
begin
WriteLn(outp);
WriteLn(outp, '<div align="right" dir="ltr" class="made"><small>');
{$IfNDef NoProjectLink}
  WriteLn(outp, msg_made, ' <a '+
    'href="', Homepage, msg_URI, '" target="_blank">'+
    AKFQuizName + '</a>');
{$EndIf}
if author<>'' then Write(outp, BR, msg_author, author);
if translator<>'' then 
  Write(outp, BR, msg_translator, translator);
if copyright<>'' then 
  Write(outp, BR, 'Copyright &copy; ', copyright);
WriteLn(outp, '</small></div>');
WriteLn(outp);
WriteLn(outp, '</body>');
WriteLn(outp, '</html>')
end;

{ close old paragraph and open a new one }
procedure Thtmlquiz.newparagraph;
begin
WriteLn(outp, '</p>');
WriteLn(outp);
WriteLn(outp, '<p>')
end;

procedure Thtmlquiz.processBlock;
var s: mystring;
begin
WriteLn(outp, '<p>');
s := readLine;
while s<>'' do
  begin
  if s<>'' 
    then begin
         if s='.' { new paragraph? }
	   then newparagraph
           else WriteLn(outp, s) { not s='.' }
        end; { s<>'' }
  s := readLine
  end;
if s<>'' then WriteLn(outp, s);
WriteLn(outp, '</p>')
end;

procedure Thtmlquiz.processComment;
begin
WriteLn(outp);
WriteLn(outp, '<div class="comment">');
processBlock;
WriteLn(outp, '</div>');
WriteLn(outp)
end;

procedure Thtmlquiz.putgraphic;
begin end;

procedure Thtmlquiz.showanswer(value: pointsType; const ans: string);
begin
inc(answerNr);

{ it must be included in block containers to make RTL work correctly }
WriteLn(outp, '<div>');
WriteLn(outp, '<input  id="q', questionNr, 'a', answerNr, '"',
              ' name="q',questionNr,'"',
              ' value="', value, '"',
              ' type="', qTypeStr(qType), '"', cet);

Write(outp, '<label for="q', questionNr, 'a', answerNr, '">');
Write(outp, ans);
WriteLn(outp, '</label>');
WriteLn(outp, '</div>');
WriteLn(outp)
end;

procedure Thtmlquiz.processDefaultAnswer;
begin
inc(answerNr);
WriteLn(outp, '<div class="defanswer">');
WriteLn(outp, '<input  id="q', questionNr, 'a', answerNr, '"',
	      ' name="q', questionNr, '"',
              ' type="radio" value="0" checked', cet); { xhtml-change }

Write(outp, '<label for="q', questionNr, 'a', answerNr, '">');
Write(outp, defanswer);
WriteLn(outp, '</label></div>', BR)
end;

procedure Thtmlquiz.processAnswer;
var 
  value: pointsType;
  s: mystring;
begin
WriteLn(outp, '<div class="answer">');

repeat
  readanswer(value, s);
  if s<>'' then showanswer(value, s);
until s='';

if (defanswer<>'') and (qType=radio) then processDefaultAnswer;

if not neutral then putgraphic;

WriteLn(outp, '</div>');
WriteLn(outp)
end;

procedure Thtmlquiz.processQuestion;
begin
inherited processQuestion;
WriteLn(outp);
WriteLn(outp, '<div class="question">');
processBlock;
WriteLn(outp, '</div>');
WriteLn(outp);
processAnswer
end;

procedure Thtmlquiz.processMulti;
begin
inherited processMulti;
WriteLn(outp);
WriteLn(outp, '<div class="question">');
processBlock;
WriteLn(outp, '</div>');
WriteLn(outp);
processAnswer
end;

procedure Thtmlquiz.processAssessment;
begin
{ don't use the assessment-block, when assessmentURI is given }
if assessmentURI<>'' 
  then ignoreBlock
  else begin
       WriteLn(outp);
       WriteLn(outp, '<div class="assessment">');
       processBlock;
       WriteLn(outp, '</div>');
       WriteLn(outp)
       end
end;

function thtmlquiz.getAssessmentURI: mystring;
begin
getAssessmentURI := assessmentURI
end;

function Thtmlquiz.geterror: integer;
begin geterror := errorcode end;

{ --------------------------------------------------------------------- }

{ @@@ need charset conversion }
function getQuizTitle(const x: string): MyString;
var 
  d: text;
  s, e : String80;
  found : boolean;
  dummy : integer;
{$IfDef FPC}
  var Buffer : array[1..1024] of char;
{$EndIf}
begin
found := false;
Assign(d, x);
{$IfDef FPC}
  SetTextBuf(d, Buffer);
{$EndIf}
reset(d);
repeat
  ReadLn(d, s);
  s := stripWhitespace(s);
  e := makeUpcase(s);
  if (POS('TITLE:', e)<>0) or
     (POS('TITEL:', e)<>0) then
     begin
     e := getvalue(s);
     found := true
     end;
until found or EOF(d) or (IOResult<>0);
close(d);
dummy := IOResult; { ignore errors }
if found then getQuizTitle := e else getQuizTitle := x
end;

end.
