//**************************************************************************
//*                     This file is the source of                         *
//*                        Audigy12 program.                               *
//*                                                                        *
//*   Audigy12 is a patch for Audigy 2 cards. It turns on the outputs      *
//*      and gives the ability of DOS audio playing on these cards         *
//*         using/with the DOS based SB16 emulator of Audigy 1.            *
//*                                                                        *
//*           written by PDSoft (Attila Padar), Nov 2003                   *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <string.h>

#ifndef uint8_t
typedef unsigned char	uint8_t;
#endif
#ifndef uint16_t
typedef unsigned short	uint16_t;
#endif
#ifndef int32_t
typedef long            int32_t;
#endif
#ifndef uint32_t
typedef unsigned long	uint32_t;
#endif

//--------------------------------------------------------------------

#define PCI_FUNCTION_ID		0xB1
#define PCI_BIOS_PRESENT	0x01
#define PCI_FIND_DEVICE		0x02
#define PCI_READ_BYTE		0x08
#define PCI_READ_WORD		0x09
#define PCI_READ_DWORD		0x0A
#define PCI_WRITE_BYTE		0x0B
#define PCI_WRITE_WORD		0x0C
#define PCI_WRITE_DWORD		0x0D
#define PCI_GET_ROUTING		0x0E
#define PCI_SET_INTERRUPT	0x0F
#define PCI_SERVICE		0x1A

#define PCI_SUCCESSFUL		0x00
#define PCI_NOT_SUPPORTED	0x81
#define PCI_BAD_VEMDERID	0x83
#define PCI_DEVICE_NOTFOUND	0x86
#define PCI_BAD_REGNUMBER	0x87
#define PCI_SET_FAILED		0x88
#define PCI_BUFFER_SMALL	0x98

#define PCIR_VID		0x00
#define	PCIR_DID		0x02
#define	PCIR_PCICMD		0x04
#define	PCIR_RID		0x08
#define	PCIR_CCODE		0x0A
#define	PCIR_HEADT		0x0E
#define	PCIR_NAMBAR		0x10
#define	PCIR_NABMBAR		0x14
#define	PCIR_SSVID		0x2C
#define	PCIR_SSID		0x2E
#define	PCIR_INTR_LN		0x3C
#define	PCIR_INTR_PIN		0x3D

typedef struct pci_config_s {
 uint8_t bBus;
 uint8_t bDev;
 uint8_t bFunc;
}pci_config_s;

#define PCIDEVNUM(bParam)      ((bParam) >> 3)
#define PCIFUNCNUM(bParam)     ((bParam) & 0x07)
#define PCIDEVFUNC(bDev,bFunc) (((bDev) << 3) | (bFunc))

#ifdef __386__ // 32-bit

static uint8_t	pcibios_FindDevice(uint16_t wVendor, uint16_t wDevice, uint8_t bIndex, pci_config_s * ppkey)
{
 union REGS reg;
 memset(&reg,0,sizeof(union REGS));

 reg.h.ah = PCI_FUNCTION_ID;
 reg.h.al = PCI_FIND_DEVICE;
 reg.w.cx = wDevice;
 reg.w.dx = wVendor;
 reg.w.si = bIndex;

 int386(PCI_SERVICE, &reg, &reg);

 ppkey->bBus = reg.h.bh;
 ppkey->bDev = PCIDEVNUM(reg.h.bl);
 ppkey->bFunc = PCIFUNCNUM(reg.h.bl);

 return reg.h.ah;
}

static uint8_t pcibios_ReadConfig_Byte(pci_config_s * ppkey, uint16_t wAdr)
{
 union REGS reg;
 memset(&reg,0,sizeof(union REGS));

 reg.h.ah = PCI_FUNCTION_ID;
 reg.h.al = PCI_READ_BYTE;
 reg.h.bh = ppkey->bBus;
 reg.h.bl = PCIDEVFUNC(ppkey->bDev, ppkey->bFunc);
 reg.w.di = wAdr;
 int386(PCI_SERVICE, &reg, &reg);

 return reg.h.cl;
}

static uint16_t pcibios_ReadConfig_Word(pci_config_s * ppkey, uint16_t wAdr)
{
 union REGS reg;
 memset(&reg,0,sizeof(union REGS));

 reg.h.ah = PCI_FUNCTION_ID;
 reg.h.al = PCI_READ_WORD;
 reg.h.bh = ppkey->bBus;
 reg.h.bl = PCIDEVFUNC(ppkey->bDev, ppkey->bFunc);
 reg.w.di = wAdr;

 int386(PCI_SERVICE, &reg, &reg);

 return reg.w.cx;
}

#else // 16-bit

static uint8_t pcibios_FindDevice(uint16_t wVendor, uint16_t wDevice, uint8_t bIndex, pci_config_s * ppkey)
{
 union REGS reg;
 memset(&reg,0,sizeof(union REGS));

 reg.h.ah = PCI_FUNCTION_ID;
 reg.h.al = PCI_FIND_DEVICE;
 reg.x.cx = wDevice;
 reg.x.dx = wVendor;
 reg.x.si = bIndex;

 int86(PCI_SERVICE, &reg, &reg);

 ppkey->bBus = reg.h.bh;
 ppkey->bDev = PCIDEVNUM(reg.h.bl);
 ppkey->bFunc = PCIFUNCNUM(reg.h.bl);

 return reg.h.ah;
}

static uint8_t pcibios_ReadConfig_Byte(pci_config_s * ppkey, uint16_t wAdr)
{
 union REGS reg;
 memset(&reg,0,sizeof(union REGS));

 reg.h.ah = PCI_FUNCTION_ID;
 reg.h.al = PCI_READ_BYTE;
 reg.h.bh = ppkey->bBus;
 reg.h.bl = PCIDEVFUNC(ppkey->bDev, ppkey->bFunc);
 reg.x.di = wAdr;
 int86(PCI_SERVICE, &reg, &reg);
 return reg.h.cl;
}

static uint16_t pcibios_ReadConfig_Word(pci_config_s * ppkey, uint16_t wAdr)
{
 union REGS reg;
 memset(&reg,0,sizeof(union REGS));

 reg.h.ah = PCI_FUNCTION_ID;
 reg.h.al = PCI_READ_WORD;
 reg.h.bh = ppkey->bBus;
 reg.h.bl = PCIDEVFUNC(ppkey->bDev, ppkey->bFunc);
 reg.x.di = wAdr;

 int86(PCI_SERVICE, &reg, &reg);
 return reg.x.cx;
}

#endif

static uint32_t pcibios_ReadConfig_Dword(pci_config_s * ppkey, uint16_t wAdr)
{
 uint32_t dwData;

 dwData  = (uint32_t)pcibios_ReadConfig_Word(ppkey, wAdr + 2) << 16;
 dwData |= (uint32_t)pcibios_ReadConfig_Word(ppkey, wAdr);

 return dwData;
}

//--------------------------------------------------------------------

#ifdef __386__
 #define outl(value,port) outpd(port,value)
 #define inl(port) inpd(port)
#else
 #ifdef __WATCOMC__
  void asm_outl(uint32_t valueh,uint32_t valuel,int port);
  #pragma aux asm_outl="shl eax,16" "mov ax,bx" "out dx,eax" parm[ax][bx][dx];
  #define outl(value,port) asm_outl(value>>16,value&0xffff,port)
  #define inl(port) ((((uint32_t)inpw((port)+2))<<16) | ((uint32_t)inpw(port)))
 #else // Borland C 16-bit (enable the "Generate assembler source" and "Compile via assembler")
  static void outl(uint32_t data,int reg)
  {
   asm mov eax,data;
   asm mov edx,reg;
   asm out dx,eax;
  }
  static uint32_t inl(int reg)
  {
   uint32_t value;
   asm mov edx,reg;
   asm in eax,dx;
   asm mov value,eax;
   return value;
  }
 #endif
#endif

#define PCI_VENDOR_ID_CREATIVE	       0x1102UL
#define PCI_DEVICE_ID_CREATIVE_AUDIGY  0x0004UL

#define A_PTR_ADDRESS_MASK      0x0fff0000UL
#define PTR_CHANNELNUM_MASK     0x0000003fUL
#define PTR			0x00UL
#define DATA			0x04UL

#define A_SPDIF_SAMPLERATE	0x76
#define A_IOCFG			0x18
#define A_IOCFG_GPOUT0		0x0044UL

#define HCFG			0x14
#define HCFG_AUDIOENABLE	0x00000001UL
#define HCFG_LOCKSOUNDCACHE	0x00000008UL
#define HCFG_LOCKTANKCACHE_MASK	0x00000004UL
#define HCFG_MUTEBUTTONENABLE	0x00000002UL

#define HCFG_AC3ENABLE_CDSPDIF	0x00000040UL
#define HCFG_AC3ENABLE_GPSPDIF  0x00000020UL
#define HCFG_AUTOMUTE		0x00000010UL
#define HCFG_JOYENABLE      	0x00000200UL

#define A_MICROCODEBASE		0x600UL
#define A_FXGPREGBASE		0x400UL
#define A_DBG			0x53
#define A_DBG_SINGLE_STEP	0x00020000UL

#define iMAC0	 0x00	/* R = A + (X * Y >> 31)   ; saturation */
#define iACC3	 0x06	/* R = A + X + Y	   ; saturation */

#define A_FXBUS(x)	(0x00 + (x))
#define A_EXTIN(x)	(0x40 + (x))
#define A_EXTOUT(x)	(0x60 + (x))
#define A_GPR(x)	(A_FXGPREGBASE + (x))

#define FXBUS_PCM_LEFT		0x00UL
#define FXBUS_PCM_RIGHT		0x01UL
#define FXBUS_MIDI_LEFT		0x04UL
#define FXBUS_MIDI_RIGHT	0x05UL

#define A_EXTIN_SPDIF_CD_L	0x02UL	/* digital CD left */
#define A_EXTIN_SPDIF_CD_R	0x03UL	/* digital CD left */

#define A_EXTOUT_FRONT_L	0x00UL	/* digital front left */
#define A_EXTOUT_FRONT_R	0x01UL	/*               right */
#define A_EXTOUT_HEADPHONE_L	0x04UL	/* headphone audigy drive left */
#define A_EXTOUT_HEADPHONE_R	0x05UL	/*                        right */
#define A_EXTOUT_REAR_L		0x06UL	/* digital rear left */
#define A_EXTOUT_REAR_R		0x07UL	/*              right */
#define A_EXTOUT_AFRONT_L	0x08UL	/* analog front left */
#define A_EXTOUT_AFRONT_R	0x09UL	/*              right */
#define A_EXTOUT_AREAR_L	0x0eUL	/* analog rear left */
#define A_EXTOUT_AREAR_R	0x0fUL	/*             right */

#define A_WRITE_EFX(a, b, c) emu10k1_writeptr((a), A_MICROCODEBASE + (b), 0, (c))

#define A_OP(op, z, w, x, y) \
{ \
 A_WRITE_EFX(iobase, (pc) * 2, (((uint32_t)(x)) << 12) | ((uint32_t)(y))); \
 A_WRITE_EFX(iobase, (pc) * 2 + 1, (((uint32_t)(op)) << 24 ) | (((uint32_t)(z)) << 12) | ((uint32_t)(w))); \
 ++pc; \
}

static void emu10k1_writefn0(uint32_t iobase, uint32_t reg, uint32_t data)
{
 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;
  data = (data << offset) & mask;
  reg &= 0x7f;

  data |= inl(iobase + reg) & ~mask;
  outl(data, iobase + reg);
 }else{
  outl(data, iobase + reg);
 }
}

static uint32_t emu10k1_readfn0(uint32_t iobase, uint32_t reg)
{
 uint32_t val;

 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;
  reg &= 0x7f;
  val = inl(iobase + reg);
  val = (val & mask) >> offset;
 }else{
  val = inl(iobase + reg);
 }
 return val;
}

static void emu10k1_writeptr(uint32_t iobase, uint32_t reg, uint32_t channel, uint32_t data)
{
 uint32_t regptr;

 regptr = ((reg << 16) & A_PTR_ADDRESS_MASK) | (channel & PTR_CHANNELNUM_MASK);

 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;
  data = (data << offset) & mask;

  outl(regptr, iobase + PTR);
  data |= inl(iobase + DATA) & ~mask;
  outl(data, iobase + DATA);
 }else{
  outl(regptr, iobase + PTR);
  outl(data, iobase + DATA);
 }
}

static uint32_t emu10k1_readptr(uint32_t iobase, uint32_t reg, uint32_t channel)
{
 uint32_t regptr, val;

 regptr = ((reg << 16) & A_PTR_ADDRESS_MASK) | (channel & PTR_CHANNELNUM_MASK);

 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;

  outl(regptr, iobase + PTR);
  val = inl(iobase + DATA);
  val = (val & mask) >> offset;
 }else{
  outl(regptr, iobase + PTR);
  val = inl(iobase + DATA);
 }
 return val;
}

int main(void)
{
 uint32_t tmp;
 uint32_t iobase,pc,i,model;
 uint32_t irq,chiprev;
 struct pci_config_s pci_dev;

 fprintf(stdout,"Audigy12 written by PDSoft (Attila Padar, http://mpxplay.cjb.net) Nov 2003\n");
 fprintf(stdout,"Hardware config program for Audigy 2 cards using the DOS drivers of Audigy 1.\n");
 fprintf(stdout,"See readme.txt for more infos.\n\n");

 if(pcibios_FindDevice(PCI_VENDOR_ID_CREATIVE,PCI_DEVICE_ID_CREATIVE_AUDIGY,0,&pci_dev)!=PCI_SUCCESSFUL){
  fprintf(stdout,"Audigy card not found!\n");
  exit(-1);
 }

 iobase = pcibios_ReadConfig_Dword(&pci_dev, PCIR_NAMBAR)&0xfff0;
 irq    = pcibios_ReadConfig_Byte(&pci_dev, PCIR_INTR_LN);
 chiprev= pcibios_ReadConfig_Byte(&pci_dev, PCIR_RID);
 model  = pcibios_ReadConfig_Word(&pci_dev, PCIR_SSID);

 fprintf(stdout,"Audigy found on port:%4.4X irq:%d chiprev:%2.2X model:%4.4X \n",
         (int)iobase,(int)irq,(int)chiprev,(int)model);

 if(!iobase || !irq){
  fprintf(stdout,"Wrong Audigy detection!\n");
  exit(-2);
 }

 if(chiprev!=4)
  fprintf(stdout,"This is not an Audigy 2 card!\n");

 //routing
 pc=0;
 emu10k1_writeptr(iobase, A_DBG, 0, A_DBG_SINGLE_STEP); // stop fx

 for (i = 0; i < 512 ; i++)
  emu10k1_writeptr(iobase, A_FXGPREGBASE+i,0,0);  // clear gpr

 // PCM + CD-digital + MIDI
 A_OP(iACC3, A_GPR(0),A_FXBUS(FXBUS_PCM_LEFT) ,A_EXTIN(A_EXTIN_SPDIF_CD_L),A_FXBUS(FXBUS_MIDI_LEFT));
 A_OP(iACC3, A_GPR(1),A_FXBUS(FXBUS_PCM_RIGHT),A_EXTIN(A_EXTIN_SPDIF_CD_R),A_FXBUS(FXBUS_MIDI_RIGHT));

 // Analog front output
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_AFRONT_L),   0xc0, 0x0c, A_GPR(0) );
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_AFRONT_R),   0xc0, 0x0c, A_GPR(1) );

 // Digital front output
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_FRONT_L),    0xc0, 0x0c, A_GPR(0) );
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_FRONT_R),    0xc0, 0x0c, A_GPR(1) );

 // Audigy Drive, Headphone output
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_HEADPHONE_L),0xc0, 0x0c, A_GPR(0) );
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_HEADPHONE_R),0xc0, 0x0c, A_GPR(1) );

 // Analog Rear output
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_AREAR_L),    0xc0, 0xc0, A_GPR(0) );
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_AREAR_R),    0xc0, 0xc0, A_GPR(1) );

 // Digital Rear output
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_REAR_L),     0xc0, 0xc0, A_GPR(0) );
 A_OP(iACC3, A_EXTOUT(A_EXTOUT_REAR_R),     0xc0, 0xc0, A_GPR(1) );

 for( ; pc<1024; pc++)
  A_OP(0xf, 0xc0, 0xc0, 0xcf, 0xc0);

 emu10k1_writeptr(iobase, A_DBG, 0, 0); // start fx

 if(chiprev==4){
  //Setup SRCMulti_I2S SamplingRate
  tmp = emu10k1_readptr(iobase, A_SPDIF_SAMPLERATE, 0);
  tmp &= 0xfffff1ffUL;
  tmp |= (0x2UL<<9);
  emu10k1_writeptr(iobase, A_SPDIF_SAMPLERATE, 0, tmp);

  // Setup SRCSel (Enable Spdif,I2S SRCMulti)
  emu10k1_writefn0(iobase, 0x20, 0x00600000UL);
  emu10k1_writefn0(iobase, 0x24, 0x00000014UL);

  // Setup SRCMulti Input Audio Enable
  emu10k1_writefn0(iobase, 0x20, 0x006E0000UL);
  emu10k1_writefn0(iobase, 0x24, 0xFF00FF00UL);

  tmp=emu10k1_readfn0(iobase,HCFG);
  tmp|=HCFG_AC3ENABLE_CDSPDIF | HCFG_AC3ENABLE_GPSPDIF | HCFG_AUTOMUTE | HCFG_JOYENABLE | HCFG_AUDIOENABLE;
  emu10k1_writefn0(iobase,HCFG,tmp);
 }

 // Enable analog/digital outs on audigy
 tmp=emu10k1_readfn0(iobase, A_IOCFG);
 tmp&=~A_IOCFG_GPOUT0;
 if(chiprev==4)
  tmp|=0x0040UL;
 emu10k1_writefn0(iobase,A_IOCFG,tmp);

 fprintf(stdout,"Patch ready...\n");

 return 0;
}
