//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: WavPack file handling in DLL
//requires the libwavpa\libwavpa.lib and wavpack.h files

#include <in_file.h>
#include "lib_mod\modplug.h"
#include <string.h>

#define INMOD_MAX_FILESIZE       16000000

#define INMOD_DEFAULT_FLAGS      (MODPLUG_ENABLE_OVERSAMPLING|MODPLUG_ENABLE_NOISE_REDUCTION)
#define INMOD_DEFAULT_RESAMPLING MODPLUG_RESAMPLE_LINEAR

typedef struct inmod_decoder_data{
 ModPlugFile *mpf;
 void *filebuffer;
 long seek_msec;
 unsigned int bytes_per_sample;
 ModPlug_Settings settings;
}inmod_decoder_data;

static struct module_type_s{
 char *name;
 unsigned long num;
}moduletypes[]={
{"MOD",0x01},{"S3M",0x02},{"XM ",0x04},{"MED",0x08},{"MTM",0x10},
{"IT ",0x20},{"669",0x40},{"ULT",0x80},{"STM",0x100},{"FAR",0x200},
{"WAV",0x400},{"AMF",0x800},{"AMS",0x1000},{"DSM",0x2000},{"MDL",0x4000},
{"OKT",0x8000},{"DMF",0x20000},{"PTM",0x40000},{"DBM",0x80000},
{"MT2",0x100000},{"AM0",0x200000},{"PSM",0x400000},{"J2B",0x800000}};

static int inmod_assign_values(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct inmod_decoder_data *inmi,struct mpxplay_infile_info_s *miis)
{
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 int moduletype,i;

 ModPlug_GetSettings(&inmi->settings);

 adi->filechannels=adi->outchannels=inmi->settings.mChannels;
 if((adi->outchannels<PCM_MIN_CHANNELS) || (adi->outchannels>PCM_MAX_CHANNELS))
  return 0;
 adi->freq=inmi->settings.mFrequency;
 if(!adi->freq)
  return 0;
 adi->bits=inmi->settings.mBits;
 if((adi->bits<PCM_MIN_BITS) || (adi->bits>PCM_MAX_BITS))
  return 0;
 inmi->bytes_per_sample=(adi->bits+7)/8;

 inmi->settings.mFlags         =INMOD_DEFAULT_FLAGS;
 inmi->settings.mResamplingMode=INMOD_DEFAULT_RESAMPLING;
 ModPlug_SetSettings(&inmi->settings);

 miis->timemsec=ModPlug_GetLength(inmi->mpf);
 moduletype=ModPlug_GetModuleType(inmi->mpf);

 miis->longname=malloc(MPXPLAY_ADITEXTSIZE_LONGNAME+8);
 if(!miis->longname)
  return 0;
 strcpy(miis->longname,"MOD->???");
 for(i=0;i<(sizeof(moduletypes)/sizeof(struct module_type_s));i++){
  if(moduletypes[i].num==moduletype){
   strcpy(&miis->longname[sizeof("MOD->")-1],moduletypes[i].name);
   break;
  }
 }

 return 1;
}

static int INMOD_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct inmod_decoder_data *inmi=NULL;
 long filesize;

 if(!fbfs->fopen_read(fbds,filename,0))
  return MPXPLAY_ERROR_INFILE_CANTOPEN;

 miis->filesize=fbfs->filelength(fbds);
 if((miis->filesize<4) || (miis->filesize>INMOD_MAX_FILESIZE))  // ???
  goto err_out_check;

 inmi=(struct inmod_decoder_data *)calloc(1,sizeof(struct inmod_decoder_data));
 if(!inmi)
  goto err_out_check;
 miis->private_data=inmi;

 inmi->filebuffer=malloc(miis->filesize);
 if(!inmi->filebuffer)
  goto err_out_check;

 filesize=fbfs->fread(fbds,inmi->filebuffer,miis->filesize);
 if(!filesize)
  goto err_out_check;

 inmi->mpf=ModPlug_Load(inmi->filebuffer,filesize);
 if(!inmi->mpf)
  goto err_out_check;

 if(!inmod_assign_values(fbfs,fbds,inmi,miis))
  goto err_out_check;

 inmi->seek_msec=-1;

 return MPXPLAY_ERROR_INFILE_OK;

err_out_check:
 return MPXPLAY_ERROR_INFILE_CANTOPEN;
}

static void INMOD_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis)
{
 struct inmod_decoder_data *inmi=miis->private_data;
 if(inmi){
  if(inmi->mpf)
   ModPlug_Unload(inmi->mpf);
  if(inmi->filebuffer)
   free(inmi->filebuffer);
  if(miis->longname)
   free(miis->longname);
  free(inmi);
 }
 fbfs->fclose(fbds);
}

static int INMOD_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis)
{
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 struct inmod_decoder_data *inmi=miis->private_data;

 adi->pcm_samplenum=ModPlug_Read(inmi->mpf,adi->pcm_bufptr,adi->pcm_framelen)/inmi->bytes_per_sample;
 if(!adi->pcm_samplenum)
  return MPXPLAY_ERROR_INFILE_NODATA;

 return MPXPLAY_ERROR_INFILE_OK;
}

static void INMOD_infile_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis,unsigned int seektype)
{
 struct inmod_decoder_data *inmi=miis->private_data;
 if(inmi->seek_msec>=0){
  ModPlug_Seek(inmi->mpf,inmi->seek_msec);
  inmi->seek_msec=-1;
 }
}

static long INMOD_infile_fseek(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis,long newmpxframenum)
{
 struct inmod_decoder_data *inmi=miis->private_data;

 inmi->seek_msec=(long)((float)newmpxframenum*(float)miis->timemsec/(float)miis->allframes);

 return newmpxframenum;
}

static char *INMOD_infile_tag_get(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 struct inmod_decoder_data *inmi=miis->private_data;
 if(inmi && inmi->mpf){
  if(!id3ip[I3I_TITLE]){
   id3ip[I3I_TITLE]=id3p;
   strcpy(id3p,(char *)ModPlug_GetName(inmi->mpf));
   id3p+=mpxplay_textconv_funcs->all_to_char(id3p,strlen(id3p),0)+1;
  }
 }
 return id3p;
}

struct mpxplay_infile_func_s IN_MOD_funcs={
 0,
 NULL,
 NULL,
 NULL,//&INMOD_infile_open, // no autodetect, program would be too slow
 &INMOD_infile_open,
 &INMOD_infile_open,
 &INMOD_infile_close,
 &INMOD_infile_decode,
 &INMOD_infile_fseek,
 &INMOD_infile_clearbuffs,
 &INMOD_infile_tag_get,
 NULL,
 NULL,
 {"669","AMF","AMS","DBM","DMF","DSM","FAR","IT" ,"J2B","MDL","MED","MOD",
  "MT2","MTM","OKT","PSM","PTM","S3M","STM","ULT","UMX","XM" ,//"WAV",
 NULL}
};

static mpxplay_module_entry_s inmod_module_entry={
 MPXPLAY_DLLMODULETYPE_FILEIN_PARSER,
 0,
 "InMOD",
 MPXPLAY_DLLMODULEVER_FILEIN_PARSER,
 (void *)&IN_MOD_funcs
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &inmod_module_entry,
  NULL
 }
};

#ifdef __DOS__
extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 //mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin);
 return (long)(&mpxplay_dll_entry_structure);
}

#else

__declspec( dllexport ) mpxplay_dll_entry_s *mpxplay_dll_entrypoint(void)
{
 return (&mpxplay_dll_entry_structure);
}

#endif
