/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: basic shared codebook operations
 last mod: $Id: sharedbook.c,v 1.28 2002/08/31 00:00:00 PDSoft Exp $

 ********************************************************************/

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "ogg.h"
#include "os.h"
#include "misc.h"
#include "codec.h"
#include "codebook.h"
#include "scales.h"

/**** pack/unpack helpers ******************************************/
int _ilog(unsigned int v){
  int ret=0;
  while(v){
    ret++;
    v>>=1;
  }
  return(ret);
}

/* 32 bit float (not IEEE; nonnormalized mantissa +
   biased exponent) : neeeeeee eeemmmmm mmmmmmmm mmmmmmmm
   Why not IEEE?  It's just not that important here. */

#define VQ_FEXP 10
#define VQ_FMAN 21
#define VQ_FEXP_BIAS 768 /* bias toward values smaller than 1. */

/* doesn't currently guard under/overflow */
long _float32_pack(float val){
  int sign=0;
  long exp;
  long mant;
  if(val<0){
    sign=0x80000000;
    val= -val;
  }
  exp= floor(log(val)/log(2.f));
  mant=rint(ldexp(val,(VQ_FMAN-1)-exp));
  exp=(exp+VQ_FEXP_BIAS)<<VQ_FMAN;

  return(sign|exp|mant);
}

float _float32_unpack(long val){
  double mant=val&0x1fffff;
  int    sign=val&0x80000000;
  long   exp =(val&0x7fe00000L)>>VQ_FMAN;
  if(sign)mant= -mant;
  return(ldexp(mant,exp-(VQ_FMAN-1)-VQ_FEXP_BIAS));
}

static ogg_uint32_t *_make_words(long *l,long n,long used_entries)
{
 long i,j,ecounter=0;
 ogg_uint32_t marker[33];
 ogg_uint32_t *r=_ogg_malloc(((used_entries)? used_entries:n)*sizeof(*r));
 memset(marker,0,sizeof(marker));

 for(i=0;i<n;i++){
  long length=l[i];
  if(length>0){
   long entry=marker[length],temp;

   if(length<32 && (entry>>length)){
    _ogg_free(r);
    return(NULL);
   }

   temp=0;
   for(j=0;j<length;j++){
    temp<<=1;
    temp|=(entry>>j)&1;
   }
   r[ecounter]=temp;

   for(j=length;j>0;j--){
    if(marker[j]&1){
     if(j==1)
      marker[1]++;
     else
      marker[j]=marker[j-1]<<1;
     break;
    }
    marker[j]++;
   }

   for(j=length+1;j<33;j++){
    if((marker[j]>>1) == entry){
     entry=marker[j];
     marker[j]=marker[j-1]<<1;
    }else
     break;
   }

   ecounter++;
  }else{
   if(!used_entries)
    ecounter++;
  }
 }

 return(r);
}

//#define SHARBOOK_DEBUG 1

#ifdef SHARBOOK_DEBUG
 #include <stdio.h>
 static unsigned int taberror,allerror,atn;
#endif

static decode_aux *_make_decode_tree(codebook *c,const static_codebook *s)
{
 long top,ptrsize,i,j,n,ccount;
 ogg_uint32_t *codelist;
 short *ptr0,*ptr1;
 decode_aux *t=_ogg_calloc(1,sizeof(*t));

 if(!t)
  return (NULL);

#ifdef SHARBOOK_DEBUG
 top=0;
#endif

 if(c->used_entries)
  c->entries=c->used_entries;
 else
  c->entries=2;

 ptrsize=(c->entries+1)&(~1);
 ptr0=t->ptr0=_ogg_calloc(ptrsize,sizeof(*(t->ptr0)));
 ptr1=t->ptr1=_ogg_calloc(ptrsize,sizeof(*(t->ptr1)));
 if(!ptr0 || !ptr1){
  codelist=NULL;
  goto tree_err_out;
 }

 if(c->used_entries){
  codelist=_make_words(s->lengthlist,s->entries,c->used_entries);

  if(!codelist)
   goto tree_err_out;

  ccount=0;
  top=0;
  for(i=0;i<s->entries;i++){
   if(s->lengthlist[i]>0){
    long ptr=0;
    for(j=0;j<s->lengthlist[i]-1;j++){
     int bit=(codelist[ccount]>>j)&1;
     if(!bit){
      if(!ptr0[ptr])
       ptr0[ptr]= ++top;
      ptr=ptr0[ptr];
     }else{
      if(!ptr1[ptr])
       ptr1[ptr]= ++top;
      ptr=ptr1[ptr];
     }
     if(top>=ptrsize) // I hope this never happens
      goto tree_err_out;
    }
    if(!((codelist[ccount]>>j)&1))
     ptr0[ptr]=-i;
    else
     ptr1[ptr]=-i;
    ccount++;
   }
  }
  _ogg_free(codelist);
  codelist=NULL;

  t->tab_maxlen = _ilog(s->entries);

  if(t->tab_maxlen<c->maxbits){
   if(c->maxbits<8)
    t->tab_maxlen=c->maxbits;
  }else{
   if(t->tab_maxlen>c->maxbits)
    if(c->maxbits>2)
     t->tab_maxlen=c->maxbits;
    else
     t->tab_maxlen=2;
  }

  if(t->tab_maxlen>14)
   t->tab_maxlen=14;
 }else
  t->tab_maxlen=1;

 n = 1<<t->tab_maxlen;
 t->tab_ptr    = _ogg_calloc((n+1)&(~1),sizeof(*t->tab_ptr));
 t->tab_codelen= _ogg_calloc((n+3)&(~3),sizeof(*t->tab_codelen));

 if(!t->tab_ptr || !t->tab_codelen)
  goto tree_err_out;

#ifdef SHARBOOK_DEBUG
  atn+=n;
  taberror=0;
#endif

 for (i = 0; i < n; i++) {
  long p=0;
  for (j = 0; j < t->tab_maxlen && (p > 0 || j == 0); j++) {
   if(i & (1 << j))
    p = ptr1[p];
   else
    p = ptr0[p];
  }

  t->tab_ptr[i] = p;
  t->tab_codelen[i] = j;

#ifdef SHARBOOK_DEBUG
  if(p>0)
   taberror++;
#endif
 }

#ifdef SHARBOOK_DEBUG
 allerror+=taberror;
 fprintf(stdout,"ce:%4d ue:%4d d:%2d tn:%2d ml:%2d atn:%5d err:%4d allerr:%5d top:%d\n",
   s->entries,c->used_entries,s->dim,t->tab_maxlen,c->maxbits,atn,taberror,allerror,top);
 taberror=0;
#endif

 return(t);

tree_err_out:
 if(codelist) _ogg_free(codelist);
 if(t->ptr0)  _ogg_free(t->ptr0);
 if(t->ptr1)  _ogg_free(t->ptr1);
 if(t->tab_ptr)    _ogg_free(t->tab_ptr);
 if(t->tab_codelen) _ogg_free(t->tab_codelen);
 _ogg_free(t);
 return (NULL);
}

long _book_maptype1_quantvals(const static_codebook *s)
{
 long vals=floor(pow((float)s->entries,1.f/s->dim));

 while(1){
  long acc=1;
  long acc1=1;
  int i;
  for(i=0;i<s->dim;i++){
   acc*=vals;
   acc1*=vals+1;
  }
  if(acc<=s->entries && acc1>s->entries){
   return(vals);
  }else{
   if(acc>s->entries){
    vals--;
   }else{
    vals++;
   }
  }
 }
}

static unsigned int _book_unquantize(codebook *c,const static_codebook *s)
{
 long j,k;
 if(s->maptype==1 || s->maptype==2){
  int quantvals;
  float mindel=_float32_unpack(s->q_min);
  float delta=_float32_unpack(s->q_delta);
  float *r=_ogg_calloc(s->entries*s->dim,sizeof(*r));

  if(!r)
   return 0;
  c->valuelist=r;

  switch(s->maptype){
    case 1:
      quantvals=_book_maptype1_quantvals(s);
      for(j=0;j<s->entries;j++){
	float last=0.f;
	int indexdiv=1;
	for(k=0;k<s->dim;k++){
	  int index= (j/indexdiv)%quantvals;
	  float val=s->quantlist[index];
	  val=fabs(val)*delta+mindel+last;
	  if(s->q_sequencep)
	   last=val;
	  *r++=val;
	  indexdiv*=quantvals;
	}
      }
      break;
    case 2:
      for(j=0;j<s->entries;j++){
	float last=0.f;
	for(k=0;k<s->dim;k++){
	  float val=s->quantlist[j*s->dim+k];
	  val=fabs(val)*delta+mindel+last;
	  if(s->q_sequencep)
	   last=val;
	  *r++=val;
	}
      }
      break;
  }
 }
 return 1;
}

void vorbis_staticbook_clear(static_codebook *b){
  if(b->allocedp){
    if(b->quantlist)_ogg_free(b->quantlist);
    if(b->lengthlist)_ogg_free(b->lengthlist);
    if(b->nearest_tree){
      _ogg_free(b->nearest_tree->ptr0);
      _ogg_free(b->nearest_tree->ptr1);
      _ogg_free(b->nearest_tree->p);
      _ogg_free(b->nearest_tree->q);
      memset(b->nearest_tree,0,sizeof(*b->nearest_tree));
      _ogg_free(b->nearest_tree);
    }
    if(b->thresh_tree){
      _ogg_free(b->thresh_tree->quantthresh);
      _ogg_free(b->thresh_tree->quantmap);
      memset(b->thresh_tree,0,sizeof(*b->thresh_tree));
      _ogg_free(b->thresh_tree);
    }

    memset(b,0,sizeof(*b));
  }
}

void vorbis_staticbook_destroy(static_codebook *b){
  if(b->allocedp){
    vorbis_staticbook_clear(b);
    _ogg_free(b);
  }
}

void vorbis_book_clear(codebook *c)
{
  if(c->decode_tree){
   struct decode_aux *t=c->decode_tree;
   if(t->tab_ptr)     _ogg_free(t->tab_ptr);
   if(t->tab_codelen) _ogg_free(t->tab_codelen);
   if(t->ptr0) _ogg_free(t->ptr0);
   if(t->ptr1) _ogg_free(t->ptr1);
   _ogg_free(c->decode_tree);
  }
  if(c->valuelist)_ogg_free(c->valuelist);
  if(c->codelist)_ogg_free(c->codelist);

  memset(c,0,sizeof(*c));
}

int vorbis_book_init_encode(codebook *c,const static_codebook *s){

  memset(c,0,sizeof(*c));
  c->c=s;
  c->entries=s->entries;
  c->used_entries=s->entries;
  c->dim=s->dim;
  c->codelist=_make_words(s->lengthlist,s->entries,0);
  _book_unquantize(c,s);

  return(0);
}

int vorbis_book_init_decode(codebook *c,const static_codebook *s)
{
 unsigned int i;

 memset(c,0,sizeof(*c));
 c->dim=s->dim;

 for(i=0;i<s->entries;i++){
  if(s->lengthlist[i]){
   if(s->lengthlist[i]>c->maxbits)
    c->maxbits=s->lengthlist[i];
   c->used_entries++;
  }
 }

 if(c->used_entries>CODEBOOK_MAX_ENTRIES)
  goto err_out;

 if(c->used_entries){
  c->decode_tree=_make_decode_tree(c,s);
  if(c->decode_tree==NULL)
   goto err_out;
  if(!_book_unquantize(c,s))
   goto err_out;
 }

 return(0);
err_out:
 vorbis_book_clear(c);
 return(-1);
}
